/**
* @license Apache-2.0
*
* Copyright (c) 2019 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var proxyquire = require( 'proxyquire' );
var randu = require( '@stdlib/random/iter/randu' );
var array2iterator = require( '@stdlib/array/to-iterator' );
var iteratorSymbol = require( '@stdlib/symbol/iterator' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var itermsumabs2 = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof itermsumabs2, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if provided an iterator argument which is not an iterator protocol-compliant object', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			itermsumabs2( value, 3 );
		};
	}
});

tape( 'the function throws an error if provided a second argument which is not a positive integer', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		-5,
		0,
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			itermsumabs2( randu(), value );
		};
	}
});

tape( 'the function returns an iterator protocol-compliant object which iteratively computes a moving sum of squared absolute values', function test( t ) {
	var expected;
	var actual;
	var values;
	var it;
	var v;
	var i;

	values = [ 2.0, -3.0, 2.0, -4.0, -3.0, 4.0 ];
	expected = [ 4.0, 13.0, 17.0, 29.0, 29.0, 41.0 ];

	it = itermsumabs2( array2iterator( values ), 3 );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < values.length; i++ ) {
		v = it.next();
		t.equal( typeof v.value, 'number', 'returns a number' );
		t.equal( typeof v.done, 'boolean', 'returns a boolean' );
		actual.push( v.value );
	}
	t.deepEqual( actual, expected, 'returns expected results' );

	v = it.next();
	t.equal( v.value, void 0, 'returns expected value' );
	t.equal( v.done, true, 'returns expected value' );

	t.end();
});

tape( 'the function returns an iterator protocol-compliant object which iteratively computes a moving sum of squared absolute values (value+done)', function test( t ) {
	var expected;
	var actual;
	var values;
	var it;
	var v;
	var i;

	values = [ 2.0, -3.0, 2.0, -4.0, -3.0, 4.0 ];
	expected = [ 4.0, 13.0, 17.0, 29.0, 29.0, 41.0 ];

	it = itermsumabs2( createIterator( values ), 3 );
	t.equal( it.next.length, 0, 'has zero arity' );

	actual = [];
	for ( i = 0; i < values.length; i++ ) {
		v = it.next();
		t.equal( typeof v.value, 'number', 'returns a number' );
		t.equal( typeof v.done, 'boolean', 'returns a boolean' );
		actual.push( v.value );
	}
	t.deepEqual( actual, expected, 'returns expected values' );

	v = it.next();
	t.equal( v.value, void 0, 'returns expected value' );
	t.equal( v.done, true, 'returns expected value' );

	t.end();

	function createIterator( arr ) {
		var len;
		var it;
		var i;

		len = arr.length;
		i = -1;

		it = {};
		it.next = next;

		return it;

		function next() {
			var out;
			i += 1;
			if ( i < len ) {
				out = {};
				out.value = arr[ i ];
				out.done = ( i === len-1 );
				return out;
			}
			return {
				'done': true
			};
		}
	}
});

tape( 'if an iterated value is a non-numeric value, the computed sum is `NaN` for at least `W` invocations', function test( t ) {
	var expected;
	var values;
	var it;
	var v;
	var i;

	values = [
		NaN, // NaN
		2.0, // NaN, 2.0
		2.0, // NaN, 2.0, 2.0
		'beep',  // 2.0, 2.0, 'beep'
		2.0, // 2.0, 'beep', 2.0
		2.0, // 'beep', 2.0, 2.0
		2.0, // 2.0, 2.0, 2.0
		NaN, // 2.0, 2.0, NaN
		2.0, // 2.0, NaN, 2.0
		2.0, // NaN, 2.0, 2.0
		2.0, // 2.0, 2.0, 2.0
		NaN, // 2.0, 2.0, NaN
		2.0, // 2.0, NaN, 2.0
		2.0, // NaN, 2.0, 2.0
		NaN, // 2.0, 2.0, NaN
		NaN, // 2.0, NaN, NaN
		NaN, // NaN, NaN, NaN
		NaN, // NaN, NaN, NaN
		2.0  // NaN, NaN, 2.0
	];

	expected = [
		NaN,
		NaN,
		NaN,
		NaN,
		NaN,
		NaN,
		12.0,
		NaN,
		NaN,
		NaN,
		12.0,
		NaN,
		NaN,
		NaN,
		NaN,
		NaN,
		NaN,
		NaN,
		NaN
	];

	it = itermsumabs2( array2iterator( values ), 3 );

	for ( i = 0; i < values.length; i++ ) {
		v = it.next();
		if ( isnan( expected[ i ] ) ) {
			t.equal( isnan( v.value ), true, 'returns expected value for window '+i );
		} else {
			t.equal( v.value, expected[ i ], 'returns expected value for window '+i );
		}
	}
	t.end();
});

tape( 'the returned iterator has a `return` method for closing an iterator (no argument)', function test( t ) {
	var it;
	var r;

	it = itermsumabs2( randu(), 3 );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.return();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	r = it.next();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	t.end();
});

tape( 'the returned iterator has a `return` method for closing an iterator (argument)', function test( t ) {
	var it;
	var r;

	it = itermsumabs2( randu(), 3 );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.next();
	t.equal( typeof r.value, 'number', 'returns a number' );
	t.equal( r.done, false, 'returns expected value' );

	r = it.return( 'finished' );
	t.equal( r.value, 'finished', 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	r = it.next();
	t.equal( r.value, void 0, 'returns expected value' );
	t.equal( r.done, true, 'returns expected value' );

	t.end();
});

tape( 'if an environment supports `Symbol.iterator` and the provided iterator is iterable, the returned iterator is iterable', function test( t ) {
	var itermsumabs2;
	var opts;
	var rand;
	var it1;
	var it2;
	var i;

	itermsumabs2 = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': '__ITERATOR_SYMBOL__'
	});

	opts = {
		'seed': 12345
	};
	rand = randu( opts );
	rand[ '__ITERATOR_SYMBOL__' ] = factory;

	it1 = itermsumabs2( rand, 3 );
	t.equal( typeof it1[ '__ITERATOR_SYMBOL__' ], 'function', 'has method' );
	t.equal( it1[ '__ITERATOR_SYMBOL__' ].length, 0, 'has zero arity' );

	it2 = it1[ '__ITERATOR_SYMBOL__' ]();
	t.equal( typeof it2, 'object', 'returns an object' );
	t.equal( typeof it2.next, 'function', 'has method' );
	t.equal( typeof it2.return, 'function', 'has method' );

	for ( i = 0; i < 100; i++ ) {
		t.equal( it2.next().value, it1.next().value, 'returns expected value' );
	}
	t.end();

	function factory() {
		return randu( opts );
	}
});

tape( 'if an environment does not support `Symbol.iterator`, the returned iterator is not "iterable"', function test( t ) {
	var itermsumabs2;
	var it;

	itermsumabs2 = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': false
	});

	it = itermsumabs2( randu(), 3 );
	t.equal( it[ iteratorSymbol ], void 0, 'does not have property' );

	t.end();
});

tape( 'if a provided iterator is not iterable, the returned iterator is not iterable', function test( t ) {
	var itermsumabs2;
	var rand;
	var it;

	itermsumabs2 = proxyquire( './../lib/main.js', {
		'@stdlib/symbol/iterator': '__ITERATOR_SYMBOL__'
	});

	rand = randu();
	rand[ '__ITERATOR_SYMBOL__' ] = null;

	it = itermsumabs2( rand, 3 );
	t.equal( it[ iteratorSymbol ], void 0, 'does not have property' );

	t.end();
});
