import subprocess
import sys
import shutil
from typing import Optional


class TerminalController:
    """
    Provides methods to manage terminal operations.
    """
    @staticmethod
    def find_terminal_emulator() -> Optional[str]:
        """
        Find a suitable terminal emulator executable.
        """
        linux_terminals = [
            'gnome-terminal',
            'x-terminal-emulator',
            'konsole',
            'xfce4-terminal',
            'lxterminal',
            'mate-terminal'
        ]

        for terminal in linux_terminals:
            terminal_path = shutil.which(terminal)
            if terminal_path:
                return terminal_path
        return None

    @staticmethod
    def is_supported_platform() -> bool:
        """
        Check if the current platform is supported.
        """
        return any(sys.platform.startswith(p) for p in ['win', 'darwin', 'linux'])

    @staticmethod
    def run(command: str) -> None:
        """
        Run a command in the appropriate terminal based on the operating system.
        """
        if not TerminalController.is_supported_platform():
            print("Platform not supported")
            return

        if sys.platform.startswith('win'):
            subprocess.Popen(['start', 'cmd', '/k', command], shell=True)
        elif sys.platform.startswith('darwin'):
            script = f'''
                tell application "Terminal"
                    do script "{command}"
                end tell
            '''
            subprocess.Popen(['osascript', '-e', script])
        elif sys.platform.startswith('linux'):
            terminal_emulator = TerminalController.find_terminal_emulator()
            if terminal_emulator:
                subprocess.Popen(
                    [terminal_emulator, '-e', f'bash -c "{command}; exec $SHELL"'])
            else:
                print("No terminal emulator found.") 