use ui;
use core:geometry;
use progvis:net;
use graphics;

SolidBrush tableText on Render = SolidBrush(white * 0.1);
SolidBrush tableFg on Render = SolidBrush(white * 0.4);
SolidBrush tableBg on Render = SolidBrush(white);
SolidBrush tableActive on Render = SolidBrush(blue + white * 0.8);
Duration panelAnimation on Render = 150 ms;
Float panelMargin on Render = 50;
Float tableHSpace on Render = 10;
Float tableVSpace on Render = 4;

class Table on Render {
	private Row[] rows;
	private Float[] minColSize;
	private Size lastSize;

	init(View view) {
		init {}

		Font f = defaultFont;
		{
			Font bold = f.clone;
			bold.weight = 500;

			Heading heading(view.labels, bold);
			rows << heading;
			heading.layout(minColSize);
		}

		for (r in view.rows) {
			Body row(view, r, f);
			rows << row;
			row.layout(minColSize);
		}
	}

	Float height() {
		Float h;
		for (row in rows)
			h += row.height;
		h;
	}

	// "draw" is what part of the table is currently visible (0, 0) is the top. We use this for
	// clipping, we don't transform anything.
	Bool render(Rect draw, Graphics g) {
		lastSize = draw.size;

		if (minColSize.empty)
			return false;

		Float width = draw.size.w;

		// Expand the first column.
		Float cols;
		for (x in minColSize)
			cols += x;
		if (minColSize.count > 1)
			minColSize[1] += width - cols;
		else
			minColSize[0] += width - cols;

		Bool repaint = false;
		Point current;
		for (row in rows) {
			Float h = row.height;
			Rect c(current, Size(width, h));
			if (c.intersects(draw))
				repaint |= row.render(c.p0, minColSize, width, g);

			current.y += h;
		}

		repaint;
	}

	// Clicked. "painter" is needed for delayed repaints.
	void onClick(Point pt, Painter painter) {
		Bool found = false;
		Point current;
		for (row in rows) {
			Float h = row.height;
			if (!found & (current.y + h > pt.y)) {
				row.onClick(pt - current, lastSize.w, painter);
				found = true;
			} else {
				row.deactivate();
			}

			current.y += h;
		}
	}
}

class Row on Render {
	private Text[] cols;
	private Float colHeight;

	// This is "just" the background color of the row.
	protected Bool active;

	init(Str[] text, Font f) {
		init {}

		for (x in text)
			cols << Text(x, f);

		for (c in cols)
			colHeight = colHeight.max(c.size.h);
		colHeight = (colHeight + 0.5).int.float; // Round to int.
	}

	void layout(Float[] w) {
		while (w.count < cols.count)
			w << 0;

		for (i, c in cols)
			w[i] = w[i].max(c.size.w + tableHSpace);
	}

	Float height() {
		baseHeight();
	}

	protected Float baseHeight() {
		colHeight + tableVSpace;
	}

	Bool render(Point offset, Float[] colWidth, Float totalWidth, Graphics g) {
		Point at = offset;
		Float h = baseHeight;
		Rect border(offset, Size(totalWidth, h));
		g.fill(border, if (active) tableActive; else tableBg;);
		g.draw(border, tableFg);

		Float dx = tableHSpace * 0.5;
		Float dy = tableVSpace * 0.5;
		at.y += dy;
		at.x += dx;
		for (i, c in cols) {
			if (i > 0) {
				g.line(Point(at.x - dx, offset.y), Point(at.x - dx, offset.y + h), tableFg);
			}

			g.push(Rect(at, Size(colWidth[i], h)));
			g.draw(c, tableText, at);
			g.pop();

			at.x += colWidth[i];
		}

		false;
	}

	void onClick(Point pt, Float width, Painter p) {}

	void deactivate() {
		active = false;
	}
}

class Heading extends Row {
	init(Str[] labels, Font f) {
		init(labels, f) {}
	}
}

class Body extends Row {
	private View view;
	private View:Row src;

	// Panel to show, if any.
	private Panel? panel;

	// Animation position for the panel (float, 0-1).
	private Float animationPos;

	// Start time of the animation for the panel.
	private Moment? animationStart;

	init(View view, View:Row p, Font f) {
		init(p.data, f) {
			view = view;
			src = p;
		}
	}

	Float height() : override {
		Float base = baseHeight();
		if (panel)
			base += panel.height * animationPos;
		base;
	}

	void onClick(Point pt, Float width, Painter p) : override {
		if (!active) {
			active = true;
			(spawn findPanel(p)).detach();
		} else if (panel) {
			// Click the panel?
			Float base = baseHeight();
			if (pt.y >= base) {
				if (pt.x >= panelMargin & pt.x < width - panelMargin) {
					pt.y -= base;
					pt.x -= panelMargin;
					panel.onClick(pt, width - 2*panelMargin, p);
				}
			} else {
				deactivate();
			}
		}
	}

	Bool render(Point offset, Float[] colWidth, Float totalWidth, Graphics g) : override {
		Bool repaint = false;

		// Update our animation (if any).
		// Note: Doing this here means that we will have a small amount of overdraw during
		// animations (the Table checks our heights before drawing us). That is fine.
		if (animationStart) {
			Float pos = (Moment() - animationStart) / panelAnimation;
			if (active) {
				animationPos = pos;
				if (animationPos > 1.0) {
					animationPos = 1.0;
					this.animationStart = null;
				}
			} else {
				animationPos = 1.0 - pos;
				if (animationPos <= 0.0) {
					animationPos = 0.0;
					this.animationStart = null;
					this.panel = null;
				}
			}
			repaint = true;
		}

		// Draw everything!
		repaint |= super:render(offset, colWidth, totalWidth, g);

		// Draw the panel, if any.
		if (panel) {
			// If we don't have this, text rendering behaves strange.
			if (animationPos > 0.01) {
				offset.y += baseHeight();
				offset.x += panelMargin;
				Float width = totalWidth - 2*panelMargin;

				g.push();
				g.transform = scale(Size(1.0, animationPos), offset);

				repaint |= panel.render(offset, width, g);

				g.pop();
			}
		}

		repaint;
	}

	void deactivate() : override {
		super:deactivate();
		clearPanel();
	}

	void setPanel(Panel panel) {
		if (!active)
			return;

		this.panel = panel;
		animationStart = Moment();
		animationPos = 0.0;
	}

	private void findPanel(Painter p) {
		setPanel(view.details(src.id));
		// Force repaint!
		p.repaint();
	}

	void clearPanel() {
		active = false;
		animationStart = Moment();
		animationPos = 1.0;
	}
}
