/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * audio_file.c - Stuff to read audio samples from a sound file
 *
 * Implemented using libsndfile which can read wav, ogg, flac and maybe mp3
 * using libmpg123 when libsndfile can't read MP3 or --with-mpg123
 * and libav which works badly on most audio files but can do M4A and videos.
 */

#include "spettro.h"
#include "a_file.h"		/* Our header file */

#if HAVE_LIBSNDFILE
#include "snd_file.h"
#endif
#if HAVE_LIBMPG123
#include "mpg_123.h"
#endif
#if HAVE_LIBAV
#include "libav.h"
#endif
#if HAVE_LIBSOX
#include "lib_sox.h"
#endif

#if EMOTION_AUDIO
#include <Emotion.h>	/* for emotion_object_play_length_get() */
extern Evas_Object *em;
#endif

#include "convert.h"
#include "lock.h"
#include "ui.h"			/* for disp_time, disp_offset and secpp */

static audio_file_t *audio_file = NULL;
freq_t sr;	/* Current sample rate */

audio_file_t *
current_audio_file(void)
{
    return audio_file;
}

/* Open the audio file to find out sampling rate, length and to be able
 * to fetch pixel data to be converted into spectra.
 *
 * Emotion seems not to let us get the raw sample data or sampling rate
 * and doesn't know the file length until the "open_done" event arrives
 * so we use libsndfile for that.
 */
audio_file_t *
open_audio_file(char *filename)
{
    audio_file_t *af = Malloc(sizeof(*af));

#if HAVE_LIBSNDFILE
    af->sndfile = NULL;
#endif
#if HAVE_LIBMPG123
    af->mh = NULL;
#endif

    /* Prefer libmpg123 for MP3, libsndfile for everything else and
     * libav if all else fails (.m4a and video files are most common).
     */
#if HAVE_LIBMPG123
    if (libmpg123_open(af, filename)) {
	af->opened_with = libmpg123;
    } else
#endif
#if HAVE_LIBSNDFILE
    if (libsndfile_open(af, filename)) {
	af->opened_with = libsndfile;
    } else
#endif
#if HAVE_LIBSOX
    if (libsox_open(af, filename)) {
	af->opened_with = libsox;
    } else
#endif
#if HAVE_LIBAV
    if (libav_open(af, filename)) {
	af->opened_with = libav;
    } else
#endif
    {   /* No sound libraries are available, or all failed to open the file */
	free(af);
	af = NULL;
    }
    audio_file = af;
    if (af) sr = af->sample_rate;
    return af;
}

/* Return the length of an audio file in seconds. */
secs_t
audio_file_length(void)
{
#if EMOTION_AUDIO
    double length = emotion_object_play_length_get(em);
    /* "Returns 0 if called before the "length_change" signal has been emitted."
     */
    if (length > 0.0) {
	frames_t frames = length * audio_file->sample_rate;
	if (frames != audio_file->frames) {
	    fprintf(stderr, "Revising file length from %ld to %ld frames\n",
		    audio_file->frames, frames);
	    audio_file->frames = frames;
	}
	return length;
    }
#endif
    return audio_file == NULL ? 0 :
	   (secs_t)audio_file->frames / audio_file->sample_rate;
}

/*
 * read_audio_file(): Read sample frames from the audio file,
 * returning them as mono floats for the graphics or with the
 * original number of channels as 16-bit system-native bytendianness
 * for the sound player.
 *
 * "data" is where to put the audio data.
 * "format" is one of af_float or af_signed
 * "channels" is the number of desired channels, 1 to monoise or copied from
 *		the WAV file to play as-is.
 * "start" is the index of the first sample frame to read.
 *	It may be negative if we are reading data for FFT transformation,
 *	in which case we invent some 0 data for the leading silence.
 * "frames_to_read" is the number of multi-sample frames to fill "data" with.
 *
 * The return value is the number of sample frames read,
 * 0 if we are already at end-of-file or
 * a negative value if some kind of read error occurred.
 *
 * To prevent locking the audio player out when we read a big area,
 * read the audio in blocks so that the audio player can get in between
 * reads. This eliminates almost all the audio player dropouts at
 * very low X zoom.
 */

frames_t
read_audio_file(audio_file_t *af, char *data,
		af_format_t format, int channels,
		frames_t start, frames_t frames_to_read)
{
    /* size of one frame of output data in bytes */
    int framesize = (format == af_float ? sizeof(float) : sizeof(short))
		    * channels;
    int frames_written = 0;	/* How many frames have we filled? */
    char *write_to = data;	/* Where to write next data */

    /* Is the whole area before time 0 or after end-of-file? */
    if (start + frames_to_read <= 0 || start >= af->frames) {
	/* All silence */
	bzero(write_to, frames_to_read * framesize);
	return frames_to_read;
    }
    /* Is it partly before time 0? */
    if (start < 0) {
	/* Fill before time 0.0 with silence */
	frames_t silence = -start;	/* How many silent frames to fill */

	bzero(write_to, silence * framesize);
	write_to += silence * framesize;
	frames_written += silence;
	frames_to_read -= silence;
	start = 0;	/* Read audio data from start of file */
    }

    /* Does it overlap the end of the file? */
    if (start + frames_to_read > af->frames) {
	/* How many frames to fill with silence */
	frames_t silence = start + frames_to_read - af->frames;

        bzero(write_to + (frames_to_read - silence) * framesize,
	      silence * framesize);
	frames_written += silence;
	frames_to_read -= silence;
    }

    switch (af->opened_with) {
#if HAVE_LIBAV
    case libav:
	while (frames_to_read > 0) {
	    frames_t frames_read;

	    lock_audio_file();
	    frames_read = libav_read_frames(af, write_to, start,
					    frames_to_read, format);
	    unlock_audio_file();

	    if (frames_read < 0) {
		fprintf(stderr, "Failed to read %ld frames at %ld from AV file\n",
			frames_to_read, start);
		return -1;
	    }
	    if (frames_read > 0) {
		write_to += frames_read * framesize;
		frames_written += frames_read;
		frames_to_read -= frames_read;
	    } else {  /* frames_read == 0 */
		fprintf(stderr, "libav_read_frames returned 0 for %ld frames at %ld\n",
			frames_to_read, start);
		break;
	    }
	}
	break;
#endif

#if HAVE_LIBMPG123
    case libmpg123:
	if (libmpg123_seek(af, start) == FALSE) {
	    fprintf(stderr, "Failed to seek to frame %ld in MPEG file.\n",
		    start);
	    return -1;
	}
	while (frames_to_read > 0) {
	    frames_t frames_read;

	    lock_audio_file();
	    frames_read = libmpg123_read_frames(af, write_to, frames_to_read,
						format);
	    unlock_audio_file();

	    if (frames_read < 0) return -1;
	    if (frames_read > 0) {
		write_to += frames_read * framesize;
		frames_written += frames_read;
		frames_to_read -= frames_read;
	    } else {  /* frames_read == 0 */
		/* We ask it to read past EOF so failure is normal */
		break;
	    }
	}
	break;
#endif

#if HAVE_LIBSNDFILE
    case libsndfile:
	if (!libsndfile_seek(af, start)) {
	    fprintf(stderr, "Failed to seek to frame %ld in audio file.\n",
		start);
	    return -1;
	}

	while (frames_to_read > 0) {
	    frames_t frames_read;

	    lock_audio_file();
	    frames_read = libsndfile_read_frames(af, write_to, frames_to_read,
						 format);
	    unlock_audio_file();

	    if (frames_read < 0) return -1;
	    if (frames_read > 0) {
		frames_written += frames_read;
		write_to += frames_read * framesize;
		frames_to_read -= frames_read;
	    } else {  /* frames_read == 0 */
		/* We ask it to read past EOF so failure is normal */
		break;
	    }
	}
	break;
#endif

#if HAVE_LIBSOX
    case libsox:
	if (!libsox_seek(af, start)) {
	    fprintf(stderr, "Failed to seek to frame %ld in audio file.\n",
		start);
	    return -1;
	}

	while (frames_to_read > 0) {
	    frames_t frames_read;

	    lock_audio_file();
	    frames_read = libsox_read_frames(af, write_to, frames_to_read,
					     format);
	    unlock_audio_file();

	    if (frames_read < 0) return -1;
	    if (frames_read > 0) {
		frames_written += frames_read;
		write_to += frames_read * framesize;
		frames_to_read -= frames_read;
	    } else {  /* frames_read == 0 */
		/* We ask it to read past EOF so failure is normal */
		break;
	    }
	}
	break;
#endif

    default:
	fprintf(stderr, "Internal error: What was %s opened with?\n",
		af->filename);
	exit(1);
    }

    /* If it stopped before reading all frames, fill the rest with silence */
    if (frames_to_read > 0) {
        bzero(write_to, frames_to_read * framesize);
	frames_written += frames_to_read;
	frames_to_read = 0;
    }

    return frames_written;
}

void
close_audio_file(audio_file_t *af)
{
    if (af == NULL) return;

    switch (af->opened_with) {
#if HAVE_LIBAV
    case libav:
	libav_close(af);
	break;
#endif
#if HAVE_LIBSNDFILE
    case libsndfile:
	libsndfile_close(af);
	break;
#endif
#if HAVE_LIBMPG123
    case libmpg123:
	libmpg123_close(af);
	break;
#endif
    default:
	/* Who cares, we're only closing it */
	break;
    }

    free(af);
}

/* Create an audio file ready for a left-to-right-bar-marker audio dump. */
/* Returns: TRUE if it succeeded, FALSE if it failed. */
bool
dump_to_audio_file(char *filename, frames_t start, frames_t nframes)
{
#if !HAVE_LIBSNDFILE
    fprintf(stderr, "Dumping to an audio file needs libsndfile\n");
    return FALSE;
#else
    SNDFILE *outfile;
    SF_INFO sfinfo;
    short *buffer;
    sf_count_t status;

    buffer = malloc(nframes * audio_file->channels * sizeof(short));
    if (buffer == NULL) {
	fprintf(stderr, "Not enough memory to dump %g seconds of audio\n",
		nframes / audio_file->sample_rate);
	return FALSE;
    }

    status = read_audio_file(audio_file, (char *)buffer,
			     af_signed, audio_file->channels,
			     start, nframes);
    if (status != nframes) {
	fprintf(stderr, "Short read (%ld out of %ld) while dumping audio file.\n",
		(long)status, (long)nframes);
	free(buffer);
	return FALSE;
    }

    bzero(&sfinfo, sizeof (sfinfo)) ;
    sfinfo.samplerate	= audio_file->sample_rate;
    sfinfo.frames	= nframes;
    sfinfo.channels	= audio_file->channels;
    sfinfo.format	= (SF_FORMAT_WAV | SF_FORMAT_PCM_16);

    if (!(outfile = sf_open(filename, SFM_WRITE, &sfinfo))) {
	fprintf(stderr, "Can't create output sound file\n");
	free(buffer);
	return FALSE;
    }

    /* If the left or right bar line is far off-screen, the required audio
     * may not be in the cache, so just read it all from the audio file.
     */
    status = sf_writef_short(outfile, buffer, (sf_count_t) nframes);
    if (status != nframes) {
	fprintf(stderr, "Short write to audio file of %ld instead of %ld frames\n",
		(long) status, (long)nframes);

	sf_close(outfile);
	free(buffer);
	return FALSE;
    }

    sf_close(outfile);
    free(buffer);

    printf("Dumped the audio selection to '%s'\n", filename);

    return TRUE;
#endif
}
