package prefuse.util.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import prefuse.util.StringLib;


/**
 * Swing component that contains a slider, and title label, and editable
 * text box displaying the slider value.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class JValueSlider extends JComponent {

    private Number     m_min, m_max, m_value;
    private boolean    m_ignore = false;
    
    private JLabel     m_label;
    private JSlider    m_slider;
    private JTextField m_field;
    private List       m_listeners;
    
    private int m_smin = 0;
    private int m_srange = 100;
    
    /**
     * Create a new JValueSlider.
     * @param title the title label of the slider component
     * @param min the value associated with the minimum slider position
     * @param max the value associated with the maximum slider position
     * @param value the value associated with the starting slider position
     */
    public JValueSlider(String title, double min, double max, double value) {
        this(title, new Double(min), new Double(max), new Double(value));
    }
    
    /**
     * Create a new JValueSlider.
     * @param title the title label of the slider component
     * @param min the value associated with the minimum slider position
     * @param max the value associated with the maximum slider position
     * @param value the value associated with the starting slider position
     */
    public JValueSlider(String title, float min, float max, float value) {
        this(title, new Float(min), new Float(max), new Float(value));
    }
    
    /**
     * Create a new JValueSlider.
     * @param title the title label of the slider component
     * @param min the value associated with the minimum slider position
     * @param max the value associated with the maximum slider position
     * @param value the value associated with the starting slider position
     */
    public JValueSlider(String title, int min, int max, int value) {
        this(title, new Integer(min), new Integer(max), new Integer(value));
        this.m_smin = min;
        this.m_srange = max-min;
        this.m_slider.setMinimum(min);
        this.m_slider.setMaximum(max);
        setValue(new Integer(value));
    }
    
    /**
     * Create a new JValueSlider.
     * @param title the title label of the slider component
     * @param min the value associated with the minimum slider position
     * @param max the value associated with the maximum slider position
     * @param value the value associated with the starting slider position
     */
    public JValueSlider(String title, long min, long max, long value) {
        this(title, new Long(min), new Long(max), new Long(value));
    }
    
    /**
     * Create a new JValueSlider.
     * @param title the title label of the slider component
     * @param min the value associated with the minimum slider position
     * @param max the value associated with the maximum slider position
     * @param value the value associated with the starting slider position
     */
    public JValueSlider(String title, Number min, Number max, Number value) {
        this.m_min    = min;
        this.m_max    = max;
        this.m_value  = value;
        this.m_slider = new JSlider();
        this.m_label  = new JLabel(title);
        this.m_field  = new JTextField();
        this.m_listeners = new ArrayList();
        
        this.m_field.setBorder(null);
        
        setSliderValue();
        setFieldValue();
        
        initUI();
    }
    
    /**
     * Initialize the UI
     */
    protected void initUI() {
        this.m_slider.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                if ( JValueSlider.this.m_ignore ) return;
                JValueSlider.this.m_ignore = true;
                // update the value
                JValueSlider.this.m_value = getSliderValue();
                // set text field value
                setFieldValue();
                // fire event
                fireChangeEvent();
                JValueSlider.this.m_ignore = false;
            }
        });
        this.m_field.addActionListener(new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                if ( JValueSlider.this.m_ignore ) return;
                JValueSlider.this.m_ignore = true;
                Number v = getFieldValue();
                if ( v != JValueSlider.this.m_value ) {
                    // update the value
                    JValueSlider.this.m_value = v;
                    // set slider value
                    setSliderValue();
                }
                // fire event
                fireChangeEvent();
                JValueSlider.this.m_ignore = false;
            }
        });
        this.m_field.addMouseListener(new MouseAdapter() {
            public void mouseEntered(MouseEvent e) {
                String s = JValueSlider.this.m_field.getText();
                if ( isTextObscured(JValueSlider.this.m_field, s) )
                    JValueSlider.this.m_field.setToolTipText(s);
            }
            public void mouseExited(MouseEvent e) {
                JValueSlider.this.m_field.setToolTipText(null);
            }
        });
        this.m_label.addMouseListener(new MouseAdapter() {
            public void mouseEntered(MouseEvent e) {
                String s = JValueSlider.this.m_label.getText();
                if ( isTextObscured(JValueSlider.this.m_label, s) )
                    JValueSlider.this.m_label.setToolTipText(s);
            }
            public void mouseExited(MouseEvent e) {
                JValueSlider.this.m_label.setToolTipText(null);
            }
        });
        
        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        add(this.m_label);
        add(this.m_slider);
        add(this.m_field);
    }
    
    /**
     * Check if any label text is obscured.
     */
    private static boolean isTextObscured(JComponent c, String s) {
        Graphics g = c.getGraphics();
        FontMetrics fm = g.getFontMetrics(c.getFont());
        int sw = fm.stringWidth(s);
        return ( sw > c.getWidth() );
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * Get the current value ssociated with the slider position.
     * @return the current value
     */
    public Number getValue() {
        return this.m_value;
    }

    /**
     * Set the current value ssociated with the slider position.
     * @param value the current value to set
     */
    public void setValue(Number value) {
        this.m_value = value;
        setSliderValue();
        setFieldValue();
    }
    
    /**
     * Compute the current slider value from the current slider position
     * @return the current value
     */
    private Number getSliderValue() {
        if ( this.m_value instanceof Integer ) {
            int val = this.m_slider.getValue();
            int min = this.m_min.intValue();
            int max = this.m_max.intValue();
            return new Integer(min + (val-this.m_smin)*(max-min)/this.m_srange);
        } else if ( this.m_value instanceof Long ) {
            int val = this.m_slider.getValue();
            long min = this.m_min.longValue();
            long max = this.m_max.longValue();
            return new Long(min + (val-this.m_smin)*(max-min)/this.m_srange);
        } else {
            double f = (this.m_slider.getValue()-this.m_smin)/(double)this.m_srange;
            double min = this.m_min.doubleValue();
            double max = this.m_max.doubleValue();
            double val = min + f*(max-min);
            return (this.m_value instanceof Double ? (Number)new Double(val)
                                              : new Float((float)val));
        }
    }
    
    /**
     * Private set the slider position based upon the current value
     */
    private void setSliderValue() {
        int val;
        if ( this.m_value instanceof Double || this.m_value instanceof Float ) {
            double value = this.m_value.doubleValue();
            double min = this.m_min.doubleValue();
            double max = this.m_max.doubleValue();
            val = this.m_smin + (int)Math.round(this.m_srange*((value-min)/(max-min)));
        } else {
            long value = this.m_value.longValue();
            long min = this.m_min.longValue();
            long max = this.m_max.longValue();
            val = this.m_smin + (int)((this.m_srange*(value-min))/(max-min));
        }
        this.m_slider.setValue(val);
    }
    
    /**
     * Get the value in the text field.
     * @return the current text field value
     */
    private Number getFieldValue() {
        if ( this.m_value instanceof Double || this.m_value instanceof Float ) {
            double v;
            try {
                v = Double.parseDouble(this.m_field.getText());
            } catch ( Exception e ) {
                // TODO handle exception
                return this.m_value;
            }
            if ( v < this.m_min.doubleValue() || v > this.m_max.doubleValue() ) {
                // TODO handle exception
                return this.m_value;
            }
            return this.m_value instanceof Double ? (Number)new Double(v) 
                                             : new Float((float)v);
        } else {
            long v;
            try {
                v = Long.parseLong(this.m_field.getText());
            } catch ( Exception e ) {
                // TODO handle exception
                return this.m_value;
            }
            if ( v < this.m_min.longValue() || v > this.m_max.longValue() ) {
                // TODO handle exception
                return this.m_value;
            }
            return this.m_value instanceof Long ? (Number)new Long(v) 
                                           : new Integer((int)v);
        }
    }
    
    /**
     * Set the text field value based upon the current value.
     */
    private void setFieldValue() {
        String text;
        if ( this.m_value instanceof Double || this.m_value instanceof Float )
            text = StringLib.formatNumber(this.m_value.doubleValue(),3);
        else
            text = String.valueOf(this.m_value.longValue());
        this.m_field.setText(text);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * Add a change listener to listen to this component.
     * @param cl the change listener to add
     */
    public void addChangeListener(ChangeListener cl) {
        if ( !this.m_listeners.contains(cl) )
            this.m_listeners.add(cl);
    }

    /**
     * Remove a change listener listening to this component.
     * @param cl the change listener to remove
     */
    public void removeChangeListener(ChangeListener cl) {
        this.m_listeners.remove(cl);
    }
    
    /**
     * Fire a change event to listeners.
     */
    protected void fireChangeEvent() {
        Iterator iter = this.m_listeners.iterator();
        ChangeEvent evt = new ChangeEvent(this); 
        while ( iter.hasNext() ) {
            ChangeListener cl = (ChangeListener)iter.next();
            cl.stateChanged(evt);
        }
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see java.awt.Component#setBackground(java.awt.Color)
     */
    public void setBackground(Color c) {
        this.m_field.setBackground(c);
        this.m_label.setBackground(c);
        this.m_slider.setBackground(c);
        super.setBackground(c);
    }
    
    /**
     * @see java.awt.Component#setForeground(java.awt.Color)
     */
    public void setForeground(Color c) {
        this.m_field.setForeground(c);
        this.m_label.setForeground(c);
        this.m_slider.setForeground(c);
        super.setForeground(c);
    }
    
    /**
     * @see java.awt.Component#setFont(java.awt.Font)
     */
    public void setFont(Font f) {
        this.m_field.setFont(f);
        this.m_label.setFont(f);
        this.m_slider.setFont(f);
        super.setFont(f);
    }
    
    /**
     * @see javax.swing.JComponent#setPreferredSize(java.awt.Dimension)
     */
    public void setPreferredSize(Dimension d) {
        int fw = Math.min(40, d.width/5);
        int lw = Math.min(100, (d.width-fw)/2);
        int sw = d.width-fw-lw;
        super.setPreferredSize(d);
        Dimension dd = new Dimension(lw, d.height);
        this.m_label.setPreferredSize(dd);
        dd = new Dimension(sw, d.height);
        this.m_slider.setPreferredSize(dd);
        dd = new Dimension(fw, d.height);
        this.m_field.setPreferredSize(dd);
    }
    
} // end of class JValueSlider
