# -*- Mode: Python; coding: iso-8859-1 -*-
# vi:si:et:sw=4:sts=4:ts=4
#
# Copyright (C) 2005 Async Open Source
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

"""
Implementao de classe base para configurao da pgina e desenho de
elementos fixos de cada pgina.
"""

import datetime

from reportlab.lib.units import mm

# sibling imports
from stoqlib.reporting.template import BaseReportTemplate
from stoqlib.reporting.default_style import (HIGHLIGHT_COLOR, SPACING,
                                             TEXT_COLOR)

SMALL_FONT = ("Helvetica", 12)

class ReportTemplate(BaseReportTemplate):
    """ 
    Classe responsvel pela configurao da pgina e desenho de elementos
    fixos de cada pgina. 
    """
    header_height = 10 * mm
    footer_height = 7 * mm
    def __init__(self, filename, report_name, timestamp=0, do_header=1, 
                 do_footer=1, **kwargs):
        """
        Classe responsvel pela configurao/desenho bsico de cada pgina.
        Seus parmetros so:

            - filename: o nome do arquivo onde o relatrio deve ser
            desenhado. Esse nome de arquivo  passado como primeiro
            parmetro para a classe do usurio atravs da funo
            build_report().
            - report_name: o nome do relatrio, utilizado, basicamente, na
            construo do rodap da pgina.
            - timestamp: define se a hora de criao do relatrio deve ser
            especificada no rodap.
            - do_header: se definido como True, chama o mtodo draw_header()
            da classe do usurio para o desenho do cabealho do relatrio.
            Esse mtodo  chamado para cada pgina criada.
            - do_footer: se definido como True, insere um rodap em cada
            pgina criada.
        """
        self.timestamp = timestamp
        BaseReportTemplate.__init__(self, filename, report_name,
                                    do_header=do_header, do_footer=do_footer,
                                    **kwargs)

    def draw_header(self, canvas):
        """
        Definido para fins de compatibilidade. Quando o usurio especificar um
        argumento True para o parmetro do_header, o mtodo draw_header() da
        classe do usurio  chamado. Se este mtodo no existir, o mtodo
        desta classe  chamado para evitar o levantamento de excesso.
        """
        return
       
    def draw_footer(self, canvas):
        """
        Mtodo chamado para o desenho do rodap de pginas. Esse mtodo 
        chamado para cada pgina criada se o parmetro 'do_footer' da classe  
        esteja definido como TRUE (valor padro assumido caso o usurio no o
        especifique). O rodap  constitudo basicamente do nome do relatrio
        (parmetro report_class da classe), a data de gerao, a hora (caso o
        parmetro time_stamp da classe seja definido como TRUE) e o nmero da
        pgina atual.
        """
        if not self.do_footer:
            return

        ts = datetime.datetime.now()
        if self.timestamp:
            date_string = ts.strftime('%d/%m/%Y   %H:%M:%S')
        else:
            date_string = ts.strftime('%d/%m/%Y')

        page_number = "Pgina: % 2d" % self.get_page_number()

        # Let's start drawing
        
        canvas.setFillColor(HIGHLIGHT_COLOR)
        canvas.rect(self.leftMargin, self.bottomMargin, self.width,
                    self.footer_height, stroke=0, fill=1)
        text_y = self.bottomMargin + 0.5 * SPACING
        canvas.setFillColor(TEXT_COLOR)
        canvas.setFont(*SMALL_FONT)
        canvas.drawString(self.leftMargin + 0.5 * SPACING, text_y,
                          self.report_name)
        canvas.drawRightString(self._rightMargin - 75, text_y, date_string)
        canvas.drawRightString(self._rightMargin - 0.5 * SPACING, text_y,
                               page_number)

