# -*- Mode: Python; coding: iso-8859-1 -*-
# vi:si:et:sw=4:sts=4:ts=4
#
# Copyright (C) 2005 Async Open Source
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

"""
Este mdulo implementa a classe BaseReportTemplate, onde todos os mtodos para
insero de elementos no relatrio esto definidos e, parcialmente,
implementados.
"""

from reportlab.lib import pagesizes
from reportlab import platypus

from stoqlib.reporting import tables, flowables
from stoqlib.reporting.default_style import (DOC_DEFAULTS, SPACING,
                                             STYLE_SHEET, TABLE_STYLE,
                                             DEFAULT_MARGIN, TABLE_LINE)

class BaseReportTemplate(platypus.BaseDocTemplate):
    """ 
    Classe responsvel pela implementao dos mtodos para insero de
    elementos no relatrio.
    """
    header_height = 0
    footer_height = 0
    def __init__(self, filename, report_name, pagesize=pagesizes.A4,
                 landscape=0, do_header=0, do_footer=0, **kwargs):
        """
        Classe responsvel pela implementao dos mtodos para insero de
        elementos no relatrio. Os parmetros para esta classe, que podem (e
        devem) ser passados  classe ReportTemplate (a qual sua classe deve
        herdar), so:

            - filename: o nome do arquivo onde o relatrio deve ser
            construdo; esse nome de arquivo  passado por build_report  
            classe do usurio e  obrigatrio.
            - report_name: o nome do relatrio, parmetro obrigatrio; o nome
            do relatrio  utilizado, basicamente, para a construo do rodap
            do relatrio.
            - pagesize: o tamanho da pgina; os tamanhos disponveis podem ser
            encontrados em reportlab.lib.pagesizes.
            - landscape: define se o relatrio deve ser gerado no formato
            paisagem; o padro  o formato "retrato" (landscape=0).
            - do_footer: define se o rodap deve ser desenhado.
        """
        self.do_header = do_header
        self.do_footer = do_footer
        self.report_name = report_name

        doc_kwargs = DOC_DEFAULTS.copy()
        doc_kwargs.update(kwargs)

        if landscape:
            pagesize = pagesizes.landscape(pagesize)

        platypus.BaseDocTemplate.__init__(self, filename, pagesize=pagesize,
                                          title=report_name, **doc_kwargs)
        self.flowables = []
        self.grouping = 0
        # Group of flowables wich shouldn't be separated on different pages
        self._together_flowables = []
        # Number of flowables to include in the current group.
        self._together_count = 0

    #
    # External API
    #
    
    def save(self):
        """ 
        Construo e salvamento do relatrio. Mtodo chamado internamente.
        """
        self.build()

    def build(self):
        """
        Mtodo chamado internamente para construo do relatrio. Inicializa
        as pginas do relatrio e constri os elementos.
        """
        # Adds forgotten flowables
        self.end_group()
        
        # If page size has changed, we try to make ReportLab work
        self._calc()

        self.setup_page_templates()
        platypus.BaseDocTemplate.build(self, self.flowables)

    #
    # Doc structure
    #

    def setup_page_templates(self):
        """ 
        Inicializao das pginas do relatrio. Temos, basicamente, neste
        mtodo a definio do espao vertical disponvel para desenho,
        baseado no tamanho do rodap e cabealho do relatrio.
        """
        frame_y = self.bottomMargin
        height = self.height

        if self.do_header:
            height -= self.header_height

        if self.do_footer:
            height -= self.footer_height
            frame_y += self.footer_height

        main_frame = platypus.Frame(self.leftMargin, frame_y,
                                    self.width, height,
                                    bottomPadding=SPACING,
                                    topPadding=SPACING)

        template = platypus.PageTemplate(id='Normal', frames=main_frame,
                                         pagesize=self.pagesize,
                                         onPage=self.paint_page_canvas)
        self.addPageTemplates([template])

    #
    # Internal API
    #
    
    def add(self, flowable):
        """ 
        Mtodo chamado para insero de elementos no relatrio. Cada elemento
        criado, tal como um pargrafo, uma tabela, um titulo ou uma assinatura
        deve ser inserido no "relatrio" atrves deste mtodo para que ele
        possa ser desenhado quando uma chamada ao mtodo build for feita. Esse
        mtodo pertence  API interna e no deve ser chamado na maioria dos
        casos (a menos que voc esteja criando um novo tipo de flowable :)
        """
        if self.grouping:
            self._together_flowables.append(flowable)
            self.end_group(self._together_count - 1)
        else:
            self.flowables.append(flowable)

    def start_group(self):
        """ 
        Utilizado para agrupar elementos, como por exemplo,  necessrio no
        caso do mtodo para insero de ttulos no relatrio; se uma nota de
        ttulo for provida, ela deve ser agrupada junto com o ttulo, pois 
        tanto ela quanto o ttulo so, basicamente, um nico elemento: um
        ttulo. 
        """
        self.grouping = 1

    def end_group(self, min_flowables=0):
        """
        Termina o agrupamento de elementos, isto , todos os elementos que
        deviam ser agrupados j o foram. 
        """
        # Updating _together_count
        if min_flowables >= 0:
            self._together_count = min_flowables
        # If there is not more flowables, close the group and add it.
        if not min_flowables:
            self.grouping = 0
            if self._together_flowables:
                self.add(platypus.KeepTogether(self._together_flowables))
            self._together_flowables = []

    def get_usable_width(self):
        """
        Retorna o espao horizontal ainda disponvel para insero/desenho de
        elementos 
        """
        return self._rightMargin - self.leftMargin 

    def get_usable_height(self):
        """
        Retorna o espao vertical ainda disponvel para insero/desenho de
        elementos 
        """
        return self._topMargin - self.bottomMargin

    def set_page_number(self, number):
        """ Define o nmero da pgina atual """
        self.add(flowables.PageNumberChanger(number))

    def get_page_number(self):
        """ 
        Retorna o nmero da pgina atual, isto , a pgina que est sendo
        construda. 
        """
        return self.page
        
    #
    # Features
    #
        
    def add_page_break(self):
        """ Adiciona uma simples quebra de pgina. """
        self.add(platypus.PageBreak())

    def add_document_break(self):
        """
        Basicamente insere uma quebra de pgina e inicia um novo documento.  
        como se tivessemos dois documentos no mesmo relatrio.
        """
        self.set_page_number(0)
        self.add_page_break()
        
    def add_blank_space(self, height=10, width=-1):
        """ 
        Adiciona um espao branco na posio atual. Parametros:

           - height: o tamanho do espao a ser inserido
           - width: o comprimento do espao

        Atravs dos parametros height e width podemos definir o tipo de 
        espacamento que queremos, ou seja, se queremos um espaamento vertical,
        neste caso definimos height=-1 e width=X (X=tamanho do espacamento) ou
        se queremos um espaamento horizontal, neste caso, height=X e with=-1;
        espaamento vertical  o padro. 
        """
        self.add(platypus.Spacer(width, height))

    def add_signatures(self, labels, *args, **kwargs):
        """
        Adiciona uma assinatura no relatrio. Parmetros:

            - labels: Uma lista de strings de assinatura, cada item da lista
            ser uma assinatura e ser inserida no relatrio lado a lado;
            dependendo do tamanho da pgina e do parametro landscape, so
            permitidos de 2 a 4 assinaturas na mesma linha.
            - align: define o alinhamento do elemento, deve-se utilizar as
            constantes LEFT, CENTER ou RIGHT definidas neste mdulo.
            - line_width: comprimento da linha de assinatura.
            - height: espao vertical disponibilizado acima da linha.
            - text_align: alinhamento do texto de assinatura.
            - style_data: permite utilizar estilos de pargrafos para o texto
            de assinatura, caso no seja especificado o padro (definido no
            mdulo default_style) ser utilizado.
        """
        self.add(flowables.Signature(labels, *args, **kwargs))

    def add_preformatted_text(self, text, style='Raw', *args, **kwargs):
        """
        Adiciona um texto pr-formatado ao relatrio. Parmetros:

            - text: o texto a ser inserido
            - style: define o estilo a ser utilizado. Como padro o estilo
            'Raw' (consulte o mdulo default_style para mais detalhes) 
            utilizado.

        Voc pode utilizar esse mtodo para insero de textos com
        espaamento.

        Note que parmetros extras podem ser passados para essa
        funo, nesse caso eles sero repassados diretamente para a
        classe Preformatted do ReportLab. 
        """
        style = STYLE_SHEET[style]
        self.add(platypus.flowables.Preformatted(text, style, *args, **kwargs))

    def add_paragraph(self, text, style='Normal', **kwargs):
        """ 
        Adiciona um pargrafo. Parametros:

            - text: o texto a ser inserido no relatrio.
            - style: define o estilo a ser utilizado; vrios deles esto
            definidos no mdulo default_style.
        """
        style = STYLE_SHEET[style]
        self.add(platypus.Paragraph(text, style, **kwargs))

    def add_report_table(self, data, header=None, style=TABLE_STYLE,
                         margins=DEFAULT_MARGIN, align=flowables.CENTER,
                         extra_row=None, table_line=TABLE_LINE,
                         highlight=tables.HIGHLIGHT_ODD, *args, **kwargs):
        """
        Inserco de uma tabela relatrio na lista de elementos. Os
        parmetros para este tipo de tabela, so:

            - data: uma lista de listas contendo as linhas da tabela, cada
            lista interna representa uma linha, enquanto seus elementos
            representam as colunas desta linha.
            - header: uma lista que, se especificada, ser utilizada como
            cabealho da tabela; o tamanho desta lista deve ser o mesmo
            das listas internas especificadas no parmetro data.
            - style: permite a especificao de estilos (TableStyle)
            prprios para uso na tabela.
            - margins: margens verticais antes e aos tabela.
            - align: alinhamento da tabela; voc pode encontrar as
            constantes para alinhamento no mdulo flowables.
            - extra_row: uma lista com a linha extra  ser inserida. Assim 
            como o parmetro header, a lista especificada como argumento
            deve possuir o mesmo tamanho das listas internas especificadas
            ao parmetro data.
            - table_line: define o tipo de linha a ser utilizada na tabela.
            Stoqlib Reporting fornece os tipos TABLE_LINE (linhas simples) e
            TABLE_LINE_BLANK (sem linhas).
            - highlight: habilita (constante HIGHLIGHT_ODD) ou desabilita
            (HIGHLIGHT_NEVER) o uso do estilo zebrado nas linhas da tabela.
            O padro  habilitado (HIGHLIGHT_ODD).
        """
        self.add_blank_space(margins)
        table_builder = tables.ReportTableBuilder(data, style, header,
                                                  table_line,
                                                  extra_row=extra_row)
        kwargs["align"] = align
        table_builder.set_highlight(highlight)
        self.add(table_builder.create_table(*args, **kwargs))
        self.add_blank_space(margins)
    
    def add_column_table(self, data, columns, style=TABLE_STYLE,
                         margins=DEFAULT_MARGIN, align=flowables.CENTER,
                         extra_row=None, table_line=TABLE_LINE, do_header=1,
                         highlight=tables.HIGHLIGHT_ODD, *args, **kwargs):
        """
        Insero de uma tabela coluna na lista de elementos. Os parmetros
        para este tipo de tabela, so:

            - data: uma lista de listas, onde cada lista internO ideal seria ter a opcao de definir o comportamento: truncar os caracteres OU
rolar para linha abaixo.a representa
            uma lista e cada elemento representa o valor  ser inserido em
            uma coluna.
            - columns: uma lista de instncias TableColumn representando as
            colunas da tabela.
            - style: estilo de tabela a ser utilizado, permite a
            especificao de estilos (TableStyle) prprios para uso na
            tabela.
            - margins: margens verticais antes e aps a tabela.
            - align: alinhamento da tabela; voc pode encontrar as
            constantes para alinhamento no mdulo flowables.
            - extra_row: uma lista com a linha extra  ser inserida. A lista
            especificada como argumento deve possuir o mesmo tamanho das
            listas internas especificadas ao parmetro data.
            - table_line: define o tipo de linha a ser utilizado na tabela.
            Stoqlib Reporting fornece os tipos TABLE_LINE (linhas simples)
            e TABLE_LINE_BLANK (sem linhas). O tipo TABLE_LINE  o padro.
            - do_header: se definido como True, o cabealho da tabela ser
            desenhado. O nome de cada coluna  obtida atravs do atributo
            'name' de cada instncia especificada lista do argumento
            columns.
            - highlight: habilita (constante HIGHLIGHT_ODD) ou desabilita
            (HIGHLIGHT_NEVER) o uso do estilo zebrado nas linhas da
            tabela. O padro  habilitado (HIGHLIGHT_ODD).
        """
        self.add_blank_space(margins)
        table_builder = tables.ColumnTableBuilder(data, columns, style=style, 
                                                  table_line=table_line,
                                                  do_header=do_header,
                                                  extra_row=extra_row)
        kwargs["align"] = align
        table_builder.set_highlight(highlight)
        self.add(table_builder.create_table(*args, **kwargs))
        self.add_blank_space(margins)

    def add_object_table(self, objs, cols, expand=0, width=0, 
                         style=TABLE_STYLE, margins=DEFAULT_MARGIN,
                         extra_row=None, align=flowables.CENTER, 
                         table_line=TABLE_LINE, highlight=tables.HIGHLIGHT_ODD,
                         *args, **kwargs):
        """
        Insero de uma tabela objeto na lista de elementos. Os parmetros
        para este tipo de tabela, so:

            - objs: uma lista de objetos na qual a lista de linhas ser
            construda.
            - cols: uma lista de colunas ObjectTableColumn.
            - expand:
            - width: utilizado para permitir ao usurio especificar o
              tamanho da tabela.
            - style: parmetro opcional, permite ao usurio definir um
              estilo de tabela (TableStyle) prprio.
            - margins: margens verticais antes e aps a tabela.
            - extra_row: uma lista de valores representado uma linha extra.
              Nem todos os elementos precisam estar preenchidos, mas 
              necessrio que eles existam, isto ,  necessrio que o
              tamanho desta lista seja o mesmo das listas internas do
              parmetro data.
            - align: alinhamento da tabela; voc pode encontrar as
              constantes para alinhamento no mdulo flowables.
            - table_line: define o tipo de linha a ser utilizado na tabela.
              Stoqlib Reporting fornece os tipos TABLE_LINE (linhas simples)
              e TABLE_LINE_BLANK (sem linhas).
            - highlight: habilita (constante HIGHLIGHT_ODD) ou desabilita
              (HIGHLIGHT_NEVER) o uso do estilo zebrado nas linhas da
              tabela. O padro  habilitado (HIGHLIGHT_ODD).
            
        """
        assert not (expand and width), \
            'Use only expand _OR_ only width at once'
        if expand:
            width = self.get_usable_width()

        self.add_blank_space(margins)
        table_builder = tables.ObjectTableBuilder(objs, cols, style,
                                                  width=width,
                                                  extra_row=extra_row,
                                                  table_line=table_line)
        kwargs["align"] = align
        table_builder.set_highlight(highlight)
        self.add(table_builder.create_table(*args, **kwargs))
        self.add_blank_space(margins)

    def add_grouping_table(self, objs, column_groups, column_widths,
                           header=None, style=TABLE_STYLE,
                           margins=DEFAULT_MARGIN, align=flowables.CENTER,
                           extra_row=None, *args, **kwargs):
        # """We need to set the table header directly for GroupingTableBuilder
        # because the Columns used with it does not have a name. Note that we
        # have one header for each column width defined and you can use a false
        # value (None, '', 0) to make the previous header span over it."""
        self.add_blank_space(margins)
        table_builder = tables.GroupingTableBuilder(objs, column_groups,
                                                    column_widths,
                                                    style=style,
                                                    header=header, 
                                                    extra_row=extra_row)
        kwargs["align"] = align
        self.add(table_builder.create_table(*args, **kwargs))
        self.add_blank_space(margins)
    
    def add_data_table(self, data, style=TABLE_STYLE, margins=DEFAULT_MARGIN, 
                       align=flowables.LEFT, *args, **kwargs):
        """ 
        Insero de uma tabela simples. Os parametros so:

            - data: uma lista de listas, onde cada lista interna representa uma
            linha da tabela, e cada item desta lista as colunas.
            - style: define o estilo que a tabela deve seguir, mais estilos 
            voc pode encontrar em stoqlib.reporting.default_style
            - margins: margens verticais antes e depois da tabela.
            - align: alinhamento da tabela
        """
            
        self.add_blank_space(margins)
        table_builder = tables.DataTableBuilder(data, style)
        kwargs["align"] = align
        self.add(table_builder.create_table(*args, **kwargs))
        self.add_blank_space(margins)
    
    def add_line(self, *args, **kwargs):
        """ Adiciona uma simples linha na posio atual """
        line = flowables.ReportLine(*args, **kwargs)
        self.add(line)

    def add_title(self, title, note=None, space_before=SPACING,
                  style='Title', note_style='Title-Note'):
        """
        Adiciona um ttulo na posio atual. Parmetros:

            - title: o texto que ser o ttulo
            - note: se especificado, ser inserido como uma nota ao ttulo
            - space_before: define o tamanho do espaamento a ser inserido
            aps o ttulo
            - style: define o estilo a ser utilizado para o pargrafo 'title';
            no  recomendado sua alterao, uma vez que isto quebra o padro
            utilizado em todo o documento (a no ser que um novo padro seja
            especificado em um atributo da classe)
            - note_style: define o estilo a ser utilizado para o pargrafo
            'note'
        """
        self.add_blank_space(space_before)
        self.start_group()
        self.add_line(v_margins=1)
        self.add_paragraph(title, style=style)
        if note:
            self.add_paragraph(note, style=note_style)
        self.add_line(v_margins=1)
        self.end_group(1)

    #
    # Handlers
    #

    def paint_page_canvas(self, canvas, doc):
        """
        Mtodo chamado quando uma nova pgina  criada; basicamente o
        processamento feito aqui  a insero do rodap e cabealho. 
        """
        if self.do_header:
            self.draw_header(canvas)
        if self.do_footer:
            self.draw_footer(canvas)
            
    def draw_header(self, canvas):
        raise NotImplementedError

    def draw_footer(self, canvas):
        raise NotImplementedError

