/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.storage.relational;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.Entity;
import org.apache.gravitino.storage.relational.RelationalBackend;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RelationalGarbageCollector
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(RelationalGarbageCollector.class);
    private final RelationalBackend backend;
    private final long storeDeleteAfterTimeMillis;
    private final long versionRetentionCount;
    @VisibleForTesting
    final ScheduledExecutorService garbageCollectorPool = new ScheduledThreadPoolExecutor(2, r -> {
        Thread t = new Thread(r, "RelationalBackend-Garbage-Collector");
        t.setDaemon(true);
        return t;
    }, new ThreadPoolExecutor.AbortPolicy());

    public RelationalGarbageCollector(RelationalBackend backend, Config config) {
        this.backend = backend;
        this.storeDeleteAfterTimeMillis = config.get(Configs.STORE_DELETE_AFTER_TIME);
        this.versionRetentionCount = config.get(Configs.VERSION_RETENTION_COUNT);
    }

    public void start() {
        long dateTimelineMinute = this.storeDeleteAfterTimeMillis / 1000L / 60L;
        long frequency = Math.max(dateTimelineMinute / 10L, 10L);
        this.garbageCollectorPool.scheduleAtFixedRate(this::collectAndClean, 5L, frequency, TimeUnit.MINUTES);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void collectAndClean() {
        long threadId = Thread.currentThread().getId();
        LOG.debug("Thread {} start to collect garbage...", (Object)threadId);
        try {
            long deletedCount;
            LOG.debug("Start to collect and delete legacy data by thread {}", (Object)threadId);
            long legacyTimeline = System.currentTimeMillis() - this.storeDeleteAfterTimeMillis;
            for (Entity.EntityType entityType : Entity.EntityType.values()) {
                deletedCount = Long.MAX_VALUE;
                LOG.debug("Try to physically delete {} legacy data that has been marked deleted before {}", (Object)entityType, (Object)legacyTimeline);
                try {
                    while (deletedCount > 0L) {
                        deletedCount = this.backend.hardDeleteLegacyData(entityType, legacyTimeline);
                    }
                }
                catch (RuntimeException e) {
                    LOG.error("Failed to physically delete type of " + (Object)((Object)entityType) + "'s legacy data: ", (Throwable)e);
                }
            }
            LOG.info("Start to collect and delete old version data by thread {}", (Object)threadId);
            for (Entity.EntityType entityType : Entity.EntityType.values()) {
                deletedCount = Long.MAX_VALUE;
                LOG.debug("Try to softly delete {} old version data that has been over retention count {}", (Object)entityType, (Object)this.versionRetentionCount);
                try {
                    while (deletedCount > 0L) {
                        deletedCount = this.backend.deleteOldVersionData(entityType, this.versionRetentionCount);
                    }
                }
                catch (RuntimeException e) {
                    LOG.error("Failed to softly delete type of " + (Object)((Object)entityType) + "'s old version data: ", (Throwable)e);
                }
            }
        }
        catch (Exception e) {
            LOG.error("Thread {} failed to collect and clean garbage.", (Object)threadId, (Object)e);
        }
        finally {
            LOG.debug("Thread {} finish to collect garbage.", (Object)threadId);
        }
    }

    @Override
    public void close() throws IOException {
        this.garbageCollectorPool.shutdown();
        try {
            if (!this.garbageCollectorPool.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.garbageCollectorPool.shutdownNow();
            }
        }
        catch (InterruptedException ex) {
            this.garbageCollectorPool.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}

