/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.salesforcedelete;

import java.util.List;
import lombok.Getter;
import lombok.Setter;
import org.apache.hop.core.CheckResult;
import org.apache.hop.core.Const;
import org.apache.hop.core.ICheckResult;
import org.apache.hop.core.annotations.Transform;
import org.apache.hop.core.exception.HopTransformException;
import org.apache.hop.core.row.IRowMeta;
import org.apache.hop.core.variables.IVariables;
import org.apache.hop.i18n.BaseMessages;
import org.apache.hop.metadata.api.HopMetadataProperty;
import org.apache.hop.metadata.api.IHopMetadataProvider;
import org.apache.hop.pipeline.PipelineMeta;
import org.apache.hop.pipeline.transform.TransformMeta;
import org.apache.hop.pipeline.transforms.salesforce.SalesforceTransformMeta;

@Transform(
    id = "SalesforceDelete",
    name = "i18n::SalesforceDelete.TypeLongDesc.SalesforceDelete",
    description = "i18n::SalesforceDelete.TypeTooltipDesc.SalesforceDelete",
    categoryDescription = "i18n:org.apache.hop.pipeline.transform:BaseTransform.Category.Output",
    image = "SFD.svg",
    keywords = "i18n::SalesforceDeleteMeta.keyword",
    documentationUrl = "/pipeline/transforms/salesforcedelete.html")
@Getter
@Setter
public class SalesforceDeleteMeta
    extends SalesforceTransformMeta<SalesforceDelete, SalesforceDeleteData> {
  private static final Class<?> PKG = SalesforceDeleteMeta.class;

  /** Deletefield */
  @HopMetadataProperty(key = "DeleteField")
  private String deleteField;

  /** Batch size */
  @HopMetadataProperty(key = "batchSize")
  private String batchSize;

  @HopMetadataProperty(key = "rollbackAllChangesOnError")
  private boolean rollbackAllChangesOnError;

  public SalesforceDeleteMeta() {
    super(); // allocate BaseTransformMeta
  }

  /**
   * @return Returns the rollbackAllChangesOnError.
   */
  public boolean isRollbackAllChangesOnError() {
    return rollbackAllChangesOnError;
  }

  /**
   * @param rollbackAllChangesOnError The rollbackAllChangesOnError to set.
   */
  public void setRollbackAllChangesOnError(boolean rollbackAllChangesOnError) {
    this.rollbackAllChangesOnError = rollbackAllChangesOnError;
  }

  /**
   * @param deleteField The DeleteField to set.
   */
  public void setDeleteField(String deleteField) {
    this.deleteField = deleteField;
  }

  /**
   * @return Returns the DeleteField.
   */
  public String getDeleteField() {
    return this.deleteField;
  }

  /**
   * @param batchSize
   */
  public void setBatchSize(String batchSize) {
    this.batchSize = batchSize;
  }

  /**
   * @return Returns the batchSize.
   */
  public String getBatchSize() {
    return this.batchSize;
  }

  public int getBatchSizeInt() {
    return Const.toInt(this.batchSize, 10);
  }

  @Override
  public Object clone() {
    SalesforceDeleteMeta retval = (SalesforceDeleteMeta) super.clone();

    return retval;
  }

  @Override
  public void setDefault() {
    super.setDefault();
    setModule("Account");
    setDeleteField(null);
    setBatchSize("10");
    setRollbackAllChangesOnError(false);
  }

  /* This function adds meta data to the rows being pushed out */
  @Override
  public void getFields(
      IRowMeta r,
      String name,
      IRowMeta[] info,
      TransformMeta nextTransform,
      IVariables variables,
      IHopMetadataProvider metadataProvider)
      throws HopTransformException {
    // Do nothing
  }

  @Override
  public void check(
      List<ICheckResult> remarks,
      PipelineMeta pipelineMeta,
      TransformMeta transformMeta,
      IRowMeta prev,
      String[] input,
      String[] output,
      IRowMeta info,
      IVariables variables,
      IHopMetadataProvider metadataProvider) {
    super.check(
        remarks,
        pipelineMeta,
        transformMeta,
        prev,
        input,
        output,
        info,
        variables,
        metadataProvider);
    CheckResult cr;

    // See if we get input...
    if (input != null && input.length > 0) {
      cr =
          new CheckResult(
              ICheckResult.TYPE_RESULT_ERROR,
              BaseMessages.getString(PKG, "SalesforceDeleteMeta.CheckResult.NoInputExpected"),
              transformMeta);
    } else {
      cr =
          new CheckResult(
              ICheckResult.TYPE_RESULT_OK,
              BaseMessages.getString(PKG, "SalesforceDeleteMeta.CheckResult.NoInput"),
              transformMeta);
    }
    remarks.add(cr);
  }

  @Override
  public boolean supportsErrorHandling() {
    return true;
  }
}
