/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.index;

import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.apache.ignite.internal.catalog.CatalogService;
import org.apache.ignite.internal.catalog.descriptors.CatalogIndexDescriptor;
import org.apache.ignite.internal.catalog.descriptors.CatalogIndexStatus;
import org.apache.ignite.internal.catalog.events.CatalogEvent;
import org.apache.ignite.internal.catalog.events.RemoveIndexEventParameters;
import org.apache.ignite.internal.catalog.events.StartBuildingIndexEventParameters;
import org.apache.ignite.internal.close.ManuallyCloseable;
import org.apache.ignite.internal.event.Event;
import org.apache.ignite.internal.hlc.ClockService;
import org.apache.ignite.internal.hlc.HybridTimestamp;
import org.apache.ignite.internal.index.IndexBuilder;
import org.apache.ignite.internal.index.IndexManagementUtils;
import org.apache.ignite.internal.index.IndexManager;
import org.apache.ignite.internal.lang.IgniteInternalException;
import org.apache.ignite.internal.network.ClusterService;
import org.apache.ignite.internal.placementdriver.PlacementDriver;
import org.apache.ignite.internal.placementdriver.PrimaryReplicaAwaitTimeoutException;
import org.apache.ignite.internal.placementdriver.ReplicaMeta;
import org.apache.ignite.internal.placementdriver.event.PrimaryReplicaEvent;
import org.apache.ignite.internal.placementdriver.event.PrimaryReplicaEventParameters;
import org.apache.ignite.internal.replicator.ReplicationGroupId;
import org.apache.ignite.internal.replicator.TablePartitionId;
import org.apache.ignite.internal.storage.MvPartitionStorage;
import org.apache.ignite.internal.storage.engine.MvTableStorage;
import org.apache.ignite.internal.storage.index.IndexStorage;
import org.apache.ignite.internal.util.CompletableFutures;
import org.apache.ignite.internal.util.ExceptionUtils;
import org.apache.ignite.internal.util.IgniteSpinBusyLock;
import org.apache.ignite.internal.util.IgniteUtils;
import org.apache.ignite.lang.ErrorGroups;
import org.apache.ignite.network.ClusterNode;
import org.jetbrains.annotations.Nullable;

class IndexBuildController
implements ManuallyCloseable {
    private final IndexBuilder indexBuilder;
    private final IndexManager indexManager;
    private final CatalogService catalogService;
    private final ClusterService clusterService;
    private final PlacementDriver placementDriver;
    private final ClockService clockService;
    private final IgniteSpinBusyLock busyLock = new IgniteSpinBusyLock();
    private final AtomicBoolean closeGuard = new AtomicBoolean();
    private final Set<TablePartitionId> primaryReplicaIds = ConcurrentHashMap.newKeySet();

    IndexBuildController(IndexBuilder indexBuilder, IndexManager indexManager, CatalogService catalogService, ClusterService clusterService, PlacementDriver placementDriver, ClockService clockService) {
        this.indexBuilder = indexBuilder;
        this.indexManager = indexManager;
        this.catalogService = catalogService;
        this.clusterService = clusterService;
        this.placementDriver = placementDriver;
        this.clockService = clockService;
    }

    public void start() {
        IgniteUtils.inBusyLock((IgniteSpinBusyLock)this.busyLock, this::addListeners);
    }

    public void close() {
        if (!this.closeGuard.compareAndSet(false, true)) {
            return;
        }
        this.busyLock.block();
        this.indexBuilder.close();
    }

    private void addListeners() {
        this.catalogService.listen((Event)CatalogEvent.INDEX_BUILDING, parameters -> this.onIndexBuilding((StartBuildingIndexEventParameters)parameters).thenApply(unused -> false));
        this.catalogService.listen((Event)CatalogEvent.INDEX_REMOVED, parameters -> this.onIndexRemoved((RemoveIndexEventParameters)parameters).thenApply(unused -> false));
        this.placementDriver.listen((Event)PrimaryReplicaEvent.PRIMARY_REPLICA_ELECTED, parameters -> this.onPrimaryReplicaElected((PrimaryReplicaEventParameters)parameters).thenApply(unused -> false));
    }

    private CompletableFuture<?> onIndexBuilding(StartBuildingIndexEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync((IgniteSpinBusyLock)this.busyLock, () -> {
            CatalogIndexDescriptor indexDescriptor = this.catalogService.index(parameters.indexId(), parameters.catalogVersion());
            ArrayList<CompletionStage> startBuildIndexFutures = new ArrayList<CompletionStage>();
            for (TablePartitionId primaryReplicaId : this.primaryReplicaIds) {
                if (primaryReplicaId.tableId() != indexDescriptor.tableId()) continue;
                CompletionStage startBuildIndexFuture = this.getMvTableStorageFuture(parameters.causalityToken(), primaryReplicaId).thenCompose(mvTableStorage -> this.awaitPrimaryReplica(primaryReplicaId, this.clockService.now()).thenAccept(replicaMeta -> this.tryScheduleBuildIndex(primaryReplicaId, indexDescriptor, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta)));
                startBuildIndexFutures.add(startBuildIndexFuture);
            }
            return CompletableFuture.allOf((CompletableFuture[])startBuildIndexFutures.toArray(CompletableFuture[]::new));
        });
    }

    private CompletableFuture<?> onIndexRemoved(RemoveIndexEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync((IgniteSpinBusyLock)this.busyLock, () -> {
            this.indexBuilder.stopBuildingIndexes(parameters.indexId());
            return CompletableFutures.nullCompletedFuture();
        });
    }

    private CompletableFuture<?> onPrimaryReplicaElected(PrimaryReplicaEventParameters parameters) {
        return IgniteUtils.inBusyLockAsync((IgniteSpinBusyLock)this.busyLock, () -> {
            TablePartitionId primaryReplicaId = (TablePartitionId)parameters.groupId();
            if (IndexManagementUtils.isLocalNode(this.clusterService, parameters.leaseholderId())) {
                this.primaryReplicaIds.add(primaryReplicaId);
                int catalogVersion = this.catalogService.latestCatalogVersion();
                if (this.catalogService.table(primaryReplicaId.tableId(), catalogVersion) == null) {
                    return CompletableFutures.nullCompletedFuture();
                }
                return this.getMvTableStorageFuture(parameters.causalityToken(), primaryReplicaId).thenCompose(mvTableStorage -> this.awaitPrimaryReplica(primaryReplicaId, parameters.startTime()).thenAccept(replicaMeta -> this.tryScheduleBuildIndexesForNewPrimaryReplica(catalogVersion, primaryReplicaId, (MvTableStorage)mvTableStorage, (ReplicaMeta)replicaMeta)));
            }
            this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
            return CompletableFutures.nullCompletedFuture();
        });
    }

    private void tryScheduleBuildIndexesForNewPrimaryReplica(int catalogVersion, TablePartitionId primaryReplicaId, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta) {
        IgniteUtils.inBusyLock((IgniteSpinBusyLock)this.busyLock, () -> {
            if (this.isLeaseExpire(replicaMeta)) {
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            for (CatalogIndexDescriptor indexDescriptor : this.catalogService.indexes(catalogVersion, primaryReplicaId.tableId())) {
                if (indexDescriptor.status() == CatalogIndexStatus.BUILDING) {
                    this.scheduleBuildIndex(primaryReplicaId, indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta));
                    continue;
                }
                if (indexDescriptor.status() != CatalogIndexStatus.AVAILABLE) continue;
                this.scheduleBuildIndexAfterDisasterRecovery(primaryReplicaId, indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta));
            }
        });
    }

    private void tryScheduleBuildIndex(TablePartitionId primaryReplicaId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, ReplicaMeta replicaMeta) {
        IgniteUtils.inBusyLock((IgniteSpinBusyLock)this.busyLock, () -> {
            if (this.isLeaseExpire(replicaMeta)) {
                this.stopBuildingIndexesIfPrimaryExpired(primaryReplicaId);
                return;
            }
            this.scheduleBuildIndex(primaryReplicaId, indexDescriptor, mvTableStorage, IndexBuildController.enlistmentConsistencyToken(replicaMeta));
        });
    }

    private void stopBuildingIndexesIfPrimaryExpired(TablePartitionId replicaId) {
        if (this.primaryReplicaIds.remove(replicaId)) {
            this.indexBuilder.stopBuildingIndexes(replicaId.tableId(), replicaId.partitionId());
        }
    }

    private CompletableFuture<MvTableStorage> getMvTableStorageFuture(long causalityToken, TablePartitionId replicaId) {
        return this.indexManager.getMvTableStorage(causalityToken, replicaId.tableId()).thenApply(mvTableStorage -> IndexBuildController.requireMvTableStorageNonNull(mvTableStorage, replicaId.tableId()));
    }

    private static MvTableStorage requireMvTableStorageNonNull(@Nullable MvTableStorage mvTableStorage, int tableId) {
        if (mvTableStorage == null) {
            throw new IgniteInternalException(ErrorGroups.Common.INTERNAL_ERR, "Table storage for the specified table cannot be null [tableId = {}]", new Object[]{tableId});
        }
        return mvTableStorage;
    }

    private CompletableFuture<ReplicaMeta> awaitPrimaryReplica(TablePartitionId replicaId, HybridTimestamp timestamp) {
        return ((CompletableFuture)this.placementDriver.awaitPrimaryReplica((ReplicationGroupId)replicaId, timestamp, 10L, TimeUnit.SECONDS).handle((replicaMeta, throwable) -> {
            if (throwable != null) {
                Throwable unwrapThrowable = ExceptionUtils.unwrapCause((Throwable)throwable);
                if (unwrapThrowable instanceof PrimaryReplicaAwaitTimeoutException) {
                    return this.awaitPrimaryReplica(replicaId, timestamp);
                }
                return CompletableFuture.failedFuture(unwrapThrowable);
            }
            return CompletableFuture.completedFuture(replicaMeta);
        })).thenCompose(Function.identity());
    }

    private void scheduleBuildIndex(TablePartitionId replicaId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken) {
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, replicaId);
        assert (mvPartition != null) : "Partition storage is missing, replicaId=" + String.valueOf(replicaId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, replicaId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndex(replicaId.tableId(), replicaId.partitionId(), indexDescriptor.id(), indexStorage, mvPartition, this.localNode(), enlistmentConsistencyToken);
    }

    private void scheduleBuildIndexAfterDisasterRecovery(TablePartitionId replicaId, CatalogIndexDescriptor indexDescriptor, MvTableStorage mvTableStorage, long enlistmentConsistencyToken) {
        MvPartitionStorage mvPartition = IndexBuildController.mvPartitionStorage(mvTableStorage, replicaId);
        assert (mvPartition != null) : "Partition storage is missing, replicaId=" + String.valueOf(replicaId);
        IndexStorage indexStorage = IndexBuildController.indexStorage(mvTableStorage, replicaId, indexDescriptor);
        this.indexBuilder.scheduleBuildIndexAfterDisasterRecovery(replicaId.tableId(), replicaId.partitionId(), indexDescriptor.id(), indexStorage, mvPartition, this.localNode(), enlistmentConsistencyToken);
    }

    private ClusterNode localNode() {
        return IndexManagementUtils.localNode(this.clusterService);
    }

    private boolean isLeaseExpire(ReplicaMeta replicaMeta) {
        return !IndexManagementUtils.isPrimaryReplica(replicaMeta, this.localNode(), this.clockService.now());
    }

    private static long enlistmentConsistencyToken(ReplicaMeta replicaMeta) {
        return replicaMeta.getStartTime().longValue();
    }

    private static MvPartitionStorage mvPartitionStorage(MvTableStorage mvTableStorage, TablePartitionId replicaId) {
        MvPartitionStorage mvPartition = mvTableStorage.getMvPartition(replicaId.partitionId());
        assert (mvPartition != null) : "Partition storage is missing, replicaId=" + String.valueOf(replicaId);
        return mvPartition;
    }

    private static IndexStorage indexStorage(MvTableStorage mvTableStorage, TablePartitionId replicaId, CatalogIndexDescriptor indexDescriptor) {
        IndexStorage indexStorage = mvTableStorage.getIndex(replicaId.partitionId(), indexDescriptor.id());
        assert (indexStorage != null) : "replicaId=" + String.valueOf(replicaId) + ", indexId=" + indexDescriptor.id();
        return indexStorage;
    }
}

