/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.queryengine.execution;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.iotdb.db.queryengine.execution.FutureStateChange;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class StateMachine<T> {
    private static final String LOCK_HELD_ERROR_MSG = "Cannot set state while holding the lock";
    private static final String STATE_IS_NULL = "newState is null";
    private static final Logger LOGGER = LoggerFactory.getLogger(StateMachine.class);
    private final String name;
    private final Executor executor;
    private final Object lock = new Object();
    private final Set<T> terminalStates;
    @GuardedBy(value="lock")
    private T state;
    @GuardedBy(value="lock")
    private final List<StateChangeListener<T>> stateChangeListeners = new ArrayList<StateChangeListener<T>>();
    private final AtomicReference<FutureStateChange<T>> futureStateChange = new AtomicReference(new FutureStateChange());

    public StateMachine(String name, Executor executor, T initialState) {
        this(name, executor, initialState, (Iterable<T>)ImmutableSet.of());
    }

    public StateMachine(String name, Executor executor, T initialState, Iterable<T> terminalStates) {
        this.name = Objects.requireNonNull(name, "name is null");
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.state = Objects.requireNonNull(initialState, "initialState is null");
        this.terminalStates = ImmutableSet.copyOf(Objects.requireNonNull(terminalStates, "terminalStates is null"));
    }

    public T get() {
        return this.state;
    }

    public T set(T newState) {
        T oldState = this.trySet(newState);
        Preconditions.checkState((oldState.equals(newState) || !this.isTerminalState(oldState) ? 1 : 0) != 0, (String)"%s cannot transition from %s to %s", (Object)this.name, this.state, newState);
        return oldState;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public T trySet(T newState) {
        ImmutableList curStateChangeListeners;
        FutureStateChange oldFutureStateChange;
        T oldState;
        Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)LOCK_HELD_ERROR_MSG);
        Objects.requireNonNull(newState, STATE_IS_NULL);
        Object object = this.lock;
        synchronized (object) {
            if (this.state.equals(newState) || this.isTerminalState(this.state)) {
                return this.state;
            }
            oldState = this.state;
            this.state = newState;
            oldFutureStateChange = this.futureStateChange.getAndSet(new FutureStateChange());
            curStateChangeListeners = ImmutableList.copyOf(this.stateChangeListeners);
            if (this.isTerminalState(this.state)) {
                this.stateChangeListeners.clear();
            }
        }
        this.fireStateChanged(newState, oldFutureStateChange, (List<StateChangeListener<T>>)curStateChangeListeners);
        return oldState;
    }

    public boolean setIf(T newState, Predicate<T> predicate) {
        T currentState;
        Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)LOCK_HELD_ERROR_MSG);
        Objects.requireNonNull(newState, STATE_IS_NULL);
        do {
            if ((currentState = this.get()).equals(newState)) {
                return false;
            }
            if (predicate.test(currentState)) continue;
            return false;
        } while (!this.compareAndSet(currentState, newState));
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean compareAndSet(T expectedState, T newState) {
        ImmutableList curStateChangeListeners;
        FutureStateChange oldFutureStateChange;
        Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)LOCK_HELD_ERROR_MSG);
        Objects.requireNonNull(expectedState, "expectedState is null");
        Objects.requireNonNull(newState, STATE_IS_NULL);
        Object object = this.lock;
        synchronized (object) {
            if (!this.state.equals(expectedState)) {
                return false;
            }
            if (this.state.equals(newState)) {
                return false;
            }
            Preconditions.checkState((!this.isTerminalState(this.state) ? 1 : 0) != 0, (String)"%s cannot transition from %s to %s", (Object)this.name, this.state, newState);
            this.state = newState;
            oldFutureStateChange = this.futureStateChange.getAndSet(new FutureStateChange());
            curStateChangeListeners = ImmutableList.copyOf(this.stateChangeListeners);
            if (this.isTerminalState(this.state)) {
                this.stateChangeListeners.clear();
            }
        }
        this.fireStateChanged(newState, oldFutureStateChange, (List<StateChangeListener<T>>)curStateChangeListeners);
        return true;
    }

    private void fireStateChanged(T newState, FutureStateChange<T> futureStateChange, List<StateChangeListener<T>> stateChangeListeners) {
        Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)"Cannot fire state change event while holding the lock");
        Objects.requireNonNull(newState, STATE_IS_NULL);
        this.safeExecute(() -> {
            Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)"Cannot notify while holding the lock");
            try {
                futureStateChange.complete(newState);
            }
            catch (Throwable e) {
                LOGGER.error("Error setting future state for {}", (Object)this.name, (Object)e);
            }
            for (StateChangeListener stateChangeListener : stateChangeListeners) {
                this.fireStateChangedListener(newState, stateChangeListener);
            }
        });
    }

    private void fireStateChangedListener(T newState, StateChangeListener<T> stateChangeListener) {
        try {
            stateChangeListener.stateChanged(newState);
        }
        catch (Throwable e) {
            LOGGER.error("Error notifying state change listener for {}", (Object)this.name, (Object)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ListenableFuture<T> getStateChange(T currentState) {
        Preconditions.checkState((!Thread.holdsLock(this.lock) ? 1 : 0) != 0, (Object)"Cannot wait for state change while holding the lock");
        Objects.requireNonNull(currentState, "currentState is null");
        Object object = this.lock;
        synchronized (object) {
            if (!this.state.equals(currentState) || this.isTerminalState(this.state)) {
                return Futures.immediateFuture(this.state);
            }
            return this.futureStateChange.get().createNewListener();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addStateChangeListener(StateChangeListener<T> stateChangeListener) {
        Object currentState;
        Objects.requireNonNull(stateChangeListener, "stateChangeListener is null");
        Object object = this.lock;
        synchronized (object) {
            currentState = this.state;
            if (!this.isTerminalState(currentState)) {
                this.stateChangeListeners.add(stateChangeListener);
            }
        }
        this.safeExecute(() -> stateChangeListener.stateChanged(currentState));
    }

    @VisibleForTesting
    boolean isTerminalState(T state) {
        return this.terminalStates.contains(state);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    List<StateChangeListener<T>> getStateChangeListeners() {
        Object object = this.lock;
        synchronized (object) {
            return ImmutableList.copyOf(this.stateChangeListeners);
        }
    }

    public String toString() {
        return this.get().toString();
    }

    private void safeExecute(Runnable command) {
        try {
            this.executor.execute(command);
        }
        catch (RejectedExecutionException e) {
            if (this.executor instanceof ExecutorService && ((ExecutorService)this.executor).isShutdown()) {
                throw new RuntimeException("Server is shutting down", e);
            }
            throw e;
        }
    }

    public static interface StateChangeListener<T> {
        public void stateChanged(T var1);
    }
}

