/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.header.ConnectHeaders;
import org.apache.kafka.connect.header.Headers;
import org.apache.kafka.connect.mirror.MirrorSourceConnector;
import org.apache.kafka.connect.mirror.MirrorSourceMetrics;
import org.apache.kafka.connect.mirror.MirrorSourceTaskConfig;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.OffsetSyncWriter;
import org.apache.kafka.connect.mirror.ReplicationPolicy;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorSourceTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(MirrorSourceTask.class);
    private KafkaConsumer<byte[], byte[]> consumer;
    private String sourceClusterAlias;
    private Duration pollTimeout;
    private ReplicationPolicy replicationPolicy;
    private MirrorSourceMetrics metrics;
    private boolean stopping = false;
    private Semaphore consumerAccess;
    private OffsetSyncWriter offsetSyncWriter;

    public MirrorSourceTask() {
    }

    MirrorSourceTask(KafkaConsumer<byte[], byte[]> consumer, MirrorSourceMetrics metrics, String sourceClusterAlias, ReplicationPolicy replicationPolicy, OffsetSyncWriter offsetSyncWriter) {
        this.consumer = consumer;
        this.metrics = metrics;
        this.sourceClusterAlias = sourceClusterAlias;
        this.replicationPolicy = replicationPolicy;
        this.consumerAccess = new Semaphore(1);
        this.offsetSyncWriter = offsetSyncWriter;
    }

    public void start(Map<String, String> props) {
        MirrorSourceTaskConfig config = new MirrorSourceTaskConfig(props);
        this.consumerAccess = new Semaphore(1);
        this.sourceClusterAlias = config.sourceClusterAlias();
        this.metrics = config.metrics();
        this.pollTimeout = config.consumerPollTimeout();
        this.replicationPolicy = config.replicationPolicy();
        if (config.emitOffsetSyncsEnabled()) {
            this.offsetSyncWriter = new OffsetSyncWriter(config);
        }
        this.consumer = MirrorUtils.newConsumer(config.sourceConsumerConfig("replication-consumer"));
        Set<TopicPartition> taskTopicPartitions = config.taskTopicPartitions();
        this.initializeConsumer(taskTopicPartitions);
        log.info("{} replicating {} topic-partitions {}->{}: {}.", new Object[]{Thread.currentThread().getName(), taskTopicPartitions.size(), this.sourceClusterAlias, config.targetClusterAlias(), taskTopicPartitions});
    }

    public void commit() {
        if (this.offsetSyncWriter != null) {
            this.offsetSyncWriter.promoteDelayedOffsetSyncs();
            this.offsetSyncWriter.firePendingOffsetSyncs();
        }
    }

    public void stop() {
        long start = System.currentTimeMillis();
        this.stopping = true;
        this.consumer.wakeup();
        try {
            this.consumerAccess.acquire();
        }
        catch (InterruptedException e) {
            log.warn("Interrupted waiting for access to consumer. Will try closing anyway.");
        }
        Utils.closeQuietly(this.consumer, (String)"source consumer");
        Utils.closeQuietly((AutoCloseable)this.offsetSyncWriter, (String)"offset sync writer");
        Utils.closeQuietly((AutoCloseable)this.metrics, (String)"metrics");
        log.info("Stopping {} took {} ms.", (Object)Thread.currentThread().getName(), (Object)(System.currentTimeMillis() - start));
    }

    public String version() {
        return new MirrorSourceConnector().version();
    }

    public List<SourceRecord> poll() {
        List<SourceRecord> list;
        if (!this.consumerAccess.tryAcquire()) {
            return null;
        }
        if (this.stopping) {
            return null;
        }
        try {
            Object object;
            ConsumerRecords records = this.consumer.poll(this.pollTimeout);
            ArrayList<SourceRecord> sourceRecords = new ArrayList<SourceRecord>(records.count());
            for (ConsumerRecord record : records) {
                SourceRecord converted = this.convertRecord((ConsumerRecord<byte[], byte[]>)record);
                sourceRecords.add(converted);
                TopicPartition topicPartition = new TopicPartition(converted.topic(), converted.kafkaPartition().intValue());
                this.metrics.recordAge(topicPartition, System.currentTimeMillis() - record.timestamp());
                this.metrics.recordBytes(topicPartition, MirrorSourceTask.byteSize((byte[])record.value()));
            }
            if (sourceRecords.isEmpty()) {
                object = null;
                return object;
            }
            log.trace("Polled {} records from {}.", (Object)sourceRecords.size(), (Object)records.partitions());
            object = sourceRecords;
            return object;
        }
        catch (WakeupException e) {
            list = null;
            return list;
        }
        catch (KafkaException e) {
            log.warn("Failure during poll.", (Throwable)e);
            list = null;
            return list;
        }
        catch (Throwable e) {
            log.error("Failure during poll.", e);
            throw e;
        }
        finally {
            this.consumerAccess.release();
        }
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) {
        if (this.stopping) {
            return;
        }
        if (metadata == null) {
            log.debug("No RecordMetadata (source record was probably filtered out during transformation) -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        if (!metadata.hasOffset()) {
            log.error("RecordMetadata has no offset -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        TopicPartition topicPartition = new TopicPartition(record.topic(), record.kafkaPartition().intValue());
        long latency = System.currentTimeMillis() - record.timestamp();
        this.metrics.countRecord(topicPartition);
        this.metrics.replicationLatency(topicPartition, latency);
        if (this.offsetSyncWriter != null) {
            TopicPartition sourceTopicPartition = MirrorUtils.unwrapPartition(record.sourcePartition());
            long upstreamOffset = MirrorUtils.unwrapOffset(record.sourceOffset());
            long downstreamOffset = metadata.offset();
            this.offsetSyncWriter.maybeQueueOffsetSyncs(sourceTopicPartition, upstreamOffset, downstreamOffset);
            this.offsetSyncWriter.firePendingOffsetSyncs();
        }
    }

    private Map<TopicPartition, Long> loadOffsets(Set<TopicPartition> topicPartitions) {
        return topicPartitions.stream().collect(Collectors.toMap(x -> x, this::loadOffset));
    }

    private Long loadOffset(TopicPartition topicPartition) {
        Map<String, Object> wrappedPartition = MirrorUtils.wrapPartition(topicPartition, this.sourceClusterAlias);
        Map wrappedOffset = this.context.offsetStorageReader().offset(wrappedPartition);
        return MirrorUtils.unwrapOffset(wrappedOffset);
    }

    void initializeConsumer(Set<TopicPartition> taskTopicPartitions) {
        Map<TopicPartition, Long> topicPartitionOffsets = this.loadOffsets(taskTopicPartitions);
        this.consumer.assign(topicPartitionOffsets.keySet());
        log.info("Starting with {} previously uncommitted partitions.", (Object)topicPartitionOffsets.values().stream().filter(this::isUncommitted).count());
        topicPartitionOffsets.forEach((topicPartition, offset) -> {
            if (this.isUncommitted((Long)offset)) {
                log.trace("Skipping seeking offset for topicPartition: {}", topicPartition);
                return;
            }
            long nextOffsetToCommittedOffset = offset + 1L;
            log.trace("Seeking to offset {} for topicPartition: {}", (Object)nextOffsetToCommittedOffset, topicPartition);
            this.consumer.seek(topicPartition, nextOffsetToCommittedOffset);
        });
    }

    SourceRecord convertRecord(ConsumerRecord<byte[], byte[]> record) {
        String targetTopic = this.formatRemoteTopic(record.topic());
        Headers headers = this.convertHeaders(record);
        return new SourceRecord(MirrorUtils.wrapPartition(new TopicPartition(record.topic(), record.partition()), this.sourceClusterAlias), MirrorUtils.wrapOffset(record.offset()), targetTopic, Integer.valueOf(record.partition()), Schema.OPTIONAL_BYTES_SCHEMA, record.key(), Schema.BYTES_SCHEMA, record.value(), Long.valueOf(record.timestamp()), (Iterable)headers);
    }

    private Headers convertHeaders(ConsumerRecord<byte[], byte[]> record) {
        ConnectHeaders headers = new ConnectHeaders();
        for (Header header : record.headers()) {
            headers.addBytes(header.key(), header.value());
        }
        return headers;
    }

    private String formatRemoteTopic(String topic) {
        return this.replicationPolicy.formatRemoteTopic(this.sourceClusterAlias, topic);
    }

    private static int byteSize(byte[] bytes) {
        if (bytes == null) {
            return 0;
        }
        return bytes.length;
    }

    private boolean isUncommitted(Long offset) {
        return offset == null || offset < 0L;
    }
}

