/*!
 * \file
 * \brief Random Number generation core tests
 * \author Andy Panov
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2012  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <limits>
#include <cmath>
#include <iostream>
#include <fstream>
#include <vector>
#include <itpp/base/random.h>
#ifdef _OPENMP
//should work on both gcc & msvc (it is an omp standard requirement)
#include <omp.h>
#endif


//tests vectors are taken from dSFMT package (v 2.2)
//Here is the package license
/*
Copyright (c) 2007, 2008, 2009 Mutsuo Saito, Makoto Matsumoto
and Hiroshima University.
Copyright (c) 2011, 2002 Mutsuo Saito, Makoto Matsumoto, Hiroshima
University and The University of Tokyo.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of the Hiroshima University nor the names of
      its contributors may be used to endorse or promote products
      derived from this software without specific prior written
      permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

//init_gen_rand(0) generated randoms [1, 2)
double ref_result1[] = {
  1.030581026769374, 1.213140320067012, 1.299002525016001, 1.381138853044628,
  1.863488397063594, 1.133443440024236, 1.073465290616508, 1.924735636004535,
  1.642876178923654, 1.135658049473512, 1.614181131134505, 1.758658396495576,
  1.629675660963117, 1.280639430198043, 1.450236428147610, 1.493660282874514,
  1.814271115361793, 1.167373211634437, 1.029478262306546, 1.995645230878212,
  1.571796427913721, 1.820991494675444, 1.342215733241735, 1.623155152394482,
  1.770877090925340, 1.971604009986542, 1.291307503466583, 1.042482713804030,
  1.489892341518318, 1.846505153340657, 1.464614144455080, 1.722998990286586,
  1.261140462560469, 1.466608076171318, 1.284476538780147, 1.566267595424500,
  1.214877812942994, 1.814025145094890, 1.875364841982453, 1.420058799919090,
  1.185941915813226, 1.613435647553408, 1.261307303980314, 1.575227905977901,
  1.368593422560852, 1.178339827374882, 1.617026203567302, 1.605806966367909,
  1.688420982162701, 1.271452757436403, 1.380651074471749, 1.923694230059664,
  1.183689647854999, 1.813616738156070, 1.674316771173891, 1.388616443854270,
  1.112619958524579, 1.143443226388526, 1.044504544026019, 1.059792215140743,
  1.473318469170684, 1.907724040304600, 1.840641748107302, 1.880247376378763,
  1.662300761822745, 1.756077893716310, 1.794902408257739, 1.208089693203264,
  1.573346980275286, 1.343298418696786, 1.290031288673300, 1.785887855614908,
  1.469555312775962, 1.303565840269790, 1.821346312600028, 1.464630963746065,
  1.350918117062525, 1.608975946390805, 1.218770519593969, 1.553737984201091,
  1.799048445169658, 1.934409566880829, 1.518706598932114, 1.806743174028226,
  1.445770445550219, 1.853994985463463, 1.294940116346864, 1.846933188597791,
  1.585285299858689, 1.436399337803629, 1.984097327010455, 1.104286003167359,
  1.650974773825423, 1.509620107806257, 1.257788148487937, 1.343504084778931,
  1.949774691002465, 1.575826707171932, 1.334309761521545, 1.970145826389876,
  1.634179511150286, 1.117295679338137, 1.975634802490159, 1.876312964236097,
  1.588415777998304, 1.655714874553697, 1.333606641414210, 1.534065546092020,
  1.430491671060673, 1.948237416318885, 1.221039675791948, 1.002582148951994,
  1.154359196718931, 1.871051234557714, 1.608929525044202, 1.314753139712890,
  1.024921298185855, 1.269548213986646, 1.311905073508091, 1.820250418624237,
  1.530899807500864, 1.382415501216911, 1.723050693688422, 1.273273073610244,
  1.313899532538009, 1.788939639818993, 1.876299000001360, 1.991333199989678,
  1.991820254950511, 1.030433951689168, 1.676262472193616, 1.508094775671104,
  1.379478805615782, 1.669062560664250, 1.454294977515323, 1.058637649351106,
  1.762305416761906, 1.259329140543268, 1.069306898227167, 1.416672040871633,
  1.692817632828360, 1.325016179312241, 1.894413502990435, 1.065827451446854,
  1.648472088974056, 1.136729079291305, 1.577029131655685, 1.770630083651844,
  1.312969529843681, 1.120742660491037, 1.381749920757092, 1.532127565299400,
  1.430576333290474, 1.408675485986594, 1.664550700542242, 1.808244181927591,
  1.449745796589231, 1.561376218313360, 1.823412610420372, 1.175033367831775,
  1.302133406247658, 1.209386996910797, 1.958852403617822, 1.327413109121052,
  1.905326191931480, 1.243341689973321, 1.171964300933144, 1.947795532949568,
  1.406088608658097, 1.352799736126945, 1.038567042456604, 1.952366632464022,
  1.271894317271382, 1.276879063784723, 1.962232899932226, 1.635977123253224,
  1.427227475039741, 1.902767774021188, 1.348976205109293, 1.098551616320939,
  1.761303753087471, 1.725151135219154, 1.973647280723666, 1.438406762461431,
  1.976338460962363, 1.080691466989757, 1.302440271222677, 1.524443530330429,
  1.677973975664305, 1.899365828945253, 1.758064697654786, 1.664994321814212,
  1.702509799458075, 1.542285497404383, 1.583785005757146, 1.364775251860427,
  1.914911483830064, 1.492662121733521, 1.803292510850298, 1.913612505233413,
  1.990030445920955, 1.265653421547230, 1.371464451146616, 1.812272753750019,
  1.468314372775775, 1.962074516808028, 1.071993018163613, 1.362232923167970,
  1.286098054243162, 1.166494117866721, 1.190627308126986, 1.336522203701874,
  1.589002579609417, 1.989085299207313, 1.886248617042378, 1.494809620088929,
  1.210511688798404, 1.373900882339293, 1.529044652411248, 1.599366524611929,
  1.436552769988409, 1.577285005915829, 1.688182285296122, 1.277637471607439,
  1.929554385968616, 1.823441682648089, 1.922013798503081, 1.645761293216693,
  1.431349251849024, 1.762959265242032, 1.878537838979557, 1.999467237190877,
  1.485220104894368, 1.716411480065391, 1.412097187018955, 1.552070069949518,
  1.892433293954831, 1.503681065768480, 1.856696961604681, 1.652070478210699,
  1.941264820695600, 1.747443060523082, 1.340963648345302, 1.115605969659878,
  1.350907250724354, 1.815702983867849, 1.598890428370514, 1.439048661943484,
  1.517663378502000, 1.304964768398997, 1.079026972256675, 1.524811537337526,
  1.316547675188607, 1.138964396377513, 1.662996685355875, 1.267094641341935,
  1.501239512928825, 1.102282578729245, 1.681064147709950, 1.806571233091203,
  1.723576229776429, 1.736972959477229, 1.912237952025255, 1.660774311421941,
  1.025816042908814, 1.438547835995945, 1.739013210613549, 1.413224917267124,
  1.358475319974022, 1.358765432047162, 1.593140545500734, 1.172245654090473,
  1.930726901246793, 1.162761062259565, 1.318549211468092, 1.092783981301198,
  1.555764474221963, 1.708252060788802, 1.269806348212832, 1.852955157375948,
  1.907379745865761, 1.486699553976457, 1.515744472393222, 1.620128750267234,
  1.622430346507121, 1.668132516832666, 1.796788360374040, 1.636602348582785,
  1.773593519549434, 1.258248394226789, 1.246140312082405, 1.015163448535963,
  1.521503953242567, 1.140634382438487, 1.702118160723025, 1.132078836187980,
  1.963663308022001, 1.452227189409805, 1.832632887982046, 1.261616333794024,
  1.803214901972612, 1.388570383844696, 1.643765598586203, 1.619544270246625,
  1.307755937472623, 1.857004438119476, 1.398636660692930, 1.556031209947539,
  1.641338721048474, 1.115939103561434, 1.654221123345289, 1.150482842848706,
  1.494319794557959, 1.788401632203792, 1.025831928414729, 1.042464682154734,
  1.886776958738527, 1.968746864493441, 1.159635330475675, 1.956557704773203,
  1.284243453753799, 1.572925045526015, 1.859321130727941, 1.005189956275669,
  1.005337392745185, 1.442294781323312, 1.890812262600537, 1.206329358583467,
  1.026098286977865, 1.047660975111641, 1.037196173307690, 1.743350801362402,
  1.274336569674040, 1.719733191271306, 1.241944533170590, 1.761242705869024,
  1.087057719955586, 1.923433325400596, 1.769554882850761, 1.422355508046144,
  1.335735282689950, 1.969166433701194, 1.309113579529401, 1.934488263782031,
  1.363050605738272, 1.205012399069315, 1.702636620368798, 1.377299901239738,
  1.082603967707894, 1.707678198781335, 1.357839171792981, 1.249151205817995,
  1.414407072442705, 1.758876753195856, 1.947897139275915, 1.873737405944571,
  1.864011982004539, 1.197733232912529, 1.708750747026161, 1.584026807407247,
  1.464232596280593, 1.910020147544669, 1.107379834907888, 1.431311422166039,
  1.656031749095368, 1.583060053927119, 1.338823782349183, 1.675841451609077,
  1.863125002352507, 1.383321945211853, 1.437552865439453, 1.323329267004908,
  1.510393439678810, 1.381858897252844, 1.488904172419540, 1.703038148781844,
  1.363689874508434, 1.782902379399008, 1.207151750281740, 1.134081954739325,
  1.786230025908168, 1.301807943529430, 1.818037084625490, 1.954083924684191,
  1.503136817574214, 1.166174793570306, 1.905851898350567, 1.861611595448786,
  1.912073551209336, 1.311539667277574, 1.348138517947969, 1.835216223115592,
  1.848638059772834, 1.508211065551719, 1.501558584290350, 1.841635694216024,
  1.695119173143315, 1.454484117280340, 1.625646404371770, 1.119869639433384,
  1.444195622035266, 1.399038475609640, 1.911871066122697, 1.393715521875472,
  1.634940751730062, 1.299965953553803, 1.625354065027734, 1.880393059283658,
  1.690675478017034, 1.628589134880479, 1.012241730322596, 1.151619785119961,
  1.951158889711496, 1.797539238310482, 1.755218167276459, 1.713478006991327,
  1.571668224641071, 1.319684419283076, 1.470800535782670, 1.106150064396913,
  1.759426082965527, 1.413902064615626, 1.663209862298114, 1.688656611986246,
  1.561766088174504, 1.021593733617964, 1.104216171759727, 1.056934446307648,
  1.131101785914085, 1.025747434603107, 1.653089659161949, 1.862010102275871,
  1.700392592888141, 1.749844513072156, 1.600475707472835, 1.797824285171013,
  1.303262789973822, 1.833983051083715, 1.706436841851279, 1.583152584413344,
  1.157186002470311, 1.160819822241010, 1.212754944300861, 1.963053526130375,
  1.268811286389321, 1.374828807938443, 1.049479080410400, 1.973869788889314,
  1.280288819687655, 1.193231050947952, 1.306156625366250, 1.244778108696800,
  1.731164172106588, 1.765672206167307, 1.617163972329793, 1.333068311058873,
  1.230571993026005, 1.225697451543153, 1.277811874136488, 1.572508333797608,
  1.556415237569166, 1.209965265731652, 1.934900416408164, 1.370051485789686,
  1.026053486793906, 1.090498102405539, 1.516231184525802, 1.080879023417522,
  1.616056345268405, 1.414971598053735, 1.813169501790841, 1.946113087783638,
  1.550874962654984, 1.095276777050610, 1.022240506381006, 1.349953863208443,
  1.639335057225891, 1.086999372715814, 1.198125757163164, 1.358872097979379,
  1.864767925147218, 1.425774441879807, 1.324356243811146, 1.013779032909748,
  1.877970176001620, 1.017769324762142, 1.517328996674653, 1.156791735454128,
  1.820771116348883, 1.209713056264071, 1.854832025810033, 1.626515379727837,
  1.072074225561507, 1.621286465674130, 1.549902343511516, 1.266113813183351,
  1.450731461629206, 1.023903615692273, 1.141802126238102, 1.478789349904744,
  1.030261249106096, 1.744520411596375, 1.610339075504574, 1.734036515785284,
  1.445021690251891, 1.067695047319750, 1.044588489496833, 1.078902049240712,
  1.775656376455273, 1.529010846255880, 1.684639314731656, 1.869078008870465,
  1.196659282984713, 1.055877083218375, 1.343311406572803, 1.657804323039657,
  1.931310179843877, 1.876531128866294, 1.953312528893538, 1.751295176436919,
  1.990635408763924, 1.747960800049019, 1.625876628099381, 1.950163235823662,
  1.461443118782703, 1.075928147908412, 1.092143870551425, 1.530725871270558,
  1.134721738077522, 1.006593825427538, 1.510662628484829, 1.102117778241225,
  1.237414790616429, 1.415898757276402, 1.227818931317890, 1.803733837180969,
  1.039835919301454, 1.945637370177811, 1.014985550164521, 1.662360752310358,
  1.389442153375327, 1.067707198772765, 1.752954274484631, 1.712429083915912,
  1.798210585469825, 1.644104449048200, 1.668431996831141, 1.786974747468935,
  1.411459450430814, 1.354534709965009, 1.755820242124135, 1.186816760748136,
  1.711332994678620, 1.903597963775769, 1.127695794052451, 1.055764745873512,
  1.462173219563238, 1.601988414252792, 1.542788005387959, 1.470016225539294,
  1.053250417579233, 1.919128040049399, 1.920409133273121, 1.930196071775802,
  1.093941301487821, 1.907077053805924, 1.180006330527884, 1.855787153756457,
  1.038787604610960, 1.603390528893462, 1.675699800641335, 1.436579268097476,
  1.695216721991263, 1.667921392643477, 1.298225569286973, 1.176659205584008,
  1.484688184402969, 1.736977199998048, 1.527346818010219, 1.124337166052271,
  1.004126214409010, 1.257995351445688, 1.048750698690273, 1.600687893284987,
  1.771330853717698, 1.781338029764260, 1.791045259533474, 1.218240073554028,
  1.404476462748409, 1.263776254264821, 1.771506489056091, 1.492535106857984,
  1.942560639598830, 1.880445735782105, 1.987365393774292, 1.624806632834506,
  1.360289989229111, 1.259447098681650, 1.565201597574085, 1.724405238661507,
  1.480135931789360, 1.613728648660302, 1.148462875077473, 1.082990694586929,
  1.791549713090436, 1.011697309037590, 1.051330287711138, 1.404529653735530,
  1.255076827666297, 1.465354319402313, 1.959722467095358, 1.667465002487598,
  1.059786656048662, 1.145589007250259, 1.111653746143364, 1.118097303976153,
  1.274790361215896, 1.061022554596977, 1.650762458048159, 1.426666443801441,
  1.643025168116494, 1.881525522016726, 1.331922961270956, 1.141320506705509,
  1.835459356205120, 1.604755953946371, 1.126481102674482, 1.345361030653902,
  1.552930483800284, 1.651060315989797, 1.281924176836940, 1.478282162483309,
  1.089522082067808, 1.514671692662497, 1.062032369569581, 1.220072855001427,
  1.335222242385036, 1.917688560760201, 1.881563679176478, 1.061345187966617,
  1.932816485102730, 1.831378583403447, 1.018053903667606, 1.803752606555742,
  1.744151146745064, 1.823935602019249, 1.549411228789844, 1.852085525549937,
  1.268910953917921, 1.398103908872195, 1.115463052648461, 1.134166374832642,
  1.059662865049822, 1.913385314672057, 1.817573645089763, 1.543472539894835,
  1.879411496379644, 1.187052469310417, 1.327060124683171, 1.201797766005763,
  1.585708496011359, 1.726441950212828, 1.765366139479803, 1.575105655795392,
  1.161836204689848, 1.177060808947556, 1.991643941775680, 1.483304774937826,
  1.299878727524718, 1.860512786972423, 1.055049090865626, 1.843398581472891,
  1.092910752402670, 1.944321623196737, 1.663494470271249, 1.922888589107380,
  1.203337506107050, 1.673695668302879, 1.453053811175328, 1.005089626720857,
  1.795346999949530, 1.229821542926916, 1.748377783144588, 1.321996885703262,
  1.309371806850785, 1.716138869778725, 1.135666913050759, 1.338269469982098,
  1.970855876728981, 1.853517049886400, 1.057177758132396, 1.248940788315584,
  1.791020744082269, 1.081574247946067, 1.730996177586578, 1.009526675464715,
  1.009798100718231, 1.881102727076643, 1.670032923753426, 1.864089798899085,
  1.947800326838745, 1.525753676291635, 1.481044314974777, 1.045650816913446,
  1.343050229590432, 1.254467660379006, 1.942791560918173, 1.712818820358553,
  1.365835692787808, 1.840681238589795, 1.226240189108953, 1.322696502726162,
  1.519948924493175, 1.902086636313264, 1.831961607099442, 1.656813895374856,
  1.580562626129614, 1.628706392122500, 1.709923293711780, 1.397619640553994,
  1.955521961038447, 1.131582790546282, 1.120774954775631, 1.643383921264857,
  1.428256924538280, 1.629234961965852, 1.481973671272142, 1.355168703906496,
  1.381360123066666, 1.297353474383750, 1.998779984895075, 1.299540149195283,
  1.554953631008684, 1.022499550534725, 1.267538332848977, 1.999635534064931,
  1.216908438689689, 1.125330423218031, 1.521094226167544, 1.352444820153410,
  1.891355866547355, 1.237188250644441, 1.829163148453760, 1.010975705160555,
  1.389099307599609, 1.166572697139759, 1.209619877947629, 1.957432819952394,
  1.171713118207805, 1.995544396463686, 1.113738650685683, 1.954952493800097,
  1.309046349979008, 1.125405858742736, 1.334059551241411, 1.618705235720444,
  1.231177186143469, 1.553727846971735, 1.178887689593680, 1.264866781514123,
  1.091971065285239, 1.056441862383996, 1.283527024623163, 1.834607590365417,
  1.962802242863336, 1.228730340204497, 1.842991683633420, 1.697545988525402,
  1.844710792546847, 1.192417542034665, 1.697508607262110, 1.784896632338091,
  1.034112834139585, 1.422033286751813, 1.453652054620216, 1.058379823639922,
  1.507638941358602, 1.047993066417010, 1.516397461578060, 1.302372136540590,
  1.461998564058737, 1.554611626737750, 1.134122374348743, 1.667716941372084,
  1.433151414564393, 1.319857533959847, 1.583420071116494, 1.478250465560727,
  1.466784271564917, 1.923843580047532, 1.176048670365018, 1.518583880443583,
  1.126284343690593, 1.309458927538403, 1.254116920997212, 1.221286762203130,
  1.314794812269592, 1.865355356312181, 1.875368141725648, 1.478554967746866,
  1.289949440091931, 1.240447745313523, 1.101700136534686, 1.332573248950360,
  1.167135678363469, 1.604936747884104, 1.828477968610324, 1.981787275210698,
  1.528961708093550, 1.844833688431583, 1.292908212260039, 1.526421493343533,
  1.477684242274037, 1.520192575343653, 1.407146775494489, 1.433258876432486,
  1.140499612461638, 1.224429147839875, 1.413739425285328, 1.939104296647277,
  1.669514052983343, 1.732909897181110, 1.751457852668096, 1.242198740016763,
  1.909522621661234, 1.248802188164318, 1.785821368840928, 1.444035896141739,
  1.408015292232671, 1.097220772250296, 1.433573415928994, 1.019849701076716,
  1.192183652047805, 1.942228730880240, 1.511971120822260, 1.167847411636749,
  1.424386751141384, 1.999535966473089, 1.256446120225825, 1.758859587241302,
  1.554606274137138, 1.141112413898596, 1.872291378873514, 1.893151469654739,
  1.869646558937494, 1.826227167199755, 1.804544311254211, 1.730661202511795,
  1.226769815903244, 1.679838430928674, 1.643941207337068, 1.322059748794036,
  1.045053637669191, 1.309404672920857, 1.331474945197764, 1.790509417887939,
  1.263083376723605, 1.245939409206071, 1.759660309891394, 1.281800922487132,
  1.712213197965571, 1.803213756378482, 1.717270512776782, 1.385372857246577,
  1.255945382268436, 1.564636086727342, 1.644313176830236, 1.176523282489581,
  1.334422402615048, 1.208067092977696, 1.149552540117835, 1.771501288135706,
  1.564055713684924, 1.621614271733339, 1.081066576966760, 1.495853964383299,
  1.219063833026128, 1.084142314244871, 1.549237225541902, 1.936776127604372,
  1.845512189258520, 1.005975975876582, 1.811498259332091, 1.635990011669673,
  1.536881182554052, 1.443297379586476, 1.886119584133706, 1.275449488687871,
  1.670965527087923, 1.987232750305201, 1.544406415554266, 1.725582465417778,
  1.799340284160104, 1.608735841735920, 1.264891652333546, 1.808533254198986,
  1.974563992723363, 1.673825352257849, 1.559378666798180, 1.815513818850944,
  1.679215902745112, 1.846384704667485, 1.353621521988754, 1.236648508090431,
  1.002104116734581, 1.053298883217048, 1.430242739641769, 1.420261924778031,
  1.783120431598888, 1.331408790925515, 1.781593450829028, 1.139278556251319,
  1.925682266142099, 1.138131587567467, 1.474662993172124, 1.319063477600368,
  1.536365678407554, 1.129922971629872, 1.780938176769717, 1.257945861972239,
  1.206107632090796, 1.055490753863072, 1.161054816125216, 1.058802624223267,
  1.756641825929552, 1.754378196150022, 1.827313268250450, 1.951406696895395,
  1.315659554078758, 1.926535624123573, 1.711614142494693, 1.408317616914680,
  1.167503504129387, 1.221579231246995, 1.772499449844828, 1.229590547241117,
  1.523859240135010, 1.750638838700595, 1.294969757722743, 1.239218469359550,
  1.895331049984825, 1.351848818389126, 1.953628356200589, 1.556987376063896,
  1.742382503539244, 1.831314773712437, 1.931187779890391, 1.627064275586785,
  1.092521265403294, 1.056269664216268, 1.697419776942093, 1.020621151020278,
  1.016460701790075, 1.663349935983337, 1.700240650962897, 1.294295439244268,
  1.217283432848958, 1.724633019876359, 1.611555526017567, 1.976769804845470,
  1.322482963420171, 1.119851357883444, 1.188940717720834, 1.639649300982932,
  1.430594405952054, 1.528763701984112, 1.677238553046157, 1.804257644847276,
  1.373695667911955, 1.041607859915277, 1.292586152868963, 1.204343281283561,
  1.219648415953772, 1.118298775504544, 1.489325174543727, 1.223311365895364,
  1.215428634537803, 1.544464527855300, 1.211873174969368, 1.800882969641407,
  1.395620323924559, 1.513764352278001, 1.855296462052718, 1.311627987537891,
  1.106545691943606, 1.006292260028909, 1.266506021169479, 1.061150534120227,
  1.552364586842264, 1.302709575585848, 1.821775657431182, 1.391750759396551,
  1.365952489392163, 1.337000387140509, 1.740591934294086, 1.185662060689487,
  1.200293174238620, 1.390126083634700, 1.802185187529872, 1.091847733098113
};
size_t test1_dataset_size = sizeof(ref_result1) / sizeof(double);

//init_gen_rand(5) generated randoms (0, 1]
double ref_result2[] = {
  0.576676137904379, 0.994304310329450, 0.127959185148864, 0.178972277952344,
  0.501233068055063, 0.859840653012468, 0.546161779037789, 0.805380602145160,
  0.381324178276622, 0.229028865043289, 0.666794531244068, 0.821723842976139
};
size_t test2_dataset_size = sizeof(ref_result2) / sizeof(double);

using namespace itpp;

int main()
{
  bool test1_failed = false;
  bool test2_failed = false;
  bool test3_failed = false;
  //set test tolerance;
  double tol = 5 * std::numeric_limits<double>::epsilon();
  //verify correcness of RNG core algorithms first
  {
    //genrand_close1_open2 test with seed = 0
    RNG_reset(0);
    Random_Generator rng;

    for(size_t i = 0; i < test1_dataset_size; ++i) {
      double r = rng.genrand_close1_open2();
      if(std::abs(r - ref_result1[i]) > tol) {
        test1_failed = true;
      }
    }
  }

  {
    //genrand_open_close test with seed = 5
    RNG_reset(5);
    Random_Generator rng;
    for(size_t i = 0; i < test2_dataset_size; ++i) {
      double r = rng.genrand_open_close();
      if(std::abs(r - ref_result2[i]) > tol) {
        test2_failed = true;
      }
    }
  }

#ifdef _OPENMP
  //set number of threads in the team to the maximum available value
  const int threads_cnt = omp_get_max_threads();
  omp_set_num_threads(threads_cnt);
#else
const int threads_cnt = 1;
#ifndef _MSC_VER
#pragma message "no OMP support"
#else
#pragma message ("no OMP support")
#endif
#endif
  //repeat test 1 in multithreaded environment
  //(repeat 3 times for each thread to test a possible clashes on shared data).
  std::vector<int> err_counters(threads_cnt);
  #pragma omp parallel
  {
    #pragma omp for
    for(int j = 0; j < threads_cnt; ++j) {
      //genrand_close1_open2 test with seed = 0
      RNG_reset(0);
      Random_Generator rng;
      for(size_t i = 0; i < test1_dataset_size; ++i) {
        double r = rng.genrand_close1_open2();
        if(std::abs(r - ref_result1[i]) > tol) {
          //can't use gtest macro here, omp does not allow to execute the code with flow breaks in parallel loop iterations
          err_counters[j]++;
        }
      }

      RNG_reset(0);
      for(size_t i = 0; i < test1_dataset_size; ++i) {
        double r = rng.genrand_close1_open2();
        if(std::abs(r - ref_result1[i]) > tol) {
          //can't use gtest macro here, omp does not allow to execute the code with flow breaks in parallel loop iterations
          err_counters[j]++;
        }
      }

      RNG_reset(0);
      for(size_t i = 0; i < test1_dataset_size; ++i) {
        double r = rng.genrand_close1_open2();
        if(std::abs(r - ref_result1[i]) > tol) {
          //can't use gtest macro here, omp does not allow to execute the code with flow breaks in parallel loop iterations
          err_counters[j]++;
        }
      }
    }
  }
  //check results when single-threaded again
  for(int j = 0; j < threads_cnt; ++j) {
    if(err_counters[j] != 0) {
      test3_failed = true;
    }
  }

  //output results
  std::cout << "-----------------------------------------" << std::endl;
  std::cout << "RNG core tests:" << std::endl;
  std::cout << "-----------------------------------------" << std::endl;
  std::cout << "genrand_close1_open2() test: " << (test1_failed ? "FAILED" : "PASSED") << std::endl;
  std::cout << "genrand_open_close() test:   " << (test2_failed ? "FAILED" : "PASSED") << std::endl;
  std::cout << "multithreading test:         " << (test3_failed ? "FAILED" : "PASSED") << std::endl;
  std::cout << "-----------------------------------------" << std::endl;

  return 0;
}
