/* -------------------------------------------------------------------------- *
 *                                   OpenMM                                   *
 * -------------------------------------------------------------------------- *
 * This is part of the OpenMM molecular simulation toolkit originating from   *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org.               *
 *                                                                            *
 * Portions copyright (c) 2019 Stanford University and the Authors.           *
 * Authors: Peter Eastman                                                     *
 * Contributors:                                                              *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS, CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,    *
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR      *
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE  *
 * USE OR OTHER DEALINGS IN THE SOFTWARE.                                     *
 * -------------------------------------------------------------------------- */

/**
 * This tests the CUDA implementation of HippoNonbondedForce.
 */

#include "openmm/internal/AssertionUtilities.h"
#include "openmm/Context.h"
#include "OpenMMAmoeba.h"
#include "openmm/System.h"
#include "openmm/VerletIntegrator.h"
#include <iostream>
#include <vector>

using namespace OpenMM;
using namespace std;

extern "C" void registerAmoebaCudaKernelFactories();

double forceTol, energyTol, consistencyTol;

void buildWaterSystem(System& system, int numWaters, HippoNonbondedForce* hippo) {
    system.addForce(hippo);
    hippo->setExtrapolationCoefficients({0.042, 0.635, 0.414});
    for (int mol = 0; mol < numWaters; mol++) {
        system.addParticle(15.995);
        system.addParticle(1.008);
        system.addParticle(1.008);
        double bohr = 0.52917720859;
        double ds = 0.1*bohr;
        double qs = 0.01*bohr*bohr/3.0;
        double c6s = sqrt(4.184)*0.001;
        double ps = sqrt(4.184*0.1);
        hippo->addParticle(-0.38280, {0.0, 0.0, ds*0.05477}, {qs*0.69866, 0.0, 0.0, 0.0, qs*-0.60471, 0.0, 0.0, 0.0, qs*-0.09395}, 6.0,
                    10*4.7075, 4.184*1326.0, 10*40.0, c6s*18.7737, ps*2.7104, -2.4233, 10*4.3097,
                    0.001*0.795, HippoNonbondedForce::Bisector, 3*mol+1, 3*mol+2, -1);
        hippo->addParticle(0.19140, {0.0, 0.0, ds*-0.20097}, {qs*0.03881, 0.0, 0.0, 0.0, qs*0.02214, 0.0, 0.0, 0.0, qs*-0.06095}, 1.0,
                    10*4.7909, 0.0, 10*3.5582, c6s*4.5670, ps*2.0037, -0.8086, 10*4.6450,
                    0.001*0.341, HippoNonbondedForce::ZThenX, 3*mol, 3*mol+2, -1);
        hippo->addParticle(0.19140, {0.0, 0.0, ds*-0.20097}, {qs*0.03881, 0.0, 0.0, 0.0, qs*0.02214, 0.0, 0.0, 0.0, qs*-0.06095}, 1.0,
                    10*4.7909, 0.0, 10*3.5582, c6s*4.5670, ps*2.0037, -0.8086, 10*4.6450,
                    0.001*0.341, HippoNonbondedForce::ZThenX, 3*mol, 3*mol+1, -1);
        hippo->addException(3*mol, 3*mol+1, 0.0, 0.0, 0.2, 0.0, 0.0, 0.0);
        hippo->addException(3*mol, 3*mol+2, 0.0, 0.0, 0.2, 0.0, 0.0, 0.0);
        hippo->addException(3*mol+1, 3*mol+2, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0);
    }
}

void checkForceEnergyConsistency(Context& context) {
    // Find the direction and magnitude of the force.

    State state = context.getState(State::Positions | State::Forces | State::Energy);
    int numParticles = context.getSystem().getNumParticles();
    double norm = 0.0;
    for (int i = 0; i < numParticles; ++i) {
        Vec3 f = state.getForces()[i];
        norm += f[0]*f[0] + f[1]*f[1] + f[2]*f[2];
    }
    norm = std::sqrt(norm);

    // Take a small step in the direction of the force and see whether the potential energy changes by the expected amount.

    const double delta = 1e-3;
    double step = 0.5*delta/norm;
    vector<Vec3> positions2(numParticles), positions3(numParticles);
    for (int i = 0; i < numParticles; ++i) {
        Vec3 p = state.getPositions()[i];
        Vec3 f = state.getForces()[i];
        positions2[i] = p-f*step;
        positions3[i] = p+f*step;
    }
    context.setPositions(positions2);
    State state2 = context.getState(State::Energy);
    context.setPositions(positions3);
    State state3 = context.getState(State::Energy);
    ASSERT_EQUAL_TOL(state2.getPotentialEnergy()-state3.getPotentialEnergy(), norm*delta, consistencyTol)
}

void testWaterDimer() {
    System system;
    HippoNonbondedForce* hippo = new HippoNonbondedForce();
    buildWaterSystem(system, 2, hippo);
    VerletIntegrator integrator(0.001);
    Context context(system, integrator, Platform::getPlatformByName("CUDA"));
    vector<Vec3> positions = {
        0.1*Vec3(1.505434, 0.0, -0.065656),
        0.1*Vec3(0.553912, 0.0, 0.057710),
        0.1*Vec3(1.907155, 0.0, 0.801980),
        0.1*Vec3(-1.436029, 0.0, 0.060505),
        0.1*Vec3(-1.781197, 0.772272, -0.388976),
        0.1*Vec3(-1.781197, -0.772272, -0.388976)
    };
    context.setPositions(positions);
    State state = context.getState(State::Energy | State::Forces);

    // Compare the forces and energy to reference values computed with Tinker.

    ASSERT_EQUAL_TOL(-18.393623712669680, state.getPotentialEnergy(), energyTol);
    vector<Vec3> expectedForces = {
        Vec3(-162.94090034728887, 0.0, 35.06615691195519),
        Vec3(127.50063696213348, 0.0, -46.51857483822334),
        Vec3(39.59601328153432, 0.0, 11.805509637931072),
        Vec3(-73.52341534248339, 0.0, -92.08855312751808),
        Vec3(34.68383272305204, -26.35219958830841, 45.867730707927564),
        Vec3(34.68383272305204, 26.35219958830841, 45.867730707927564),
    };
    for (int i = 0; i < system.getNumParticles(); i++)
        ASSERT_EQUAL_VEC(-expectedForces[i], state.getForces()[i], forceTol);

    // Compare the permanent dipoles to reference values computed with Tinker.

    vector<Vec3> expectedLabDipoles = {
        Vec3(-1.3999971343167967e-3, 0.0, 2.5377493339976591e-3),
        Vec3(-1.0546602554698980e-2, 0.0, 1.3673800193405874e-3),
        Vec3(4.4683063920526751e-3, 0.0, 9.6506368469037353e-3),
        Vec3(-1.7652415043849052e-3, 0.0, -2.2987140077655862e-3),
        Vec3(-3.8321399176713630e-3, 8.5739534328208283e-3, -4.9902484654859164e-3),
        Vec3(-3.8321399176713630e-3, -8.5739534328208283e-3, -4.9902484654859164e-3)
    };
    vector<Vec3> labDipoles;
    hippo->getLabFramePermanentDipoles(context, labDipoles);
    for (int i = 0; i < system.getNumParticles(); i++)
        ASSERT_EQUAL_VEC(expectedLabDipoles[i], labDipoles[i], 1e-5);

    // Compare the induced dipoles to reference values computed with Tinker.

    vector<Vec3> expectedInducedDipoles = {
        Vec3(-2.6946897397455059e-3, 0.0, 5.7283370317562626e-4),
        Vec3(-3.4687164828218306e-3, 0.0, 2.3558084682622801e-4),
        Vec3(-8.3759103839743578e-4, 0.0, -4.0453541836814088e-4),
        Vec3(-4.7635598636156719e-3, 0.0, -6.2133742026147971e-4),
        Vec3(-6.3777660942845697e-4, 5.2439508203868156e-4, -4.3219965383009991e-4),
        Vec3(-6.3777660942845697e-4, -5.2439508203868156e-4, -4.3219965383009991e-4)
    };
    vector<Vec3> inducedDipoles;
    hippo->getInducedDipoles(context, inducedDipoles);
    for (int i = 0; i < system.getNumParticles(); i++)
        ASSERT_EQUAL_VEC(expectedInducedDipoles[i], inducedDipoles[i], 1e-5);

    // Check for consistency between forces and energy.

    checkForceEnergyConsistency(context);
}

void testWaterBox() {
    System system;
    HippoNonbondedForce* hippo = new HippoNonbondedForce();
    buildWaterSystem(system, 216, hippo);
    system.setDefaultPeriodicBoxVectors(Vec3(1.8643, 0, 0), Vec3(0, 1.8643, 0), Vec3(0, 0, 1.8643));
    hippo->setNonbondedMethod(HippoNonbondedForce::PME);
    hippo->setCutoffDistance(0.7);
    hippo->setSwitchingDistance(0.6);
    hippo->setPMEParameters(3.85037, 24, 24, 24);
    hippo->setDPMEParameters(3.85037, 20, 20, 20);
    VerletIntegrator integrator(0.001);
    Context context(system, integrator, Platform::getPlatformByName("CUDA"));
    double alpha;
    int nx, ny, nz;
    hippo->getPMEParametersInContext(context, alpha, nx, ny, nz);
    ASSERT_EQUAL_TOL(alpha, 3.85037, 1e-5);
    ASSERT_EQUAL(24, nx);
    ASSERT_EQUAL(24, ny);
    ASSERT_EQUAL(24, nz);
    hippo->getDPMEParametersInContext(context, alpha, nx, ny, nz);
    ASSERT_EQUAL_TOL(alpha, 3.85037, 1e-5);
    ASSERT_EQUAL(20, nx);
    ASSERT_EQUAL(20, ny);
    ASSERT_EQUAL(20, nz);
    vector<Vec3> positions = {
        Vec3(0.8679662000000001, 0.7087692, -0.0696862),
        Vec3(0.7809455000000001, 0.6755792, -0.03822590000000001),
        Vec3(0.8722232000000001, 0.6814243000000001, -0.1617561),
        Vec3(-0.0117313, 0.8244446999999999, 0.6837616),
        Vec3(0.021689200000000002, 0.7895445, 0.6050027),
        Vec3(0.0444268, 0.7826013, 0.7530196),
        Vec3(0.8379057, -0.0092611, 0.6814631000000001),
        Vec3(0.9340423, 0.0098069, 0.6734062000000001),
        Vec3(0.7939619000000001, 0.0573676, 0.6269838000000001),
        Vec3(0.6589952000000001, 0.1844323, -0.6923167000000001),
        Vec3(0.5885429000000001, 0.2402305, -0.6717934),
        Vec3(0.6181533, 0.10627470000000001, -0.7273678000000001),
        Vec3(0.7146600000000001, 0.5753582, 0.2331517),
        Vec3(0.6368123, 0.6126035000000001, 0.2862678),
        Vec3(0.7025018000000001, 0.6294645000000001, 0.15181960000000003),
        Vec3(-0.24265810000000002, -0.8504195, -0.25048339999999997),
        Vec3(-0.16920630000000003, -0.8368252, -0.3058292),
        Vec3(-0.27932070000000003, -0.7602469000000001, -0.2403097),
        Vec3(-0.8038375000000001, -0.3605589, 0.23036910000000002),
        Vec3(-0.8113753000000001, -0.42481270000000004, 0.30184940000000005),
        Vec3(-0.7619392, -0.2863004, 0.2709622),
        Vec3(-0.1480631, 0.8244085000000001, -0.8272214999999999),
        Vec3(-0.2090204, 0.8687978, -0.7676996),
        Vec3(-0.0700878, 0.8823325000000001, -0.8322213),
        Vec3(-0.37419620000000003, -0.277783, -0.2326319),
        Vec3(-0.4620456, -0.2444778, -0.23905189999999998),
        Vec3(-0.37289210000000006, -0.3160952, -0.1433101),
        Vec3(-0.6467812000000001, -0.5265942, 0.0408263),
        Vec3(-0.6585076000000001, -0.4796537, -0.0413008),
        Vec3(-0.7021252, -0.4752640000000001, 0.10079579999999999),
        Vec3(0.9273577, 0.5342431000000001, 0.405546),
        Vec3(0.8645939, 0.54665, 0.4837640000000001),
        Vec3(0.8741774, 0.5686149, 0.330282),
        Vec3(0.183016, 0.42767310000000003, -0.6993499000000001),
        Vec3(0.17032750000000002, 0.5223773, -0.6703852000000001),
        Vec3(0.24081130000000003, 0.38534470000000004, -0.6383339000000001),
        Vec3(0.1964382, 0.6832988000000001, 0.8373101000000001),
        Vec3(0.24961350000000004, 0.7215781, 0.9056298),
        Vec3(0.18112830000000002, 0.5905846, 0.8573539000000001),
        Vec3(0.5405568, 0.43889940000000005, 0.7932737000000001),
        Vec3(0.553738, 0.3438955, 0.7781311),
        Vec3(0.47551560000000004, 0.4734908, 0.7290617),
        Vec3(0.32299980000000006, 0.29284170000000004, -0.109065),
        Vec3(0.39310900000000004, 0.3198265, -0.1702769),
        Vec3(0.30044380000000004, 0.37089690000000003, -0.0493266),
        Vec3(0.74624, 0.5262829, 0.613117),
        Vec3(0.8025650000000001, 0.5588493, 0.688177),
        Vec3(0.6935076, 0.4455766, 0.6325405000000001),
        Vec3(-0.8864042, 0.7023845, -0.6659632000000001),
        Vec3(-0.8370939000000001, 0.6372557, -0.7141895),
        Vec3(-0.9489800000000002, 0.7315873, -0.7293125),
        Vec3(-0.45262990000000003, 0.3549989, 0.8030031),
        Vec3(-0.516977, 0.42241270000000003, 0.7731323),
        Vec3(-0.42620959999999997, 0.3801892, 0.8884875),
        Vec3(0.48232860000000005, -0.1386218, 0.40384640000000005),
        Vec3(0.5493155, -0.0669965, 0.4013081),
        Vec3(0.4187463, -0.1101832, 0.33384330000000006),
        Vec3(-0.215115, -0.201706, -0.8593685000000001),
        Vec3(-0.24757090000000004, -0.2320356, -0.7783564000000001),
        Vec3(-0.2650484, -0.2434614, -0.9307384000000001),
        Vec3(0.1944893, 0.49332260000000006, 0.049791),
        Vec3(0.25532210000000005, 0.5263352, 0.1205303),
        Vec3(0.2082534, 0.5590999, -0.020217100000000002),
        Vec3(-0.6827006, -0.5285917000000001, -0.2371899),
        Vec3(-0.7405873000000001, -0.6022738000000001, -0.2075913),
        Vec3(-0.6398443, -0.5453726, -0.3235054),
        Vec3(-0.8538047000000001, -0.7577917000000001, -0.1688532),
        Vec3(-0.8075591, -0.844826, -0.15134550000000002),
        Vec3(-0.9389507, -0.7836271, -0.20518350000000002),
        Vec3(-0.6916556, -0.1100882, -0.5168782000000001),
        Vec3(-0.6365888000000001, -0.1659366, -0.5746209999999999),
        Vec3(-0.6538885, -0.024937400000000002, -0.5325993000000001),
        Vec3(0.6330290000000001, -0.9323893000000001, -0.641663),
        Vec3(0.7026026000000001, -0.9680578000000001, -0.7007727),
        Vec3(0.6812488000000001, -0.8683594, -0.5925798999999999),
        Vec3(-0.24243450000000002, -0.5918126, 0.2701855),
        Vec3(-0.1613829, -0.5486394, 0.2503024),
        Vec3(-0.31447970000000003, -0.5307275, 0.2618121),
        Vec3(0.0637202, -0.6080457, 0.5849135000000001),
        Vec3(0.08923120000000001, -0.6092342000000001, 0.4907709),
        Vec3(-0.0339074, -0.5838972, 0.5853292),
        Vec3(0.5199216, -0.22649180000000002, -0.0138343),
        Vec3(0.5802838, -0.2788698, -0.06975360000000001),
        Vec3(0.567034, -0.1679393, 0.0462296),
        Vec3(0.0510145, 0.7629450000000001, 0.40545000000000003),
        Vec3(-0.007113500000000001, 0.8146563000000001, 0.3452959),
        Vec3(0.14649620000000002, 0.7740819000000001, 0.36691720000000005),
        Vec3(0.3146724, 0.8895843, 0.6526257000000001),
        Vec3(0.3506091, 0.8599906000000002, 0.7352942),
        Vec3(0.21457290000000004, 0.8929622, 0.6682029),
        Vec3(0.7308541000000001, -0.8339335000000001, -0.2471342),
        Vec3(0.6562127000000001, -0.8180601, -0.30628030000000006),
        Vec3(0.6993025, -0.8364954, -0.1531235),
        Vec3(-0.7530792000000001, 0.1069683, 0.4989387),
        Vec3(-0.7565406000000001, 0.19714220000000002, 0.4648636),
        Vec3(-0.793825, 0.0433547, 0.43702660000000004),
        Vec3(0.4452035000000001, 0.2700609, -0.5437815),
        Vec3(0.46033260000000004, 0.3058652, -0.45515900000000004),
        Vec3(0.43864530000000007, 0.1737004, -0.5180418999999999),
        Vec3(0.8427922000000001, -0.8619286000000002, 0.17846910000000002),
        Vec3(0.8340498000000001, -0.8005342000000001, 0.25362250000000003),
        Vec3(0.849672, -0.9530562, 0.2174667),
        Vec3(-0.8109456, 0.175383, -0.3096997),
        Vec3(-0.7245287, 0.1827177, -0.27214170000000004),
        Vec3(-0.8082178000000001, 0.1783171, -0.40593290000000004),
        Vec3(0.27769330000000003, -0.5701955, 0.7748213),
        Vec3(0.3287974, -0.5069688, 0.7233816000000001),
        Vec3(0.19870410000000002, -0.5817355000000001, 0.7200131000000001),
        Vec3(0.36351710000000004, -0.6953519, 0.5339628000000001),
        Vec3(0.33538510000000005, -0.7592789, 0.6031367000000001),
        Vec3(0.2875801, -0.6787975, 0.4740576),
        Vec3(0.6888027000000001, -0.41690230000000006, -0.180019),
        Vec3(0.7559735000000001, -0.4813701, -0.2004669),
        Vec3(0.6603805, -0.3759237, -0.2626496),
        Vec3(-0.4470837000000001, -0.41056400000000004, 0.3415362),
        Vec3(-0.4991607, -0.3358538, 0.31409560000000003),
        Vec3(-0.4791764, -0.43922140000000004, 0.42543870000000006),
        Vec3(0.8282263000000001, -0.0462068, -0.2560579),
        Vec3(0.8776769, -0.1293641, -0.23359450000000004),
        Vec3(0.8760297, 0.038481600000000005, -0.2651128),
        Vec3(0.4737236, 0.161643, 0.49011150000000003),
        Vec3(0.3969692, 0.21683680000000002, 0.5152631000000001),
        Vec3(0.5184651, 0.2148403, 0.4154746),
        Vec3(-0.3497332, -0.5781436000000001, -0.22027130000000003),
        Vec3(-0.4038857, -0.5773824000000001, -0.138168),
        Vec3(-0.415297, -0.6012265000000001, -0.2829379),
        Vec3(-0.2863989, -0.025933400000000002, -0.1857006),
        Vec3(-0.2132201, -0.0027953, -0.1174211),
        Vec3(-0.2873625, -0.12244250000000001, -0.18742530000000002),
        Vec3(0.11383000000000001, 0.11331000000000001, -0.2085899),
        Vec3(0.19655060000000002, 0.150352, -0.1725316),
        Vec3(0.0420521, 0.15344400000000002, -0.1551356),
        Vec3(-0.05613280000000001, 0.4590705, -0.2780017),
        Vec3(-0.0061173, 0.39193560000000005, -0.32941280000000006),
        Vec3(-0.0082763, 0.4733898, -0.19556020000000002),
        Vec3(-0.6423002000000001, -0.17052040000000002, -0.2528225),
        Vec3(-0.7233989000000001, -0.1949552, -0.1975418),
        Vec3(-0.6678416, -0.1321413, -0.33512780000000003),
        Vec3(-0.27721, 0.25522100000000003, -0.0672282),
        Vec3(-0.29074890000000003, 0.34835520000000003, -0.0385605),
        Vec3(-0.2977127, 0.2592348, -0.163582),
        Vec3(0.6708678, -0.4852016, -0.837928),
        Vec3(0.6593474, -0.5094964000000001, -0.7447187),
        Vec3(0.6582524000000001, -0.5591596, -0.8966502000000001),
        Vec3(-0.8497001, -0.0440284, 0.2803721),
        Vec3(-0.8422350000000001, 0.0331961, 0.2183451),
        Vec3(-0.9377295000000001, -0.07561600000000002, 0.2677026),
        Vec3(0.2975383, -0.08949700000000001, 0.6060783000000001),
        Vec3(0.2093991, -0.09957, 0.5670129),
        Vec3(0.34588320000000006, -0.07592480000000001, 0.5263958),
        Vec3(-0.6085023, -0.16296200000000002, 0.7970284000000001),
        Vec3(-0.6685848000000001, -0.23918300000000003, 0.7820849000000001),
        Vec3(-0.6177296000000001, -0.11905840000000001, 0.8835912000000001),
        Vec3(-0.37635230000000003, -0.3356777, 0.8285436000000002),
        Vec3(-0.46606500000000006, -0.3029414, 0.8406662000000001),
        Vec3(-0.3411834, -0.31789470000000003, 0.7431180000000001),
        Vec3(-0.110012, -0.0320162, 0.0375372),
        Vec3(-0.0527013, -0.10121830000000001, 0.0128652),
        Vec3(-0.0673226, 0.028855900000000004, 0.0934677),
        Vec3(0.09102090000000002, -0.8271801999999999, 0.14114290000000002),
        Vec3(0.0158544, -0.8759532000000001, 0.18708950000000002),
        Vec3(0.0565924, -0.7546113000000001, 0.0944383),
        Vec3(0.1554065, -0.6468033000000001, 0.3310872),
        Vec3(0.1362455, -0.7162802, 0.27223400000000003),
        Vec3(0.16162330000000003, -0.5622409, 0.28728539999999997),
        Vec3(0.054312700000000005, -0.1388652, 0.4886094),
        Vec3(0.011032, -0.0665826, 0.5311239),
        Vec3(0.04203660000000001, -0.22163670000000002, 0.5267301),
        Vec3(0.1100526, 0.10194900000000001, -0.9255318000000001),
        Vec3(0.1460815, 0.0575286, -0.8484128000000001),
        Vec3(0.0265613, 0.0594128, -0.9297750000000001),
        Vec3(-0.18423480000000003, 0.2327827, -0.5355326),
        Vec3(-0.1572941, 0.2573139, -0.6288125),
        Vec3(-0.22166790000000003, 0.14399340000000002, -0.5361006),
        Vec3(0.2452307, -0.2814686, -0.6448759000000001),
        Vec3(0.2862295, -0.3091668, -0.5589336),
        Vec3(0.29139200000000004, -0.3262923, -0.7181510000000001),
        Vec3(-0.2207998, -0.31120070000000005, 0.5945795),
        Vec3(-0.1262203, -0.31253390000000003, 0.6205467),
        Vec3(-0.22282690000000002, -0.2421858, 0.5220629),
        Vec3(0.5845471, 0.5020556, -0.6836491),
        Vec3(0.5557986, 0.5913737, -0.6753889000000001),
        Vec3(0.5089998, 0.4459153, -0.6661079000000001),
        Vec3(-0.34216430000000003, 0.4865553, 0.0731755),
        Vec3(-0.3965419, 0.44584520000000005, 0.1478214),
        Vec3(-0.3973445, 0.49583380000000005, -0.004443),
        Vec3(-0.23029500000000003, -0.2349717, 0.2112168),
        Vec3(-0.25764380000000003, -0.1873492, 0.2873191),
        Vec3(-0.1882868, -0.1715106, 0.15037820000000002),
        Vec3(0.030588500000000005, 0.4878766, 0.3791182),
        Vec3(0.0299338, 0.5855407, 0.40975780000000006),
        Vec3(0.1169911, 0.4563497, 0.4030616),
        Vec3(0.2925008, -0.36648450000000005, -0.36074500000000004),
        Vec3(0.3015896, -0.39166180000000006, -0.2644267),
        Vec3(0.2353923, -0.28894590000000003, -0.35182840000000004),
        Vec3(0.1111505, -0.40702550000000004, -0.9085692999999999),
        Vec3(0.20849050000000002, -0.4227348, -0.9180249),
        Vec3(0.0897072, -0.49022180000000004, -0.8740523),
        Vec3(-0.49929290000000004, -0.1974219, -0.7099668000000001),
        Vec3(-0.5173269, -0.11863090000000001, -0.7673625000000001),
        Vec3(-0.5070893, -0.2753273, -0.7658821),
        Vec3(0.4730983, 0.147842, 0.0720986),
        Vec3(0.4271686, 0.19882650000000002, 0.0034225),
        Vec3(0.41174130000000003, 0.0788607, 0.10618179999999999),
        Vec3(0.14215830000000002, -0.0176666, -0.6729105),
        Vec3(0.121766, 0.0394331, -0.5978213),
        Vec3(0.19484440000000003, -0.08779880000000001, -0.6400659000000001),
        Vec3(-0.7093223000000001, -0.8541193, 0.4116187),
        Vec3(-0.7074252000000001, -0.8047477000000001, 0.3242159),
        Vec3(-0.6153142, -0.8537148000000001, 0.445555),
        Vec3(0.838254, -0.6460958, 0.3765735),
        Vec3(0.9258625, -0.6141972, 0.4060763),
        Vec3(0.7637364, -0.621791, 0.44078520000000004),
        Vec3(-0.4320241, 0.40371370000000006, 0.32970920000000004),
        Vec3(-0.3750151, 0.4141786, 0.41054700000000005),
        Vec3(-0.5172925, 0.4455578, 0.34889060000000005),
        Vec3(0.5581877000000001, 0.1781994, 0.7505132),
        Vec3(0.5017196, 0.1139584, 0.8120950000000001),
        Vec3(0.5330781, 0.15912170000000003, 0.6595286),
        Vec3(-0.0734443, 0.1125031, 0.48846660000000003),
        Vec3(-0.0239224, 0.1540322, 0.41420700000000005),
        Vec3(-0.0961628, 0.1783437, 0.5554548),
        Vec3(0.6831005000000001, -0.924997, 0.4497165),
        Vec3(0.6836746, -0.9068011, 0.5427145999999999),
        Vec3(0.7668067000000001, -0.965251, 0.42499570000000003),
        Vec3(0.2714055, -0.36775810000000003, 0.1962003),
        Vec3(0.33334320000000006, -0.3851775, 0.266354),
        Vec3(0.3095021, -0.3978526, 0.1102566),
        Vec3(-0.1364034, -0.5762724, -0.5514747),
        Vec3(-0.2238584, -0.5409106, -0.5314718),
        Vec3(-0.0948185, -0.4908428, -0.5559580000000001),
        Vec3(-0.36234350000000004, -0.8061229000000001, 0.1383141),
        Vec3(-0.3132847, -0.7446722000000001, 0.18579060000000003),
        Vec3(-0.4307943, -0.7546630000000001, 0.0933298),
        Vec3(0.5717894, 0.6958376000000001, 0.4528556),
        Vec3(0.5852657, 0.7879846, 0.434179),
        Vec3(0.6443636, 0.6708413000000001, 0.5135144),
        Vec3(0.8121415000000001, 0.8262619000000001, -0.8483986),
        Vec3(0.8585190000000001, 0.8939651, -0.8974230000000001),
        Vec3(0.7560207000000001, 0.7748539000000001, -0.9044644000000001),
        Vec3(-0.5111889, -0.0352865, 0.5807903),
        Vec3(-0.5940048, 0.0102298, 0.556963),
        Vec3(-0.5349104, -0.101409, 0.6507582000000001),
        Vec3(0.0327324, 0.2563438, 0.011325100000000001),
        Vec3(-0.0633994, 0.25221360000000004, -0.0069385),
        Vec3(0.0724331, 0.3482989, 0.016056400000000002),
        Vec3(-0.6003367000000001, -0.7683112000000001, -0.8874252),
        Vec3(-0.6987751000000001, -0.7912403000000001, -0.8892689000000001),
        Vec3(-0.5699015000000001, -0.8504722, -0.8558813000000001),
        Vec3(0.0789377, -0.6357883000000001, -0.7563043),
        Vec3(-0.015110100000000001, -0.6341313000000001, -0.7440906),
        Vec3(0.10908810000000001, -0.5879281, -0.6770796),
        Vec3(0.5045666, 0.7219298000000001, -0.5151826),
        Vec3(0.5598994, 0.7846492, -0.5618276),
        Vec3(0.5480466, 0.6978976000000001, -0.4337211),
        Vec3(0.1286092, 0.6990271, -0.6806004),
        Vec3(0.048132900000000006, 0.7140371000000001, -0.7345224),
        Vec3(0.1716012, 0.7791024000000001, -0.6762706),
        Vec3(-0.31987830000000006, -0.9175883000000001, -0.6334156000000001),
        Vec3(-0.30327210000000004, -0.8198157, -0.6198635000000001),
        Vec3(-0.3084774, -0.9539992, -0.5434969),
        Vec3(-0.2002554, -0.5668583, 0.5440781),
        Vec3(-0.23474930000000002, -0.4756386, 0.5542471),
        Vec3(-0.21330269999999998, -0.6049986, 0.4540349),
        Vec3(0.6174952, -0.8574733000000001, 0.0210584),
        Vec3(0.5358389, -0.8191902000000001, 0.0557392),
        Vec3(0.6760918, -0.856171, 0.0943985),
        Vec3(-0.627861, 0.6057339, 0.40423580000000003),
        Vec3(-0.6169211, 0.6708022, 0.3383078),
        Vec3(-0.7221766000000001, 0.5771239, 0.40322050000000004),
        Vec3(0.0124733, -0.5816398, 0.0203961),
        Vec3(0.0029974000000000003, -0.5049468, 0.07868710000000001),
        Vec3(-0.0458026, -0.5646677000000001, -0.0555312),
        Vec3(0.3839311, -0.0064469, 0.861972),
        Vec3(0.35862530000000004, -0.0282978, 0.7705313),
        Vec3(0.30799360000000003, 0.0282626, 0.9065559000000001),
        Vec3(0.7375986000000001, -0.1368885, 0.4452411),
        Vec3(0.8024931000000001, -0.0682735, 0.4768078),
        Vec3(0.7477247, -0.1279372, 0.3493009),
        Vec3(0.9209092000000001, -0.1611532, -0.6423337),
        Vec3(0.9487648000000001, -0.2470973, -0.6800897),
        Vec3(0.9998917, -0.1354277, -0.5866779000000001),
        Vec3(-0.5314625000000001, -0.5613762000000001, 0.7941745),
        Vec3(-0.5589748, -0.6389969, 0.8478078),
        Vec3(-0.4444519, -0.5357513, 0.814946),
        Vec3(-0.36558840000000004, 0.2941259, -0.3295211),
        Vec3(-0.4418712, 0.25281630000000005, -0.3712668),
        Vec3(-0.29375680000000004, 0.30250050000000006, -0.3971189),
        Vec3(-0.8786626000000001, -0.31490900000000005, -0.16403790000000001),
        Vec3(-0.8441651000000001, -0.4011755, -0.1901885),
        Vec3(-0.9285717, -0.3276173, -0.08345350000000001),
        Vec3(-0.7358188000000001, 0.8239942, -0.2093781),
        Vec3(-0.7811518000000001, 0.7751233000000001, -0.2768278),
        Vec3(-0.689243, 0.7625331000000001, -0.1541437),
        Vec3(0.3770974, -0.0453172, -0.1931225),
        Vec3(0.42457570000000006, -0.09662180000000001, -0.12861140000000001),
        Vec3(0.3797803, 0.0441972, -0.15421800000000002),
        Vec3(0.2205813, 0.40988939999999996, 0.8958738),
        Vec3(0.3001973, 0.355544, 0.8746430000000001),
        Vec3(0.2103639, 0.42436680000000004, 0.9940011000000001),
        Vec3(-0.8723398, -0.8284874000000001, -0.8890692),
        Vec3(-0.9298972, -0.7537539, -0.9239127),
        Vec3(-0.8772683000000001, -0.8361037, -0.792594),
        Vec3(-0.8908952, 0.0807167, -0.7785764000000001),
        Vec3(-0.906123, -0.006191200000000001, -0.7422589),
        Vec3(-0.9534219, 0.1030144, -0.8475490000000001),
        Vec3(0.9182207, -0.3418263, -0.8430667000000001),
        Vec3(0.9046738, -0.2558299, -0.8879898000000002),
        Vec3(0.8390804, -0.39587340000000004, -0.8701949000000001),
        Vec3(0.10031620000000002, 0.9279866999999999, -0.8313986),
        Vec3(0.1167271, 1.0190392, -0.8070619000000001),
        Vec3(0.18499620000000003, 0.8761188000000001, -0.8493178000000001),
        Vec3(-0.1842572, 0.4893383, 0.8198130000000001),
        Vec3(-0.246821, 0.5316242, 0.8833179),
        Vec3(-0.1155294, 0.438941, 0.8716281000000001),
        Vec3(-0.40050549999999996, -0.3953707, 0.040234700000000005),
        Vec3(-0.36821480000000006, -0.3305443, 0.10587690000000001),
        Vec3(-0.4738199, -0.44524340000000007, 0.0841144),
        Vec3(0.35118150000000004, -0.7590537, -0.9004707000000001),
        Vec3(0.3415066, -0.7281286, -0.8094635000000001),
        Vec3(0.325155, -0.6835442, -0.9579171000000001),
        Vec3(-0.7614136, -0.7773463, 0.1262079),
        Vec3(-0.7489497, -0.6927033, 0.0753779),
        Vec3(-0.8528840000000001, -0.8050689000000001, 0.11647099999999999),
        Vec3(-0.5419143000000001, 0.706188, -0.5745275000000001),
        Vec3(-0.4504905, 0.6840455000000001, -0.5750772000000001),
        Vec3(-0.5895918, 0.6221393000000001, -0.5676189),
        Vec3(-0.9017162000000001, 0.18887810000000002, 0.09378130000000001),
        Vec3(-0.9868157000000001, 0.21150560000000002, 0.0623903),
        Vec3(-0.8466599000000001, 0.1708818, 0.0191067),
        Vec3(-0.8037596000000001, -0.5106016000000001, 0.47508020000000006),
        Vec3(-0.7322228000000001, -0.5707104000000001, 0.5043673000000001),
        Vec3(-0.8230169, -0.44810510000000003, 0.5475067),
        Vec3(-0.43905459999999996, 0.16915020000000003, -0.7756245),
        Vec3(-0.35588440000000005, 0.12649660000000001, -0.7996466),
        Vec3(-0.41361210000000004, 0.2629917, -0.7760619000000001),
        Vec3(-0.3975194, -0.6218233, -0.692733),
        Vec3(-0.3085645, -0.6066352, -0.7325296),
        Vec3(-0.46408150000000004, -0.6676349, -0.7524295000000001),
        Vec3(0.8240648, 0.5082183, -0.8857912000000001),
        Vec3(0.7453645, 0.5101728, -0.8259448000000001),
        Vec3(0.8368223000000001, 0.4228825, -0.9227974),
        Vec3(0.1598931, -0.18297960000000002, -0.0316752),
        Vec3(0.1992252, -0.1475465, -0.11498150000000001),
        Vec3(0.1726926, -0.2749881, -0.0397199),
        Vec3(0.1433819, -0.13867020000000002, -0.3304673),
        Vec3(0.1323094, -0.0486757, -0.306711),
        Vec3(0.0623143, -0.18020160000000002, -0.3264828),
        Vec3(0.3735991, 0.5554990000000001, 0.6219603),
        Vec3(0.33209160000000004, 0.6265941, 0.6761937),
        Vec3(0.4195666, 0.6053207, 0.5526652000000001),
        Vec3(-0.6285976, -0.9228132000000001, -0.4460410000000001),
        Vec3(-0.5683885000000001, -1.0005163, -0.47408700000000004),
        Vec3(-0.6707036000000001, -0.9488426000000001, -0.36084380000000005),
        Vec3(0.40362000000000003, -0.3378299, 0.6905724000000001),
        Vec3(0.34574410000000005, -0.2618106, 0.6794504),
        Vec3(0.4929409, -0.2991177, 0.6666963),
        Vec3(0.0606517, 0.8561279000000002, -0.37824060000000004),
        Vec3(0.049809900000000004, 0.9526035, -0.36992990000000003),
        Vec3(0.1288545, 0.8500162, -0.4475972),
        Vec3(-0.5666717, 0.8368298, -0.8299623),
        Vec3(-0.49266610000000005, 0.7904359000000001, -0.8744783),
        Vec3(-0.5742748, 0.8095703000000001, -0.7389592),
        Vec3(-0.3344692, 0.7636718, 0.34758300000000003),
        Vec3(-0.4140634000000001, 0.7105012, 0.32964720000000003),
        Vec3(-0.33507970000000004, 0.7470949, 0.44333500000000003),
        Vec3(-0.13221860000000002, 0.6638182, 0.0028906),
        Vec3(-0.18950880000000003, 0.6054034, 0.0554933),
        Vec3(-0.1928852, 0.7271511, -0.0442137),
        Vec3(0.8793551000000002, 0.692545, 0.7841717),
        Vec3(0.9639675000000001, 0.7325848, 0.785068),
        Vec3(0.8776882000000001, 0.6398711, 0.8674035999999999),
        Vec3(-0.6322101, 0.0607808, 0.11740990000000001),
        Vec3(-0.5647986, 0.1150825, 0.15883480000000003),
        Vec3(-0.5852992, -0.0060471, 0.066073),
        Vec3(0.5685139, -0.35460460000000005, -0.41285489999999997),
        Vec3(0.47279660000000007, -0.3473549, -0.4165755),
        Vec3(0.6024761000000001, -0.2749532, -0.4610993),
        Vec3(0.38219990000000004, 0.6538527000000001, 0.167803),
        Vec3(0.3549684, 0.7331168, 0.1175762),
        Vec3(0.3816423, 0.6788603000000001, 0.2602522),
        Vec3(0.3892622, -0.7034239000000001, -0.6415978),
        Vec3(0.4720509, -0.6539827, -0.6295004),
        Vec3(0.32628100000000004, -0.6550918000000001, -0.589037),
        Vec3(0.5454437000000001, 0.3881722, -0.2767521),
        Vec3(0.5906266, 0.32404890000000003, -0.32722500000000004),
        Vec3(0.6116652, 0.4657253, -0.2669082),
        Vec3(-0.4179373, -0.8326009000000001, 0.43950050000000007),
        Vec3(-0.3737878, -0.8754474999999999, 0.5134173),
        Vec3(-0.39345840000000004, -0.882532, 0.3534465),
        Vec3(0.7949488, -0.13534980000000002, 0.9244688000000001),
        Vec3(0.7234465000000001, -0.1100736, 0.9869694000000001),
        Vec3(0.8010855, -0.0754076, 0.8476271000000001),
        Vec3(0.44986030000000005, -0.42192670000000004, 0.3963173),
        Vec3(0.48952260000000003, -0.3311215, 0.4147093),
        Vec3(0.48023710000000003, -0.48716980000000004, 0.4671292),
        Vec3(-0.430725, 0.7384668, -0.28794620000000004),
        Vec3(-0.3402854, 0.7237543, -0.3252031),
        Vec3(-0.41432310000000006, 0.7834459, -0.2019767),
        Vec3(0.28687520000000005, -0.041838900000000005, 0.19093400000000002),
        Vec3(0.2544472, -0.0880398, 0.1179279),
        Vec3(0.20828410000000003, -0.0227627, 0.24579640000000003),
        Vec3(0.7961389, 0.1611971, 0.4525253),
        Vec3(0.7498497000000001, 0.18331740000000002, 0.3728113),
        Vec3(0.884188, 0.20509960000000002, 0.4576281),
        Vec3(0.2692467, -0.45179460000000005, -0.0884665),
        Vec3(0.17894350000000003, -0.48356170000000004, -0.0587396),
        Vec3(0.3330887, -0.5126025000000001, -0.04746790000000001),
        Vec3(0.0711797, 0.1680547, -0.4780435),
        Vec3(0.08545960000000001, 0.15679110000000002, -0.3850059),
        Vec3(-0.0187286, 0.1868256, -0.4936632000000001),
        Vec3(0.45582000000000006, -0.0044434, -0.4673875),
        Vec3(0.5376141, -0.0551202, -0.4913295),
        Vec3(0.44096, -0.0197728, -0.37395550000000005),
        Vec3(0.7109730000000001, 0.045347500000000006, -0.026884400000000003),
        Vec3(0.6996444, 0.009859900000000001, -0.1119484),
        Vec3(0.6343115, 0.1050849, -0.0110689),
        Vec3(-0.2549577, -0.0200852, -0.4492507),
        Vec3(-0.3199039, -0.0856439, -0.48887520000000007),
        Vec3(-0.260512, -0.0151501, -0.3534),
        Vec3(-0.8527353, 0.4524174000000001, -0.23474080000000003),
        Vec3(-0.7662213000000001, 0.43424660000000004, -0.21048520000000004),
        Vec3(-0.8794374, 0.36967930000000004, -0.279574),
        Vec3(-0.28204310000000005, -0.07004780000000001, 0.4413266),
        Vec3(-0.20883430000000003, -0.0069797, 0.45305210000000007),
        Vec3(-0.3535913, -0.0413547, 0.5069151),
        Vec3(0.6924598, -0.1285634, -0.4901833),
        Vec3(0.7496477, -0.083151, -0.4206076000000001),
        Vec3(0.7527035, -0.1446676, -0.5677236),
        Vec3(-0.5353763, 0.6572088, 0.6734366),
        Vec3(-0.4428432, 0.6663173, 0.6834862),
        Vec3(-0.5603600000000001, 0.6385787, 0.5822425),
        Vec3(0.7661597000000001, 0.2386104, 0.9175259000000001),
        Vec3(0.7321694000000001, 0.2395921, 1.0055536),
        Vec3(0.6959141, 0.21022, 0.8533079000000001),
        Vec3(-0.349659, -0.3765912, -0.4994838),
        Vec3(-0.36121810000000004, -0.3620689, -0.4036115),
        Vec3(-0.4323649, -0.35278160000000003, -0.5463213),
        Vec3(-0.9351953000000001, 0.8343146000000001, 0.3704719),
        Vec3(-0.8465363, 0.8758592000000001, 0.3736785),
        Vec3(-0.9159223000000001, 0.7399420000000001, 0.3880808),
        Vec3(-0.0957571, -0.5174275, -0.21901700000000002),
        Vec3(-0.19073450000000003, -0.5209294, -0.23031640000000003),
        Vec3(-0.0758582, -0.4314608, -0.2636469),
        Vec3(0.5048831, -0.7988284000000001, 0.25875149999999997),
        Vec3(0.5870168, -0.8296198000000001, 0.30438160000000003),
        Vec3(0.455197, -0.7498875, 0.32292800000000005),
        Vec3(0.37056580000000006, 0.8368298, -0.8304931),
        Vec3(0.38606070000000003, 0.9297256000000002, -0.8425405),
        Vec3(0.3866516, 0.8146007000000002, -0.7371475000000001),
        Vec3(-0.25351340000000006, 0.7116414, 0.6338590000000001),
        Vec3(-0.19188840000000001, 0.7705593000000001, 0.6804078),
        Vec3(-0.19752250000000002, 0.6330832000000001, 0.6121365000000001),
        Vec3(-0.3440263, 0.7005018000000001, 0.9111704),
        Vec3(-0.2633953, 0.7421761, 0.9465848),
        Vec3(-0.3654896, 0.7513753000000001, 0.825647),
        Vec3(0.6452501, -0.5478051, -0.5722874),
        Vec3(0.7297698, -0.5914122, -0.5586895000000001),
        Vec3(0.6253267, -0.4919338, -0.4937095),
        Vec3(-0.6528475, 0.4497861000000001, -0.5573636),
        Vec3(-0.625784, 0.46840560000000003, -0.4656222),
        Vec3(-0.7247764000000001, 0.3931701, -0.5623068),
        Vec3(-0.7310706, 0.8183648, 0.7852201000000001),
        Vec3(-0.6552879, 0.7735013000000001, 0.7436342),
        Vec3(-0.6877865000000001, 0.8451022, 0.8686616000000001),
        Vec3(-0.1476441, -0.7867001, 0.7421787000000001),
        Vec3(-0.0852398, -0.8570669000000001, 0.7084405),
        Vec3(-0.136374, -0.7190554, 0.6737331000000001),
        Vec3(0.5228019, -0.049587, -0.7592953),
        Vec3(0.4737616, -0.053813900000000005, -0.6742036),
        Vec3(0.4625354, -0.09013290000000002, -0.8269022),
        Vec3(0.7566531000000001, -0.10067340000000001, 0.18082130000000002),
        Vec3(0.7657411000000001, -0.1866188, 0.1428564),
        Vec3(0.7358547, -0.0388051, 0.11399429999999999),
        Vec3(0.18509240000000002, -0.5717449, -0.5035416),
        Vec3(0.172104, -0.6227449, -0.4217213),
        Vec3(0.2199319, -0.48563780000000006, -0.46928919999999996),
        Vec3(-0.8904325000000001, 0.6791585000000001, -0.38428660000000003),
        Vec3(-0.8690637000000001, 0.5952111999999999, -0.3490118),
        Vec3(-0.8676001, 0.6844077000000001, -0.4770651),
        Vec3(-0.3434304, 0.43971980000000005, -0.7863568000000001),
        Vec3(-0.26347770000000004, 0.3892027, -0.7779256000000001),
        Vec3(-0.34328470000000005, 0.5161485, -0.7310788),
        Vec3(-0.7446738, 0.6124823, 0.0150068),
        Vec3(-0.7404893000000001, 0.5207959, 0.0481504),
        Vec3(-0.8405563000000001, 0.6215466000000001, -0.0024439),
        Vec3(0.019533000000000002, -0.7307128, -0.3501714),
        Vec3(-0.0240912, -0.6769004000000001, -0.42436550000000006),
        Vec3(-0.019491500000000002, -0.6981989, -0.267986),
        Vec3(0.7068819000000001, 0.9130323000000001, 0.7260588000000001),
        Vec3(0.6559855, 0.8400987000000001, 0.7699406),
        Vec3(0.7972497000000001, 0.8829223000000002, 0.7213747),
        Vec3(-0.0185218, -0.319229, -0.6289346),
        Vec3(0.0776651, -0.3073203, -0.6391948000000001),
        Vec3(-0.0497426, -0.2964088, -0.7170893),
        Vec3(-0.20736300000000002, 0.4369672, 0.5251074),
        Vec3(-0.18686380000000002, 0.3811508, 0.6060261),
        Vec3(-0.1253174, 0.4326708, 0.478168),
        Vec3(-0.0855003, 0.30691080000000004, -0.7935226000000001),
        Vec3(0.007575800000000001, 0.3132202, -0.7617253),
        Vec3(-0.10098450000000002, 0.22074870000000002, -0.8320426),
        Vec3(-0.6328833, 0.0064083000000000005, -0.8467228000000001),
        Vec3(-0.7268416000000001, 0.0384648, -0.8383978000000001),
        Vec3(-0.5825246000000001, 0.09193040000000001, -0.8320175000000001),
        Vec3(-0.8276352, -0.8809851000000001, -0.634491),
        Vec3(-0.7541457, -0.8792368, -0.5688031),
        Vec3(-0.8510208000000001, -0.9731116000000001, -0.6353248),
        Vec3(0.0463654, 0.28267760000000003, 0.7018388),
        Vec3(0.08753240000000001, 0.3588447, 0.7482107),
        Vec3(0.0770257, 0.2126646, 0.758365),
        Vec3(0.0478371, -0.3407198, 0.6910039000000001),
        Vec3(0.0585298, -0.299675, 0.7756467),
        Vec3(0.055653100000000004, -0.4390636, 0.7183527000000001),
        Vec3(-0.16895590000000002, 0.7002511, -0.33035610000000004),
        Vec3(-0.10091520000000001, 0.7635432, -0.3553133),
        Vec3(-0.1333699, 0.6195680000000001, -0.2919485),
        Vec3(0.9000097, -0.6227871, 0.8137681000000001),
        Vec3(0.9373858, -0.6821152, 0.7497417000000001),
        Vec3(0.8024020000000001, -0.6011019000000001, 0.8033727000000002),
        Vec3(0.1272378, 0.1954065, 0.3058033),
        Vec3(0.1260842, 0.2615112, 0.2341207),
        Vec3(0.1890312, 0.2223439, 0.37329340000000005),
        Vec3(0.4139066, -0.3527427, -0.9022608999999999),
        Vec3(0.4070028, -0.34991170000000005, -0.9985525000000002),
        Vec3(0.4960292, -0.39423640000000004, -0.8778433),
        Vec3(0.7176276, 0.2178662, -0.407594),
        Vec3(0.6995275000000001, 0.1684022, -0.48732230000000004),
        Vec3(0.7621313000000001, 0.2984315, -0.43939940000000005),
        Vec3(-0.0034836, -0.3929919, 0.21480950000000001),
        Vec3(0.08380030000000001, -0.3513346, 0.22596630000000004),
        Vec3(-0.0654999, -0.32885810000000004, 0.2426748),
        Vec3(-0.8701616, 0.27952370000000004, -0.5700842),
        Vec3(-0.8848959000000001, 0.2145603, -0.6420186000000001),
        Vec3(-0.9464654, 0.3362988, -0.5610379),
        Vec3(0.28044440000000004, 0.8909551000000001, -0.5635389000000001),
        Vec3(0.3063615, 0.9805993000000002, -0.5770957),
        Vec3(0.3512196, 0.8486179, -0.5165468),
        Vec3(0.3236002, 0.8357402, 0.36366390000000004),
        Vec3(0.34508540000000004, 0.9071799, 0.3073263),
        Vec3(0.3476445, 0.8586691000000001, 0.4607935000000001),
        Vec3(0.5628532, 0.7185941, 0.8510937000000001),
        Vec3(0.5467837000000001, 0.6318153, 0.8138885),
        Vec3(0.4964153, 0.7477925000000001, 0.9150467000000001),
        Vec3(-0.3767261, 0.12603899999999998, 0.2017681),
        Vec3(-0.37077340000000003, 0.20195200000000002, 0.2565591),
        Vec3(-0.33646020000000004, 0.1609935, 0.1212351),
        Vec3(0.8228917, -0.34417610000000004, 0.0850139),
        Vec3(0.9022130000000002, -0.3709208, 0.1334881),
        Vec3(0.7553662000000001, -0.3976731, 0.1276651),
        Vec3(-0.3004292, 0.1345869, 0.7236734),
        Vec3(-0.3715311, 0.0724731, 0.6912689000000001),
        Vec3(-0.3525684, 0.21119010000000002, 0.7538936),
        Vec3(0.49014230000000003, -0.5547141, -0.0119795),
        Vec3(0.532755, -0.5560969, 0.076977),
        Vec3(0.5447642, -0.49307520000000005, -0.0617088),
        Vec3(-0.33062100000000005, 0.8469477, -0.04950570000000001),
        Vec3(-0.2836855, 0.9122642000000001, -0.1080706),
        Vec3(-0.3311215, 0.8916887, 0.0353875),
        Vec3(-0.6572180000000001, 0.3018421, -0.026207900000000003),
        Vec3(-0.6711444000000001, 0.26599560000000005, 0.060646000000000005),
        Vec3(-0.6040782, 0.2473886, -0.0843542),
        Vec3(-0.5429147000000001, -0.5967833, 0.5177682),
        Vec3(-0.4896648, -0.6608203, 0.46925340000000004),
        Vec3(-0.5386594, -0.6130690000000001, 0.6117931),
        Vec3(0.6054430000000001, 0.7035601, 0.0031519),
        Vec3(0.5939348, 0.7951184, -0.014109400000000001),
        Vec3(0.5390949, 0.6866654, 0.0702638),
        Vec3(-0.08902290000000002, -0.2615376, -0.3621726),
        Vec3(-0.1695442, -0.2097314, -0.35379760000000005),
        Vec3(-0.0895128, -0.2637531, -0.4561712),
        Vec3(-0.5446979, 0.0994907, -0.21067140000000004),
        Vec3(-0.4494883, 0.09979120000000001, -0.1881595),
        Vec3(-0.5759736000000001, 0.009419900000000002, -0.1919247),
        Vec3(-0.7517688, -0.40783400000000003, 0.7202707),
        Vec3(-0.8242348, -0.4324693, 0.7729813000000001),
        Vec3(-0.6924399000000001, -0.48174530000000004, 0.726943),
        Vec3(-0.1134884, 0.8628488000000001, 0.20810689999999998),
        Vec3(-0.1962897, 0.8634713, 0.2554947),
        Vec3(-0.1119731, 0.7882422, 0.15160030000000002),
        Vec3(-0.6288317000000001, 0.8011683, 0.20221290000000003),
        Vec3(-0.6404647, 0.8858198, 0.1546109),
        Vec3(-0.6669138000000001, 0.7315199, 0.14826140000000002),
        Vec3(-0.6766934, -0.4026505, -0.8306645),
        Vec3(-0.6542552, -0.4855702, -0.8828363),
        Vec3(-0.7743726, -0.3954163, -0.8170031),
        Vec3(0.6895244, 0.7052113000000001, -0.30312890000000003),
        Vec3(0.7810039, 0.7265357, -0.3330751),
        Vec3(0.6666757, 0.7541675, -0.22021780000000002),
        Vec3(0.800326, 0.4735929, -0.5168656),
        Vec3(0.7233375, 0.4730204, -0.5685772),
        Vec3(0.8406708, 0.562958, -0.5300038),
        Vec3(-0.523016, -0.6461196, -0.4390836),
        Vec3(-0.5701558999999999, -0.7313756, -0.4377348),
        Vec3(-0.48555450000000006, -0.632797, -0.5273443999999999),
        Vec3(-0.3803496, -0.9221115, 0.7305476),
        Vec3(-0.2934288, -0.8877999000000001, 0.7425047),
        Vec3(-0.4416066, -0.8566250000000001, 0.7573191),
        Vec3(-0.5990438, -0.1574415, 0.30725210000000003),
        Vec3(-0.5805447, -0.11192930000000001, 0.39337310000000003),
        Vec3(-0.6651651000000001, -0.0947206, 0.2634773),
        Vec3(0.11554510000000001, 0.7138377, -0.11783170000000001),
        Vec3(0.0330359, 0.7363227000000001, -0.0724568),
        Vec3(0.10551450000000001, 0.7478676000000001, -0.2100793),
        Vec3(-0.5886817, 0.5150957, -0.2997276),
        Vec3(-0.5191777000000001, 0.4556623, -0.2611465),
        Vec3(-0.5401622, 0.6003350000000001, -0.3064194),
        Vec3(0.2539927, 0.3387568, 0.49761800000000006),
        Vec3(0.1760904, 0.3182324, 0.5589217),
        Vec3(0.2841539, 0.4315465, 0.5278853),
        Vec3(0.8331859, -0.7344673, -0.5188191),
        Vec3(0.8424401, -0.7586778000000001, -0.4198621),
        Vec3(0.8911675000000001, -0.7764424000000001, -0.5823267000000001),
        Vec3(-0.5774081, 0.1315144, -0.5304553000000001),
        Vec3(-0.5222992, 0.1580336, -0.6077077000000001),
        Vec3(-0.6219878, 0.21784409999999998, -0.509336),
        Vec3(0.6340084, -0.49260640000000006, 0.2149626),
        Vec3(0.5639784, -0.4766017, 0.2829591),
        Vec3(0.6883511000000001, -0.5598794, 0.256282),
        Vec3(-0.2722394, 0.6614441000000001, -0.5843805),
        Vec3(-0.23324140000000002, 0.7251953000000001, -0.6403722000000001),
        Vec3(-0.2221682, 0.6596615, -0.5034993),
        Vec3(-0.1813152, 0.07128240000000001, -0.9048176),
        Vec3(-0.17639210000000002, -0.0271744, -0.8916841),
        Vec3(-0.20977960000000004, 0.08605000000000002, -0.9970314),
        Vec3(0.6146244000000001, -0.6879929, 0.8376712),
        Vec3(0.5324465, -0.7183905, 0.8841506000000001),
        Vec3(0.6642264, -0.7749945, 0.8201756),
        Vec3(0.2887544, 0.8612615, 0.0453821),
        Vec3(0.2452117, 0.8047637000000001, -0.0251289),
        Vec3(0.2348751, 0.9285636, 0.08956420000000001),
        Vec3(-0.8475558, -0.8180718000000001, 0.6501232),
        Vec3(-0.803431, -0.8945692, 0.6891221000000001),
        Vec3(-0.8173606, -0.8179435, 0.5569319),
        Vec3(0.6714205, -0.2947903, 0.6551671),
        Vec3(0.7143284000000001, -0.2459194, 0.7270891),
        Vec3(0.7212365000000001, -0.2637046, 0.5791018),
        Vec3(0.6445003, -0.5462306, 0.5577966),
        Vec3(0.6730099, -0.4696457, 0.6073312),
        Vec3(0.6099428, -0.6099736, 0.6222859000000001),
        Vec3(-0.1818761, -0.6080111, -0.880542),
        Vec3(-0.1644534, -0.6777931, -0.9477669000000001),
        Vec3(-0.2078582, -0.5258591, -0.9215838000000001),
        Vec3(0.6037377, 0.2950576, 0.2863541),
        Vec3(0.6409766000000001, 0.3757001, 0.2570754),
        Vec3(0.5577033, 0.2617474, 0.20905870000000004),
        Vec3(-0.7731645, 0.33376680000000003, 0.3301121),
        Vec3(-0.8345957, 0.40272220000000003, 0.3589257),
        Vec3(-0.8130328000000001, 0.28452380000000005, 0.2577949)
    };
    context.setPositions(positions);
    State state = context.getState(State::Energy | State::Forces);
    
    // Compare the forces and energy to reference values computed with Tinker.

    ASSERT_EQUAL_TOL(-8689.0456938244151, state.getPotentialEnergy(), energyTol);
    vector<Vec3> expectedForces = {
        Vec3(-335.0172933029931, 22.266752620867532, -311.0298423949193),
        Vec3(125.33189469021171, -48.64909489077264, 173.68519722812303),
        Vec3(-68.82118100602865, 201.02869856438417, -111.6897919454047),
        Vec3(-61.153379096776426, -616.9805120862311, -185.54939132041835),
        Vec3(-36.81384802661533, 127.429744136704, 389.96946400173465),
        Vec3(-149.5327745486792, 304.0904995695656, -167.18743050384256),
        Vec3(-5.774742682153585, 373.60988946029994, -361.47500169984926),
        Vec3(-181.40211600742782, -247.91603688859874, 134.01205148261855),
        Vec3(21.198833848005762, -339.8691795138551, 45.802133678195986),
        Vec3(-572.6283376708045, -212.1053261376012, 252.67837117733575),
        Vec3(217.70654441990092, 196.93954185286114, -214.91066333815215),
        Vec3(301.133383969155, 213.93807850760453, 28.9133028668968),
        Vec3(-299.08443712963157, 434.0122811695415, 0.013121440693021303),
        Vec3(59.52664104608579, -247.94912563119672, -230.14506152824993),
        Vec3(90.95139727044368, -62.875870942678866, -7.637291617312463),
        Vec3(88.67134147030124, 257.7041314274848, -28.532734429050475),
        Vec3(-70.26814245017862, -105.14173766805607, -18.25952793194643),
        Vec3(-70.28148862947461, -261.8105030692827, 28.87470048624826),
        Vec3(209.77702595833722, 416.2567365924144, 632.57753195419),
        Vec3(-190.08312952734263, -18.294772882839077, -505.8441797104047),
        Vec3(-179.7429255793455, -92.01351813231925, -160.07155590063837),
        Vec3(-26.02384439787512, 330.90933534519417, -140.8958215021164),
        Vec3(109.63698884057686, -571.6693436875398, -139.96868562036477),
        Vec3(-137.30575212422485, -18.590762562776803, 26.55457387413793),
        Vec3(-470.7904899631753, -245.44713111184723, 266.2903954223516),
        Vec3(472.8083986505288, -89.0282869998406, -203.89855802595156),
        Vec3(320.6697244615793, 384.48206499977783, -301.7069190335516),
        Vec3(-52.803387093321916, 371.49950630788226, -134.22826294908592),
        Vec3(36.99526542998541, -48.845037544586425, 191.51299040052936),
        Vec3(128.42355577787023, -408.4164655600154, -173.3124425056217),
        Vec3(-878.269059321722, -405.611635336188, 58.17112139996775),
        Vec3(311.21777122108915, 348.87955985753723, -41.081660659626756),
        Vec3(616.7791174948256, 246.05437434041508, 135.81742400623813),
        Vec3(103.45047101349174, 200.32520291949814, 272.8589209780621),
        Vec3(-9.490187025589702, -290.6683465976385, -73.40438173785563),
        Vec3(-55.4759120118965, 162.87656847420035, -230.05485334711796),
        Vec3(109.52717578132189, -336.2359129283262, 497.0416460397142),
        Vec3(-75.0006907250051, 202.17953028474363, -292.48773052352334),
        Vec3(-262.4420239324133, 234.94106079413922, -125.27331602906199),
        Vec3(-14.202580814281575, -262.83218804797286, -229.84220570272163),
        Vec3(212.05109735217303, 99.3513392917139, -63.47217888422316),
        Vec3(69.01293955243082, 367.7778083972, 163.89342426801701),
        Vec3(-147.74894232097182, 622.1821269496787, 90.0489458605032),
        Vec3(-296.7521624967717, -446.6548799522563, 184.76666740616145),
        Vec3(213.50832760333225, -163.1447531897083, -133.1253079707448),
        Vec3(-263.12394827367666, -272.8332624893659, 627.1233508940367),
        Vec3(130.22756305721452, -174.11906923075196, -141.92061497589265),
        Vec3(29.750796355350722, -46.36135673544522, -249.62285120478313),
        Vec3(-9.13445457731486, -104.19390094140452, -664.7908266736111),
        Vec3(-6.098628627937129, 75.3690598663586, 214.20148433669854),
        Vec3(93.85773518954444, 70.31730678831572, 419.69479715633724),
        Vec3(153.08875466165682, 481.7270969394399, 57.00333898487111),
        Vec3(-40.77543925453837, -220.28139776477002, -1.1259274322172284),
        Vec3(-4.0235594287871335, -383.61042422379546, -71.23508976663497),
        Vec3(418.0952203126167, 258.72039518106965, -47.349109649340804),
        Vec3(0.04695838952295269, 74.96915961237585, -51.56319329013884),
        Vec3(29.043347800433725, -121.56146857432789, 448.008737586852),
        Vec3(-635.3406965422951, -130.49158863352883, -16.637291453440223),
        Vec3(357.06426206105454, -79.87073091942342, -79.98429428457332),
        Vec3(259.79092998995526, 176.66734932402687, -68.32489685158053),
        Vec3(41.586627904206296, 911.714192439227, 348.02477261221054),
        Vec3(-160.06817726684196, -346.7078180859659, 70.76436888328635),
        Vec3(234.35149116384792, -460.58513477969444, -73.426128633765),
        Vec3(157.4675603840564, -424.1782014191973, -372.6497703158724),
        Vec3(119.26625771982772, 531.6272342992715, 112.81649088166586),
        Vec3(-34.07274116756812, 110.6794010683938, 210.08473261060942),
        Vec3(-198.40995599088754, -570.8909011764729, 115.21000416227702),
        Vec3(51.41637305126889, 346.8967211007784, 46.042161875197216),
        Vec3(143.69207010587456, 232.12347501298257, -111.85121609580156),
        Vec3(587.8256947940419, 167.03440041641014, -344.5262841142951),
        Vec3(-294.0302209549635, -102.8160127286171, 278.0538118178713),
        Vec3(-155.3754267902171, -109.36554731955172, 119.48308135091925),
        Vec3(334.73319863236054, 377.32048036240354, -77.85868301095428),
        Vec3(-227.14891011932315, -91.8088898622582, 287.0865066368314),
        Vec3(-203.96567623951984, -185.40142586146987, -38.135012396093074),
        Vec3(-20.4293898817084, 592.6694193025131, -22.971076219709346),
        Vec3(-115.34910930011313, -232.3372229865608, 196.22480672888415),
        Vec3(68.25750021108932, -475.02332746194145, -164.93581903944687),
        Vec3(-333.7383046001966, 132.27112439169812, -761.0324474778728),
        Vec3(216.86879805383205, 53.879327754001224, 149.678361921911),
        Vec3(147.69195577766362, -257.43256468417815, 562.7838289823908),
        Vec3(381.03219261827206, -306.7234704626524, 223.53742098150207),
        Vec3(-303.05669134128937, 47.5927486894213, -37.253967643358976),
        Vec3(-65.56006757720597, 57.30446123065876, -99.92542301755618),
        Vec3(16.45016285279712, 176.9512834170706, -749.5247703975105),
        Vec3(8.8502897582912, 134.11245318402337, 256.66351522740337),
        Vec3(-172.84287762857088, -320.6199115936171, 55.79038894807629),
        Vec3(-164.2330289485781, -76.7611788656553, 310.152391940976),
        Vec3(-2.2175942260194152, -78.54632557153865, -80.75861093555594),
        Vec3(98.54858566877967, -12.069149082363449, -137.1999706358292),
        Vec3(-484.3780436682154, -19.13689509690661, 310.75160467834843),
        Vec3(171.0767536821465, -34.60959395997303, -115.3288869482195),
        Vec3(304.9942121214169, -98.24199876126227, -359.2513692773626),
        Vec3(-46.917816002077096, 342.99848016254066, -710.7609279934222),
        Vec3(-255.7733125462255, -130.61213124346506, 333.6902423638203),
        Vec3(130.3665876050882, 72.45464900777448, 407.67413132012115),
        Vec3(362.6959350832374, -625.6289663832105, 534.8145739471075),
        Vec3(109.9449670148772, 34.02485255261072, -29.333173337415655),
        Vec3(10.609296002267028, 319.0453059680116, -209.23668662392288),
        Vec3(489.92703773110577, -90.97242966996552, 470.6492704242861),
        Vec3(-148.6591729184633, -262.71858457289375, -401.94526102641487),
        Vec3(-282.1158812260683, 19.459896854715428, -211.49662848026315),
        Vec3(368.7383546475742, -20.484173872570658, -219.36806133763213),
        Vec3(-270.8205798617515, 311.9392165851896, 27.643007573580448),
        Vec3(140.56187148787137, -87.90210361076366, 151.4747821114451),
        Vec3(-60.16866721698693, 302.97054215665713, -819.4553589015721),
        Vec3(35.79598871589246, -122.39126281358682, 210.26426229337034),
        Vec3(-66.10611807867937, 157.5398290851135, 448.861116021134),
        Vec3(-498.3505587804024, 32.47643839609101, 501.6056415549782),
        Vec3(21.42926264054984, 35.45731305548093, -32.13558813571573),
        Vec3(176.69598167411476, -8.818539210212053, 82.33055000489443),
        Vec3(311.69676437401915, -180.34661299766054, -374.47482858089404),
        Vec3(-55.6362284394929, -23.143898952474096, 0.6568904901292746),
        Vec3(97.0788452798702, 279.8982155045794, 202.65171870217557),
        Vec3(-378.30748347232014, 127.61358171776939, 290.9515219066664),
        Vec3(197.2613702769855, -292.04013928277976, -142.4748620807829),
        Vec3(282.63019847216043, 122.84174235642894, 20.537956693491076),
        Vec3(357.0756608486188, 80.38597643570732, 293.7034380840931),
        Vec3(-206.42247820888818, 234.27232340296675, -90.7074846249461),
        Vec3(-307.70713998461395, -196.037137219103, 27.15152638460597),
        Vec3(-64.97305840292616, 432.6702312462856, -377.1701790062745),
        Vec3(192.39934479522014, -209.42186728613302, 321.17117572988707),
        Vec3(-34.070401388556284, -242.92310385438617, -161.37927007292356),
        Vec3(-210.06535904776248, 475.94338248099257, 39.573057705113),
        Vec3(196.6392201325647, -178.23021411913012, -52.06018603864398),
        Vec3(366.96563306664547, -75.50449018422802, 234.11118983874277),
        Vec3(-87.45285537955262, -292.1388809885602, 125.9245047297564),
        Vec3(-2.6097614349141502, 61.38814525645277, -238.2875371484905),
        Vec3(224.299624134995, 129.03756654996485, -209.3802240227379),
        Vec3(130.06059150251497, 383.3327075111684, 747.1711009467334),
        Vec3(175.21625830299126, -416.5788853289343, -243.601499615591),
        Vec3(-364.4521367639826, -106.04727942609144, -330.2576636999889),
        Vec3(99.5129561601014, -7.0737493392099795, -171.21060084342537),
        Vec3(-73.40659512011415, 32.77319925171829, 116.94825700535361),
        Vec3(-141.09592782476182, 91.41482357556552, -122.03803728317106),
        Vec3(-607.1238052871533, -332.23592016130823, -493.8941057121413),
        Vec3(243.01331892218482, 308.991212840969, 92.08162012563092),
        Vec3(185.80085793917712, 216.10179468907106, 302.86857771039007),
        Vec3(-401.0890281173852, 388.4391946603248, -367.9309797163247),
        Vec3(-100.81625054195577, 110.90568711955605, -60.98830214711071),
        Vec3(57.5827034562495, -200.39318297705665, 221.16502473467597),
        Vec3(15.479198217011403, -294.8476780974793, 313.05202536753137),
        Vec3(-22.68966762034828, -16.883065945378583, -222.31114691758262),
        Vec3(60.131175414220685, 259.0717252907813, 43.812674683776464),
        Vec3(-244.29545015574348, -13.003701760585107, -521.0596730034538),
        Vec3(253.30455540194777, -53.74416045641287, 322.4507796394325),
        Vec3(59.53271516593827, -172.86192268690564, 428.91232924831996),
        Vec3(-94.18810336875917, -213.91761035510586, -774.2959734446436),
        Vec3(-42.66916874467921, 67.67886910782568, 261.31730399501794),
        Vec3(-135.18139010892307, 182.14511102572936, 39.19620603730564),
        Vec3(-267.48903981479583, 323.93677093152576, 155.57926646179976),
        Vec3(265.68057580662486, 237.15137320106408, -73.8073231300024),
        Vec3(163.6422175243033, -344.1016778626731, -98.9553131951134),
        Vec3(-417.57769260569216, 372.9724079786472, -354.9144199200316),
        Vec3(174.60383046684865, -295.19215783245784, 245.12914586006102),
        Vec3(-89.98904496649995, -87.2775956819704, 353.4653488579693),
        Vec3(286.75766928201676, 187.08878816951588, 7.134623737454445),
        Vec3(-658.3418339179785, 0.8204879635860092, 161.42627569588382),
        Vec3(-179.1795611511418, -123.25665912137428, -27.946948054390216),
        Vec3(-503.57735364056236, 43.16812666561538, -255.79483040075291),
        Vec3(402.9967717966019, 165.9311189465745, 269.5058302465449),
        Vec3(293.37065377232165, -273.6332290743603, 320.63041436390574),
        Vec3(-262.10212008659437, -46.7955813969605, -581.4088275555852),
        Vec3(-30.2320497047736, -9.079949804681405, 147.56286166198885),
        Vec3(-43.93173355284762, -132.38098797288382, 270.16597187974565),
        Vec3(-168.36792036071273, -53.98374514539546, 225.78668913751721),
        Vec3(334.7967152701796, -263.8406209142154, 166.4503781499664),
        Vec3(-186.90785272583642, 182.96718206224855, -256.451054140323),
        Vec3(-394.49463290750947, -146.77389944500393, 594.9063097843305),
        Vec3(377.0132355528018, 172.2165949478846, -658.2310181509444),
        Vec3(87.99398155638868, 95.45955377229726, -224.7212655405849),
        Vec3(91.35908549169696, -283.11956651302893, -158.37895642205956),
        Vec3(-22.156629024982795, -67.2336785087731, 193.14902926414084),
        Vec3(41.72481944476615, 343.97275897996354, -77.0897828280966),
        Vec3(-227.92503372467664, -139.8115628103093, -139.55704139377036),
        Vec3(-154.60617592523425, 6.595226244996243, -26.050724130230446),
        Vec3(-202.81419832720695, 11.338982518476884, 309.4286928931821),
        Vec3(435.77631768531353, 20.104427710837115, -230.79703011625188),
        Vec3(-280.3722173344336, -5.317699878558584, -173.66993018163438),
        Vec3(118.23315941393871, -405.4829647949468, 37.80228471408086),
        Vec3(-458.5673721340151, 112.65705030065294, 409.6634223706711),
        Vec3(280.331368333651, 22.57328190427317, -384.40378205387753),
        Vec3(185.38078021818973, 9.77483717142891, -103.57091253465515),
        Vec3(-278.14912325432425, -36.20838885976491, 267.73561222615245),
        Vec3(-34.830853667667476, -18.943233607275854, -335.7974137018089),
        Vec3(478.4250193364935, -316.0844609317257, -106.9495030481627),
        Vec3(-206.10138317271623, 479.25253681999743, 26.423499841774987),
        Vec3(237.7517285969004, -408.2186042733612, -104.62920759536269),
        Vec3(237.47608164868515, -396.53375556819947, -62.01554416514412),
        Vec3(246.13264323273938, 299.2182541382955, -101.60304862993172),
        Vec3(-88.3888451297483, -234.54057850287776, 249.45343181489312),
        Vec3(74.20321439970564, 32.783072104780274, 27.84285720322591),
        Vec3(-227.73218962858508, 25.955765828861534, 768.9909275501468),
        Vec3(218.04593781599476, 68.13904535430797, -373.67055762036176),
        Vec3(-70.85593969488238, -465.75757735272515, -386.9288173904391),
        Vec3(337.4971335703799, -610.401933199198, 41.7376937632877),
        Vec3(-128.19389877563995, 225.52623728832899, 120.85438004154167),
        Vec3(-197.36890509908193, 149.33425686805785, -73.66235648792265),
        Vec3(-529.8844689771222, 136.96655160230569, -520.4899602281171),
        Vec3(358.12374781973904, -33.68519613067586, 90.92035558030787),
        Vec3(449.57843590236314, 102.2033934671651, 148.3933944651018),
        Vec3(-573.5551815876588, -168.7468657009597, 91.64110217984988),
        Vec3(438.7190198207128, -172.82187813986562, -87.74461307839385),
        Vec3(346.16872244254057, 67.52112917677196, -299.6681425397538),
        Vec3(-81.42562048805756, -169.45181146484416, 686.7534291826855),
        Vec3(-48.77863644337722, -57.02529540436205, -239.34917988192007),
        Vec3(98.97471436009545, 66.29734927568607, -147.2160842675108),
        Vec3(571.4868223675801, 109.37640359293688, -241.82852144361874),
        Vec3(-27.302491797473444, -112.70580053011595, 310.4651161763384),
        Vec3(-296.45138867950146, 16.11991536911067, 190.39253933211316),
        Vec3(244.1906180549886, 445.984116282156, 730.4335661266872),
        Vec3(-24.92525709360532, -287.34251883027173, -224.27905729827484),
        Vec3(-52.28210358734295, -112.55664731155846, -406.45432093102903),
        Vec3(-35.93533629631095, 303.0684473948616, 544.1190864213654),
        Vec3(-280.26263220953695, -144.32736752996388, -186.22540145446177),
        Vec3(139.8295597293426, -77.8076576832115, -270.5449832550807),
        Vec3(-355.6661141215022, -154.03179563848596, -206.73818107160739),
        Vec3(243.1742601175141, 213.0696851632588, 33.94148512402239),
        Vec3(-15.957294605174972, 143.9105658917933, 300.125564885917),
        Vec3(177.7591556586813, 492.52778610498217, -74.34220357918292),
        Vec3(-226.830631991299, -155.6469051553185, 87.86685132810285),
        Vec3(-64.02516018286668, -92.32797627847043, -62.480563960283256),
        Vec3(492.8294568021554, -161.5459500935738, 549.2843326732387),
        Vec3(-110.76036948748197, 94.9855493501626, -234.97576425629097),
        Vec3(-449.1619568111018, -102.48984366369703, -200.96370925656765),
        Vec3(570.9163346756443, 82.30432988258791, -186.75547798110142),
        Vec3(-167.3224429913125, 124.82326502308682, -16.057485298110592),
        Vec3(-223.22476312479617, -204.6397809328862, -53.18667195553344),
        Vec3(-271.5556269469767, 683.6702538648904, -102.07762941144526),
        Vec3(134.62606565263354, -410.44767192891493, -37.45205679258815),
        Vec3(74.3886024417582, -380.2265778392809, 167.12841474834616),
        Vec3(51.97506596324779, 526.4093096850187, 29.1896719191959),
        Vec3(-120.11128754642573, -355.2133509248956, 32.492509082662444),
        Vec3(98.11863631229781, -144.1612884646125, -98.15665591329949),
        Vec3(622.8039474941972, 627.8848271429141, 309.5672683525694),
        Vec3(-224.65594596550676, -207.01495109188178, -268.61904935094833),
        Vec3(-145.45934471329684, 68.77728500682441, -184.11358107811122),
        Vec3(305.8113859131425, -40.76685071110852, -999.2423905417055),
        Vec3(-200.5858959306156, -3.212783801977652, 104.99355118038582),
        Vec3(65.75975451314979, -6.185961208359154, 260.7217381132384),
        Vec3(-465.325630006053, 57.243087558111455, 244.48446186690254),
        Vec3(284.39071908119865, -403.30368265456985, -227.93324395547202),
        Vec3(319.566870539745, -154.25285215328194, -244.75552729911286),
        Vec3(113.95989360867209, 210.3673617061578, 24.4869768189808),
        Vec3(460.4983861239349, -226.50133066112335, -88.59685197216878),
        Vec3(-98.09787243132699, -244.04610698615312, -163.59809551548102),
        Vec3(-585.0908382278882, -889.1343804189501, 261.8360010792674),
        Vec3(-5.758449030052453, 117.8568682173512, -39.074275143409196),
        Vec3(211.12632405257287, 426.9945188148396, 26.337265835984113),
        Vec3(-261.1123357379434, 397.37927572746406, 385.80418113770185),
        Vec3(193.80057938382404, -156.03070640112747, -164.3047139303805),
        Vec3(95.64032398633752, 130.1828205456674, -356.2199481014633),
        Vec3(613.4334217010296, 72.2340712870241, 388.29832354067554),
        Vec3(-298.17301652117567, -485.9210453626593, -198.3924134013901),
        Vec3(-327.28584175027964, -69.94548015665167, -100.92794190916767),
        Vec3(-95.35204059147279, 758.8346740109748, -194.78919552260624),
        Vec3(100.98823322759415, -213.1945318796383, 64.24599810971583),
        Vec3(225.68708404163152, -77.61888875340239, -192.77536729645487),
        Vec3(-246.52466151347878, 190.681831372716, 252.93416398953948),
        Vec3(129.60771170235893, -186.19056827830264, -68.5035780563697),
        Vec3(93.5709211341456, -118.66636057842628, -199.0551553620597),
        Vec3(-234.7503041241301, 650.242143690075, -687.8965017480454),
        Vec3(-75.97170398650412, -40.86957278265794, 221.02893728463457),
        Vec3(224.04760000079307, -161.68890748532917, 235.32139129294472),
        Vec3(-74.00277073089084, 549.5541839324454, 907.3249138777222),
        Vec3(-41.53734323654314, -87.68222100616845, -283.5794117156566),
        Vec3(-163.26973194049333, -119.24675888406907, 58.022839403782754),
        Vec3(-361.68978882794306, 203.22693867650017, -228.11712287648672),
        Vec3(354.5436461015289, -398.4125169812179, 73.95001180998987),
        Vec3(212.09479737847641, -38.76073634658328, 29.716200325408277),
        Vec3(-482.44796651544783, 450.52466265861904, -49.53658099063551),
        Vec3(264.8763703413555, -176.1523824510752, -179.5364728358089),
        Vec3(25.934723044885093, -296.1137082397114, 153.9122506945108),
        Vec3(-454.9541821120418, -54.12890962459669, -206.31506876664034),
        Vec3(447.0240999051737, 243.6131266820493, 281.33572023230363),
        Vec3(278.5724047530407, -82.33441841810905, 288.83145603927267),
        Vec3(-205.86073760331593, 252.96028131048593, -327.951075003245),
        Vec3(-8.102335280217162, -21.25603560331037, 282.5453411439364),
        Vec3(103.04228102611414, -82.98598255630152, 84.99892701613994),
        Vec3(277.886957196108, -390.7010994011783, -20.81660705735313),
        Vec3(38.85218525806698, 65.47184623788992, 145.50643468675383),
        Vec3(-389.44531367623784, 209.15181976831985, -267.86500262317975),
        Vec3(341.2590622095847, 355.96996723289317, 297.3182674750273),
        Vec3(-143.21413877558632, 435.64524536899194, -402.5198602393208),
        Vec3(-9.1574004574585, -214.91977155827013, -89.97013896442718),
        Vec3(88.10148161888606, -381.39540185252184, -519.6040717554135),
        Vec3(-99.01610917808237, 367.8255314129375, 132.8955373395401),
        Vec3(-18.737962788115055, 57.01667717253856, 255.84577394353622),
        Vec3(-99.0955840838852, -530.6962970253314, 65.53533222740258),
        Vec3(-237.92161335829448, 67.03743308303048, 110.37769253762544),
        Vec3(309.8838329638783, 132.35911932869914, -138.5522797283263),
        Vec3(-96.6853311031437, -710.2881975293326, 58.34437189638875),
        Vec3(213.86805033080176, 288.2392680555765, -105.81975964779848),
        Vec3(22.772096275540605, 235.6350351773449, -65.47097586247526),
        Vec3(-94.00421852809512, -192.87250405359734, 246.73260365131748),
        Vec3(-247.11680926114641, 225.04786856058493, -210.8380653655543),
        Vec3(-127.8970001261775, 57.72938451084627, -159.15484814975818),
        Vec3(375.5728222836788, -381.28196818590226, 439.7283099955979),
        Vec3(-217.16080164887964, -53.01981186700791, -90.89745186747565),
        Vec3(-95.68140738731651, 305.01411462510595, -279.0369626510706),
        Vec3(371.37239377715747, 317.440137340223, 160.40093833792014),
        Vec3(-94.21505282887951, -350.9893416523208, 78.21621943988582),
        Vec3(-222.54281618460953, -171.3855599524479, -146.0877533579349),
        Vec3(-323.21815571674796, -342.9579687506031, -88.45352114131323),
        Vec3(388.04238540141733, 255.65859010927807, 14.001619399996128),
        Vec3(213.20205319044996, -32.28586591346377, 162.39314984533453),
        Vec3(-805.8058051882732, 372.3107473646138, -203.15301715170082),
        Vec3(395.93943462922255, 57.12474515707026, 45.644819760124264),
        Vec3(293.94722535676993, -37.328318287498064, -182.27190610751978),
        Vec3(532.0401793893479, -77.39412079621457, 52.21771947192521),
        Vec3(-161.227850710132, -328.78876308408985, 112.0919283045154),
        Vec3(-41.03660306291223, -359.6277647538153, 14.983452501856092),
        Vec3(-569.13236331891, 185.63153390797075, 228.2621086521638),
        Vec3(22.441031195862237, -60.82374909827159, 100.45918016682288),
        Vec3(117.5885525974438, 87.53966101612221, -203.05548955437476),
        Vec3(-214.58602140325254, -36.579422801152866, 605.2727706004744),
        Vec3(-200.27682635530584, 28.830562583079605, -289.31910886246925),
        Vec3(159.49590951095988, -121.40513788929071, -28.819158731892223),
        Vec3(-409.8711277579596, 516.8007911315127, 132.53563908239533),
        Vec3(-167.0747576875269, -336.21944443805734, -85.74302903364367),
        Vec3(324.6782925866288, -227.62736204544572, 239.66332745765882),
        Vec3(-522.3235597701688, 283.2165343551196, -406.29658609135186),
        Vec3(-47.239971300907676, -17.319738387750075, 117.69492699127983),
        Vec3(139.0179249406367, -65.17933149069559, 56.22262714627035),
        Vec3(194.92258039190506, -495.6735324881299, 191.00496219578724),
        Vec3(-155.58871524729037, 158.85939335788166, -59.76117566037533),
        Vec3(-46.97301692216656, 118.25286655504905, 206.041971459776),
        Vec3(-162.37161796443198, -113.6059127535774, -604.8797596743224),
        Vec3(57.7537729943822, 204.73071565228176, 214.1526142931406),
        Vec3(48.971414258803634, -68.10961851908384, 209.61380632419835),
        Vec3(592.3172324044921, 161.93944862884157, 731.2515728272998),
        Vec3(-373.4260162119567, -153.043454184166, 124.54463246735587),
        Vec3(-280.80249648048533, 65.32981422622129, -212.025129334847),
        Vec3(186.31448691865677, -78.90028853688928, -384.906890722121),
        Vec3(-286.0808210223127, 11.502688246210578, 233.87274807182425),
        Vec3(-27.574013807316593, -237.75767920620768, 24.276091840117044),
        Vec3(173.97345602308616, 47.625454491439996, -438.22872656915433),
        Vec3(-105.84156806925954, 117.86387370805265, 269.1195268405124),
        Vec3(151.41030640361936, 82.54193589475396, 299.8083517509004),
        Vec3(-584.9051666003791, -540.6482923909504, -96.85519120238199),
        Vec3(273.94930496542975, 217.0974436280911, 33.49000156469636),
        Vec3(348.7666456141812, 303.4735198887688, 41.90434154552069),
        Vec3(452.56750856888493, -375.1900613590575, -509.7678128241393),
        Vec3(56.47335585171018, 89.7725082191558, 260.75623048626727),
        Vec3(-359.37837810269326, -10.624492892056933, 87.6887444718302),
        Vec3(-377.6978561428832, 616.8984655801014, 133.2059803025605),
        Vec3(40.134048794209875, -78.89965459234843, -159.62634062699297),
        Vec3(360.66030946972086, -288.2118563365973, 108.3758118381954),
        Vec3(318.5607438464587, 96.50621836570497, 292.56313106969037),
        Vec3(249.26781863372435, -222.2491090376534, 21.576020205573677),
        Vec3(-340.5686506786288, -418.5024552015164, 91.60690985250798),
        Vec3(277.7794522103271, -892.1351514277258, 372.7040292998588),
        Vec3(116.95787510292236, 322.6731113652018, -107.63726638764396),
        Vec3(-308.3546601856363, 237.35078259429588, -294.1483889125617),
        Vec3(164.1738590858144, 438.0549652370004, -425.9657274016055),
        Vec3(-63.05481952284951, -304.942673686842, 398.5654893635808),
        Vec3(-69.89996880598419, -90.5574199449692, 71.62949433934261),
        Vec3(401.94154181825047, 527.4927748047301, -441.3641269369436),
        Vec3(-212.44927885097604, -334.32524893063555, 60.89200958920557),
        Vec3(-55.705402265470276, -215.19833979078726, 185.46660493215018),
        Vec3(421.74746596498085, -400.5066571353421, 406.065002095627),
        Vec3(-312.65620354632034, 273.92078627718433, -173.64786975397487),
        Vec3(-185.49903825078508, 278.9923405014473, -205.32705783231194),
        Vec3(-194.09762863805068, -341.91084014807365, 241.45607142436228),
        Vec3(-46.74969004931445, 6.173596837525256, -16.88792508920924),
        Vec3(-109.8945867456066, 203.78117837060773, -271.0692141914807),
        Vec3(-688.5161850407885, -192.27352781951956, -40.526530898570634),
        Vec3(264.0441921794354, 207.8029017969357, 48.022138260367136),
        Vec3(510.5755183071709, 39.212284510797396, -132.72271330720213),
        Vec3(58.91843480738649, 48.16337984914947, 428.38999683203394),
        Vec3(-488.55847606571126, 41.551085099982664, -50.65864762524458),
        Vec3(-70.79796461361309, 184.62653852151868, -173.03256792603696),
        Vec3(736.4806590328083, -182.66692835105675, 74.59070581354057),
        Vec3(-596.9188933568696, 381.18771263768883, -140.21476278402355),
        Vec3(-204.72373982572086, -20.43531170016827, -71.06998781726801),
        Vec3(-48.55272200733267, 81.05798542574162, -20.543518122230417),
        Vec3(212.30179451958665, -92.24219154270519, -248.2649876337345),
        Vec3(-53.92002968681649, -223.7934791614247, -88.83796848312214),
        Vec3(-552.4083690241632, 729.0134449872447, 121.0187961635613),
        Vec3(388.52435559625064, -300.507650523301, -428.23107647492037),
        Vec3(56.60142788506428, -403.9019799029175, -120.16275565049197),
        Vec3(308.37156127951414, 393.6081020430882, 14.959894152546308),
        Vec3(-290.8469062326849, -126.10075403283811, -202.179648519888),
        Vec3(135.281948695074, -349.368436260298, -125.09978807611155),
        Vec3(731.1720290468502, 148.9220767745155, -125.78362223638345),
        Vec3(-634.5063246495208, 170.0384301552944, -192.212272470312),
        Vec3(-140.27435992540563, -168.1995921499923, 79.48356744390934),
        Vec3(250.804188818791, -441.1486773375977, -36.70551482612697),
        Vec3(25.085877970121736, 207.87404818412796, -163.67099199580164),
        Vec3(26.950675730895643, 234.0932818942474, 28.56397358876569),
        Vec3(-9.454435747271129, -20.917863986387033, 301.2174328607685),
        Vec3(237.82781821871458, -52.948950459254156, -107.70003032789225),
        Vec3(61.73475120535793, -296.08429582596233, 143.95438124074064),
        Vec3(170.14773327424908, 158.21624761087693, 462.41826264479556),
        Vec3(-10.212404180033044, -342.449842395608, -179.30692529008857),
        Vec3(-7.9008953016637955, -27.386307692329506, -279.06223556604647),
        Vec3(623.8039636448013, 204.85655792186034, 327.2389520583137),
        Vec3(-151.40310120225476, -102.53356803519355, -373.67932569927854),
        Vec3(-114.60108510899096, 49.56765040252641, -92.43163950075933),
        Vec3(-655.4535985329173, -231.14956837314008, 3.116939149128992),
        Vec3(296.310271026448, 486.70967996995785, 35.39255793924772),
        Vec3(172.08210633165635, -230.4114817253959, 67.2429770869512),
        Vec3(-117.51457238759482, 245.89189704794484, -365.4507607803772),
        Vec3(137.983421929608, -238.63953239741107, 15.905738135326954),
        Vec3(-20.615187840314757, -85.26834402580137, 141.97729391004376),
        Vec3(268.74586567330704, -444.5911718723675, 320.8208665983761),
        Vec3(186.01833330852335, 581.0752468784148, -18.836715858421186),
        Vec3(-186.37150359219203, 105.4510492534819, 102.23449649830738),
        Vec3(-378.59501900893537, -212.52931276037933, 213.47242953695846),
        Vec3(34.54860314915441, 136.27116508160643, -172.85833667790186),
        Vec3(82.46135646309041, 35.78982735165146, -192.3643679675803),
        Vec3(362.6764195504091, -310.1156978932209, 464.7454537873777),
        Vec3(-390.3172393695492, -77.78660095363016, -339.32454199292266),
        Vec3(138.9725052448885, 253.2950341885662, -195.8551343953917),
        Vec3(-77.77699415519196, -418.58538413391784, -74.28670560809776),
        Vec3(-194.84445301328364, -7.877080531814001, 165.5212090227259),
        Vec3(357.5114108099747, 169.62849296589957, -58.7318256241748),
        Vec3(-205.0719400595654, 74.5439957691607, 459.81989777117826),
        Vec3(225.79111975528735, 26.27998014053224, 38.169840613421854),
        Vec3(225.28920964919845, -302.2653713591056, -67.27751628149855),
        Vec3(607.5108973708401, -504.1751214140174, 160.94819170794239),
        Vec3(-69.93045100299327, -36.799354260333494, 30.49281294423247),
        Vec3(-223.07693729429133, 321.7339859642227, -119.94994217496966),
        Vec3(-386.152206081632, 783.636531768563, 107.55137251024011),
        Vec3(9.610827297518053, -347.14092395716125, -74.09274261126966),
        Vec3(158.34185609366665, -11.999361448019783, 222.95633958634434),
        Vec3(577.7097100440483, 48.895330528079, 241.95854692308532),
        Vec3(-215.62481585775666, 111.85868020159693, -231.96801104109238),
        Vec3(-465.5774615091408, -100.35047728153442, -27.60777043752202),
        Vec3(349.00841247283637, 135.21443424697273, -361.5545055351252),
        Vec3(-240.73972384220488, 65.15458605945898, 418.44623667785595),
        Vec3(-7.185843457200288, 9.124662863122898, 252.5855944998341),
        Vec3(-670.9137873329693, 15.628041411782526, -69.24683889488085),
        Vec3(281.5562132077926, 125.61210072926009, -207.65352575771374),
        Vec3(12.288853384156322, 46.72417772340007, 88.99682617128613),
        Vec3(-310.16819954567063, 75.12792236473321, 146.45728694979894),
        Vec3(74.81025034077423, -67.3523719690747, -192.46124232071003),
        Vec3(80.39367736970732, -12.269727817000502, 21.517057746940612),
        Vec3(504.6565508295855, 145.38275628042413, 41.84222177852041),
        Vec3(-343.5623653676825, -52.04010867142658, -115.60974758310779),
        Vec3(-274.03281448918597, -22.6974453925755, -111.28736000953147),
        Vec3(-196.35502702123955, 337.8740247082284, -360.83640000130913),
        Vec3(39.71851872750599, -176.69454550309882, 53.6900525850605),
        Vec3(171.77374557034238, -286.48483828517936, 176.38100740537547),
        Vec3(402.9083736209138, -35.23596085508224, 314.0963205000684),
        Vec3(6.208976548032057, 121.898605161116, -482.97911323925143),
        Vec3(-106.38659258059808, -122.98007642890884, -323.82127140573044),
        Vec3(-126.78592336340877, 612.4875853611072, 714.8845442538958),
        Vec3(-130.95742751268318, -327.8035588626552, 13.737051202125194),
        Vec3(43.71022005636412, -234.93308610993728, -157.91845354232677),
        Vec3(764.7690122403798, -156.77912679824576, 607.6853461644266),
        Vec3(-279.82785463996305, 211.25912315893075, 124.66847330052909),
        Vec3(31.54241986777193, 259.96481449469456, -226.62496190811885),
        Vec3(634.6703333518747, 134.76599836974347, -452.5086582950348),
        Vec3(-67.58336133802118, -144.24401501768097, 184.37864194783114),
        Vec3(-297.7952840660286, -303.8271815484487, 0.30101761797852833),
        Vec3(328.4835557514249, 283.13839189958037, 435.2983771026576),
        Vec3(-43.43474055359954, 249.38873432379353, -265.557800712022),
        Vec3(-83.87619214816716, -170.04197577715345, 151.66596729141796),
        Vec3(-258.36384461101056, 23.057091916769785, 238.25614025098204),
        Vec3(146.07267917397894, 103.8552762455757, -480.2259138132139),
        Vec3(372.1197039472768, -49.14203788650857, 85.21854473285518),
        Vec3(621.1763220000987, -250.40437867264362, 309.00536151120116),
        Vec3(-275.5795704453378, 231.3583108255857, -88.82107171408583),
        Vec3(-189.29412828047478, 151.36733815176305, -221.13733462711662),
        Vec3(128.94381918230957, -33.18758467435246, -422.22565139314895),
        Vec3(-352.2990515271668, 82.92506167420726, 34.68838626963287),
        Vec3(133.53902986709014, 51.727136668888996, 204.40305583162265),
        Vec3(-497.37923463749974, -121.76607553558905, -49.41904612249853),
        Vec3(305.44954752398826, 112.78595133385349, 4.749053765389011),
        Vec3(125.83710649669833, 13.163438021461062, 149.83453651819747),
        Vec3(157.06912791278268, -0.5171966142119424, -678.1427777935485),
        Vec3(-360.19887193863485, 124.22102885526218, 294.9326731520692),
        Vec3(-255.04962761056123, 111.62485842221265, 203.03660695470904),
        Vec3(-292.9522603455384, 137.08735478655865, 621.8258240445643),
        Vec3(114.45010624125372, -59.64977272268843, -191.2132501685082),
        Vec3(-12.411578194131781, 103.20865834708889, -506.930988926099),
        Vec3(290.14080300222923, -375.58836058229264, -151.2870847893579),
        Vec3(-117.54170340541573, 469.116879143056, -152.40985286627517),
        Vec3(-178.55829097854638, -49.900688136833786, 220.5830222523799),
        Vec3(391.14454811810003, 531.5288051680091, 249.98791692405663),
        Vec3(-583.7409446744587, -341.1384248320302, -18.439901520528462),
        Vec3(-82.62162728437389, -35.02370226948587, -200.97065983524251),
        Vec3(-404.588325394559, -293.3517374583258, -355.19998309556837),
        Vec3(47.3362991211429, 220.2421315309823, 223.6934819900016),
        Vec3(191.57160728928307, -69.89142137133237, 85.45212433443848),
        Vec3(-307.1788543156219, 898.603579545786, 57.777382167454284),
        Vec3(436.0276793597111, -288.27080875909036, 128.92230922633965),
        Vec3(-25.696018868319396, -609.6668554680576, 27.354330973549036),
        Vec3(134.53485728281302, -863.1490487463003, 7.55358533767775),
        Vec3(5.82649846805037, 222.45609829127227, 163.53219858266843),
        Vec3(-43.712515913246115, 281.64431832406166, -87.08799262286338),
        Vec3(637.7742866308085, 304.6587993385723, -642.0124184924257),
        Vec3(126.97153318910993, -85.32640272260798, 246.00996641890754),
        Vec3(-300.6884250888344, -91.77690391919501, 246.4295213323357),
        Vec3(387.27044011243385, -82.48255245008419, 217.47527107587888),
        Vec3(-14.940331308857358, -164.74177925605835, -287.2327509247083),
        Vec3(-257.87748881436744, 11.809289741603672, 138.3446665319014),
        Vec3(577.2679078626843, -250.21001338910187, -277.3369530946959),
        Vec3(-282.0363468890284, -36.17855136726721, 22.048373133570482),
        Vec3(-134.7451993216848, 280.74108286050057, 218.47129336793103),
        Vec3(-193.8962746036012, 485.2982947874612, 299.4412942973369),
        Vec3(17.16241104600595, -210.14712380790678, -189.10103821046442),
        Vec3(76.92194834694459, -31.337624771612827, 96.68889655327172),
        Vec3(630.0668312787265, -400.23005935079476, 181.36518690260235),
        Vec3(-250.09130296853994, 305.6744427499183, -210.93746077607477),
        Vec3(-383.2488674743835, 327.6804240583259, -172.94242028992176),
        Vec3(73.23368257902733, -131.4472136725954, 752.3484326841066),
        Vec3(-57.69978616889563, 96.88548556561102, -553.4908020962104),
        Vec3(-166.43100257299437, -11.695430425660156, -351.7235364565684),
        Vec3(491.228097123417, -542.4077749028473, 710.1342661566871),
        Vec3(-91.2119044228842, 408.3221870989447, -387.0742839973131),
        Vec3(-104.23747094053064, -194.80935879057512, -146.52341438529993),
        Vec3(313.0040153135171, -135.3888007619903, 220.4997248835101),
        Vec3(-487.3267874613191, -43.92592526959597, -25.44129044175576),
        Vec3(-154.285006296245, 127.40939857390963, 254.38116439651583),
        Vec3(-116.95731351105863, -237.0365834340773, -301.6290164354321),
        Vec3(-102.71983863601567, 281.34141019219516, 429.95906087048684),
        Vec3(98.60524549006234, 69.34468577352828, 79.48869874911504),
        Vec3(279.8574704659307, 547.4574046928512, 7.74649169850069),
        Vec3(62.828239428798646, -35.644496681660314, 67.55144834910989),
        Vec3(-124.16274013373202, -266.10917000985285, 171.01805953382186),
        Vec3(467.07610186013306, -137.81362696517292, -708.9007529966486),
        Vec3(-179.17422437424494, 13.025404631925724, 168.11275807641016),
        Vec3(-262.2255109426203, 56.42206232341501, 117.09032776785075),
        Vec3(58.20833329696738, 48.613368504855416, -572.7032892022513),
        Vec3(27.96132210619347, -101.65864258202168, 256.21802343532363),
        Vec3(-138.76741655453398, -21.2341274466359, 93.92135436389086),
        Vec3(78.99152809446993, 361.12363776076154, 26.62134492435415),
        Vec3(-268.0120697505899, -109.08303805052158, 1.3635238691132316),
        Vec3(186.37358587599041, -163.87403602761174, 126.94521216192753),
        Vec3(-483.20013472222496, 101.44448086225994, -266.99308870651856),
        Vec3(81.88348785839946, 112.63875606345701, 279.06738748752235),
        Vec3(57.4830077303275, -229.22969619673017, 200.29706555483068),
        Vec3(322.24722802404386, 104.24996045823842, -76.70125948877903),
        Vec3(-303.3238597300328, -229.16713804079458, 25.0240171259158),
        Vec3(-307.45777387177907, 135.19078133981913, -89.10952709869348),
        Vec3(285.09317684199146, 395.7647971144089, 88.26577084274554),
        Vec3(-290.9840438847235, -226.53830623585532, -16.796689636714003),
        Vec3(-23.83913870765369, -220.6526293716373, -187.4772525940614),
        Vec3(-87.01301510311579, -374.8631668085156, 54.85654373558141),
        Vec3(-250.258540694562, 274.1321785560846, -282.868527504452),
        Vec3(156.57046895421834, 30.488710446415638, -203.29654567913954),
        Vec3(138.88414554474255, 36.705825153250224, 63.58758294392938),
        Vec3(-4.151571944920135, -206.61221790456332, -99.658948111508),
        Vec3(-70.19767894734504, -138.86098184015916, 213.00848447002988),
        Vec3(-76.39918265300803, -150.6542911790245, 463.5555072133937),
        Vec3(-124.82936097661748, -183.16458718145077, -432.2498258464293),
        Vec3(107.61052644067013, 196.24722683573492, -80.25788683696474),
        Vec3(-639.7206172413605, 159.83837252039942, -96.40857828147178),
        Vec3(319.3869620018208, -95.85515371593043, 46.62835038957132),
        Vec3(342.52961175614536, -176.3485447649931, 132.1060961515659),
        Vec3(520.7563945995084, 432.15659620947457, 111.7784151280868),
        Vec3(-300.9403100085422, -238.89318795815262, -37.69035433643073),
        Vec3(-347.8083865480956, 67.95605091292002, 13.157759331918607),
        Vec3(1.5749012469905102, 430.3719470570706, 197.47929117707534),
        Vec3(-78.26989547040932, -340.30657694098545, 43.88020100777593),
        Vec3(73.85568916021155, -383.5418100622305, -60.7756672464542),
        Vec3(90.01352035714248, -698.8095371136719, 161.2599804433718),
        Vec3(-150.47445153358203, 173.9490213227927, 96.11280656668865),
        Vec3(-122.86959495054164, 382.11089136243186, -40.81352305302425),
        Vec3(314.9315882143728, -258.37212719399747, -42.139726323323345),
        Vec3(-148.25848969940404, 289.61038370201067, -70.75381164996946),
        Vec3(-308.11282405564015, -52.44439012540086, -144.48587589718113),
        Vec3(-374.71942537327294, 343.3335092983839, 164.04489669219944),
        Vec3(93.81792620376217, -515.6354264427762, -413.51432518435627),
        Vec3(461.9949240574166, -0.09008484391950301, 110.3541715874114),
        Vec3(-218.62882048951369, 425.32378382006, -513.5901548412263),
        Vec3(10.963230050428185, 31.56447804567961, 294.93650039424455),
        Vec3(32.18094429272624, -99.32424744128294, 352.54963154899485),
        Vec3(451.407785318005, -769.9530349296572, -185.2498205291093),
        Vec3(-157.0319740102172, 391.84102588800323, 55.71567257894609),
        Vec3(-99.11311391591164, 294.5711001972031, 187.28407776299665),
        Vec3(265.6825036786936, -526.649964721742, 217.34022020671608),
        Vec3(87.40422739093127, 258.1004620432469, -273.0363388629197),
        Vec3(-201.84490114860725, -49.40784641125451, -300.6952856923296),
        Vec3(-521.814880804564, -615.3258874534415, -256.4726239810002),
        Vec3(300.0336167913912, 427.5500550949781, 14.325919378935538),
        Vec3(140.70851604331958, 225.52329767742208, 171.16777412027113),
        Vec3(-236.32025130170496, 439.2270680201855, -390.7403710451149),
        Vec3(156.78573037355073, -162.48705980757663, 43.74351661511086),
        Vec3(213.72101149918407, -17.284880435513287, 604.9972686468564),
        Vec3(-494.73667110153434, -68.89668838318043, -222.68431916369033),
        Vec3(78.9821487923666, 9.468356163031238, -25.498020388630966),
        Vec3(212.25737233103033, -17.014340050902813, 267.5905554210995),
        Vec3(453.8067723566274, 364.4637246448333, 149.99069571746608),
        Vec3(-177.47592105862327, -92.47170385751126, 169.70729912876791),
        Vec3(-104.09745588910576, -123.90105537632645, 29.522603476622823),
        Vec3(-107.90878722010852, 633.7491206220252, -240.4464663940923),
        Vec3(159.34660603668124, -288.6290805893232, 229.07768586571527),
        Vec3(82.68827196740764, -7.161172237208839, 97.78218824458624),
        Vec3(81.29530506356198, -376.6140571239579, -574.5689313963147),
        Vec3(-87.68559126725243, 462.3667113764172, 184.41737271069144),
        Vec3(-123.19231964696777, -10.559140090610708, 185.36834732329862),
        Vec3(243.01796459337638, 282.25591797878195, 181.47417858162524),
        Vec3(45.2898345551462, -413.43208301813036, -74.19004547178974),
        Vec3(11.348452030736503, -36.93955436131568, -142.26172505197883),
        Vec3(-141.46232685056015, 742.5898479355589, 33.381914772132724),
        Vec3(-160.53203784824868, -63.65230818640389, 32.061533600221324),
        Vec3(-40.00480367990102, -133.40392761241463, -151.80158045713034),
        Vec3(-626.2242891551515, 167.54149648585627, -59.1097468593287),
        Vec3(254.89919627881363, 268.61581103224705, -24.795078403231052),
        Vec3(157.28962521631803, -295.6151357122448, 226.90365007432416),
        Vec3(275.5351313879823, 295.9876800692807, 63.89447055579131),
        Vec3(-322.70702417452617, -39.0452311440029, 111.25885306794342),
        Vec3(-349.06739199083313, -223.66742515428317, -177.45885174916506),
        Vec3(-74.92000429420452, 110.12864306681907, 235.00430137274853),
        Vec3(254.890722497277, 65.05627636369265, -384.8900072500228),
        Vec3(-190.73124035400733, 22.30064920161101, -432.4659619095808),
        Vec3(490.14912212769315, -70.1432315160618, 167.97112704968632),
        Vec3(-83.0307607982963, 110.28831525177955, -116.1565512974211),
        Vec3(-548.089261697656, 126.87402362774641, -211.47775932847372),
        Vec3(-107.93505397529555, 17.030788494915257, -283.1388171720635),
        Vec3(-3.4809026660346523, -27.953928581550617, 227.67373947798032),
        Vec3(-45.50885518571216, -134.74602913718573, 314.7601381833695),
        Vec3(-88.62115952094847, 164.37013541780107, 466.10807016366635),
        Vec3(-69.60286837324972, -159.56832569587237, -99.77915939835437),
        Vec3(-116.51859911169545, 98.82107579980642, -246.1109912835994),
        Vec3(330.91464120411547, 253.24094200709567, 278.32557143034154),
        Vec3(-230.72408649508694, 75.75839302811679, 5.5286568710674295),
        Vec3(-119.75342432690866, -28.727415427120302, -163.8918033736238),
        Vec3(-392.13986021968697, -406.41792298821673, -363.12164150302135),
        Vec3(265.1049055053364, 309.5074043470523, -64.34976323639101),
        Vec3(335.55814506611324, 53.213670844593665, 291.0120981627222),
        Vec3(-256.17287309248604, -432.08147957460494, 43.21907098068741),
        Vec3(275.356818222935, 224.93852052245637, -30.26875753305559),
        Vec3(65.29543353236605, 406.3291811412344, 128.85286503051742),
        Vec3(-604.9833265293979, -48.42128327343298, 102.39782144299821),
        Vec3(476.45168231702087, -120.11194342860455, 219.4027004166261),
        Vec3(333.913117764101, -123.58365613445721, 51.02492093651193),
        Vec3(475.3975678272806, -433.5265405655237, -316.3981032215494),
        Vec3(-212.10667007171378, 196.09782456397198, -84.82998433177549),
        Vec3(-200.20660716300733, 158.67182460151994, 124.8362392636121),
        Vec3(-19.303927332973775, 540.3592637874732, -155.01453617600419),
        Vec3(-157.5646877458501, -125.41942170628774, -48.404235316186345),
        Vec3(95.18633129549345, -584.9352579495654, 187.15265106316366),
        Vec3(-128.76536264812614, 370.2794308351056, 288.99352710822967),
        Vec3(156.4187369167104, -153.43507609852202, -224.87846656142537),
        Vec3(125.18419924488227, 23.91919275428273, -314.1009483097722),
        Vec3(131.17329953709526, -36.97312115913363, -306.54336244017543),
        Vec3(-21.286261187231993, 156.10334206080432, 306.58378116609765),
        Vec3(-98.98290263405354, -141.13219958169117, 191.02386737902773),
        Vec3(-156.3949247096025, 244.45355232257597, -63.09411046638876),
        Vec3(-35.367487762748, -291.56419465924836, 87.10776038858218),
        Vec3(83.68999913747565, 84.08039214623521, 335.84533905135044),
        Vec3(-247.3465381366986, 197.8175231855145, -196.2466922112647),
        Vec3(240.57262994423147, -12.631455037628108, 55.997914104948784),
        Vec3(233.29776444327882, 35.00686695783241, 265.51562401167655)
    };
    for (int i = 0; i < system.getNumParticles(); i++)
        ASSERT_EQUAL_VEC(-expectedForces[i], state.getForces()[i], forceTol);
    checkForceEnergyConsistency(context);
}

void testChangingParameters() {
    System system;
    HippoNonbondedForce* hippo = new HippoNonbondedForce();
    buildWaterSystem(system, 2, hippo);
    VerletIntegrator integrator(0.001);
    Context context(system, integrator, Platform::getPlatformByName("CUDA"));
    vector<Vec3> positions = {
        0.1*Vec3(1.505434, 0.0, -0.065656),
        0.1*Vec3(0.553912, 0.0, 0.057710),
        0.1*Vec3(1.907155, 0.0, 0.801980),
        0.1*Vec3(-1.436029, 0.0, 0.060505),
        0.1*Vec3(-1.781197, 0.772272, -0.388976),
        0.1*Vec3(-1.781197, -0.772272, -0.388976)
    };
    context.setPositions(positions);
    State state1 = context.getState(State::Energy | State::Forces);
    
    // Change some of the parameters and make sure the results change.

    hippo->setParticleParameters(0, -0.2, {0.0, 0.0, 0.005}, {0.001, 0.0, 0.0, 0.0, -0.001, 0.0, 0.0, 0.0, 0.0}, 6.0,
                20, 4.184*1326.0, 10*40.0, 0.03, 2.0, -2.4233, 10*4.3097,
                0.001*0.795, HippoNonbondedForce::Bisector, 1, 2, -1);
    hippo->setExceptionParameters(3, 3, 4, 0.1, 0.1, 0.3, 0.05, 0.05, 0.15);
    hippo->updateParametersInContext(context);
    State state2 = context.getState(State::Energy | State::Forces);
    ASSERT(fabs(state1.getPotentialEnergy()-state2.getPotentialEnergy()) > 1.0)

    // Create a new context with the altered parameters and see if it agrees.

    VerletIntegrator integrator2(0.001);
    Context context2(system, integrator2, Platform::getPlatformByName("CUDA"));
    context.setPositions(positions);
    State state3 = context.getState(State::Energy | State::Forces);
    ASSERT_EQUAL_TOL(state2.getPotentialEnergy(), state3.getPotentialEnergy(), 1e-5);
    for (int i = 0; i < system.getNumParticles(); i++)
        ASSERT_EQUAL_VEC(state2.getForces()[i], state3.getForces()[i], 1e-5);
}

int main(int argc, char* argv[]) {
    try {
        registerAmoebaCudaKernelFactories();
        if (argc > 1)
            Platform::getPlatformByName("CUDA").setPropertyDefaultValue("Precision", std::string(argv[1]));
        string precision = Platform::getPlatformByName("CUDA").getPropertyDefaultValue("Precision");
        if (precision == "double") {
            forceTol = 1e-5;
            energyTol = 1e-5;
            consistencyTol = 5e-3;
        }
        else {
            forceTol = 5e-3;
            energyTol = 1e-3;
            consistencyTol = 5e-2;
        }
        testWaterDimer();
        testWaterBox();
        testChangingParameters();
    }
    catch (const std::exception& e) {
        std::cout << "exception: " << e.what() << std::endl;
        return 1;
    }
    std::cout << "Done" << std::endl;
    return 0;
}
