/*
 * @(#)Skewb3d.c
 *
 * Copyright 1994 - 2010  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Methods file for Skewb3d */

#include "SkewbP.h"
#include "Skewb3dP.h"

#ifndef WINVER
static Boolean setValuesPuzzle3D(Widget current, Widget request, Widget renew);
static void resizePuzzle3D(Skewb3DWidget w);
static void initializePuzzle3D(Widget request, Widget renew);
static void exposePuzzle3D(Widget renew, XEvent *event, Region region);
static void movePuzzle3DTl(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DTop(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DTr(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DLeft(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DRight(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DBl(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DBottom(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzle3DBr(Skewb3DWidget w,
	XEvent *event, char **args, int nArgs);

static char translations3D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>r: Read()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>e: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsList3D[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) movePuzzle3DTl},
	{(char *) "MoveTop", (XtActionProc) movePuzzle3DTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzle3DTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzle3DLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) movePuzzle3DRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzle3DBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzle3DBottom},
	{(char *) "MoveBr", (XtActionProc) movePuzzle3DBr},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Read", (XtActionProc) readPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resources3D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[2]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[4]),
	 XtRString, (caddr_t) "Pink"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[5]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentControl),
	 XtRString, (caddr_t) "0"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(SkewbWidget, skewb.select),
	 XtRCallback, (caddr_t) NULL}
};

Skewb3DClassRec skewb3dClassRec =
{
	{
		(WidgetClass) & skewbClassRec,	/* superclass */
		(char *) "Skewb3D",	/* class name */
		sizeof (Skewb3DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzle3D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsList3D,	/* actions */
		XtNumber(actionsList3D),	/* num actions */
		resources3D,	/* resources */
		XtNumber(resources3D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzle3D,	/* resize */
		(XtExposeProc) exposePuzzle3D,	/* expose */
		(XtSetValuesFunc) setValuesPuzzle3D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translations3D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass skewb3dWidgetClass = (WidgetClass) & skewb3dClassRec;
#endif

static Point faceLoc3D[MAX_FACES][MAX_ORIENT];
static Point cubeLoc3D[MAX_FACES][MAX_ORIENT];
static Point diamondLoc3D[MAX_FACES][MAX_ORIENT + 1];
static Point triangleLoc3D[MAX_FACES][MAX_ORIENT][4];
static Point letter3DList[MAX_FACES][MAX_CUBES];
static Point orientDiamond[MAX_FACES][MAX_ORIENT][2];
static Point orientTriangle[MAX_FACES][MAX_ORIENT][2];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
diamondOffset3D(Skewb3DWidget w, int face, int *dx, int *dy)
{
	char *buf;

	if (w->skewb.vertical) {
		switch (face) {
		case 0:
			*dx = w->skewb3d.viewMiddle.x + w->skewb.delta - 1;
			*dy = w->skewb3d.viewMiddle.y - w->skewb.delta - 2;
			break;
		case 1:
			*dx = w->skewb3d.viewMiddle.x - 2 * w->skewb.delta;
			*dy = w->skewb3d.viewMiddle.y;
			break;
		case 2:
			*dx = w->skewb3d.viewMiddle.x + w->skewb.delta - 1;
			*dy = w->skewb3d.viewMiddle.y + 2 * w->skewb.delta - 1;
			break;
		case 3:
			*dx = w->skewb3d.viewMiddle.x + 2 * w->skewb.delta;
			*dy = w->skewb3d.viewSize.y + w->skewb3d.viewMiddle.y +
				w->skewb.delta - 1;
			break;
		case 4:
			*dx = w->skewb3d.viewMiddle.x - w->skewb.delta + 1;
			*dy = w->skewb3d.viewSize.y + w->skewb3d.viewMiddle.y -
				w->skewb.delta + 1;
			break;
		case 5:
			*dx = w->skewb3d.viewMiddle.x - 2;
			*dy = w->skewb3d.viewSize.y + w->skewb3d.viewMiddle.y +
				2 * w->skewb.delta + 2;
			break;
		default:
			intCat(&buf, "diamondOffset3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	} else {
		switch (face) {
		case 0:
			*dx = w->skewb3d.viewMiddle.x;
			*dy = w->skewb3d.viewMiddle.y - 2 * w->skewb.delta + 1;
			break;
		case 1:
			*dx = w->skewb3d.viewMiddle.x - w->skewb.delta - 2;
			*dy = w->skewb3d.viewMiddle.y + w->skewb.delta;
			break;
		case 2:
			*dx = w->skewb3d.viewMiddle.x + 2 * w->skewb.delta - 1;
			*dy = w->skewb3d.viewMiddle.y + w->skewb.delta;
			break;
		case 3:
			*dx = w->skewb3d.viewSize.x + w->skewb3d.viewMiddle.x -
				w->skewb.delta + 1;
			*dy = w->skewb3d.viewMiddle.y - w->skewb.delta;
			break;
		case 4:
			*dx = w->skewb3d.viewSize.x + w->skewb3d.viewMiddle.x +
				w->skewb.delta - 1;
			*dy = w->skewb3d.viewMiddle.y + 2 * w->skewb.delta;
			break;
		case 5:
			*dx = w->skewb3d.viewSize.x + w->skewb3d.viewMiddle.x +
				2 * w->skewb.delta + 2;
			*dy = w->skewb3d.viewMiddle.y - w->skewb.delta;
			break;
		default:
			intCat(&buf, "diamondOffset3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

static void
cubeOffset3D(Skewb3DWidget w, int face, int corner, int *dx, int *dy)
{
	int side;

	diamondOffset3D(w, face, dx, dy);
	for (side = 1; side <= corner; side++) {
		*dx += cubeLoc3D[face][side].x;
		*dy += cubeLoc3D[face][side].y;
	}
}

static void
mapTo3D(Skewb3DWidget w, int face, int side, int *corner)
{
	switch (face) {
	case 0:
		*corner = (side + 2) % MAX_ORIENT;
		break;
	case 1:
	case 5:
		*corner = side;
		break;
	case 2:
		*corner = (side + 1) % MAX_ORIENT;
		break;
	case 3:
	case 4:
		*corner = (side + 3) % MAX_ORIENT;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "mapTo3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

static void
mapFrom3D(Skewb3DWidget w, int face, int corner, int *side)
{
	switch (face) {
	case 0:
		*side = (corner + 2) % MAX_ORIENT;
		break;
	case 1:
	case 5:
		*side = corner;
		break;
	case 2:
		*side = (corner + 3) % MAX_ORIENT;
		break;
	case 3:
	case 4:
		*side = (corner + 1) % MAX_ORIENT;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "mapFrom3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

void
drawTriangle3D(Skewb3DWidget w, int face, int position, int offset)
{
	GC faceGC, borderGC;
	int side = 0, dx, dy, rotation, letterX, letterY;

	mapTo3D(w, face, position, &side);
	cubeOffset3D(w, face, side, &dx, &dy);
	rotation = w->skewb.cubeLoc[face][position].rotation;
	letterX = dx + letter3DList[face][side].x;
	letterY = dy + letter3DList[face][side].y;
	triangleLoc3D[face][side][0].x = dx;
	triangleLoc3D[face][side][0].y = dy;
	if (offset) {
		borderGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][position].face];
		if (w->skewb.mono) {
			faceGC = w->skewb.inverseGC;
		} else {
			faceGC = w->skewb.borderGC;
		}
	} else {
		faceGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][position].face];
		borderGC = w->skewb.borderGC;
	}
	POLYGON((SkewbWidget) w, dr, faceGC, borderGC,
		triangleLoc3D[face][side], 3, True, False);
	if (w->skewb.mono) {
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->skewb.faceChar[w->skewb.cubeLoc
				[face][position].face];
#else
			w->skewb.faceName[w->skewb.cubeLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		if (offset) {
			borderGC = w->skewb.borderGC;
		} else {
			borderGC = w->skewb.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC,
			letterX + w->skewb.letterOffset.x,
			letterY + w->skewb.letterOffset.y,
			buf, 1);
	}
	if (w->skewb.orient) {
		DRAWLINE(w, dr, borderGC,
			letterX +
			orientTriangle[face][rotation][0].x,
			letterY +
			orientTriangle[face][rotation][0].y,
			letterX +
			orientTriangle[face][rotation][1].x,
			letterY +
			orientTriangle[face][rotation][1].y);
	}
}

void
drawDiamond3D(Skewb3DWidget w, int face, int offset)
{
	GC faceGC, borderGC;
	int dx, dy, rotation;

	diamondOffset3D(w, face, &dx, &dy);
	rotation = w->skewb.cubeLoc[face][MAX_ORIENT].rotation;
	diamondLoc3D[face][0].x = dx + cubeLoc3D[face][1].x / 2;
	diamondLoc3D[face][0].y = dy + cubeLoc3D[face][1].y / 2;
	if (offset) {
		borderGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][MAX_ORIENT].face];
		if (w->skewb.mono) {
			faceGC = w->skewb.inverseGC;
		} else {
			faceGC = w->skewb.borderGC;
		}
	} else {
		faceGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][MAX_ORIENT].face];
		borderGC = w->skewb.borderGC;
	}
	POLYGON((SkewbWidget) w, dr, faceGC, borderGC,
		diamondLoc3D[face], 4, True, False);
	if (w->skewb.mono) {
		int letterX, letterY;
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->skewb.faceChar[w->skewb.cubeLoc
				[face][MAX_ORIENT].face];
#else
			w->skewb.faceName[w->skewb.cubeLoc
				[face][MAX_ORIENT].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + letter3DList[face][MAX_ORIENT].x + w->skewb.letterOffset.x;
		letterY = dy + letter3DList[face][MAX_ORIENT].y + w->skewb.letterOffset.y;
		if (offset) {
			borderGC = w->skewb.borderGC;
		} else {
			borderGC = w->skewb.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC,
			letterX, letterY, buf, 1);
	}
	if (w->skewb.orient) {
		DRAWLINE(w, dr, borderGC,
			dx +
			orientDiamond[face][rotation][0].x,
			dy +
			orientDiamond[face][rotation][0].y,
			dx +
			orientDiamond[face][rotation][1].x,
			dy +
			orientDiamond[face][rotation][1].y);
	}
}

void
drawFrame3D(Skewb3DWidget w, Boolean focus)
{
	int face, dx, dy;
	GC gc = (focus) ? w->skewb.frameGC : w->skewb.borderGC;

	dx = w->skewb3d.viewSize.x + w->skewb.puzzleOffset.x;
	dy = w->skewb3d.viewSize.y + w->skewb.puzzleOffset.y;
	if (w->skewb.vertical) {
		DRAWLINE(w, dr, w->skewb.frameGC,
			0, dy, dx + w->skewb.puzzleOffset.x + 1, dy);
		DRAWTEXT(w, dr, w->skewb.borderGC,
			(int) (2 * w->skewb.delta),
			(int) (3 * w->skewb.delta + w->skewb.letterOffset.y),
			"Front", 5);
		DRAWTEXT(w, dr, w->skewb.borderGC, (int)
			(-4 * w->skewb.delta + 2 * 6 * w->skewb.letterOffset.x + w->core.width),
			(int) (-w->skewb.delta - 2 * w->skewb.letterOffset.y + w->core.height),
			"Back", 4);
	} else {
		DRAWLINE(w, dr, w->skewb.frameGC,
			dx, 0, dx, dy + w->skewb.puzzleOffset.y + 1);
		DRAWTEXT(w, dr, w->skewb.borderGC,
			(int) (2 * w->skewb.delta),
			(int) (3 * w->skewb.delta + w->skewb.letterOffset.y),
			"Front", 5);
		DRAWTEXT(w, dr, w->skewb.borderGC, (int)
			(-4 * w->skewb.delta + 2 * 6 * w->skewb.letterOffset.x + w->core.width),
			(int) (-w->skewb.delta - 2 * w->skewb.letterOffset.y + w->core.height),
			"Back", 4);
	}
	for (face = 0; face < MAX_FACES; face++) {
		POLYLINE((SkewbWidget) w, dr, gc,
			faceLoc3D[face], MAX_ORIENT, False);
	}
}

static void
resizePieces(Skewb3DWidget w)
{
	int face, orient, side, corner;
	Point subcubeLoc3D[MAX_FACES][MAX_ORIENT];
	Point orientCubeLoc3D[2][MAX_FACES][MAX_ORIENT];
	Point subdiamondLoc3D[MAX_FACES][MAX_ORIENT];
	Point orientDiamondLoc3D[2][MAX_FACES][MAX_ORIENT];

	w->skewb.letterOffset.x = -2;
	w->skewb.letterOffset.y = 3;
	w->skewb3d.viewMiddle.x = w->skewb3d.faceSize.x +
		w->skewb.puzzleOffset.x;
	w->skewb3d.viewMiddle.y = w->skewb3d.faceSize.y +
		w->skewb.puzzleOffset.y;
	for (face = 0; face < MAX_FACES; face++) {
		faceLoc3D[face][0].x = w->skewb3d.viewMiddle.x;
		faceLoc3D[face][0].y = w->skewb3d.viewMiddle.y;
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			faceLoc3D[face][orient].x = w->skewb3d.faceSize.x;
			faceLoc3D[face][orient].y = w->skewb3d.faceSize.y;
		}
	}
	if (w->skewb.vertical) {
		faceLoc3D[0][1].x /= -2;
		faceLoc3D[0][1].y /= -1;
		faceLoc3D[0][2].y = 0;
		faceLoc3D[0][3].x /= 2;

		faceLoc3D[1][1].x /= -2;
		faceLoc3D[1][2].x /= -2;
		faceLoc3D[1][2].y /= -1;
		faceLoc3D[1][3].x /= 2;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y = 0;
		faceLoc3D[2][2].x /= -2;
		faceLoc3D[2][3].x /= -1;
		faceLoc3D[2][3].y = 0;

		for (face = MAX_FACES / 2; face < MAX_FACES; face++)
			faceLoc3D[face][0].y += w->skewb3d.viewSize.y + 3;

		faceLoc3D[3][1].x /= 2;
		faceLoc3D[3][1].y /= -1;
		faceLoc3D[3][2].x /= 2;
		faceLoc3D[3][3].x /= -2;

		faceLoc3D[4][1].x /= -1;
		faceLoc3D[4][1].y = 0;
		faceLoc3D[4][2].x /= 2;
		faceLoc3D[4][2].y /= -1;
		faceLoc3D[4][3].y = 0;

		faceLoc3D[5][1].x /= 2;
		faceLoc3D[5][2].x /= -1;
		faceLoc3D[5][2].y = 0;
		faceLoc3D[5][3].x /= -2;
		faceLoc3D[5][3].y /= -1;
	} else {
		faceLoc3D[0][1].x /= -1;
		faceLoc3D[0][1].y /= -2;
		faceLoc3D[0][2].y /= -2;
		faceLoc3D[0][3].y /= 2;

		faceLoc3D[1][1].x = 0;
		faceLoc3D[1][2].x /= -1;
		faceLoc3D[1][2].y /= -2;
		faceLoc3D[1][3].x = 0;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y /= -2;
		faceLoc3D[2][2].x = 0;
		faceLoc3D[2][3].y /= 2;
		faceLoc3D[2][3].x /= -1;

		for (face = MAX_FACES / 2; face < MAX_FACES; face++)
			faceLoc3D[face][0].x += w->skewb3d.viewSize.x + 3;

		faceLoc3D[3][1].x /= -1;
		faceLoc3D[3][1].y /= 2;
		faceLoc3D[3][2].x = 0;
		faceLoc3D[3][2].y /= -1;
		faceLoc3D[3][3].y /= -2;

		faceLoc3D[4][1].y /= 2;
		faceLoc3D[4][2].x /= -1;
		faceLoc3D[4][2].y /= 2;
		faceLoc3D[4][3].x /= -1;
		faceLoc3D[4][3].y /= -2;

		faceLoc3D[5][1].x = 0;
		faceLoc3D[5][1].y /= -1;
		faceLoc3D[5][2].y /= 2;
		faceLoc3D[5][3].x = 0;
	}

	for (face = 0; face < MAX_FACES; face++) {
		cubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		cubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		subcubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		subcubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		orientCubeLoc3D[0][face][0].x = orientCubeLoc3D[1][face][0].x = 0;
		orientCubeLoc3D[0][face][0].y = orientCubeLoc3D[1][face][0].y = 0;
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			cubeLoc3D[face][orient].x = faceLoc3D[face][orient].x - 3 *
				w->skewb.delta * faceLoc3D[face][orient].x /
				w->skewb3d.faceSize.x;
			cubeLoc3D[face][orient].y = faceLoc3D[face][orient].y - 3 *
				w->skewb.delta * faceLoc3D[face][orient].y /
				w->skewb3d.faceSize.y;
			subcubeLoc3D[face][orient].x = (faceLoc3D[face][orient].x -
				5 * faceLoc3D[face][orient].x * w->skewb.delta /
				w->skewb3d.faceSize.x) / 2;
			subcubeLoc3D[face][orient].y = (faceLoc3D[face][orient].y -
				5 * faceLoc3D[face][orient].y * w->skewb.delta /
				w->skewb3d.faceSize.y) / 2;
			orientCubeLoc3D[0][face][orient].x = (faceLoc3D[face][orient].x -
				5 * faceLoc3D[face][orient].x * w->skewb.delta /
				w->skewb3d.faceSize.x) / 4;
			orientCubeLoc3D[0][face][orient].y = (faceLoc3D[face][orient].y -
				5 * faceLoc3D[face][orient].y * w->skewb.delta /
				w->skewb3d.faceSize.y) / 4;
			orientCubeLoc3D[1][face][orient].x = (faceLoc3D[face][orient].x -
				7 * faceLoc3D[face][orient].x * w->skewb.delta /
				w->skewb3d.faceSize.x) / 6;
			orientCubeLoc3D[1][face][orient].y = (faceLoc3D[face][orient].y -
				7 * faceLoc3D[face][orient].y * w->skewb.delta /
				w->skewb3d.faceSize.y) / 6;
		}
		triangleLoc3D[face][0][1].x = subcubeLoc3D[face][1].x;
		triangleLoc3D[face][0][2].x = subcubeLoc3D[face][2].x -
			subcubeLoc3D[face][1].x;
		triangleLoc3D[face][0][1].y = subcubeLoc3D[face][1].y;
		triangleLoc3D[face][0][2].y = subcubeLoc3D[face][2].y -
			subcubeLoc3D[face][1].y;
		triangleLoc3D[face][1][1].x = subcubeLoc3D[face][2].x;
		triangleLoc3D[face][1][2].x = -subcubeLoc3D[face][1].x -
			subcubeLoc3D[face][2].x;
		triangleLoc3D[face][1][1].y = subcubeLoc3D[face][2].y;
		triangleLoc3D[face][1][2].y = -subcubeLoc3D[face][1].y -
			subcubeLoc3D[face][2].y;
		triangleLoc3D[face][2][1].x = -subcubeLoc3D[face][1].x;
		triangleLoc3D[face][2][2].x = subcubeLoc3D[face][1].x -
			subcubeLoc3D[face][2].x;
		triangleLoc3D[face][2][1].y = -subcubeLoc3D[face][1].y;
		triangleLoc3D[face][2][2].y = subcubeLoc3D[face][1].y -
			subcubeLoc3D[face][2].y;
		triangleLoc3D[face][3][1].x = -subcubeLoc3D[face][2].x;
		triangleLoc3D[face][3][2].x = subcubeLoc3D[face][1].x +
			subcubeLoc3D[face][2].x;
		triangleLoc3D[face][3][1].y = -subcubeLoc3D[face][2].y;
		triangleLoc3D[face][3][2].y = subcubeLoc3D[face][1].y +
			subcubeLoc3D[face][2].y;
		for (orient = 0; orient < MAX_ORIENT; orient++) {
			letter3DList[face][orient].x =
				(2 * triangleLoc3D[face][orient][1].x +
				 triangleLoc3D[face][orient][2].x) / 3;
			letter3DList[face][orient].y =
				(2 * triangleLoc3D[face][orient][1].y +
				 triangleLoc3D[face][orient][2].y) / 3;
			triangleLoc3D[face][orient][3].x =
				-triangleLoc3D[face][orient][1].x - triangleLoc3D[face][orient][2].x;
			triangleLoc3D[face][orient][3].y =
				-triangleLoc3D[face][orient][1].y - triangleLoc3D[face][orient][2].y;
		}
	}
	w->skewb3d.cubeSize.x = w->skewb3d.faceSize.x - 2 * w->skewb.delta;
	w->skewb3d.cubeSize.y = w->skewb3d.faceSize.y - 2 * w->skewb.delta;
	w->skewb3d.cubeDiagonal = w->skewb3d.faceDiagonal - 2 * w->skewb.delta;
	w->skewb3d.cubeDiag = w->skewb3d.faceDiagonal + 2 * w->skewb.delta;

	if (w->skewb.vertical) {
		letter3DList[0][MAX_ORIENT].x = w->skewb3d.cubeSize.x / 4;
		letter3DList[0][MAX_ORIENT].y = -w->skewb3d.cubeSize.y / 2 + 2;
		letter3DList[1][MAX_ORIENT].x = -w->skewb3d.cubeDiagonal;
		letter3DList[1][MAX_ORIENT].y = 0;
		letter3DList[2][MAX_ORIENT].x = w->skewb3d.cubeSize.x / 4;
		letter3DList[2][MAX_ORIENT].y = w->skewb3d.cubeSize.y / 2 - 2;
		letter3DList[3][MAX_ORIENT].x = w->skewb3d.cubeDiagonal;
		letter3DList[3][MAX_ORIENT].y = 0;
		letter3DList[4][MAX_ORIENT].x = -w->skewb3d.cubeSize.x / 4;
		letter3DList[4][MAX_ORIENT].y = -w->skewb3d.cubeSize.y / 2 + 2;
		letter3DList[5][MAX_ORIENT].x = -w->skewb3d.cubeSize.x / 4;
		letter3DList[5][MAX_ORIENT].y = w->skewb3d.cubeSize.y / 2 - 2;
	} else {
		letter3DList[0][MAX_ORIENT].x = 0;
		letter3DList[0][MAX_ORIENT].y = -w->skewb3d.cubeDiagonal;
		letter3DList[1][MAX_ORIENT].x = -w->skewb3d.cubeSize.x / 2 + 2;
		letter3DList[1][MAX_ORIENT].y = w->skewb3d.cubeSize.y / 4;
		letter3DList[2][MAX_ORIENT].x = w->skewb3d.cubeSize.x / 2 - 2;
		letter3DList[2][MAX_ORIENT].y = w->skewb3d.cubeSize.y / 4;
		letter3DList[3][MAX_ORIENT].x = -w->skewb3d.cubeSize.x / 2 + 2;
		letter3DList[3][MAX_ORIENT].y = -w->skewb3d.cubeSize.y / 4;
		letter3DList[4][MAX_ORIENT].x = 0;
		letter3DList[4][MAX_ORIENT].y = w->skewb3d.cubeDiagonal;
		letter3DList[5][MAX_ORIENT].x = w->skewb3d.cubeSize.x / 2 - 2;
		letter3DList[5][MAX_ORIENT].y = -w->skewb3d.cubeSize.y / 4;
	}

	for (face = 0; face < MAX_FACES; face++) {
		for (orient = 0; orient < MAX_ORIENT - 1; orient++) {
			diamondLoc3D[face][orient].x = (cubeLoc3D[face][orient].x +
				cubeLoc3D[face][orient + 1].x) / 2;
			diamondLoc3D[face][orient].y = (cubeLoc3D[face][orient].y +
				cubeLoc3D[face][orient + 1].y) / 2;
			subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][orient].x +
				subcubeLoc3D[face][orient + 1].x) / 2;
			subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][orient].y +
				subcubeLoc3D[face][orient + 1].y) / 2;
			orientDiamondLoc3D[0][face][orient].x =
				(orientCubeLoc3D[0][face][orient].x +
				orientCubeLoc3D[0][face][orient + 1].x) / 2;
			orientDiamondLoc3D[0][face][orient].y =
				(orientCubeLoc3D[0][face][orient].y +
				orientCubeLoc3D[0][face][orient + 1].y) / 2;
			orientDiamondLoc3D[1][face][orient].x =
				(orientCubeLoc3D[1][face][orient].x +
				orientCubeLoc3D[1][face][orient + 1].x) / 2;
			orientDiamondLoc3D[1][face][orient].y =
				(orientCubeLoc3D[1][face][orient].y +
				orientCubeLoc3D[1][face][orient + 1].y) / 2;
		}
		/* Its a parallelagram so take advantage of that */
		diamondLoc3D[face][orient].x = (cubeLoc3D[face][MAX_ORIENT - 1].x -
			cubeLoc3D[face][MAX_ORIENT / 2].x) / 2;
		diamondLoc3D[face][orient].y = (cubeLoc3D[face][MAX_ORIENT - 1].y -
			cubeLoc3D[face][MAX_ORIENT / 2].y) / 2;
		diamondLoc3D[face][MAX_ORIENT].x = -diamondLoc3D[face][1].x -
			diamondLoc3D[face][2].x - diamondLoc3D[face][3].x;
		diamondLoc3D[face][MAX_ORIENT].y = -diamondLoc3D[face][1].y -
			diamondLoc3D[face][2].y - diamondLoc3D[face][3].y;

		subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][MAX_ORIENT - 1].x -
			subcubeLoc3D[face][MAX_ORIENT / 2].x) / 2;
		subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][MAX_ORIENT - 1].y -
			subcubeLoc3D[face][MAX_ORIENT / 2].y) / 2;
		orientDiamondLoc3D[0][face][orient].x =
			(orientCubeLoc3D[0][face][MAX_ORIENT - 1].x -
			orientCubeLoc3D[0][face][MAX_ORIENT / 2].x) / 2;
		orientDiamondLoc3D[0][face][orient].y =
			(orientCubeLoc3D[0][face][MAX_ORIENT - 1].y -
			orientCubeLoc3D[0][face][MAX_ORIENT / 2].y) / 2;
		orientDiamondLoc3D[1][face][orient].x =
			(orientCubeLoc3D[1][face][MAX_ORIENT - 1].x -
			orientCubeLoc3D[1][face][MAX_ORIENT / 2].x) / 2;
		orientDiamondLoc3D[1][face][orient].y =
			(orientCubeLoc3D[1][face][MAX_ORIENT - 1].y -
			orientCubeLoc3D[1][face][MAX_ORIENT / 2].y) / 2;

		mapFrom3D(w, face, 1, &corner);
		orientDiamond[face][corner][0].x = cubeLoc3D[face][1].x / 2;
		orientDiamond[face][corner][0].y = cubeLoc3D[face][1].y / 2;
		orientDiamond[face][corner][1].x = orientDiamond[face][corner][0].x +
			(cubeLoc3D[face][2].x - subcubeLoc3D[face][2].x) / 2;
		orientDiamond[face][corner][1].y = orientDiamond[face][corner][0].y +
			(cubeLoc3D[face][2].y - subcubeLoc3D[face][2].y) / 2;
		orientTriangle[face][corner][0].x = -orientCubeLoc3D[0][face][2].x / 2;
		orientTriangle[face][corner][0].y = -orientCubeLoc3D[0][face][2].y / 2;
		orientTriangle[face][corner][1].x = -orientCubeLoc3D[1][face][2].x / 2;
		orientTriangle[face][corner][1].y = -orientCubeLoc3D[1][face][2].y / 2;
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			side = corner;
			mapFrom3D(w, face, (orient + 1) % MAX_ORIENT, &corner);
			orientDiamond[face][corner][0].x =
				orientDiamond[face][side][0].x +
				diamondLoc3D[face][orient].x;
			orientDiamond[face][corner][0].y =
				orientDiamond[face][side][0].y +
				diamondLoc3D[face][orient].y;
			orientDiamond[face][corner][1].x =
				orientDiamond[face][side][1].x +
				subdiamondLoc3D[face][orient].x;
			orientDiamond[face][corner][1].y =
				orientDiamond[face][side][1].y +
				subdiamondLoc3D[face][orient].y;
			orientTriangle[face][corner][0].x =
				orientTriangle[face][side][0].x +
				orientDiamondLoc3D[0][face][orient].x;
			orientTriangle[face][corner][0].y =
				orientTriangle[face][side][0].y +
				orientDiamondLoc3D[0][face][orient].y;
			orientTriangle[face][corner][1].x =
				orientTriangle[face][side][1].x +
				orientDiamondLoc3D[1][face][orient].x;
			orientTriangle[face][corner][1].y =
				orientTriangle[face][side][1].y +
				orientDiamondLoc3D[1][face][orient].y;
		}
	}
}

Boolean
selectPieces3D(Skewb3DWidget w, int x, int y, int *face, int *position)
{
	int u, v, front, tl, ur, ul, found, side, dx, dy;
	int x_1, y_1, x_2, y_2;

	x_1 = x;
	y_1 = y;
	if (w->skewb.vertical) {
		x -= w->skewb3d.viewMiddle.x;
		front = (y < w->skewb3d.viewSize.y + w->skewb.puzzleOffset.y);
		if (!front)
			y -= (w->skewb3d.viewSize.y);
		tl = (y < w->skewb3d.viewMiddle.y);
		y -= w->skewb3d.viewMiddle.y;
		u = -w->skewb3d.faceSize.y * x + w->skewb3d.faceDiagonal * y;
		v = w->skewb3d.faceSize.y * x + w->skewb3d.faceDiagonal * y;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 0 : 1;
			else
				*face = (ul) ? 1 : 2;
		} else {
			if (tl)
				*face = (ul) ? 4 : 3;
			else
				*face = (ur) ? 3 : 5;
		}
	} else {
		y -= w->skewb3d.viewMiddle.y;
		front = (x < w->skewb3d.viewSize.x + w->skewb.puzzleOffset.x);
		if (!front)
			x -= (w->skewb3d.viewSize.x);
		tl = (x < w->skewb3d.viewMiddle.x);
		x -= w->skewb3d.viewMiddle.x;
		u = -w->skewb3d.faceSize.x * y + w->skewb3d.faceDiagonal * x;
		v = w->skewb3d.faceSize.x * y + w->skewb3d.faceDiagonal * x;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 1 : 0;
			else
				*face = (ul) ? 0 : 2;
		} else {
			if (tl)
				*face = (ul) ? 3 : 4;
			else
				*face = (ur) ? 4 : 5;
		}
	}
	x = x_1;
	y = y_1;
	found = 0;
	diamondOffset3D(w, *face, &dx, &dy);
	for (side = 0; side < MAX_ORIENT; side++) {
		x_1 = dx + orientDiamond[*face][side][0].x;
		y_1 = dy + orientDiamond[*face][side][0].y;
		x_2 = dx + orientDiamond[*face][(side + 1) % MAX_ORIENT][0].x;
		y_2 = dy + orientDiamond[*face][(side + 1) % MAX_ORIENT][0].y;
		if ((x_2 - x_1) * (y - y_1) <= (y_2 - y_1) * (x - x_1)) {
			*position = side;
			found++;
		}
	}
	if (found == 0)
		*position = MAX_ORIENT;
	else if (found > 1)
		return False;
	return True;
}

Boolean
narrowSelection3D(Skewb3DWidget w, int *face, int *position, int *direction)
{
	switch (*direction) {
	case TR:
	case BR:
	case BL:
	case TL:
		if (w->skewb.vertical) {
			if (*face == 1)
				*direction = *direction + 2 * MAX_ORIENT;
			else if (*face == 3)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
			else if (*face == 4)
				*direction = (*direction + 2) % MAX_ORIENT +
					2 * MAX_ORIENT;
			else if (*face == 5)
				*direction = (*direction + 2) % MAX_ORIENT;
		}
		break;
	case CCW:
	case CW:
		break;
	case TOP:
	case RIGHT:
	case BOTTOM:
	case LEFT:
		if (w->skewb.vertical) {
			if (*face == 1)
				*direction = (TL + *direction) % MAX_ORIENT;
			else if (*face == 3)
				*direction = (BL + *direction) % MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (TR + *direction) % MAX_ORIENT;
			else if (*face == 4)
				*direction = (BR + *direction) % MAX_ORIENT;
			else if (*face == 5)
				*direction = (BL + *direction) % MAX_ORIENT +
					2 * MAX_ORIENT;
		}
		break;
	default:
		return False;
	}
	if (*position != MAX_ORIENT) {
		if (*direction == CW)
			*direction = (*position + 3) % MAX_ORIENT;
		else if (*direction == CCW)
			*direction = (*position + 1) % MAX_ORIENT;
		else if (*direction < MAX_ORIENT &&
				!((*direction + *position) % 2))
			return False;
	}
	return True;
}

#ifndef WINVER
static Boolean
setValuesPuzzle3D(Widget current, Widget request, Widget renew)
{
	Skewb3DWidget c = (Skewb3DWidget) current, w = (Skewb3DWidget) renew;
	Boolean redraw = False;

	if (w->skewb3d.cubeSize.x != c->skewb3d.cubeSize.x) {
		resizePuzzle3D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzle3D(Skewb3DWidget w)
{
	Point tempSize;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->skewb.delta = 4;
	w->skewb.vertical = (w->core.height >= w->core.width);
	if (w->skewb.vertical) {
		tempSize.y = w->core.height / MAX_VIEWS;
		tempSize.x = w->core.width;
		if (tempSize.x >= DIVIDE(tempSize.y)) {
			w->skewb3d.cubeSize.y = MAX((tempSize.y - 3 * w->skewb.delta) / 2 -
				w->skewb.delta - 2, 0);
			w->skewb3d.cubeSize.x = DIVIDE(w->skewb3d.cubeSize.y);
		} else {
			w->skewb3d.cubeSize.x = MAX((tempSize.x - 2 * w->skewb.delta - 7) / 2 -
				w->skewb.delta, 0);
			w->skewb3d.cubeSize.y = MULTIPLY(w->skewb3d.cubeSize.x);
		}
		w->skewb3d.cubeDiagonal = w->skewb3d.cubeSize.x / 2;
		w->skewb3d.faceSize.x = w->skewb3d.cubeSize.x + 2 * w->skewb.delta + 1;
		w->skewb3d.faceSize.y = w->skewb3d.cubeSize.y + 2 * w->skewb.delta + 1;
		w->skewb3d.faceDiagonal = w->skewb3d.faceSize.x / 2;
		w->skewb3d.viewSize.x = 2 * w->skewb3d.faceSize.x + 3;
		w->skewb3d.viewSize.y = 2 * w->skewb3d.faceSize.y + 3;
		w->skewb.puzzleSize.x = w->skewb3d.viewSize.x + 1;
		w->skewb.puzzleSize.y = MAX_VIEWS * w->skewb3d.viewSize.y + 1;
	} else {
		tempSize.x = w->core.width / MAX_VIEWS;
		tempSize.y = w->core.height;
		if (tempSize.y >= DIVIDE(tempSize.x)) {
			w->skewb3d.cubeSize.x = MAX((tempSize.x - 3 * w->skewb.delta) / 2 -
				w->skewb.delta - 2, 0);
			w->skewb3d.cubeSize.y = DIVIDE(w->skewb3d.cubeSize.x);
		} else {
			w->skewb3d.cubeSize.y = MAX((tempSize.y - 2 * w->skewb.delta - 7) / 2 -
				w->skewb.delta, 0);
			w->skewb3d.cubeSize.x = MULTIPLY(w->skewb3d.cubeSize.y);
		}
		w->skewb3d.cubeDiagonal = w->skewb3d.cubeSize.y / 2;
		w->skewb3d.faceSize.y = w->skewb3d.cubeSize.y + 2 * w->skewb.delta + 1;
		w->skewb3d.faceSize.x = w->skewb3d.cubeSize.x + 2 * w->skewb.delta + 1;
		w->skewb3d.faceDiagonal = w->skewb3d.faceSize.y / 2;
		w->skewb3d.viewSize.y = 2 * w->skewb3d.faceSize.y + 3;
		w->skewb3d.viewSize.x = 2 * w->skewb3d.faceSize.x + 3;
		w->skewb.puzzleSize.y = w->skewb3d.viewSize.y + 1;
		w->skewb.puzzleSize.x = MAX_VIEWS * w->skewb3d.viewSize.x + 1;
	}
	w->skewb.puzzleOffset.x = ((int) w->core.width - w->skewb.puzzleSize.x) / 2;
	w->skewb.puzzleOffset.y = ((int) w->core.height - w->skewb.puzzleSize.y) /
		2;
	resizePieces(w);
}

#ifndef WINVER
static
#endif
void
initializePuzzle3D(
#ifdef WINVER
Skewb3DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Skewb3DWidget w = (Skewb3DWidget) renew;

	setAllColors((SkewbWidget) w);
#endif
	w->skewb.dim = 3;
	resizePuzzle3D(w);
}

#ifndef WINVER
static
#endif
void
exposePuzzle3D(
#ifdef WINVER
Skewb3DWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	Skewb3DWidget w = (Skewb3DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, dr, w->skewb.inverseGC,
		0, 0, w->core.width, w->core.height);
	drawFrame3D(w, w->skewb.focus);
	drawAllPieces((SkewbWidget) w);
}

#ifndef WINVER
static void
movePuzzle3DTl(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DTop(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DTr(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DLeft(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DRight(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DBl(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DBottom(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzle3DBr(Skewb3DWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}
#endif
