/* GIMP Plugin Preview Widget                                                 
 * Copyright (C) 1998-1999 Shawn T. Amundson                
 * 
 * Some of the checkerboard & image combination code is based of
 * code Copyright 1997 (C) Federico Mena Quintero and is used
 * here by permission.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.           
 *                                                                            
 * This library is distributed in the hope that it will be useful,             
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */      

#include <stdio.h>
#include <gtk/gtk.h>
#include "gimppreview.h"

static void gimp_preview_init            (GimpPreview *preview);
static void gimp_preview_class_init      (GimpPreviewClass *klass);

static void gimp_preview_plus_callback   (GtkWidget *widget,
				          gpointer data);
static void gimp_preview_minus_callback  (GtkWidget *widget,
				          gpointer data);
static gint gimp_preview_event           (GtkWidget *widget, GdkEvent *event, 
					  gpointer data);
static void gimp_preview_recompute_sizes (GimpPreview *preview);
static void gimp_preview_update_preview  (GimpPreview *preview);
static void gimp_preview_marshal_signal  (GtkObject     *object,
					  GtkSignalFunc func,
					  gpointer      func_data,
					  GtkArg        *args);


#define PREVIEW_SIZE 100
 
enum {
  UPDATE_PREVIEW,
  PREVIEW_CHANGED,
  LAST_SIGNAL
};

#define PREVIEW_SCALE_DEFAULT 5
#define PREVIEW_SCALE_LAST 20
static gdouble preview_scale[PREVIEW_SCALE_LAST+1] = {  0.16,   0.20,  0.25,
						        0.33,   0.50,  1.00,
						        2.00,   3.00,  4.00,  
							5.00,   6.00,  7.00,  
							8.00,   9.00, 10.00, 
							11.00, 12.00, 13.00, 
							14.00, 15.00, 16.00 };

static GtkVBox *parent_class;
static guint gimp_preview_signals[LAST_SIGNAL] = { 0 };


#define PREVIEW_DATA(preview) ((GimpPreviewData*)(GIMP_PREVIEW (preview)->private_data))

typedef struct {
  gint scale_n;

  GtkWidget *label;
  GtkWidget *button_minus;
  GtkWidget *button_plus;

  gboolean in_drag;
  gint drag_x;
  gint drag_y;

  guint orig_image_x; 
  guint orig_image_y; 

  guint offset_x;
  guint offset_y;
  guint orig_offset_x;
  guint orig_offset_y;

  guchar *preview_buffer_na;
  
} GimpPreviewData;


GtkType
gimp_preview_get_type (void)
{
  static GtkType preview_type = 0;

  if (!preview_type)
    {
      GtkTypeInfo preview_info =
      {
	"GimpPreview",
	sizeof (GimpPreview),
	sizeof (GimpPreviewClass),
	(GtkClassInitFunc) gimp_preview_class_init,
	(GtkObjectInitFunc) gimp_preview_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      preview_type = gtk_type_unique (gtk_vbox_get_type (), &preview_info);
    }

  return preview_type;
}

static void
gimp_preview_class_init (GimpPreviewClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkVBoxClass *vbox_class;

  object_class = GTK_OBJECT_CLASS (klass);
  widget_class = GTK_WIDGET_CLASS (klass);
  container_class = GTK_CONTAINER_CLASS (klass);
  vbox_class = GTK_VBOX_CLASS (klass);

  parent_class = gtk_type_class (gtk_vbox_get_type ());

  gimp_preview_signals[UPDATE_PREVIEW] =
    gtk_signal_new("update_preview",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET (GimpPreviewClass, update_preview),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gimp_preview_signals[PREVIEW_CHANGED] =
    gtk_signal_new("preview_changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET (GimpPreviewClass, preview_changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, gimp_preview_signals, 
				LAST_SIGNAL);

  klass->update_preview = NULL;
  klass->preview_changed = NULL;
}

static void
gimp_preview_init (GimpPreview *preview)
{
  gchar buffer[10];
  guint x;
  guint bpp_minus_alpha;

  preview->private_data = g_malloc(sizeof(GimpPreviewData));

  PREVIEW_DATA (preview)->scale_n = PREVIEW_SCALE_DEFAULT;
  preview->scale = preview_scale[PREVIEW_DATA (preview)->scale_n];

  PREVIEW_DATA (preview)->label = gtk_label_new("");
  sprintf(buffer, "%d%%", (gint) (preview->scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  PREVIEW_DATA (preview)->in_drag = FALSE;
  PREVIEW_DATA (preview)->drag_x = 0;
  PREVIEW_DATA (preview)->drag_y = 0;
  PREVIEW_DATA (preview)->offset_x = 0;
  PREVIEW_DATA (preview)->offset_y = 0;
  PREVIEW_DATA (preview)->orig_offset_x = 0;
  PREVIEW_DATA (preview)->orig_offset_y = 0;

  preview->check_row_0 = g_malloc (PREVIEW_SIZE * sizeof (guchar));
  preview->check_row_1 = g_malloc (PREVIEW_SIZE * sizeof (guchar));
  preview->check_size = 16;
  preview->check_dark = ((int) (1.3 / 3.0 * 255));
  preview->check_light = ((int) (1.8 / 3.0 * 255));

  for (x = 0; x < PREVIEW_SIZE; x++)
    {
      if ((x / preview->check_size) & 1)
	{
	  preview->check_row_0[x] = preview->check_dark;
	  preview->check_row_1[x] = preview->check_light;
	}
      else
	{
	  preview->check_row_0[x] = preview->check_light;
	  preview->check_row_1[x] = preview->check_dark;
	}
    }

  PREVIEW_DATA (preview)->preview_buffer_na = g_malloc (sizeof (guchar)
							* PREVIEW_SIZE 
							* 3);
  preview->buffer = g_malloc (sizeof (guchar)
			      * (PREVIEW_SIZE + 16)
			      * (PREVIEW_SIZE + 16) 
			      * 4);

  preview->preview = NULL;
  preview->progress_bar = NULL;
}

GtkWidget*
gimp_preview_new (GDrawable *drawable)
{
  GimpPreview *preview;
  GtkWidget *frame;
  GtkWidget *packer;
  GtkWidget *hbox;
  guchar *color_cube;

  preview = GIMP_PREVIEW (gtk_type_new (gimp_preview_get_type ()));
  preview->drawable = drawable;

  frame = gtk_frame_new (NULL);
  gtk_box_pack_start (GTK_BOX (preview), frame, FALSE, FALSE, 0);
  
  packer = gtk_packer_new ();
  gtk_container_add(GTK_CONTAINER (frame), packer);

  preview->preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size(GTK_PREVIEW (preview->preview), PREVIEW_SIZE, PREVIEW_SIZE);
  gtk_container_add(GTK_CONTAINER (packer), preview->preview);
  
  /* Some of this may get moved up to GimpPluginWindow widget instead. */
  gtk_preview_set_gamma (gimp_gamma ());
  gtk_preview_set_install_cmap (gimp_install_cmap());
  color_cube = gimp_color_cube();
  gtk_preview_set_color_cube(color_cube[0], color_cube[1], color_cube[3],
			     color_cube[4]); 
  gtk_widget_set_default_visual (gtk_preview_get_visual ());
  gtk_widget_set_default_colormap (gtk_preview_get_cmap ());

  gtk_widget_set_events (preview->preview,
			 GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK |
			 GDK_POINTER_MOTION_HINT_MASK |
			 GDK_BUTTON_MOTION_MASK);
  gtk_signal_connect (GTK_OBJECT (preview->preview), "event",
		      GTK_SIGNAL_FUNC (gimp_preview_event),
		      (gpointer) preview);

  preview->progress_bar = gtk_progress_bar_new ();
  gtk_widget_set_usize (preview->progress_bar, PREVIEW_SIZE, 10);
  gtk_box_pack_start (GTK_BOX (preview), preview->progress_bar, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (0,0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (preview), hbox, FALSE, FALSE, 0);

  PREVIEW_DATA (preview)->button_minus = gtk_button_new_with_label ("-");
  PREVIEW_DATA (preview)->button_plus = gtk_button_new_with_label ("+");

  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->button_minus, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->label, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->button_plus, TRUE, FALSE, 0);

  gtk_signal_connect (GTK_OBJECT (PREVIEW_DATA (preview)->button_minus), "clicked",
		      GTK_SIGNAL_FUNC (gimp_preview_minus_callback),
		      (gpointer) preview);
  gtk_signal_connect (GTK_OBJECT (PREVIEW_DATA (preview)->button_plus), "clicked",
		      GTK_SIGNAL_FUNC (gimp_preview_plus_callback),
		      (gpointer) preview);

  gtk_widget_show (PREVIEW_DATA (preview)->button_minus);
  gtk_widget_show (PREVIEW_DATA (preview)->label);
  gtk_widget_show (PREVIEW_DATA (preview)->button_plus);
  gtk_widget_show (hbox);
  gtk_widget_show (preview->progress_bar);
  gtk_widget_show (preview->preview);
  gtk_widget_show (packer);
  gtk_widget_show (frame);

  gimp_preview_recompute_sizes (preview);
  gimp_preview_update_preview (preview);

  return GTK_WIDGET (preview);
}

void
gimp_preview_update (GimpPreview *preview)
{
  gimp_preview_recompute_sizes (preview);
  gimp_preview_update_preview (preview);
}

static void
gimp_preview_plus_callback (GtkWidget *widget, gpointer data)
{
  GimpPreview *preview;
  gchar buffer[10];

  preview = GIMP_PREVIEW (data);
  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST)
    return;

  PREVIEW_DATA (preview)->scale_n++;
  preview->scale = preview_scale[PREVIEW_DATA (preview)->scale_n];
  sprintf(buffer, "%d%%", (gint) (preview->scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST)
    gtk_widget_set_sensitive (widget, FALSE);

  if (PREVIEW_DATA (preview)->scale_n == 1)
    gtk_widget_set_sensitive (PREVIEW_DATA (preview)->button_minus, TRUE);

  gimp_preview_recompute_sizes (preview);
  gimp_preview_update_preview (preview);
}

static void
gimp_preview_minus_callback (GtkWidget *widget, gpointer data)
{
  GimpPreview *preview;
  gchar buffer[10];

  preview = GIMP_PREVIEW (data);
  if (PREVIEW_DATA (preview)->scale_n == 0)
    return;

  PREVIEW_DATA (preview)->scale_n--;
  preview->scale = preview_scale[PREVIEW_DATA (preview)->scale_n];
  sprintf(buffer, "%d%%", (gint) (preview->scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  if (PREVIEW_DATA (preview)->scale_n == 0)
    gtk_widget_set_sensitive (widget, FALSE);

  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST - 1)
    gtk_widget_set_sensitive (PREVIEW_DATA (preview)->button_plus, TRUE);

  gimp_preview_recompute_sizes (preview);
  gimp_preview_update_preview (preview);
}

static gint
gimp_preview_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  GimpPreview *preview;
  GdkEventButton *button_event;
  gint x, y;
  gint dx, dy;
  gdouble image_x, image_y;
  gint offset_x;
  gint offset_y;

  preview = GIMP_PREVIEW (data);
  button_event = (GdkEventButton*) event;

  switch (event->type)
    {
    case GDK_BUTTON_PRESS:
      if (button_event->button == 1)
	{
	  gtk_widget_get_pointer(widget, &x, &y);

	  PREVIEW_DATA (preview)->in_drag = TRUE;
	  PREVIEW_DATA (preview)->drag_x = x;
	  PREVIEW_DATA (preview)->drag_y = y;

	  PREVIEW_DATA (preview)->orig_image_x = preview->image_x;
	  PREVIEW_DATA (preview)->orig_image_y = preview->image_y;

	  PREVIEW_DATA (preview)->orig_offset_x = PREVIEW_DATA (preview)->offset_x;
	  PREVIEW_DATA (preview)->orig_offset_y = PREVIEW_DATA (preview)->offset_y;

	  gtk_grab_add (widget);

	  gimp_preview_update_preview (preview);
	}
      break;

    case GDK_BUTTON_RELEASE:

      if (PREVIEW_DATA (preview)->in_drag)
	{
	  gtk_grab_remove (widget);
	  PREVIEW_DATA (preview)->in_drag = FALSE;

	  gimp_preview_update_preview (preview);
	}
      
      break;
    case GDK_MOTION_NOTIFY:

      gtk_widget_get_pointer(widget, &x, &y);

      dx = x - PREVIEW_DATA (preview)->drag_x;
      dy = y - PREVIEW_DATA (preview)->drag_y;
      

      image_x = (PREVIEW_DATA (preview)->orig_image_x 
		 - ((gdouble) dx) / preview->scale);
      image_y = (PREVIEW_DATA (preview)->orig_image_y 
		 - ((gdouble) dy) / preview->scale);

      image_x = MAX (image_x, 0);
      image_y = MAX (image_y, 0);
      image_x = MIN (image_x, preview->drawable->width - preview->image_width);
      image_y = MIN (image_y, preview->drawable->height - preview->image_height);

      if (preview->scale > 1)
	{
	  
	  offset_x = (gint) ((gdouble) ((gint) PREVIEW_DATA (preview)->orig_image_x - (gint) image_x)
			     * preview->scale) - (dx - PREVIEW_DATA (preview)->orig_offset_x);
	  offset_y = (gint) ((gdouble) ((gint) PREVIEW_DATA (preview)->orig_image_y - (gint) image_y)
			     * preview->scale) - (dy - PREVIEW_DATA (preview)->orig_offset_y);

	  while (offset_x > preview->scale)
	    {
	      image_x++;
	      offset_x -= preview->scale;
	    }

	  while (offset_y > preview->scale)
	    {
	      image_y++;
	      offset_y -= preview->scale;
	    }

	  while (offset_x < 0)
	    {
	      image_x--;
	      offset_x += preview->scale;
	    }

	  while (offset_y < 0)
	    {
	      image_y--;
	      offset_y += preview->scale;
	    }

	  if ((image_x < 0) || (image_x >= preview->drawable->width - preview->image_width))
	    offset_x = 0;

	  if ((image_y < 0) || (image_y >= preview->drawable->height - preview->image_height))
	    offset_y = 0;

	  image_x = MAX (image_x, 0);
	  image_y = MAX (image_y, 0);
	  image_x = MIN (image_x, preview->drawable->width - preview->image_width);
	  image_y = MIN (image_y, preview->drawable->height - preview->image_height);

	  PREVIEW_DATA (preview)->offset_x = offset_x;
	  PREVIEW_DATA (preview)->offset_y = offset_y;
	}
      else
	{
	  PREVIEW_DATA (preview)->offset_x = 0;
	  PREVIEW_DATA (preview)->offset_y = 0;
	}

      preview->image_x = (gint) image_x;
      preview->image_y = (gint) image_y;

      gimp_preview_update_preview (preview);

      break;

    default:
      break;
    }
  
  return FALSE;
}

static void
gimp_preview_recompute_sizes(GimpPreview *preview)
{
  guint real_width;
  guint real_height;

  preview->width = PREVIEW_SIZE;
  preview->height = PREVIEW_SIZE;

  /* So we can shift up to an extra image pixel. */
  if (preview->scale > 1)
    {
      preview->width += (guint) preview->scale - 1;
      preview->height += (guint) preview->scale - 1;
    }

  preview->image_width = preview->width / preview->scale;
  preview->image_height = preview->height / preview->scale;

  if (preview->image_width > preview->drawable->width)
    {
      preview->image_width = preview->drawable->width;
      preview->image_x = 0;
      preview->width = preview->image_width * preview->scale;
    }
  else
    {
      if (preview->image_x + preview->image_width > preview->drawable->width)
	preview->image_x = preview->drawable->width - preview->image_width;
    }

  if (preview->image_height > preview->drawable->height)
    {
      preview->image_height = preview->drawable->height;
      preview->image_y = 0;
      preview->height = preview->image_height * preview->scale;
    }
  else
    {
      if (preview->image_y + preview->image_height > preview->drawable->height)
	preview->image_y = preview->drawable->height - preview->image_height;
    }
  
  real_width = preview->width;
  real_height = preview->height;
  
  if (preview->scale > 1)
    {
      real_width -= (guint) preview->scale - 1;
      real_height -= (guint) preview->scale - 1;
    }

  gtk_preview_size (GTK_PREVIEW (preview->preview), 
		    real_width, real_height);

  /* This clears the preview border back to the background color. */
  if (real_width < PREVIEW_SIZE || real_height < PREVIEW_SIZE)
    {
      gdk_window_clear_area (preview->preview->window,
			     0,
			     0,
			     (GTK_WIDGET (preview->preview)->allocation.width
			      - real_width)/2,
			     GTK_WIDGET (preview->preview)->allocation.height);
      gdk_window_clear_area (preview->preview->window,
			     0,
			     0,
			     GTK_WIDGET (preview->preview)->allocation.width,
			     (GTK_WIDGET (preview->preview)->allocation.height
			      - real_height)/2);
      gdk_window_clear_area (preview->preview->window,
			     GTK_WIDGET (preview->preview)->allocation.width -
			     (GTK_WIDGET (preview->preview)->allocation.width
			      - real_width)/2,
			     0,
			     GTK_WIDGET (preview->preview)->allocation.width,
			     GTK_WIDGET (preview->preview)->allocation.height);
      gdk_window_clear_area (preview->preview->window,
			     0,
			     GTK_WIDGET (preview->preview)->allocation.height -
			     (GTK_WIDGET (preview->preview)->allocation.height
			      - real_height)/2,
			     GTK_WIDGET (preview->preview)->allocation.width,
			     GTK_WIDGET (preview->preview)->allocation.height);
    }

  PREVIEW_DATA (preview)->orig_offset_x = 0;
  PREVIEW_DATA (preview)->orig_offset_y = 0;
  PREVIEW_DATA (preview)->offset_x = 0;
  PREVIEW_DATA (preview)->offset_y = 0;

}

static void 
gimp_preview_update_preview  (GimpPreview *preview)
{
  GimpPreviewEvent *preview_event;
  GPixelRgn region;
  guint alpha;
  guint bpp;
  guint bpp_minus_alpha;
  size_t copy_size;
  guint image_x;
  guint image_y;
  guint image_width;
  guint image_height;
  guchar *image_data = NULL;
  guint dy;
  guint dx;
  guint sy;
  guint sx;
  guchar *src_ptr;
  guchar *dest_ptr;

  guint row;
  guint j;


  preview_event = g_new (GimpPreviewEvent, 1);
  preview_event->scale = preview->scale;
  preview_event->x = preview->image_x;
  preview_event->y = preview->image_y;
  preview_event->scaled_data = preview->buffer;
  
  /* FIXME: this in not accurate... */
  preview_event->width = (guint) (((double) preview->width) / preview->scale);
  preview_event->height = (guint) (((double) preview->height) / preview->scale);

  
  gimp_pixel_rgn_init (&region, preview->drawable,
		       preview_event->x, 
		       preview_event->y,
		       preview_event->x + preview_event->width,
		       preview_event->y + preview_event->height,
		       FALSE, FALSE);

  alpha = gimp_drawable_has_alpha (preview->drawable->id);
  bpp = preview->drawable->bpp;
  bpp_minus_alpha = bpp - alpha;
  copy_size = (size_t)(sizeof(guchar) * bpp);

  image_x = preview_event->x;
  image_y = preview_event->y;
  image_width = preview_event->width;
  image_height = preview_event->height;

  image_data = g_malloc (sizeof (guchar) * bpp 
			 * image_width * image_height);


  gimp_pixel_rgn_get_rect (&region, image_data,
			   image_x, image_y, image_width, image_height);
  
  /* Scale */
  src_ptr = image_data;
  dest_ptr = preview->buffer;

  if (bpp_minus_alpha == 3)
    {
      for (dy = 0; dy < preview->height; dy++)
	{
	  sy = (dy * image_width) / preview->width;
	  for (dx = 0; dx < preview->width; dx++)
	    {
	      sx = (dx * image_height) / preview->height;
	      src_ptr = image_data + (sy * image_width + sx) * bpp;

	      dest_ptr[0] = src_ptr[0];
	      dest_ptr[1] = src_ptr[1];
	      dest_ptr[2] = src_ptr[2];
	      
	      if (alpha) 
		dest_ptr[3] = src_ptr[3];
	      else
		dest_ptr[3] = 255;
	      
	      dest_ptr += 4;
	    }
	}
    } 
  else 
    {
      for (dy = 0; dy < preview->height; dy++)
	{
	  sy = dy / preview->scale;
	  for (dx = 0; dx < preview->width; dx++)
	    {
	      sx = dx / preview->scale;
	      src_ptr = image_data + (sy * image_width + sx) * bpp;

	      dest_ptr[0] = src_ptr[0];
	      dest_ptr[1] = src_ptr[0];
	      dest_ptr[2] = src_ptr[0];

	      if (alpha)
		dest_ptr[3] = src_ptr[1];
	      else
		dest_ptr[3] = 255;

	      dest_ptr += 4;
	    }
	}
    }

  if (PREVIEW_DATA (preview)->in_drag) 
    {
      for (row = 0; row < preview->height; row++)
	{
	  gimp_preview_draw_row (preview, row, &(preview->buffer[row 
								* preview->width
								* 4]));
	}
    }
  else
    {
      gtk_signal_emit (GTK_OBJECT (preview), 
		       gimp_preview_signals[UPDATE_PREVIEW],
		       preview_event);
    }

  gimp_preview_force_redraw (preview);

  g_free (image_data);
  g_free (preview_event);
  
}

void
gimp_preview_force_redraw (GimpPreview *preview)
{
  GdkRectangle req;

  req.x = (GTK_WIDGET (preview->preview)->allocation.width
	   - GTK_PREVIEW (preview->preview)->buffer_width)/2;
  req.y = ((GTK_WIDGET (preview->preview)->allocation.height
	    - GTK_PREVIEW (preview->preview)->buffer_height)/2);
  req.width = GTK_PREVIEW (preview->preview)->buffer_width;
  req.height = GTK_PREVIEW (preview->preview)->buffer_height;

  gtk_widget_draw (GTK_WIDGET (preview->preview), &req);
}

/* size_data = (preview->width) * (preview->drawable->bpp - alpha); */
void 
gimp_preview_draw_row (GimpPreview *preview, guint row, guchar *data)
{
  gint width;
  gint real_row;
  guchar *ptr;
  gint offset_x;
  gint offset_y;
  static size_t size = 0;
  static guint alpha;
  static guint bpp_minus_alpha;

  guchar *src_ptr;
  guchar *dest_ptr;

  guchar *check_row;
  guchar check;

  gint j;

  GdkRectangle req;

  width = preview->width;
  real_row = row;
  ptr = data;

  if (!size)
    {
      alpha = gimp_drawable_has_alpha (preview->drawable->id);
      bpp_minus_alpha = preview->drawable->bpp - alpha;
      size = sizeof (guchar) * preview->drawable->bpp;
    }

  if (preview->scale > 1)
    {
      real_row -= PREVIEW_DATA (preview)->offset_y;
      if (real_row < 0)
	return;
      if (real_row > preview->height - preview->scale)
	return;

      width -= (preview->scale - 1);
      ptr += PREVIEW_DATA (preview)->offset_x * 4;
    }

  src_ptr = ptr;
  dest_ptr = PREVIEW_DATA (preview)->preview_buffer_na;
  ptr = dest_ptr;

  if ((real_row / preview->check_size) & 1)
    check_row = preview->check_row_0;
  else
    check_row = preview->check_row_1;
  
  for (j = 0; j < width; j++)
    {
      check = check_row[j];
      
      dest_ptr[0] = check + ((src_ptr[0] - check) * src_ptr[3]) / 255;
      dest_ptr[1] = check + ((src_ptr[1] - check) * src_ptr[3]) / 255;
      dest_ptr[2] = check + ((src_ptr[2] - check) * src_ptr[3]) / 255;
      
      dest_ptr+=3;
      src_ptr+=4;
    }

  gtk_preview_draw_row (GTK_PREVIEW (preview->preview), ptr, 0, real_row, 
			width);

  /*
  req.x = (GTK_WIDGET (preview->preview)->allocation.width
	   - GTK_PREVIEW (preview->preview)->buffer_width)/2;
  req.y = ((GTK_WIDGET (preview->preview)->allocation.height
	    - GTK_PREVIEW (preview->preview)->buffer_height)/2 + real_row);
  req.width = GTK_PREVIEW (preview->preview)->buffer_width;
  req.height = 1;
  gtk_widget_draw (GTK_WIDGET (preview->preview), &req);

  */
}

