#include "gimp.h"
#include <GraphicsDefs.h>
#include <stdlib.h>

static void transparentize_rgb(Image, Image);
static void transparentize_indexed(Image, Image);

int main(int argc, char **argv)
{
	Image input, output;

	if (gimp_init(argc, argv))
	{
		input = gimp_get_input_image(0);
		output = gimp_get_output_image(0);
	
	
	    switch (gimp_image_type(input))
	    {
	    	case RGB_IMAGE:
				if (input && output)
				{
					transparentize_rgb(input, output);
					gimp_update_image(output);
				}
			
				if (input)
					gimp_free_image(input);
				if (output)
					gimp_free_image(output);
				break;

		    case GRAYA_IMAGE:
		    case GRAY_IMAGE:
				gimp_message("Transparentize: cannot operate on greyscale images");
				break;
		    case RGBA_IMAGE:
				gimp_message("Transparentize: cannot operate on RGBA images");
				break;
		    case INDEXED_IMAGE:
				if (input && output)
				{
					transparentize_indexed(input, output);
					gimp_update_image(output);
				}
			
				if (input)
					gimp_free_image(input);
				if (output)
					gimp_free_image(output);
				break;
		    default:
				gimp_message("Transparentize: cannot operate on unknown type images");
				break;
		}
	
		gimp_quit();
	}
	exit(0);
}

static void transparentize_rgb(Image input, Image output)
{
	long width, height;
	long channels, rowstride;
	unsigned char *src_row, *dest_row;
	unsigned char *src, *dest;
	short row, col;
	int x1, y1, x2, y2;
	int i;
	rgb_color fg;
	rgb_color pixel;
	
  
	gimp_image_area(input, &x1, &y1, &x2, &y2);
	
	// Get the size of the input image. (This will/must be the same as the size of the output image.

	width = gimp_image_width(input);
	height = gimp_image_height(input);
	channels = gimp_image_channels(input);
	rowstride = width * channels;
	
	src_row = gimp_image_data(input);
	dest_row = gimp_image_data(output);
	
	// Advance the source and destination pointers
	src_row += rowstride * y1;
	dest_row += rowstride * y1;
	
	gimp_foreground_color(&fg.red, &fg.green, &fg.blue);
	pixel.alpha = fg.alpha = 0xFF;
	
	
	for (row = y1; row < y2; row++)
	{
		src = src_row + (x1 * channels);
		dest = dest_row + (x1 * channels); 
		
		for (col = x1; col < x2; col++)
		{
			// Assume 3 channels
			pixel.red = *src++;
			pixel.green = *src++;
			pixel.blue = *src++;
			
			if (*(ulong *)&pixel == *(ulong *)&fg)
			{
				*dest++ = /*R4 BOGUS*/B_TRANSPARENT_COLOR.red;
				*dest++ = /*R4 BOGUS*/B_TRANSPARENT_COLOR.green;
				*dest++ = /*R4 BOGUS*/B_TRANSPARENT_COLOR.blue;
			}
			else
			{
				*dest++ = pixel.red;
				*dest++ = pixel.green;
				*dest++ = pixel.blue;
			}
		}	  
		
		src_row += rowstride;
		dest_row += rowstride;
	}
}



static void transparentize_indexed(Image input, Image output)
{
	long width, height;
	long channels, rowstride;
	unsigned char *src_row, *dest_row;
	unsigned char *src, *dest;
	short row, col;
	int x1, y1, x2, y2;
	int i, colors;
	rgb_color pixel;
	uchar* cmap = (uchar*) gimp_image_cmap(input);
	uchar fg = 0;
	
  	gimp_foreground_color(&pixel.red, &pixel.green, &pixel.blue);
  	colors = gimp_image_colors(input);

  	for (i = 0; i < colors; i++)
  	{
  		int j = i * 3;
  		if (cmap[j++] == pixel.red && cmap[j++] == pixel.green && cmap[j] == pixel.blue)
  		{
  			fg = i;
  			break;
  		}
  	}
	

	gimp_image_area(input, &x1, &y1, &x2, &y2);
	
	// Get the size of the input image. (This will/must be the same as the size of the output image.

	width = gimp_image_width(input);
	height = gimp_image_height(input);
	channels = gimp_image_channels(input);
	rowstride = width * channels;
	
	src_row = gimp_image_data(input);
	dest_row = gimp_image_data(output);
	
	// Advance the source and destination pointers
	src_row += rowstride * y1;
	dest_row += rowstride * y1;
	
	for (row = y1; row < y2; row++)
	{
		src = src_row + (x1 * channels);
		dest = dest_row + (x1 * channels); 
		
		for (col = x1; col < x2; col++)
		{
			uchar c = *src++;
			
			if (c == fg)
				c = B_TRANSPARENT_8_BIT;
			
			*dest++ = c;
		}	  
		
		src_row += rowstride;
		dest_row += rowstride;
	}
}



