/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __BUFFER_H__
#define __BUFFER_H__

#include "cow.h"
#include "areas.h"


/* ----------------------------------------------------------- */
/* need to reconcile Tag with ColorModel */
typedef guint Tag;
#define tag_valid(t) TRUE
#define tag_bytes(t) (t)
/* ----------------------------------------------------------- */


#define GIMP_BUFFER(b)        ((GimpBuffer*)(b))
#define GIMP_BUFFER_CLASS(b)  ((GimpBufferClass*)(b))

typedef struct _GimpBuffer GimpBuffer;
typedef struct _GimpBufferClass GimpBufferClass;


typedef enum
{
  BUFFER_NONE,
  BUFFER_FLAT,
  BUFFER_TILED,
  BUFFER_SHM,
  BUFFER_PTR
} BufferType;


typedef enum
{
  ALLOC_NONE,
  ALLOC_ALLOC,
  ALLOC_UNALLOC
} Alloc;


typedef enum
{
  VALIDATE_NONE,
  VALIDATE_VALIDATE,
  VALIDATE_INVALIDATE
} Validate;


typedef enum
{
  USE_NONE,
  USE_READ,
  USE_WRITE,
  USE_UPDATE,
  USE_RELEASE
} Use;



/* custom validator.  the void ptr is the raw memory located inside
   area in the buffer */
typedef void (*GimpValidator) (GimpBuffer *, GimpArea *, void *);


struct _GimpBufferClass
{
  /* subclass type */
  BufferType type;

  /* virtual method table */
  void         (* delete)   (GimpBuffer        *buffer);
  gboolean     (* alloc)    (GimpBuffer        *buffer,
                             GimpArea          *area,
                             Alloc              how);
  gboolean     (* map)      (GimpBuffer        *dst,
                             GimpArea          *darea,
                             GimpBuffer        *src,
                             GimpArea          *sarea);
  gboolean     (* validate) (GimpBuffer        *buffer,
                             GimpArea          *area,
                             Validate           how);
  GimpMemPtr * (* use)      (GimpBuffer        *buffer,
                             GimpPoint         *tile,
                             Use                how);
  gboolean     (* valid)    (GimpBuffer        *buffer,
                             GimpPoint         *tile);
};



/* the member vars have funny names to avoid surprises from the
   TileManager typedef */
struct _GimpBuffer
{
  /* the class object */
  void * klass;

  /* custom validator */
  GimpValidator vfunc;
  void * vdata;
  
  /* the xy dimensions */
  struct {

    struct {

      /* how many tiles */
      gint count;

      /* how big are they */
      gint size;

    } tile;

    struct {
      
      /* size of image data in pixels */
      gint size;
    
      /* offset of image data into first tile */
      gint offset;

    } image;
    
  } _x, _y;

  /* the z dimension */
  struct {
    
    /* pixel layout. merge with jays ColorModel concept */
    Tag tag;

    /* size in bytes */
    gint bpp;

  } _z;

  /* temporary, gimp expects buffers to have a position */
  gint qq_x;
  gint qq_y;
};


/* the image area */
#define gimp_buffer_width(b)   ((b)->_x.image.size)
#define gimp_buffer_height(b)  ((b)->_y.image.size)
#define gimp_buffer_xoff(b)    ((b)->_x.image.offset)
#define gimp_buffer_yoff(b)    ((b)->_y.image.offset)
#define gimp_buffer_tag(b)     ((b)->_z.tag)
#define gimp_buffer_bpp(b)     ((b)->_z.bpp)

/* the tile array */
#define gimp_buffer_tilewidth(b)   ((b)->_x.tile.size)
#define gimp_buffer_tileheight(b)  ((b)->_y.tile.size)
#define gimp_buffer_xtiles(b)      ((b)->_x.tile.count)
#define gimp_buffer_ytiles(b)      ((b)->_y.tile.count)



void        gimp_buffer_delete   (GimpBuffer        *buffer);


/* multi-tile routines */
gboolean    gimp_buffer_alloc    (GimpBuffer        *buffer,
                                  GimpArea          *area,
                                  Alloc              how);

gboolean    gimp_buffer_map      (GimpBuffer        *dst,
                                  GimpArea          *darea,
                                  GimpBuffer        *src,
                                  GimpArea          *sarea);

gboolean    gimp_buffer_validate (GimpBuffer        *buffer,
                                  GimpArea          *area,
                                  Validate           how);

/* single tile routines */
GimpMemPtr * gimp_buffer_use     (GimpBuffer        *buffer,
                                  GimpPoint         *tile,
                                  Use                how);

gboolean     gimp_buffer_valid   (GimpBuffer        *buffer,
                                  GimpPoint         *tile);

#endif
