/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "memory.h"


/* GimpMemMgr tracks, caches, and swaps all GimpMem objects in the
   system. */
struct _GimpMemMgr
{
  /* fixme: put a caching swapping memory manager here */
  int foo;
};




#define gimp_memmgr_is_joined(mem) \
          ((mem) && ((mem)->ptrcount>1))

#define gimp_memmgr_dirty(mem) \
          ((mem) && ((mem)->sharable = FALSE))

#define gimp_memmgr_ptr(mem) \
          (((mem) && (mem)->usecount) ? (mem)->data : NULL)


static GimpMem *
gimp_memmgr_new (gint size)
{
  GimpMem * mem = g_new (GimpMem, 1);
  mem->data = g_malloc (size);
  mem->size = size;
  mem->ptrcount = 0;
  mem->usecount = 0;
  mem->sharable = TRUE;
  return mem;
}


/* a ptr has just gone from 0->1 uses */
static gboolean
gimp_memmgr_use (GimpMem * mem)
{
  g_return_val_if_fail (mem != NULL, FALSE);

  if (mem->usecount == 0)
    {
      /* HOOK: swap in */
    }

  mem->usecount++;

  return TRUE;
}


/* a ptr has just gone from 1->0 uses */
static gboolean
gimp_memmgr_unuse (GimpMem * mem)
{
  g_return_val_if_fail (mem != NULL, FALSE);

  mem->usecount--;
  
  if ((mem->usecount == 0) && (mem->ptrcount != 0))
    {
      mem->sharable = TRUE;
      /* HOOK: swap out */
    }

  return TRUE;
}


/* make a copy of the mem, return new id */
static GimpMem *
gimp_memmgr_dup (GimpMem * mem)
{
  GimpMem * newmem;

  g_return_val_if_fail (mem != NULL, NULL);

  newmem = gimp_memmgr_new (mem->size);
  g_return_val_if_fail (newmem != NULL, NULL);

  /* HOOK: make sure mem is swapped in */
  if (mem->data != NULL)
    memcpy (newmem->data, mem->data, mem->size);
  else
    g_warning ("gimp_memmgr_dup copy failed");
  
  return newmem;
}


/* add a ptr to mem.  if mem doesn;t want ptr, make a copy and attach
   ptr to that.  if 'use' is true, ptr is carrying outstanding uses.
   return the id of whoever ptr got attached to */
static GimpMem *
gimp_memmgr_attach (GimpMem * mem,
                    gboolean  use)
{
  g_return_val_if_fail (mem != NULL, NULL);

  if (mem->sharable != TRUE)
    {
      mem = gimp_memmgr_dup (mem);
      g_return_val_if_fail (mem != NULL, NULL);
    }

  mem->ptrcount++;
  
  if (use == TRUE)
    if (gimp_memmgr_use (mem) != TRUE)
      g_warning ("gimp_memmgr_attach use failed");
  
  return mem;
}


static GimpMem *
gimp_memmgr_detach (GimpMem * mem,
                    gboolean  unuse)
{
  g_return_val_if_fail (mem != NULL, NULL);

  mem->ptrcount--;
  
  if (unuse == TRUE)
    if (gimp_memmgr_unuse (mem) != TRUE)
      g_warning ("gimp_memmgr_detach unuse failed");

  if (mem->ptrcount == 0)
    {
      /* HOOK: del from swap */
      g_free (mem->data);
      g_free (mem);
    }

  return mem;
}






static void
gimp_memptr_unlink (GimpMemPtr  *ptr)
{
  g_return_if_fail (ptr != NULL);

  if (d_is_alloced (ptr))
    {
      gboolean unuse = (d_usecount (ptr) != 0 ? TRUE : FALSE);

      if (gimp_memmgr_detach (ptr->mem, unuse) == NULL)
        g_warning ("gimp_memptr_unlink detach failed");
    }

  d_init (ptr);
}


void
gimp_memptr_init (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  ptr->mem = NULL;
  ptr->usecount = 0;
  ptr->joined = FALSE;
  ptr->size = 0;
  ptr->data = NULL;
}


void
gimp_memptr_uninit (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  if (d_usecount (ptr) != 0)
    g_warning ("uniniting an in-use gmp");
  
  gimp_memptr_unlink (ptr);  
}


gboolean
gimp_memptr_alloc (GimpMemPtr  *ptr,
                   gint         size)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (size > 0, FALSE);
  g_return_val_if_fail (! d_is_alloced (ptr), FALSE);
  {
    GimpMem * mem = gimp_memmgr_new (size);
    g_return_val_if_fail (mem != NULL, FALSE);
 
    mem = gimp_memmgr_attach (mem, FALSE);
    g_return_val_if_fail (mem != NULL, FALSE);

    ptr->mem = mem;
    ptr->size = size;
  }
  return TRUE;
}


gboolean 
gimp_memptr_unalloc (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount (ptr) == 0, FALSE);

  gimp_memptr_unlink (ptr);  

  return TRUE;
}


gboolean
gimp_memptr_use (GimpMemPtr *ptr,
                 gboolean    dirty)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);

  if (dirty == TRUE)
    {
      if (d_is_joined (ptr) && (d_split (ptr) != TRUE))
        {
          g_warning ("gimp_memptr_use split failed");
          return FALSE;
        }
      gimp_memmgr_dirty (ptr->mem);
    }

  if (ptr->usecount == 0)
    {
      if (gimp_memmgr_use (ptr->mem) != TRUE)
        {
          g_warning ("gimp_memptr_use use failed");
          return FALSE;
        }
      ptr->data = gimp_memmgr_ptr (ptr->mem);
    }

  ptr->usecount++;

  return TRUE;
}


gboolean
gimp_memptr_unuse (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount (ptr) != 0, FALSE);

  if (ptr->usecount == 1)
    {
      if (gimp_memmgr_unuse (ptr->mem) != TRUE)
        {
          g_warning ("gimp_memptr_unuse unuse failed");
          return FALSE;
        }
      ptr->data = NULL;
    }
  
  ptr->usecount--;
  
  return TRUE;
}


gboolean
gimp_memptr_join (GimpMemPtr *ptr,
                  GimpMemPtr *newptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (newptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);
  g_return_val_if_fail (! d_is_alloced (newptr), FALSE);

  newptr->mem = gimp_memmgr_attach (ptr->mem, FALSE);
  g_return_val_if_fail (newptr->mem != NULL, FALSE);

  if (newptr->mem == ptr->mem)
    {
      ptr->joined = TRUE;
      newptr->joined = TRUE;
    }
  else
    {
      newptr->size = d_size (ptr);
    }
  
  return TRUE;
}


gboolean
gimp_memptr_split (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);

  if (d_is_joined (ptr))
    {
      if (gimp_memmgr_is_joined (ptr->mem))
        {
          gboolean use = (ptr->usecount ? TRUE : FALSE);
          GimpMem * mem = ptr->mem;

          mem = gimp_memmgr_detach (mem, use);
          g_return_val_if_fail (mem != NULL, FALSE);

          mem = gimp_memmgr_dup (mem);
          g_return_val_if_fail (mem != NULL, FALSE);

          mem = gimp_memmgr_attach (mem, use);
          g_return_val_if_fail (mem != NULL, FALSE);

          ptr->mem = mem;
          if (ptr->usecount != 0)
            ptr->data = gimp_memmgr_ptr (mem);
        }

      ptr->joined = FALSE;
    }
  
  return TRUE;
}


