/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_tile.h"


#define DATA   (&tile_buffer->tiles[i].data)
#define VALID  (tile_buffer->tiles[i].valid)


static void
gimp_tile_buffer_delete (GimpBuffer * buffer)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i;

  for (i = 0; i < tile_buffer->ntiles; i++)
    {
      d_uninit (DATA);
    }

  g_free (tile_buffer->tiles);
  gimp_buffer_uninit (GIMP_BUFFER (tile_buffer));
  g_free (tile_buffer);
}


static gboolean
gimp_tile_buffer_focus (GimpBuffer   *buffer,
                        GimpPortion  *portion,
                        GimpArea     *roi)
{
  GimpArea * focus = &portion->focus;

  gint xtotal = (buffer->width +
                 buffer->x.phase +
                 buffer->x.period - 1) / buffer->x.period;

  gint xtiles = (focus->a.x +
                 buffer->x.phase +
                 buffer->x.period - 1) / buffer->x.period;

  gint ytiles = (focus->a.y +
                 buffer->y.phase +
                 buffer->y.period - 1) / buffer->y.period;
    
  portion->tilenum = (ytiles * xtotal) + xtiles;
  
  return TRUE;
}


static gboolean
gimp_tile_buffer_alloc (GimpBuffer   *buffer,
                        GimpPortion  *portion,
                        Alloc         how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = portion->tilenum;
  gint n = buffer->x.period * buffer->y.period * buffer->depth;

  switch (how)
    {
    case ALLOC_ALLOC:
      g_return_val_if_fail (d_alloc (DATA, n) == TRUE, FALSE);
      return TRUE;

    case ALLOC_UNALLOC:
      g_return_val_if_fail (d_unalloc (DATA) == TRUE, FALSE);
      VALID = FALSE;
      return TRUE;

    case ALLOC_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_tile_buffer_validate (GimpBuffer  *buffer,
                           GimpPortion *portion, 
                           Validate     how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = portion->tilenum;

  switch (how)
    {
    case VALIDATE_VALIDATE:
      if (VALID == TRUE)
        return TRUE;

      if (! d_is_alloced (DATA))
        {
          if (buffer->autoalloc == FALSE)
            return FALSE;
          
          if (gimp_tile_buffer_alloc (buffer, portion,
                                      ALLOC_ALLOC) != TRUE)
            {
              g_warning ("tilebuffer autoalloc failed");
              return FALSE;
            }
        }

      /* HOOK: PERFORM VALIDATION HERE */
      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      memset (DATA->data, 0, (buffer->x.period *
                              buffer->y.period *
                              buffer->depth));
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);

      VALID = TRUE;
      return TRUE;


    case VALIDATE_INVALIDATE:
      g_return_val_if_fail (d_usecount (DATA) == 0, FALSE);
      VALID = FALSE;
      return TRUE;


    case VALIDATE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}



static gboolean
gimp_tile_buffer_use (GimpBuffer  *buffer,
                      GimpPortion *portion, 
                      Use          how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = portion->tilenum;

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      if (buffer->autovalidate == FALSE)
        return FALSE;
          
      if (gimp_tile_buffer_validate (buffer, portion,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("tilebuffer autovalidate failed");
          return FALSE;
        }
    }

  switch (how)
    {
    case USE_READ:
      g_return_val_if_fail (d_read (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_UPDATE:
      g_return_val_if_fail (d_update (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_WRITE:
      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_RELEASE:
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_tile_buffer_query (GimpBuffer    *buffer,
                        GimpPortion   *portion,
                        GimpMemStatus *status)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = portion->tilenum;

  status->alloced   = (d_is_alloced (DATA) ? TRUE : FALSE);
  status->valid     = VALID;
  status->usecount  = (d_usecount (DATA));

  return TRUE;
}


static gboolean
gimp_tile_buffer_data (GimpBuffer      *buffer,
                       GimpPortion     *portion,
                       GimpPixelArray  *array)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = portion->tilenum;

  /* describe the memory layout */
  array->tag       = buffer->tag;
  array->width     = MIN (buffer->x.period,
                          portion->focus.b.x) - portion->focus.a.x;
  array->height    = MIN (buffer->y.period,
                          portion->focus.b.y) - portion->focus.a.y;
  array->pixstride = buffer->depth;
  array->rowstride = buffer->depth * buffer->x.period;

  /* get a pointer to the memory */
  array->data = DATA;

  /* and save an offset to the first pixel */
  array->offset = (portion->focus.a.y * array->rowstride) +
                  (portion->focus.a.x * array->pixstride);

  return TRUE;
}


static GimpTileBufferClass my_class =
{
  {
    BUFFER_TILED,
    gimp_tile_buffer_delete,
    gimp_tile_buffer_focus,
    gimp_tile_buffer_alloc,
    gimp_tile_buffer_validate,
    gimp_tile_buffer_use,
    gimp_tile_buffer_query,
    gimp_tile_buffer_data
  }
};






GimpTileBuffer *
gimp_tile_buffer_new (Tag   tag,
                      gint  width,
                      gint  height,
                      gint  tile_width,
                      gint  tile_height)
{
  GimpTileBuffer * tile_buffer;
  gint i;

  if (tile_width == 0)
    tile_width = 64;

  if (tile_height == 0)
    tile_height = 64;

  tile_buffer = g_new (GimpTileBuffer, 1);

  gimp_buffer_init (GIMP_BUFFER (tile_buffer),
                    tag, width, height,
                    tile_width, 0, tile_height, 0);

  tile_buffer->ntiles =
    ((width / tile_width) + 1) *
    ((height / tile_height) + 1);
  
  tile_buffer->tiles = g_new (GimpTile, tile_buffer->ntiles);

  for (i = 0; i < tile_buffer->ntiles; i++)
    {
      d_init (DATA);
      VALID = FALSE;
    }

  GIMP_BUFFER (tile_buffer)->klass = (void*) &my_class;

  return tile_buffer;
}


