﻿//  ------------------------------------------------------------------------------------
//  Copyright (c) 2015 Red Hat, Inc.
//  All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the ""License""); you may not use this
//  file except in compliance with the License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  THIS CODE IS PROVIDED *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED WARRANTIES OR
//  CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE, MERCHANTABLITY OR
//  NON-INFRINGEMENT.
//
//  See the Apache Version 2.0 License for specific language governing permissions and
//  limitations under the License.
//  ------------------------------------------------------------------------------------

// AnonymousRelay.cs uses a special feature of AMQP brokers and
// routing intermediaries called ANONYMOUS-RELAY. An anonymous relay
// is capable of sending messages to any endpoint through a single
// connection and sender link.
// 
// AnonymousRelay.cs is similar to the Interop.Server example
// as it performs an UPPERCASE service. In addition, AnonymousRelay.cs
// examines the offered capabilities of the peer when a connection 
// is established. If AnonymousRelay.cs detects an anonymous relay then it
// creates only a single sender link for all of its service replies
// regardless of the client ReplyTo address.
// 
// Sending all messages over a single SenderLink is much more efficient
// than creating a SenderLink, sending a message, and destroying the link
// on a per-message basis.
// 

// TODO:
//  Issue: Creating a sender link and sending a message in the onMessage
//         callback hangs. The first reply gets sent but the call to send
//         does not return.
//         Define conditional here to enable callback processing. If this
//         is not defined then a fallback loop to process messages in the
//         main thread is used.
//#define UseOnMessageCallback

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Amqp;
using Amqp.Framing;
using Amqp.Types;

namespace AnonymousRelay
{
    // SenderPool
    //
    // Given a session and a replyTo address
    // return the SenderLink to use for sending.
    // This version:
    //  * supports only one session
    //  * supports a 'relay' sender which, when defined,
    //    sends all messages regardless of the replyTo address
    //  * keeps senders open indefinitely
    //
    class SenderPool
    {
        // relay is a link to an anonymous_relay message router.
        // If relay exists then send all messages to it.
        private SenderLink relay = null;

        // Otherwise keep a map of sender links indexed by
        // the replyTo string.
        private Dictionary<String, SenderLink> senders =
            new Dictionary<String, SenderLink>();

        public SenderLink Relay
        {
            get { return relay; }
            set { relay = value; }
        }

        public SenderLink GetSender(Session session, String replyTo)
        {
            SenderLink sender = relay;
            if (null == sender)
            {
                if (senders.ContainsKey(replyTo))
                {
                    sender = senders[replyTo];
                }
                else
                {
                    String senderName = "PooledSender-" + senders.Count.ToString();
                    sender = new SenderLink(session, senderName, replyTo);
                    senders[replyTo] = sender;
                }
            }
            return sender;
        }
    }


    class AnonymousRelay
    {
        //
        // This server performs an UPPERCASE service
        //
        static String PerformService(String incoming)
        {
            return incoming.ToUpper();
        }

        //
        // Return message body as string.
        //
        static String GetContent(Message msg)
        {
            object body = msg.Body;
            if (body == null)
            {
                return "null";
            }
            else if (body is byte[])
            {
                return Encoding.UTF8.GetString((byte[])msg.Body);
            }
            else
            {
                return body.ToString();
            }
        }

        //
        // Sample invocation: server [urlAddress [serviceName]]
        //
        static int Main(string[] args)
        {
            Connection.DisableServerCertValidation = true;
            // uncomment the following to write frame traces
            //Trace.TraceLevel = TraceLevel.Frame;
            //Trace.TraceListener = (f, a) => Console.WriteLine(DateTime.Now.ToString("[hh:ss.fff]") + " " + string.Format(f, a));

            String urlString   = args.Length >= 1 ? args[0] : "amqp://guest:guest@localhost:5672";
            String amqpAddress = args.Length >= 2 ? args[1] : "service_queue";
            bool useRelayLink = false;

            Connection connection = null;
            Session session = null;
            bool receiverOpen = false;

            int exitStatus = 0;

            //
            // Sender pool
            //
            SenderPool senderPool = new SenderPool();

            //
            // on_connection_opened
            //

            //
            // Receiver onClosed
            //
            ClosedCallback onReceiverClosed = (ao, e) =>
            {
                Console.WriteLine("Receiver Closed: {0}", e.ToString());
                receiverOpen = false;
            };

#if UseOnMessageCallback
            //
            // on_message
            //
            MessageCallback onMessage = (link, message) =>
            {
                link.Accept(message);
                String replyTo = message.Properties.ReplyTo;
                Console.WriteLine("Received request. ReplyTo {0}", replyTo);
                SenderLink sndr = senderPool.GetSender(session, replyTo);
                Message response = new Message(PerformService(GetContent(message)));
                response.Properties = new Properties() { CorrelationId = message.Properties.MessageId };
                response.Properties.To = replyTo;
                sndr.Send(response);
                Console.WriteLine("Done sending response.");
                // 20150325 - sndr.Send never returns.
                // The response goes over the wire and received by the client with full acks.
            };
#endif

            //
            // on_start
            //

            // Process cli args
            if (args.Length > 0)
                urlString = args[0];

            try
            {
                // The address to serve. Requests for our service arrive here.
                Address urlAddress = new Address(urlString);

                // Open a connection, detect anonymous relay.
                ManualResetEvent connOpenEvent = new ManualResetEvent(false);
                OnOpened onOpened = (c, o) =>
                {
                    Symbol ar = new Symbol("ANONYMOUS-RELAY");
                    if (o.OfferedCapabilities.Contains(ar))
                    {
                        useRelayLink = true;
                    }
                    connOpenEvent.Set();
                };
                connection = new Connection(urlAddress, null, null, onOpened);
                bool connOpenContext = false;
                bool connectionOpen = connOpenEvent.WaitOne(10000, connOpenContext);
                if (!connectionOpen)
                {
                    throw new Exception("Connection did not open");
                }

                // Open a session
                Console.WriteLine("Opening session");
                session = new Session(connection);

                // If connected to an anoymous relay then create the relay sender
                OnAttached onAttached = (l, a) => { Console.WriteLine("relay sender attached"); };
                if (useRelayLink)
                {
                    senderPool.Relay = new SenderLink(session, "server-sender", new Target(), onAttached);
                }

                // Create server receiver link
                // When messages arrive, reply to them
                ReceiverLink receiver = new ReceiverLink(session, "server-receiver", amqpAddress);
                receiver.Closed = onReceiverClosed;
#if UseOnMessageCallback
                receiver.Start(10, onMessage);

                Console.WriteLine("Press enter key to exit...");
                Console.ReadLine();
#else
                receiver.Start(10);

                receiverOpen = true;

                while (true)
                {
                    Message request = receiver.Receive(500);
                    if (null != request)
                    {
                        Console.WriteLine("received request");
                        receiver.Accept(request);
                        String replyTo = request.Properties.ReplyTo;
                        Console.WriteLine("ReplyTo: {0}", replyTo);
                        SenderLink sender = senderPool.GetSender(session, replyTo);

                        Message response = new Message(PerformService(GetContent(request)));
                        Console.WriteLine("response content: {0}", response.ToString());
                        response.Properties = new Properties() { CorrelationId = request.Properties.MessageId };
                        response.Properties.To = replyTo;

                        try
                        {
                            sender.Send(response, 10000);
                        }
                        catch (Exception exception)
                        {
                            Console.WriteLine("Error waiting for response to be sent: {0} exception {1}",
                                GetContent(response), exception.Message);
                            break;
                        }
                        Console.WriteLine("Processed request: {0} -> {1}",
                            GetContent(request), GetContent(response));
                    }
                    else
                    {
                        // timed out waiting for request. This is normal.
                        if (!receiverOpen)
                        {
                            Console.WriteLine("Exiting: Receiver is closed.");
                            break;
                        }
                    }
                }
#endif
            }
            catch (System.Net.Sockets.SocketException e)
            {
                Console.WriteLine("Exception {0}.", e);
                exitStatus = 1;
            }
            catch (Exception e)
            {
                Console.WriteLine("Exception {0}.", e);
                exitStatus = 1;
            }

            if (null != session) session.Close();
            if (null != connection) connection.Close();

            return exitStatus;
        }
    }
}
