/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.activemq.transport.netty;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.X509ExtendedKeyManager;
import java.net.Socket;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;

/**
 * An X509ExtendedKeyManager wrapper which always chooses and only
 * returns the given alias, and defers retrieval to the delegate
 * key manager.
 */
public class X509AliasKeyManager extends X509ExtendedKeyManager {

   private X509ExtendedKeyManager delegate;
   private String alias;

   public X509AliasKeyManager(String alias, X509ExtendedKeyManager delegate) throws IllegalArgumentException {
      if (alias == null) {
         throw new IllegalArgumentException("The given key alias must not be null.");
      }

      this.alias = alias;
      this.delegate = delegate;
   }

   @Override
   public String chooseClientAlias(String[] keyType, Principal[] issuers, Socket socket) {
      return alias;
   }

   @Override
   public String chooseServerAlias(String keyType, Principal[] issuers, Socket socket) {
      return alias;
   }

   @Override
   public X509Certificate[] getCertificateChain(String alias) {
      return delegate.getCertificateChain(alias);
   }

   @Override
   public String[] getClientAliases(String keyType, Principal[] issuers) {
      return new String[]{alias};
   }

   @Override
   public PrivateKey getPrivateKey(String alias) {
      return delegate.getPrivateKey(alias);
   }

   @Override
   public String[] getServerAliases(String keyType, Principal[] issuers) {
      return new String[]{alias};
   }

   @Override
   public String chooseEngineClientAlias(String[] keyType, Principal[] issuers, SSLEngine engine) {
      return alias;
   }

   @Override
   public String chooseEngineServerAlias(String keyType, Principal[] issuers, SSLEngine engine) {
      return alias;
   }
}
