/*
 * Copyright 2016 Red Hat Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
'use strict';

var path = require('path');
var fs = require('fs');
var rhea = require('rhea');
var log = require("./log.js").logger();

function authenticate(credentials, options) {
    return new Promise(function(resolve, reject) {
        if( credentials && credentials.username ) {
            options.username = credentials.username;
            if (credentials.password) {
                log.debug('authenticating as %s using PLAIN', options.username);
                options.password = credentials.password;
            } else if (credentials.token) {
                log.debug('authenticating as %s using XOAUTH2', options.username);
                options.token = credentials.token;
            }
        } else {
            log.debug('authenticating as anonymous');
            options.username = "anonymous";
        }

        var authServer = rhea.connect(options);
        var handled = false;
        authServer.on("connection_open", function (context) {
            handled = true;
            context.connection.close();
            resolve(context.connection.properties);
        });
        var handleFailure = function (context) {
            if (!handled) {
                handled = true;
                if (context.error !== undefined) {
                    log.error('Authentication failed! error no: '+ context.error.errno + ' syscall: '+context.error.syscall+ ' address: '+context.error.address+ ' port: '+ context.error.port + ' code: '+ context.error.code);
                }
                reject(JSON.stringify(context.connection.get_error()));
            }
        };
        authServer.on("connection_error", handleFailure);
        authServer.on("connection_close", handleFailure);
        authServer.on("disconnected", handleFailure);

    });
}

function default_options(env) {
    var options = {
        host: env.AUTHENTICATION_SERVICE_HOST,
        port: env.AUTHENTICATION_SERVICE_PORT,
        sasl_init_hostname: env.AUTHENTICATION_SERVICE_SASL_INIT_HOST
    };
    var ca_path = env.AUTH_SERVICE_CA || path.resolve('/opt/agent/authservice-ca', 'tls.crt');
    try {
        options.ca = [fs.readFileSync(ca_path)];
        options.transport = 'tls';
        options.rejectUnauthorized = false;
    } catch (error) {
        log.warn('CA cannot be loaded from ' + ca_path + ': ' + error);
    }
    return options;
}

module.exports.authenticate = authenticate;
module.exports.default_options = default_options;

