/*
 * Copyright 2011 JBoss Inc
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */

package org.drools.guvnor.server.builder;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.drools.guvnor.client.common.AssetFormats;
import org.drools.guvnor.server.builder.DeclaredTypesSorter.DeclaredTypeAssetInheritanceInformation;
import org.drools.guvnor.server.contenthandler.ContentHandler;
import org.drools.guvnor.server.contenthandler.ContentManager;
import org.drools.guvnor.server.contenthandler.IRuleAsset;
import org.drools.guvnor.server.util.DroolsHeader;
import org.drools.repository.AssetItem;
import org.drools.repository.PackageItem;

public class PackageDRLAssembler extends AssemblerBase {

    private StringBuilder src;

    public PackageDRLAssembler(PackageItem packageItem) {
        super( packageItem );
    }

    public String getDRL() {
        src = new StringBuilder();

        loadHeader();
        loadDSLFiles();
        loadFunctions();
        loadRuleAssets();

        return src.toString();
    }

    private void loadHeader() {
        src.append( "package " ).append( this.packageItem.getName() ).append( "\n" );

        //Add package header first as declared types may depend on an import (see https://issues.jboss.org/browse/JBRULES-3133)
        final String packageHeader = DroolsHeader.getDroolsHeader( packageItem );
        src.append( DroolsHeader.getPackageHeaderImports( packageHeader ) );

        loadDeclaredTypes();

        //Globals may be a Declared type (see https://bugzilla.redhat.com/show_bug.cgi?id=756421)
        src.append( DroolsHeader.getPackageHeaderGlobals( packageHeader ) );
        src.append( DroolsHeader.getPackageHeaderMiscellaneous( packageHeader ) );
    }

    private void loadDeclaredTypes() {
        final List<AssetItem> assets = new ArrayList<AssetItem>();
        final DeclaredTypesSorter sorter = new DeclaredTypesSorter();

        //Get list of candidates
        Iterator<AssetItem> assetItemIterator = getAssetItemIterator( AssetFormats.DRL_MODEL );
        while ( assetItemIterator.hasNext() ) {
            AssetItem assetItem = assetItemIterator.next();
            if ( !assetItem.isArchived() && !assetItem.getDisabled() ) {
                assets.add( assetItem );
            }
        }

        //Order candidates by inheritance and add to source
        final List<DeclaredTypeAssetInheritanceInformation> sortedAssets = sorter.sort( assets );
        for ( DeclaredTypeAssetInheritanceInformation dt : sortedAssets ) {
            src.append( dt.getDrl() ).append( "\n\n" );
        }
        
    }

    private void loadFunctions() {
        Iterator<AssetItem> assetItemIterator = getAssetItemIterator( AssetFormats.FUNCTION );
        while ( assetItemIterator.hasNext() ) {
            addAsset( assetItemIterator.next() );
        }
    }

    private void loadRuleAssets() {
        Iterator<AssetItem> assetItemIterator = getAllAssets();
        while ( assetItemIterator.hasNext() ) {
            addRuleAsset( assetItemIterator.next() );
        }
    }

    private void addRuleAsset(AssetItem asset) {
        if ( !asset.isArchived() && !asset.getDisabled() ) {
            ContentHandler handler = ContentManager.getHandler( asset.getFormat() );
            if ( handler.isRuleAsset() ) {
                IRuleAsset ruleAsset = (IRuleAsset) handler;
                ruleAsset.assembleDRL( builder,
                                       asset,
                                       src );
            }
            src.append( "\n\n" );
        }
    }

    private void addAsset(AssetItem assetItem) {
        if ( !assetItem.isArchived() && !assetItem.getDisabled() ) {
            src.append( assetItem.getContent() ).append( "\n\n" );
        }
    }

}
