package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

class DirectoryPatcher implements Patcher {

    private static final Logger logger = LoggerFactory.getLogger(DirectoryPatcher.class);

    public final File rootDir;
    public final List<String> removeList;
    public final List<String> protectedList;
    public final List<PatchEntry> patchEntries;
    // File (path) -> checksums mapping (there can be multiple checksums for one path -- coming from different versions)
    public final Map<String, List<Checksum>> checksums;

    public DirectoryPatcher(File rootDir, List<String> removeList, List<String> protectedList,
                     List<PatchEntry> patchEntries, Map<String, List<Checksum>> checksums) {
        this.rootDir = rootDir;
        this.removeList = removeList;
        this.protectedList = protectedList;
        this.patchEntries = patchEntries;
        this.checksums = checksums;
    }

    @Override
    public void backup(File baseBackupDir) throws IOException {
        File backupDir = new File(baseBackupDir, rootDir.getName());
        logger.info("Backing-up (copying) root directory {} to {}.", rootDir, backupDir);
        FileUtils.copyDirectory(rootDir, backupDir);
    }

    @Override
    public void apply() throws IOException {
        removeOldFiles();
        applyUpdates();
    }

    private void removeOldFiles() throws IOException {
        logger.info("Removing old files from directory {}", rootDir.getAbsolutePath());
        // remove all files - beware of the protected list
        List<String> pathsToRemove = findPathsToRemove(removeList);
        for (String relativePath : pathsToRemove) {
            // by default mark the file for removal, unless decided otherwise below
            boolean shouldRemove = true;
            File entryToRemove = new File(rootDir, relativePath);
            if (protectedList.contains(relativePath)) {
                shouldRemove = shouldRemoveProtectedFile(entryToRemove, relativePath);
            }
            if (shouldRemove) {
                removeEntryIfExists(entryToRemove);
            }
        }
    }

    /**
     * Checks whether specified file included in the protectedlist should be removed.
     *
     * A file in the protectedlist can be removed only if:
     *  - it no longer exists in the new directory
     *  - AND it does not contain any user changes
     *
     * @return true if the specified file in the protectedlist should be removed
     */
    private boolean shouldRemoveProtectedFile(File protectedFile, String relativePath) throws IOException {
        // Files in the protectedlist should not usually be removed, there is just one exception:
        //  - the file was removed in the newer version and the file in current distribution does not contain any custom changes
        boolean shouldRemove = false;
        final String targetMD5Hex = Files.hash(protectedFile, Hashing.md5()).toString();
        logger.trace("Considering file [{}] included in the protectedlist with MD5 checksum [{}] for removal.", protectedFile, targetMD5Hex);

        if (isInUpdateList(relativePath)) {
            logger.debug("File {} in update list, not removing it. It will be properly handled when applying the new changes.", relativePath);
        } else {
            // the file is not in update list, which means it was removed in the new distribution
            // check whether the old file has user changes, if it does, just create marker file, otherwise remove it
            if (checksums.containsKey(relativePath) && containsMatchingChecksum(checksums.get(relativePath), targetMD5Hex)) {
                shouldRemove = true;
                logger.info("File {} is in the protectedlist, but no user changes were detected. Deleting it directly.", relativePath);
            } else {
                logger.warn("File {} is in the protectedlist and it only exists in the old directory. Creating just marker " +
                        "file with .removed suffix instead of removing it. Please investigate the change manually and " +
                        "remove the file if needed.", protectedFile);
                FileUtils.touch(new File(rootDir, relativePath + ".removed"));
                logger.trace("File {} is in the protectedlist, ignoring it.", relativePath);
            }
        }
        return shouldRemove;
    }

    private void removeEntryIfExists(File entry) throws IOException {
        if (entry.exists()) {
            logger.trace("Removing entry {}.", entry.getAbsolutePath());
            if (entry.isFile()) {
                PatchingUtils.deleteFileAndParentsIfEmpty(entry, rootDir);
            } else if (entry.isDirectory()) {
                PatchingUtils.deleteDirAndParentsIfEmpty(entry, rootDir);
            } else {
                // this should not really happen as the remove-list should only contain files and dirs. Throw exception to
                // indicate callers are passing incorrect remove-list
                throw new RuntimeException("Specified file '" + entry + "' is in fact directory. Remove list " +
                        "should only contain actual files.");
            }
        } else {
            logger.trace("File {} not found, can not remove it.", entry);
            // nothing to do here
        }
    }

    private boolean isInUpdateList(final String relativePath) {
        return patchEntries.stream().anyMatch(entry -> entry.getRelativePath().equals(relativePath));
    }

    /**
     * Creates list of concrete relative paths that should be removed from the directory.
     *
     * The input list may contain wildcards (e.g. *), aka globs.
     *
     * @param removeListWithGlobs list of relative paths, possibly with wildcards
     * @return list of relative paths to remove, these are concrete paths exist in the directory (no wildcards)
     */
    private List<String> findPathsToRemove(List<String> removeListWithGlobs) {
        logger.debug("Gathering paths that will be removed, based on remove list with wildcards.");
        Collection<Pattern> regexPatterns = removeListWithGlobs.stream().
                map(input -> Pattern.compile(PatchingUtils.createRegexFromGlob(input))).
                collect(Collectors.toList());

        // gather all paths in the directory
        Collection<File> files = FileUtils.listFilesAndDirs(rootDir, TrueFileFilter.INSTANCE, DirectoryFileFilter.INSTANCE);

        // remove the directory root as we will not delete that one in any case
        files = files.stream().filter(file -> !file.equals(rootDir)).collect(Collectors.toList());

        // transform the files into relative paths
        Collection<String> paths = files.stream().
                filter(file -> file.getAbsolutePath().length() > rootDir.getAbsolutePath().length()).
                map(this::stripRootDirFromPath).
                collect(Collectors.toList());

        return paths.stream().
                filter(path -> matchesOneOfRegexs(path, regexPatterns)).
                collect(Collectors.toList());
    }

    private String stripRootDirFromPath(File file) {
        return file.getAbsolutePath().
                substring(rootDir.getAbsolutePath().length() + 1).
                replace(File.separatorChar, '/');
    }

    private boolean matchesOneOfRegexs(final String path, Collection<Pattern> regexPatterns) {
        return regexPatterns.stream().anyMatch(pattern -> pattern.matcher(path).matches());
    }


    private void applyUpdates() throws IOException {
        logger.info("Applying updates to directory {}", rootDir.getAbsolutePath());
        for (PatchEntry patchEntry : patchEntries) {
            String relPath = patchEntry.getRelativePath();
            File entryToCopy = patchEntry.getActualFile();
            File target = new File(rootDir, relPath);
            if (protectedList.contains(relPath) && target.exists()) {
                final String sourceMD5Hex = Files.hash(entryToCopy, Hashing.md5()).toString();
                final String targetMD5Hex = Files.hash(target, Hashing.md5()).toString();
                logger.trace("Source file ({}) MD5 checksum: {}", entryToCopy, sourceMD5Hex);
                logger.trace("Target file ({}) MD5 checksum: {}", target, targetMD5Hex);
                // do not create ".new" marker file for files that are identical (it would only confuse users)
                if (sourceMD5Hex.equals(targetMD5Hex)) {
                    logger.debug("File {} in the protectedlist will be ignored, because there are no changes. The patched file" +
                            " is identical to the file being patched (current file in the directory).", entryToCopy);
                    continue;
                }
                // check if the current file has a known checksum - e.g. it was updated by one of the patches and thus
                // still has the default value shipped by one the previous versions. In this case it is safe to upgrade
                // the file automatically
                if (checksums.containsKey(relPath) && containsMatchingChecksum(checksums.get(relPath), targetMD5Hex)) {
                    logger.info("File {} is in the protectedlist, but no user changes were detected. Replacing it with the latest version.", relPath);
                } else {
                    logger.warn("File {} is in the protectedlist, creating a new file with suffix .new instead of overwriting. " +
                            "Please investigate the differences and apply them manually.", relPath);
                    target = new File(rootDir, relPath + ".new");
                }
            }
            if (entryToCopy.isFile()) {
                logger.trace("Copying file {} to {}", entryToCopy, target);
                FileUtils.copyFile(entryToCopy, target);
            } else {
                logger.trace("Copying contents of directory {} to {}", entryToCopy, target);
                FileUtils.copyDirectory(entryToCopy, target);
            }
        }
    }

    private boolean containsMatchingChecksum(List<Checksum> checksums, final String targetChecksum) {
        return checksums.stream().
                anyMatch(checksum -> checksum.getHexValue().equals(targetChecksum));
    }

}
