package org.jboss.brmsbpmsuite.patching.systemproperty;

import org.jboss.brmsbpmsuite.patching.client.ConfigurationDiffsGeneratorTest;
import org.jboss.brmsbpmsuite.patching.xml.XMLDocumentDao;
import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

public class SystemPropertyDiffExtractorTest {

    private static final File SERVER_1_CONFIG_DIR = createFileFromResource("/config-test/eap-servers/eap-version-1/standalone/configuration");
    private static final File SERVER_2_CONFIG_DIR = createFileFromResource("/config-test/eap-servers/eap-version-2/standalone/configuration");

    private XMLDocumentDao documentDao;

    @Before
    public void setUp() throws ParserConfigurationException, TransformerConfigurationException {
        documentDao = new XMLDocumentDao();
    }

    @Test
    public void shouldNotProduceActionsForUnchangedConfiguration() throws IOException, SAXException, ParserConfigurationException {
        Document server1Document = loadDocument(SERVER_1_CONFIG_DIR, "standalone.xml");
        Document server2Document = loadDocument(SERVER_2_CONFIG_DIR, "standalone.xml");

        SystemPropertyDiffExtractor diffExtractor = new SystemPropertyDiffExtractor(server1Document);
        Document diffDocument = diffExtractor.createDiff(server2Document);

        NodeList properties = diffDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(0));
    }

    @Test
    public void shouldProduceAdditionsForAddedProperties() throws IOException, SAXException, ParserConfigurationException {
        Document server1Document = loadDocument(SERVER_1_CONFIG_DIR, "standalone-1.xml");
        Document server2Document = loadDocument(SERVER_2_CONFIG_DIR, "standalone-1.xml");

        SystemPropertyDiffExtractor diffExtractor = new SystemPropertyDiffExtractor(server1Document);
        Document diffDocument = diffExtractor.createDiff(server2Document);

        NodeList properties = diffDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(1));

        NamedNodeMap attributes = properties.item(0).getAttributes();
        assertThat(attributes.getNamedItem("action").getNodeValue(), is("add"));
        assertThat(attributes.getNamedItem("name").getNodeValue(), is("property3"));
    }

    @Test
    public void shouldProduceUpdatesForChangedProperties() throws IOException, SAXException, ParserConfigurationException {
        Document server1Document = loadDocument(SERVER_1_CONFIG_DIR, "standalone-2.xml");
        Document server2Document = loadDocument(SERVER_2_CONFIG_DIR, "standalone-2.xml");

        SystemPropertyDiffExtractor diffExtractor = new SystemPropertyDiffExtractor(server1Document);
        Document diffDocument = diffExtractor.createDiff(server2Document);

        NodeList properties = diffDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(1));

        NamedNodeMap attributes = properties.item(0).getAttributes();
        assertThat(attributes.getNamedItem("action").getNodeValue(), is("update"));
        assertThat(attributes.getNamedItem("name").getNodeValue(), is("property2"));
        assertThat(attributes.getNamedItem("value").getNodeValue(), is("different-value2"));
    }

    @Test
    public void shouldProduceAnAdditionAndReplacedForChangedPropertyNames() throws IOException, SAXException, ParserConfigurationException {
        Document server1Document = loadDocument(SERVER_1_CONFIG_DIR, "standalone-3.xml");
        Document server2Document = loadDocument(SERVER_2_CONFIG_DIR, "standalone-3.xml");

        SystemPropertyDiffExtractor diffExtractor = new SystemPropertyDiffExtractor(server1Document);
        Document diffDocument = diffExtractor.createDiff(server2Document);

        NodeList properties = diffDocument.getDocumentElement().getElementsByTagName("property");
        assertThat(properties.getLength(), is(2));

        NamedNodeMap attributes = properties.item(0).getAttributes();
        assertThat(attributes.getNamedItem("action").getNodeValue(), is("remove"));
        assertThat(attributes.getNamedItem("name").getNodeValue(), is("property2"));
        assertThat(attributes.getNamedItem("value").getNodeValue(), is("value2"));

        attributes = properties.item(1).getAttributes();
        assertThat(attributes.getNamedItem("action").getNodeValue(), is("add"));
        assertThat(attributes.getNamedItem("name").getNodeValue(), is("different-property2"));
        assertThat(attributes.getNamedItem("value").getNodeValue(), is("value2"));
    }

    private static File createFileFromResource(String filePath) {
        return new File(ConfigurationDiffsGeneratorTest.class.getResource(filePath).getPath());
    }

    private Document loadDocument(File dir, String filename) throws IOException, SAXException {
        File file = new File(dir, filename);
        return documentDao.load(new FileInputStream(file));
    }

}