/*******************************************************************************
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.websphere.management.j2ee;

/**
 * The Notification model specifies the attributes that must be included in every event
 * generated by the management system. A Notification includes:
 * The name of the source managed object
 * <ul>
 * <li>A timestamp</li>
 * <li>a sequence number</li>
 * <li>a string that identifies its type</li>
 * <li>a message that provides additional information about the event</li>
 * <li>optional vendor specific user data.</li>
 * </ul>
 * All management systems that support event notification must support all of
 * the attributes of the Notification model. All events that are generated on behalf of
 * managed objects that are part of or extend the J2EE Management Model must
 * provide valid data for each of the attributes specified by the Notification model.
 */
public interface Notification {

    /**
     * The name of the managed object that generated this event. The value of the
     * source attribute must be equal to the OBJECT_NAME of the managed object
     * from which the event originated.
     */
    String getSource();

    /**
     * The time of the event represented as a long, whose value is the number of
     * milliseconds since January 1, 1970, 00:00:00.
     */
    long getTimeStamp();

    /**
     * The sequence number of the event. Identifies the position of the event in a
     * stream of events. The sequence number provides a means of determining the order
     * of sequential events that occurred with the same timestamp within the minimum
     * supported unit of time. The scope of the sequence number is the source of the
     * event.
     */
    long getSequenceNumber();

    /**
     * The type of an event, not to be confused with its Java class, is the
     * characterization of a generic event object. The type is assigned by the source
     * object and conveys the semantic meaning of a particular event. The type is given
     * as a string, which is interpreted as any number of dot-separated components. This
     * allows some structure in the naming of event types.
     * All strings prefixed by j2ee. are reserved for the standard events defined by
     * this specification. Otherwise, event sources are free to define all types they wish to
     * use when naming the events they emit. All of the event types emitted by an event
     * provider must be listed in the eventTypes attribute.
     * Standard event types are:
     * <ul>
     * <li>j2ee.object.created- a new managed object was created.</li>
     * <li>j2ee.object.deleted - a managed object was deleted.</li>
     * <li>j2ee.state.starting - a state manageable object entered the starting</li>
     * state.
     * <li>j2ee.state.running - a state manageable object entered the running state.</li>
     * <li>j2ee.state.stopping - a state manageable object entered the stopping
     * state. </li>
     * <li>j2ee.state.stopped - a state manageable object entered the stopped state.</li>
     * <li>j2ee.state.failed - a state manageable object entered the failed state.</li>
     * <li>j2ee.attribute.changed - an attribute has change value</li>
     * </ul>
     */
    String getType();

    /**
     * An informational message about the event.
     */
    String getMessage();

    /**
     * Optional data the notification broadcaster wishes to communicate to listeners.
     * The content of the data is vendor-specific. The userData attribute may be null.
     */
    Object getUserData();

}
