/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.websphere.ws.sib.unittest.ras;

import java.util.ArrayList;
import java.util.List;

import junit.framework.AssertionFailedError;

import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ws.sib.utils.ras.SibMessage;

/**
 * SibMessageListener is used when assertion support is enabled. It listens to SibMessage
 * and records the error messages flowing through it
 */  
public class SibMessageListener implements SibMessage.Listener
{
  /** The list of error messages we've seen */
  private List<LogRecord> _logRecords = new ArrayList<LogRecord>();
  
  /* -------------------------------------------------------------------------- */
  /* message method
  /* -------------------------------------------------------------------------- */
  /**
   * The method called to indicate that an error message is being generated by SibMessage
   * 
   * @param type   The type of the error message
   * @param me     The messaging engine (if any) generating the message
   * @param tc     The trace component generating the message
   * @param msgKey The message key of the message being generated
   * @param objs   The inserts of the message being generated
   * @param formattedMessage A formatted version of the message being generated
   */
  public void message(SibMessage.Listener.MessageType type, String me, TraceComponent tc, String msgKey, Object objs, Object[] formattedMessage)
  {
    switch(type)
    {
      case AUDIT  : _logRecords.add(new LogRecord(Severity.AUDIT,msgKey,objs));
                    break;       
      case ERROR  : _logRecords.add(new LogRecord(Severity.ERROR,msgKey,objs));
                    break;       
      case FATAL  : _logRecords.add(new LogRecord(Severity.FATAL,msgKey,objs));
                    break;       
      case INFO   : _logRecords.add(new LogRecord(Severity.INFO,msgKey,objs));
                    break;       
      case SERVICE: _logRecords.add(new LogRecord(Severity.SERVICE,msgKey,objs));
                    break;       
      case WARNING: _logRecords.add(new LogRecord(Severity.WARNING,msgKey,objs));
                    break;       
    }     
  }
  
  /* -------------------------------------------------------------------------- */
  /* assertLogged method
  /* -------------------------------------------------------------------------- */
  /**
   * This method can be used to assert that a log record occurred. It will throw
   * a junit.framework.AssertionFailedError if the log record hasn't been seen
   * 
   * @param record The log record
   */
  public void assertLogged(LogRecord record)
  {
    if (!_logRecords.contains(record))
    {
      throw new AssertionFailedError("The log record "+record+" did not occur in the SibMessage log");
    }
  }

  /* ------------------------------------------------------------------------ */
  /* assertNotLogged method                                    
  /* ------------------------------------------------------------------------ */
  /**
   * This method can be used to assert that a log record did not occur. It will
   * throw a junit.framework.AssertionFailedError if the log record has been
   * seen. The cause of the error will indicate the point in the code that 
   * the log entry was made.
   * 
   * @param record The log record to check was not made.
   */
  public void assertNotLogged(LogRecord record)
  {
    if (_logRecords.contains(record))
    {
      LogRecord actual = _logRecords.get(+_logRecords.indexOf(record));
      Error e = new AssertionFailedError("The log record "+record+" does occur in the SibMessage log");
      e.initCause(actual.getStackTrace());
      throw e;
    }
  }

  /* -------------------------------------------------------------------------- */
  /* reset method
  /* -------------------------------------------------------------------------- */
  /**
   * Throw away all knowledge of which LogRecords we've seen
   */
  public void reset()
  {
    _logRecords.clear();
  }
}