/*******************************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.sip.stack.transaction.transactions.st;

import jain.protocol.ip.sip.SipEvent;
import jain.protocol.ip.sip.SipParseException;
import jain.protocol.ip.sip.SipProvider;
import jain.protocol.ip.sip.header.HeaderParseException;
import jain.protocol.ip.sip.header.ViaHeader;
import jain.protocol.ip.sip.message.Request;
import jain.protocol.ip.sip.message.Response;

import com.ibm.ws.jain.protocol.ip.sip.message.RequestImpl;
import com.ibm.ws.jain.protocol.ip.sip.message.ResponseImpl;
import com.ibm.ws.sip.stack.context.MessageContext;
import com.ibm.ws.sip.stack.context.MessageContextFactory;
import com.ibm.ws.sip.stack.transaction.SIPTransactionConstants;
import com.ibm.ws.sip.stack.transaction.SIPTransactionStack;
import com.ibm.ws.sip.stack.transaction.transactions.BranchMethodKey;
import com.ibm.ws.sip.stack.transaction.transactions.MergedRequestKey;
import com.ibm.ws.sip.stack.transaction.transactions.SIPTransactionImpl;
import com.ibm.ws.sip.stack.transaction.transactions.SIPTransactionsModel;
import com.ibm.ws.sip.stack.transaction.transport.SIPTransportException;
import com.ibm.ws.sip.stack.transaction.util.ApplicationProperties;

public abstract class SIPServerTransactionImpl
	extends SIPTransactionImpl
	implements SIPServerTransaction 
{
	/**
	 * identifier of the source request per rfc3261 8.2.2.2,
	 * or null if this feature is disabled in configuration
	 */
	private MergedRequestKey m_mergedRequestKey;

	public SIPServerTransactionImpl(
		SIPTransactionStack transactionStack,
		SipProvider provider,
		Request req,
		BranchMethodKey key,
		MergedRequestKey mergedRequestKey) 
	{
		super(transactionStack, provider, req, key, -1);
		m_mergedRequestKey = mergedRequestKey;
	}
	
	/**
	 * gets the identifier of the source request per rfc3261 8.2.2.2
	 * @return identifier of the source request,
	 *  or null if this feature is disabled in configuration
	 */
	public MergedRequestKey getMergedRequestKey() {
		return m_mergedRequestKey;
	}
	
	/**
	 * sets the identifier of the source request per rfc3261 8.2.2.2
	 * @param key identifier of the source request
	 */
	public void setMergedRequestKey(MergedRequestKey key) {
		m_mergedRequestKey = key;
	}
	
	/**
	 * called when a request arrives from the network, to match a server
	 * transaction to this request per rfc 3261 section 17.2.3
	 * @param req the incoming request
	 * @return true if all request-matching rules pass, otherwise false
	 */
	public boolean isRequestPartOfTransaction(Request req) {
		Request initial = getFirstRequest();
		if (initial == null) {
			return false;
		}
		
		ViaHeader oldVia, newVia;
		try {
			oldVia = (ViaHeader)initial.getHeader(ViaHeader.name, true);
			newVia = (ViaHeader)req.getHeader(ViaHeader.name, true);
		}
		catch (HeaderParseException e) {
			return false;
		}
		catch (IllegalArgumentException e) {
			return false;
		}
		String newBranch = newVia.getBranch();
		String oldBranch = oldVia.getBranch();
		if (newBranch != null && newBranch.startsWith(SIPTransactionConstants.BRANCH_MAGIC_COOKIE)) {
			// The branch parameter in the topmost Via header field of the request
			// is examined.  If it is present and begins with the magic cookie
			// "z9hG4bK", the request was generated by a client transaction
			// compliant to this specification
			
			// todo also check sent-by and method
			boolean rc = newBranch.equals(oldBranch);
			
			try {
				if(req.getMethod().equals(Request.ACK)){
					rc = rc && (initial.getMethod().equals(Request.INVITE));
				}
				else{
					rc = rc && (initial.getMethod().equals(req.getMethod()));
				}
				
				rc = rc && (oldVia.getHost().equals(newVia.getHost()))
						&& (oldVia.getPort()== newVia.getPort());
			} catch (SipParseException e) {
				return false;
			}
			
			return rc;
		}
		
		// get here with a request from an old 2543 element
		return is2543RequestPartOfTransaction(req, oldVia, newVia);
	}
	
	
	/**
	 * called when a request with no branch arrives from the network
	 * @param req the incoming request
	 * @param oldVia top via header in initial request
	 * @param newVia top via header in new request
	 * @return true if all request-matching rules pass, otherwise false
	 */
	protected abstract boolean is2543RequestPartOfTransaction(Request req, ViaHeader oldVia, ViaHeader newVia);
	
	//send back the response to the trasport
	public void sendResponseToTransport( Response sipResponse )
		throws SIPTransportException		
	{
		// defect 496693, when original request is recieved from loopback
		// response should be returned using loopback too. 
		RequestImpl request = (RequestImpl)getFirstRequest();
		if (request != null && request.isLoopback()) {
			((ResponseImpl)sipResponse).setLoopback(true);
		}
		SipProvider provider = getProviderContext();
		MessageContext messageContext = MessageContextFactory.instance().getMessageContext(sipResponse,this);
		try{
			getParentStack().getTransportCommLayerMgr().sendMessage(messageContext, provider, getTransportConnection(), this); 
		}catch(Exception e){
			messageContext.writeError(e);
		}
	}
	

	public void sendRequestToUA( Request req )
	{
		forwardRequestToJainLayer(req, getId());
	}
	
	public void sendRequestToUA( Request req, long transactionId )
	{
		forwardRequestToJainLayer(req, transactionId);
	}
	
	/**
	 *  notify the Jain Layer 
	 */
	private void forwardRequestToJainLayer( Request req, long transactionId )
	{
		SipEvent event = new SipEvent( getProviderContext() , transactionId, req );
		getParentStack().getUACommLayer().sendEventToUA( event );
	}
	 
	/**
	 *  notify the JainLayer on the error send by the response
	 */
	public void notifyRespnseErrorToUA(Response sipResponse)
	{
		SipEvent event = new SipEvent(getProviderContext(), getId(), sipResponse);
		getParentStack().getUACommLayer().sendEventToUA( event );			
	}
	
	/**
	 *	Notify time out to the UA layer 
	 */
	public void notifyTimeOutToUA()
	{
		SipEvent event = new SipEvent(getProviderContext(), getId(), true );
		getParentStack().getUACommLayer().sendEventToUA( event );
	}
	
	
	/**
	 * remove the transaction
	 */
	public void remove()
	{
		SIPTransactionsModel.instance().remove( this );	
	}
	
	
	
	
	




}
