// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.

package v1alpha1

import (
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// ComplianceState shows the state of enforcement
type ComplianceState string

const (
	// Compliant is an ComplianceState
	Compliant ComplianceState = "Compliant"

	// NonCompliant is an ComplianceState
	NonCompliant ComplianceState = "NonCompliant"

	// UnknownCompliancy is an ComplianceState
	UnknownCompliancy ComplianceState = "UnknownCompliancy"
)

//Result of a CIS test: passed or failed
type Result string

const (
	// Passed the test
	Passed Result = "[PASS]"

	// Failed the test
	Failed Result = "[FAIL]"
)

// Severity in the policy spec. It is set by the policy provider
type Severity string

const (
	// Low Severity
	Low Severity = "low"

	// Medium Severity
	Medium Severity = "medium"

	// High Severity
	High Severity = "high"
)

// ControllerConfig includes the configuration of the controller itself
type ControllerConfig struct {
	ClusterName   string
	Namespace     string
	EventOnParent string
	CosURL        string
}

// CisPolicySpec defines the desired state of CisPolicy
type CisPolicySpec struct {
	// INSERT ADDITIONAL SPEC FIELDS - desired state of cluster
	// Important: Run "make" to regenerate code after modifying this file
	Severity            Severity            `json:"severity,omitempty"`
	CISSpecVersion      float64             `json:"cisSpecVersion"`
	KubernetesCisPolicy KubernetesCisPolicy `json:"kubernetesCisPolicy,omitempty"`
}

// KubernetesCisPolicy has the list of tests that must be verified
type KubernetesCisPolicy struct {
	MasterNodeExcludeRules []string `json:"masterNodeExcludeRules,omitempty"`
	WorkerNodeExcludeRules []string `json:"workerNodeExcludeRules,omitempty"`
}

// CisPolicyStatus defines the observed state of CisPolicy
type CisPolicyStatus struct {
	// INSERT ADDITIONAL STATUS FIELD - define observed state of cluster
	// Important: Run "make" to regenerate code after modifying this file
	CisPolicyStatus    map[string]*CisPerClusterStatus `json:"cisPolicyStatus,omitempty"`
	ObservedGeneration int64                           `json:"observedGeneration,omitempty"`

	//TODO add aggregate status if needed here for multi-clusters
}

//CisPerClusterStatus indicating whether it is compliant in its own cluster not globally
type CisPerClusterStatus struct {
	ClusterName string            `json:"clusterName,omitempty"`
	NodeStatus  map[string]string `json:"nodeStatus,omitempty"`
	Compliancy  ComplianceState   `json:"compliancy,omitempty"`
	Risk        RiskScore         `json:"riskScore,omitempty"`
}

//RiskScore a calculated score
type RiskScore struct {
	HighestRiskScore    float64 `json:"highestRiskScore,omitempty"`
	RiskCategory        string  `json:"riskCategory,omitempty"`
	CumulativeRiskScore float64 `json:"-"`
}

// +genclient
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// CisPolicy is the Schema for the cispolicies API
// +k8s:openapi-gen=true
// +kubebuilder:subresource:status
type CisPolicy struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	Spec   CisPolicySpec   `json:"spec,omitempty"`
	Status CisPolicyStatus `json:"status,omitempty"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// CisPolicyList contains a list of CisPolicy
type CisPolicyList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata,omitempty"`
	Items           []CisPolicy `json:"items"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// Policy is the Schema for the Policy API
type Policy struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// PolicyList is a list of Policy resources
type PolicyList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata"`

	Items []Policy `json:"items"`
}

func init() {

	SchemeBuilder.Register(&CisPolicy{}, &CisPolicyList{})
}

/*
- problem:
cannot use &Policy literal (type *Policy) as type runtime.Object in argument to SchemeBuilder.Register:
*Policy does not implement runtime.Object (missing DeepCopyObject
- solution:
lister-gen—creates listers for CustomResources which offer a read-only caching layer for GET and LIST requests.
client-gen—creates typed clientsets for CustomResource APIGroups
deepcopy-gen—creates a method func (t* T) DeepCopy() *T for each type T

*/
