#!/bin/bash

set -o pipefail
set -eux

display_usage() {
    cat <<EOT
Build script to start the docker build of fuse-console-operator.
Pass in the tag from the hawtio-operator repository to build.

Usage: build-image.sh [options] -t <hawtio-operator-tag> -v <hawtio-operator-version>

with options:

-d, --dry-run   Run without committing and changes or running in OSBS.
--scratch       Build OSBS image in scratch mode.
--help          This help message
EOT
}

cleanup() {
    rm -f hawtio-operator-*.tar.gz
    rm -rf hawtio-operator 2> /dev/null || true
}

strip_prod_version() {
    local version=$1

    local strippedVersion=${version%%-redhat*}
    strippedVersion=${strippedVersion%%-temporary*}

    echo $strippedVersion
}

checkout_sources() {
    local version=$1
    local tag=$2

    scmURL=`cat source-repos | grep hawtio-operator | cut -d ' ' -f 1`
    if ! git clone $scmURL hawtio-operator
    then
        echo "ERROR: Failed to clone $scmURL"
        return 1
    fi

    pushd hawtio-operator

    git checkout $tag

    if ! git archive --format=tar.gz -o ../hawtio-operator-$tag.tar.gz --prefix=hawtio-operator-$tag/ $tag
    then
        echo "ERROR: Failed to archive $tag"
        popd
        return 1
    fi

    popd

    # sed -i '/hawtio-operator/d' sources
    # if ! rhpkg upload hawtio-operator-$tag.tar.gz
    # then
    #     echo "Error uploading hawtio-operator-$tag.tar.gz to lookaside cache"
    #     exit 1
    # fi

    return 0
}

update_source_repos() {
    local hawtioOperatorTag=$1
    local dryRun=$2

    echo "Updating source-repos"
    sed -i "/hawtio-operator /{s/ [^ ]*/ $hawtioOperatorTag/1}" source-repos

    if [ "$dryRun" == "false" ]
    then
        git add source-repos
    fi
}

update_dockerfile() {
    local hawtioVersion=$1
    local hawtioTag=$2
    local dryRun=$3

    echo "Updating Dockerfile"
    sed -i "/^ENV/,/^$/s/\(.*\)HAWTIO_OPERATOR_VERSION\([ =]\)[0-9a-zA-Z\.-]\+\(.*\)/\1HAWTIO_OPERATOR_VERSION\2$hawtioVersion\3/" Dockerfile
    sed -i "/^ENV/,/^$/s/\(.*\)HAWTIO_OPERATOR_TAG\([ =]\)[0-9a-zA-Z\.-]\+\(.*\)/\1HAWTIO_OPERATOR_TAG\2$hawtioTag\3/" Dockerfile

    if [ "$dryRun" == "false" ]
    then
        git add Dockerfile
    fi
}

osbs_build() {
    local version=$1
    local scratchBuild=$2

    num_files=$(git status --porcelain  | { egrep '^\s?[MADRC]' || true; } | wc -l)
    if ((num_files > 0)) ; then
        echo "Committing $num_files"
        git commit -m"Updated for build of fuse-console-operator $version"
        git push
    else
        echo "There are no files to be committed. Skipping commit + push"
    fi

    if [ "$scratchBuild" == "false" ]
    then
        echo "Starting OSBS build"
        rhpkg container-build
    else
        local branch=$(git rev-parse --abbrev-ref HEAD)
        local build_options=""

        # If we are building on a private branch, then we need to use the correct target
        if [[ $branch == *"private"* ]] ; then
            # Remove the private part of the branch name: from private-opiske-fuse-7.4-openshift-rhel-7
            # to fuse-7.4-openshift-rhel-7 and we add the containers candidate to the options
            local target="${branch#*-*-}-containers-candidate"

            build_options="${build_options} --target ${target}"
            echo "Using target ${target} for the private container build"
        fi

        echo "Starting OSBS scratch build"
        rhpkg container-build --scratch ${build_options}
    fi
}

main() {
    TAG=
    VERSION=
    DRY_RUN=false
    SCRATCH=false

    # Parse command line arguments
    while [ $# -gt 0 ]
    do
        arg="$1"

        case $arg in
          -h|--help)
            display_usage
            return 0
            ;;
          -d|--dry-run)
            DRY_RUN=true
            ;;
          --scratch)
            SCRATCH=true
            ;;
          -t|--tag)
            shift
            TAG="$1"
            ;;
          -v|--version)
            shift
            VERSION="$1"
            ;;
          *)
            echo "Unknonwn argument: $1"
            display_usage
            return 1
            ;;
        esac
        shift
    done

    # Check that syndesis version is specified
    if [ -z "$TAG" ]
    then
        echo "ERROR: Hawtio-operator tag wasn't specified."
        return 1
    fi

    if [ "$TAG" == "master" ]
    then
        TAG=`git ls-remote https://code.engineering.redhat.com/gerrit/hawtio/hawtio-operator.git master | awk ' { print $1 } '`
    fi

    if [ -z "$VERSION" ]
    then
        echo "ERROR: Hawtio-operator version wasn't specified."
        return 1
    fi

    checkout_sources $VERSION $TAG

    update_source_repos $TAG $DRY_RUN

    update_dockerfile $VERSION $TAG $DRY_RUN

    if [ "$DRY_RUN" == "false" ]
    then
        osbs_build $VERSION $SCRATCH
    fi

    cleanup
}

main $*
