package operatorsource

import (
	"context"

	"github.com/operator-framework/operator-marketplace/pkg/apis/operators/shared"
	v1 "github.com/operator-framework/operator-marketplace/pkg/apis/operators/v1"
	"github.com/operator-framework/operator-marketplace/pkg/appregistry"
	"github.com/operator-framework/operator-marketplace/pkg/datastore"
	"github.com/operator-framework/operator-marketplace/pkg/defaults"
	"github.com/operator-framework/operator-marketplace/pkg/operatorhub"
	"github.com/operator-framework/operator-marketplace/pkg/phase"
	"github.com/operator-framework/operator-marketplace/pkg/status"
	log "github.com/sirupsen/logrus"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/manager"
)

// NewHandlerWithParams returns a new Handler.
func NewHandlerWithParams(client client.Client, datastore datastore.Writer, factory PhaseReconcilerFactory, transitioner phase.Transitioner, newCacheReconciler NewOutOfSyncCacheReconcilerFunc, ss status.SyncSender) Handler {
	return &operatorsourcehandler{
		client:             client,
		datastore:          datastore,
		factory:            factory,
		transitioner:       transitioner,
		newCacheReconciler: newCacheReconciler,
		syncSender:         ss,
	}
}

func NewHandler(mgr manager.Manager, client client.Client, ss status.SyncSender) Handler {
	return &operatorsourcehandler{
		client:    client,
		datastore: datastore.Cache,
		factory: &phaseReconcilerFactory{
			registryClientFactory: appregistry.NewClientFactory(),
			datastore:             datastore.Cache,
			client:                client,
			refresher:             cscRefresher,
		},
		transitioner:       phase.NewTransitioner(),
		newCacheReconciler: NewOutOfSyncCacheReconciler,
		syncSender:         ss,
	}
}

// Handler is the interface that wraps the Handle method
//
// Handle handles a new event associated with OperatorSource type.
//
// ctx represents the parent context.
// event encapsulates the event fired by operator sdk.
type Handler interface {
	Handle(ctx context.Context, operatorSource *v1.OperatorSource) (bool, error)
}

// operatorsourcehandler implements the Handler interface
type operatorsourcehandler struct {
	// This client, initialized using mgr.Client() above, is a split client
	// that reads objects from the cache and writes to the apiserver
	client             client.Client
	datastore          datastore.Writer
	factory            PhaseReconcilerFactory
	transitioner       phase.Transitioner
	newCacheReconciler NewOutOfSyncCacheReconcilerFunc
	syncSender         status.SyncSender
}

func (h *operatorsourcehandler) Handle(ctx context.Context, in *v1.OperatorSource) (bool, error) {
	logger := log.WithFields(log.Fields{
		"type":      in.TypeMeta.Kind,
		"namespace": in.GetNamespace(),
		"name":      in.GetName(),
	})

	defaults.New(defaults.GetGlobalDefinitions(), operatorhub.GetSingleton().Get()).RestoreSpecIfDefault(in)

	outOfSyncCacheReconciler := h.newCacheReconciler(logger, h.datastore, h.client)
	out, status, _, err := outOfSyncCacheReconciler.Reconcile(ctx, in)
	if err != nil {
		return false, err
	}

	// The out of sync cache reconciler has handled the event. So we should
	// transition into the new phase and return.
	if status != nil {
		return false, h.transition(ctx, logger, out, status, err)
	}

	// The out of sync cache reconciler has not handled the event, so let's use
	// the regular phase reconciler to handle this event.
	phaseReconciler, err := h.factory.GetPhaseReconciler(logger, in)
	if err != nil {
		return false, err
	}

	out, status, requeue, err := phaseReconciler.Reconcile(ctx, in)
	if out == nil {
		// If the reconciler didn't return an object, that means it must have been deleted.
		// In that case, we should just return without attempting to modify it.
		return false, err
	}

	return requeue, h.transition(ctx, logger, out, status, err)
}

func (h *operatorsourcehandler) transition(ctx context.Context, logger *log.Entry, opsrc *v1.OperatorSource, nextPhase *shared.Phase, reconciliationErr error) error {
	// If reconciliation threw an error, we can't quit just yet. We need to
	// figure out whether the OperatorSource object needs to be updated.

	if !h.transitioner.TransitionInto(&opsrc.Status.CurrentPhase, nextPhase) {
		// OperatorSource object has not changed, no need to update. We are done.
		return reconciliationErr
	}

	// OperatorSource object has been changed. At this point, reconciliation has
	// either completed successfully or failed.
	// In either case, we need to update the modified OperatorSource object.
	if updateErr := h.client.Update(ctx, opsrc); updateErr != nil {
		logger.Errorf("Failed to update object - %v", updateErr)

		// Error updating the object - report a failed sync.
		h.syncSender.SendSyncMessage(updateErr)

		if reconciliationErr == nil {
			// No reconciliation err, but update of object has failed!
			return updateErr
		}

		// Presence of both Reconciliation error and object update error.
		// TODO: find a way to chain the update error?
		return reconciliationErr
	}

	return reconciliationErr
}
