/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package fieldpath

import (
	"bytes"
	"io"

	jsoniter "github.com/json-iterator/go"
)

func (s *Set) ToJSON() ([]byte, error) {
	buf := bytes.Buffer{}
	stream := jsoniter.NewStream(jsoniter.ConfigCompatibleWithStandardLibrary, &buf, 4096)

	stream.WriteObjectStart()
	s.emitContents_v1(false, stream)
	stream.WriteObjectEnd()
	err := stream.Flush()
	return buf.Bytes(), err
}

func (s *Set) emitContents_v1(includeSelf bool, stream *jsoniter.Stream) {
	mi, ci := 0, 0
	first := true
	preWrite := func() {
		if first {
			first = false
			return
		}
		stream.WriteMore()
	}

	for mi < len(s.Members.members) && ci < len(s.Children.members) {
		mpe := s.Members.members[mi]
		cpe := s.Children.members[ci].pathElement

		if mpe.Less(cpe) {
			preWrite()
			str, _ := SerializePathElement(mpe)
			stream.WriteObjectField(str)
			stream.WriteEmptyObject()
			mi++
		} else if cpe.Less(mpe) {
			preWrite()
			str, _ := SerializePathElement(cpe)
			stream.WriteObjectField(str)
			stream.WriteObjectStart()
			s.Children.members[ci].set.emitContents_v1(false, stream)
			stream.WriteObjectEnd()
			ci++
		} else {
			preWrite()
			str, _ := SerializePathElement(cpe)
			stream.WriteObjectField(str)
			stream.WriteObjectStart()
			s.Children.members[ci].set.emitContents_v1(true, stream)
			stream.WriteObjectEnd()
			mi++
			ci++
		}
	}

	for mi < len(s.Members.members) {
		mpe := s.Members.members[mi]

		preWrite()
		str, _ := SerializePathElement(mpe)
		stream.WriteObjectField(str)
		stream.WriteEmptyObject()
		mi++
	}

	for ci < len(s.Children.members) {
		cpe := s.Children.members[ci].pathElement

		preWrite()
		str, _ := SerializePathElement(cpe)
		stream.WriteObjectField(str)
		stream.WriteObjectStart()
		s.Children.members[ci].set.emitContents_v1(false, stream)
		stream.WriteObjectEnd()
		ci++
	}

	if includeSelf && !first {
		preWrite()
		stream.WriteObjectField(".")
		stream.WriteEmptyObject()
	}
}

// FromJSON clears s and reads a JSON formatted set structure.
func (s *Set) FromJSON(r io.Reader) error {
	iter := jsoniter.Parse(jsoniter.ConfigCompatibleWithStandardLibrary, r, 4096)

	found, _ := readIter_v1(iter)
	if found == nil {
		*s = Set{}
	} else {
		*s = *found
	}
	return iter.Error
}

// returns true if this subtree is also (or only) a member of parent; s is nil
// if there are no further children.
func readIter_v1(iter *jsoniter.Iterator) (children *Set, isMember bool) {
	iter.ReadMapCB(func(iter *jsoniter.Iterator, key string) bool {
		if key == "." {
			isMember = true
			iter.Skip()
			return true
		}
		pe, err := DeserializePathElement(key)
		if err == ErrUnknownPathElementType {
			// Ignore these-- a future version maybe knows what
			// they are. We drop these completely rather than try
			// to preserve things we don't understand.
			iter.Skip()
			return true
		} else if err != nil {
			iter.ReportError("parsing key as path element", err.Error())
			iter.Skip()
			return true
		}
		grandchildren, childIsMember := readIter_v1(iter)
		if childIsMember {
			if children == nil {
				children = &Set{}
			}
			// TODO: append & sort afterwards
			children.Members.Insert(pe)
		}
		if grandchildren != nil {
			if children == nil {
				children = &Set{}
			}
			*children.Children.Descend(pe) = *grandchildren
		}
		return true
	})
	if children == nil {
		isMember = true
	}
	return children, isMember
}
