package openshift

import (
	"os"
	"path/filepath"

	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/templates/content"
)

const (
	infraCRDfilename = "cluster-infrastructure-crd.yaml"
)

var _ asset.WritableAsset = (*InfrastructureCRD)(nil)

// InfrastructureCRD is the custom resource definition for the openshift/api
// Infrastructure type.
type InfrastructureCRD struct {
	fileName string
	FileList []*asset.File
}

// Dependencies returns all of the dependencies directly needed by the asset
func (t *InfrastructureCRD) Dependencies() []asset.Asset {
	return []asset.Asset{}
}

// Name returns the human-friendly name of the asset.
func (t *InfrastructureCRD) Name() string {
	return "Infrastructure"
}

// Generate generates the actual files by this asset
func (t *InfrastructureCRD) Generate(parents asset.Parents) error {
	t.fileName = infraCRDfilename
	data, err := content.GetOpenshiftTemplate(t.fileName)
	if err != nil {
		return err
	}
	t.FileList = []*asset.File{
		{
			Filename: filepath.Join(content.TemplateDir, t.fileName),
			Data:     []byte(data),
		},
	}
	return nil
}

// Files returns the files generated by the asset.
func (t *InfrastructureCRD) Files() []*asset.File {
	return t.FileList
}

// Load returns the asset from disk.
func (t *InfrastructureCRD) Load(f asset.FileFetcher) (bool, error) {
	file, err := f.FetchByName(filepath.Join(content.TemplateDir, infraCRDfilename))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}
		return false, err
	}
	t.FileList = []*asset.File{file}
	return true, nil
}
