package v1

import (
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	operatorsv1api "github.com/openshift/api/operator/v1"
)

const (
	// DefaultRouteName is the name of the default route created for the registry
	// when a default route is requested from the operator
	DefaultRouteName = "default-route"

	// ImageRegistryName is the name of the image-registry workload resource (deployment)
	ImageRegistryName = "image-registry"

	// ImageRegistryResourceName is the name of the image registry config instance
	ImageRegistryResourceName = "cluster"

	// ClusterProxyResourceName is the name of the cluster proxy config instance
	ClusterProxyResourceName = "cluster"

	// CloudCredentialsName is the name of the cloud credentials secret
	CloudCredentialsName = "installer-cloud-credentials"

	// ImageRegistryCertificatesName is the name of the configmap that is managed by the
	// registry operator and mounted into the registry pod, to provide additional
	// CAs to be trusted during image pullthrough
	ImageRegistryCertificatesName = ImageRegistryName + "-certificates"

	// ImageRegistryPrivateConfiguration is the name of a secret that is managed by the
	// registry operator and which provides credentials to the registry for things like
	// accessing S3 storage
	ImageRegistryPrivateConfiguration = ImageRegistryName + "-private-configuration"

	// ImageRegistryPrivateConfigurationUser is the name of a secret that is managed by
	// the administrator and which provides credentials to the registry for things like
	// accessing S3 storage.  This content takes precedence over content the operator
	// automatically pulls from other locations, and it is merged into ImageRegistryPrivateConfiguration
	ImageRegistryPrivateConfigurationUser = ImageRegistryPrivateConfiguration + "-user"

	// ImageRegistryOperatorNamespace is the namespace containing the registry operator
	// and the registry itself
	ImageRegistryOperatorNamespace = "openshift-image-registry"

	// ImageRegistryClusterOperatorResourceName is the name of the clusteroperator resource
	// that reflects the registry operator status.
	ImageRegistryClusterOperatorResourceName = "image-registry"

	// Status Conditions

	// OperatorStatusTypeRemoved denotes that the image-registry instance has been
	// removed
	OperatorStatusTypeRemoved = "Removed"

	// StorageExists denotes whether or not the registry storage medium exists
	StorageExists = "StorageExists"

	// StorageTagged denotes whether or not the registry storage medium
	// that we created was tagged correctly
	StorageTagged = "StorageTagged"

	// StorageLabeled denotes whether or not the registry storage medium
	// that we created was labeled correctly
	StorageLabeled = "StorageLabeled"

	// StorageEncrypted denotes whether or not the registry storage medium
	// that we created has encryption enabled
	StorageEncrypted = "StorageEncrypted"

	// StoragePublicAccessBlocked denotes whether or not the registry storage medium
	// that we created has had public access to itself and its objects blocked
	StoragePublicAccessBlocked = "StoragePublicAccessBlocked"

	// StorageIncompleteUploadCleanupEnabled denotes whether or not the registry storage
	// medium is configured to automatically cleanup incomplete uploads
	StorageIncompleteUploadCleanupEnabled = "StorageIncompleteUploadCleanupEnabled"

	// VersionAnnotation reflects the version of the registry that this deployment
	// is running.
	VersionAnnotation = "release.openshift.io/version"
)

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

type ConfigList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata"`
	Items           []Config `json:"items"`
}

// +genclient
// +genclient:nonNamespaced
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// Config is the configuration object for a registry instance managed by
// the registry operator
type Config struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata"`

	Spec ImageRegistrySpec `json:"spec"`
	// +optional
	Status ImageRegistryStatus `json:"status"`
}

type ImageRegistrySpec struct {
	// ManagementState indicates whether the registry instance represented
	// by this config instance is under operator management or not.  Valid
	// values are Managed, Unmanaged, and Removed.
	ManagementState operatorsv1api.ManagementState `json:"managementState"`
	// HTTPSecret is the value needed by the registry to secure uploads, generated by default.
	// +optional
	HTTPSecret string `json:"httpSecret"`
	// Proxy defines the proxy to be used when calling master api, upstream
	// registries, etc.
	// +optional
	Proxy ImageRegistryConfigProxy `json:"proxy"`
	// Storage details for configuring registry storage, e.g. S3 bucket
	// coordinates.
	// +optional
	Storage ImageRegistryConfigStorage `json:"storage"`
	// ReadOnly indicates whether the registry instance should reject attempts
	// to push new images or delete existing ones.
	// +optional
	ReadOnly bool `json:"readOnly"`
	// DisableRedirect controls whether to route all data through the Registry,
	// rather than redirecting to the backend.
	// +optional
	DisableRedirect bool `json:"disableRedirect"`
	// Requests controls how many parallel requests a given registry instance
	// will handle before queuing additional requests.
	// +optional
	Requests ImageRegistryConfigRequests `json:"requests"`
	// DefaultRoute indicates whether an external facing route for the registry
	// should be created using the default generated hostname.
	// +optional
	DefaultRoute bool `json:"defaultRoute"`
	// Routes defines additional external facing routes which should be
	// created for the registry.
	// +optional
	Routes []ImageRegistryConfigRoute `json:"routes,omitempty"`
	// Replicas determines the number of registry instances to run.
	Replicas int32 `json:"replicas"`
	// LogLevel determines the level of logging enabled in the registry.
	LogLevel int64 `json:"logging"`
	// Resources defines the resource requests+limits for the registry pod.
	// +optional
	Resources *corev1.ResourceRequirements `json:"resources,omitempty"`
	// NodeSelector defines the node selection constraints for the registry
	// pod.
	// +optional
	NodeSelector map[string]string `json:"nodeSelector,omitempty"`
	// Tolerations defines the tolerations for the registry pod.
	// +optional
	Tolerations []corev1.Toleration `json:"tolerations,omitempty"`
}

type ImageRegistryStatus struct {
	operatorsv1api.OperatorStatus `json:",inline"`

	// StorageManaged is a boolean which denotes whether or not
	// we created the registry storage medium (such as an
	// S3 bucket).
	StorageManaged bool `json:"storageManaged"`
	// Storage indicates the current applied storage configuration of the
	// registry.
	Storage ImageRegistryConfigStorage `json:"storage"`
}

type ImageRegistryConfigProxy struct {
	// +optional
	HTTP string `json:"http"`
	// +optional
	HTTPS string `json:"https"`
	// +optional
	NoProxy string `json:"noProxy"`
}

// ImageRegistryConfigStorageS3CloudFront holds the configuration
// to use Amazon Cloudfront as the storage middleware in a registry.
// https://docs.docker.com/registry/configuration/#cloudfront
type ImageRegistryConfigStorageS3CloudFront struct {
	// BaseURL contains the SCHEME://HOST[/PATH] at which Cloudfront is served.
	BaseURL string `json:"baseURL"`
	// PrivateKey points to secret containing the private key, provided by AWS.
	PrivateKey corev1.SecretKeySelector `json:"privateKey"`
	// KeypairID is key pair ID provided by AWS.
	KeypairID string `json:"keypairID"`
	// Duration is the duration of the Cloudfront session.
	// +optional
	Duration metav1.Duration `json:"duration"`
}

type ImageRegistryConfigStorageEmptyDir struct {
}

// ImageRegistryConfigStorageS3 holds the information to configure
// the registry to use the AWS S3 service for backend storage
// https://docs.docker.com/registry/storage-drivers/s3/
type ImageRegistryConfigStorageS3 struct {
	// Bucket is the bucket name in which you want to store the registry's
	// data.
	// Optional, will be generated if not provided.
	// +optional
	Bucket string `json:"bucket"`
	// Region is the AWS region in which your bucket exists.
	// Optional, will be set based on the installed AWS Region.
	// +optional
	Region string `json:"region"`
	// RegionEndpoint is the endpoint for S3 compatible storage services.
	// Optional, defaults based on the Region that is provided.
	// +optional
	RegionEndpoint string `json:"regionEndpoint"`
	// Encrypt specifies whether the registry stores the image in encrypted
	// format or not.
	// Optional, defaults to false.
	// +optional
	Encrypt bool `json:"encrypt"`
	// KeyID is the KMS key ID to use for encryption.
	// Optional, Encrypt must be true, or this parameter is ignored.
	// +optional
	KeyID string `json:"keyID"`
	// CloudFront configures Amazon Cloudfront as the storage middleware in a
	// registry.
	// +optional
	CloudFront *ImageRegistryConfigStorageS3CloudFront `json:"cloudFront,omitempty"`
}

type ImageRegistryConfigStorageGCS struct {
	// Bucket is the bucket name in which you want to store the registry's
	// data.
	// Optional, will be generated if not provided.
	// +optional
	Bucket string `json:"bucket,omitempty"`
	// Region is the GCS location in which your bucket exists.
	// Optional, will be set based on the installed GCS Region.
	// +optional
	Region string `json:"region,omitempty"`
	// ProjectID is the Project ID of the GCP project that this bucket should
	// be associated with.
	// +optional
	ProjectID string `json:"projectID,omitempty"`
	// KeyID is the KMS key ID to use for encryption.
	// Optional, buckets are encrypted by default on GCP.
	// This allows for the use of a custom encryption key.
	// +optional
	KeyID string `json:"keyID,omitempty"`
}

// ImageRegistryConfigStorageSwift holds the information to configure
// the registry to use the OpenStack Swift service for backend storage
// https://docs.docker.com/registry/storage-drivers/swift/
type ImageRegistryConfigStorageSwift struct {
	// +optional
	AuthURL string `json:"authURL"`
	// +optional
	AuthVersion string `json:"authVersion"`
	// +optional
	Container string `json:"container"`
	// +optional
	Domain string `json:"domain"`
	// +optional
	DomainID string `json:"domainID"`
	// +optional
	Tenant string `json:"tenant"`
	// +optional
	TenantID string `json:"tenantID"`
	// +optional
	RegionName string `json:"regionName"`
}

type ImageRegistryConfigStoragePVC struct {
	// +optional
	Claim string `json:"claim"`
}

// ImageRegistryConfigStorageAzure holds the information to configure
// the registry to use Azure Blob Storage for backend storage.
type ImageRegistryConfigStorageAzure struct {
	// +optional
	AccountName string `json:"accountName"`
	// +optional
	// +kubebuilder:validation:Pattern=`^[0-9a-z]+(-[0-9a-z]+)*$`
	// +kubebuilder:validation:MinLength=3
	// +kubebuilder:validation:MaxLength=63
	Container string `json:"container"`
}

// ImageRegistryConfigStorage describes how the storage should be configured
// for the image registry.
type ImageRegistryConfigStorage struct {
	// EmptyDir represents ephemeral storage on the pod's host node.
	// This storage cannot be used with more than 1 replica and is not suitable
	// for production use. When the pod is removed from a node for any reason,
	// the data in the emptyDir is deleted forever.
	// This configuration is EXPERIMENTAL and is subject to change without notice.
	// +optional
	EmptyDir *ImageRegistryConfigStorageEmptyDir `json:"emptyDir,omitempty"`
	// S3 represents configuration that uses Amazon Simple Storage Service.
	// +optional
	S3 *ImageRegistryConfigStorageS3 `json:"s3,omitempty"`
	// GCS represents configuration that uses Google Cloud Storage.
	// +optional
	GCS *ImageRegistryConfigStorageGCS `json:"gcs,omitempty"`
	// Swift represents configuration that uses OpenStack Object Storage.
	// This configuration is EXPERIMENTAL and is subject to change without notice.
	// +optional
	Swift *ImageRegistryConfigStorageSwift `json:"swift,omitempty"`
	// PVC represents configuration that uses a PersistentVolumeClaim.
	// +optional
	PVC *ImageRegistryConfigStoragePVC `json:"pvc,omitempty"`
	// Azure represents configuration that uses Azure Blob Storage.
	// +optional
	Azure *ImageRegistryConfigStorageAzure `json:"azure,omitempty"`
}

type ImageRegistryConfigRequests struct {
	// +optional
	Read ImageRegistryConfigRequestsLimits `json:"read"`
	// +optional
	Write ImageRegistryConfigRequestsLimits `json:"write"`
}

type ImageRegistryConfigRequestsLimits struct {
	// MaxRunning sets the maximum in flight api requests to the registry.
	// +optional
	MaxRunning int `json:"maxRunning"`
	// MaxInQueue sets the maximum queued api requests to the registry.
	// +optional
	MaxInQueue int `json:"maxInQueue"`
	// MaxWaitInQueue sets the maximum time a request can wait in the queue
	// before being rejected.
	// +optional
	MaxWaitInQueue metav1.Duration `json:"maxWaitInQueue"`
}

type ImageRegistryConfigRoute struct {
	// Name of the route to be created.
	Name string `json:"name"`
	// Hostname for the route.
	// +optional
	Hostname string `json:"hostname,omitempty"`
	// SecretName points to secret containing the certificates to be used
	// by the route.
	// +optional
	SecretName string `json:"secretName,omitempty"`
}
