/****************************************************************************************
* Copyright (C) 2019 Intel Corporation
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* 1. Redistributions of source code must retain the above copyright notice,
*   this list of conditions and the following disclaimer.
* 2. Redistributions in binary form must reproduce the above copyright notice,
*   this list of conditions and the following disclaimer in the documentation
*   and/or other materials provided with the distribution.
* 3. Neither the name of the copyright holder nor the names of its contributors
*   may be used to endorse or promote products derived from this software
*   without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
* THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
* BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
* OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* SPDX-License-Identifier: BSD-3-Clause
*
****************************************************************************************/
#include "ddp.h"
#include "cmdparams.h"

bool Global_print_debug = 0;

typedef struct _supported_devices_t{
    uint16_t vendorid;
    uint16_t deviceid;
    uint16_t subvendorid;
    uint16_t subdeviceid;
    char*    branding_string;
} supported_devices_t;

supported_devices_t Global_supported_devices[] =
{
    {0x8086, 0x0CF8, 0x8086, 0x0000, "Intel(R) Ethernet Controller X710 Intel(R) FPGA Programmable Acceleration Card N3000 for Networking"},
    {0x8086, 0x0CF8, 0x8086, 0x0001, "Intel(R) Ethernet Controller X710 Intel(R) FPGA Programmable Acceleration Card N3000 for Networking"},
    {0x8086, 0x0D58, 0x8086, 0x0000, "Intel(R) Ethernet Controller XXV710 Intel(R) FPGA Programmable Acceleration Card N3000 for Networking"},
    {0x8086, 0x0D58, 0x8086, 0x0001, "Intel(R) Ethernet Controller XXV710 Intel(R) FPGA Programmable Acceleration Card N3000 for Networking"},
    {0x8086, 0x104E, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10 Gigabit SFP+"},
    {0x8086, 0x104F, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10 Gigabit backplane"},
    {0x8086, 0x154B, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 Generic ID"},
    {0x8086, 0x154C, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Virtual Function 700 Series"},
    {0x8086, 0x1572, 0x1028, 0x0000, "Intel(R) Ethernet 10G X710 rNDC"},
    {0x8086, 0x1572, 0x1028, 0x1F99, "Intel(R) Ethernet 10G 4P X710/I350 rNDC"},
    {0x8086, 0x1572, 0x1028, 0x1F9C, "Intel(R) Ethernet 10G 4P X710 SFP+ rNDC"},
    {0x8086, 0x1572, 0x103C, 0x0000, "HPE Ethernet 10Gb 562SFP+ Adapter"},
    {0x8086, 0x1572, 0x103C, 0x22FC, "HPE Ethernet 10Gb 2-port 562FLR-SFP+ Adapter"},
    {0x8086, 0x1572, 0x103C, 0x22FD, "HPE Ethernet 10Gb 2-port 562SFP+ Adapter"},
    {0x8086, 0x1572, 0x1137, 0x0000, "Cisco(R) Ethernet Converged NIC X710-DA"},
    {0x8086, 0x1572, 0x1137, 0x013B, "Cisco(R) Ethernet Converged NIC X710-DA4"},
    {0x8086, 0x1572, 0x1137, 0x020A, "Cisco(R) Ethernet Converged NIC X710-DA2"},
    {0x8086, 0x1572, 0x1590, 0x0000, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x1590, 0x0225, "HPE Eth 10Gb 4p 563SFP+ Adptr"},
    {0x8086, 0x1572, 0x1590, 0x022F, "HPE Ethernet 10Gb 2-port 564i Communication Board"},
    {0x8086, 0x1572, 0x17AA, 0x0000, "Lenovo ThinkServer X710 AnyFabric for 10Gbe SFP+"},
    {0x8086, 0x1572, 0x17AA, 0x4001, "Lenovo ThinkServer X710-4 AnyFabric for 10Gbe SFP+"},
    {0x8086, 0x1572, 0x17AA, 0x4002, "Lenovo ThinkServer X710-2 AnyFabric for 10Gbe SFP+"},
    {0x8086, 0x1572, 0x8086, 0x0000, "Intel(R) Ethernet Converged Network Adapter X710"},
    {0x8086, 0x1572, 0x8086, 0x0001, "Intel(R) Ethernet Converged Network Adapter X710-4"},
    {0x8086, 0x1572, 0x8086, 0x0002, "Intel(R) Ethernet Converged Network Adapter X710-4"},
    {0x8086, 0x1572, 0x8086, 0x0004, "Intel(R) Ethernet Converged Network Adapter X710-4"},
    {0x8086, 0x1572, 0x8086, 0x0005, "Intel(R) Ethernet Converged Network Adapter X710"},
    {0x8086, 0x1572, 0x8086, 0x0006, "Intel(R) Ethernet Converged Network Adapter X710"},
    {0x8086, 0x1572, 0x8086, 0x0007, "Intel(R) Ethernet Converged Network Adapter X710-2"},
    {0x8086, 0x1572, 0x8086, 0x0008, "Intel(R) Ethernet Converged Network Adapter X710-2"},
    {0x8086, 0x1572, 0x8086, 0x0009, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x8086, 0x000A, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x8086, 0x000B, "Intel(R) Ethernet Server Adapter X710-DA2 for OCP"},
    {0x8086, 0x1572, 0x8086, 0x000D, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x8086, 0x000E, "Intel(R) Ethernet Server Adapter OCP X710-2"},
    {0x8086, 0x1572, 0x8086, 0x000F, "Intel(R) Ethernet Server Adapter OCP X710-2"},
    {0x8086, 0x1572, 0x8086, 0x0013, "Intel(R) Ethernet 10G 2P X710 OCP"},
    {0x8086, 0x1572, 0x8086, 0x0014, "Intel(R) Ethernet 10G 4P X710 OCP"},
    {0x8086, 0x1572, 0x8086, 0x0015, "Intel(R) Ethernet Server Adapter X710-DA2 for OCP"},
    {0x8086, 0x1572, 0x8086, 0x4005, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x8086, 0x4006, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0x8086, 0x4007, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1572, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1573, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GbE SFP+"},
    {0x8086, 0x1574, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 QEMU"},
    {0x8086, 0x1580, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 for 40GbE backplane"},
    {0x8086, 0x1581, 0x1028, 0x0000, "Intel(R) Ethernet 10G X710-k bNDC"},
    {0x8086, 0x1581, 0x1028, 0x1F98, "Intel(R) Ethernet 10G 4P X710-k bNDC"},
    {0x8086, 0x1581, 0x1028, 0x1F9E, "Intel(R) Ethernet 10G 2P X710-k bNDC"},
    {0x8086, 0x1581, 0x1590, 0x00F8, "HPE Ethernet 10Gb 2-port 563i Adapter"},
    {0x8086, 0x1581, 0x1590, 0x0000, "HPE Ethernet 10Gb 2-port 563i Adapter"},
    {0x8086, 0x1581, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GbE backplane"},
    {0x8086, 0x1582, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GbE backplane"},
    {0x8086, 0x1583, 0x1028, 0x0000, "Intel(R) Ethernet 40G 2P XL710 QSFP+ rNDC"},
    {0x8086, 0x1583, 0x1028, 0x1F9F, "Intel(R) Ethernet 40G 2P XL710 QSFP+ rNDC"},
    {0x8086, 0x1583, 0x108E, 0x0000, "Oracle 10 Gb/40 Gb Ethernet Adapter"},
    {0x8086, 0x1583, 0x108E, 0x7B1B, "Oracle Quad 10Gb Ethernet Adapter"},
    {0x8086, 0x1583, 0x108E, 0x7B1D, "Oracle 10 Gb/40 Gb Ethernet Adapter"},
    {0x8086, 0x1583, 0x1137, 0x0000, "Cisco(R) Ethernet Converged NIC XL710-QDA2"},
    {0x8086, 0x1583, 0x1137, 0x013C, "Cisco(R) Ethernet Converged NIC XL710-QDA2"},
    {0x8086, 0x1583, 0x8086, 0x0000, "Intel(R) Ethernet Converged Network Adapter XL710-Q2"},
    {0x8086, 0x1583, 0x8086, 0x0001, "Intel(R) Ethernet Converged Network Adapter XL710-Q2"},
    {0x8086, 0x1583, 0x8086, 0x0002, "Intel(R) Ethernet Converged Network Adapter XL710-Q2"},
    {0x8086, 0x1583, 0x8086, 0x0003, "Intel(R) Ethernet I/O Module XL710-Q2"},
    {0x8086, 0x1583, 0x8086, 0x0004, "Intel(R) Ethernet Server Adapter XL710-Q2OCP"},
    {0x8086, 0x1583, 0x8086, 0x0006, "Intel(R) Ethernet Converged Network Adapter XL710-Q2"},
    {0x8086, 0x1583, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 for 40GbE QSFP+"},
    {0x8086, 0x1584, 0x8086, 0x0000, "Intel(R) Ethernet Converged Network Adapter XL710-Q1"},
    {0x8086, 0x1584, 0x8086, 0x0001, "Intel(R) Ethernet Converged Network Adapter XL710-Q1"},
    {0x8086, 0x1584, 0x8086, 0x0002, "Intel(R) Ethernet Converged Network Adapter XL710-Q1"},
    {0x8086, 0x1584, 0x8086, 0x0003, "Intel(R) Ethernet I/O Module XL710-Q1"},
    {0x8086, 0x1584, 0x8086, 0x0004, "Intel(R) Ethernet Server Adapter XL710-Q1OCP"},
    {0x8086, 0x1584, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 for 40GbE QSFP+"},
    {0x8086, 0x1585, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 for 10GbE QSFP+"},
    {0x8086, 0x1586, 0x108E, 0x0000, "Intel(R) Ethernet Controller X710 for 10GBASE-T"},
    {0x8086, 0x1586, 0x108E, 0x4857, "Intel(R) Ethernet Controller X710 for 10GBASE-T"},
    {0x8086, 0x1586, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GBASE-T"},
    {0x8086, 0x1587, 0x103C, 0x0000, "HPE Eth 10/20Gb 2p 660FLB Adptr"},
    {0x8086, 0x1587, 0x103C, 0x22FE, "HPE Eth 10/20Gb 2p 660FLB Adptr"},
    {0x8086, 0x1588, 0x103C, 0x0000, "HPE Eth 10/20Gb 2p 660M Adptr"},
    {0x8086, 0x1588, 0x103C, 0x22FE, "HPE Flex-20 20Gb 2-port 660M Adapter"},
    {0x8086, 0x1588, 0x103C, 0x22FF, "HPE Eth 10/20Gb 2p 660M Adptr"},
    {0x8086, 0x1589, 0x108E, 0x0000, "Oracle Quad Port 10GBase-T Adapter"},
    {0x8086, 0x1589, 0x108E, 0x7B1C, "Oracle Quad Port 10GBase-T Adapter"},
    {0x8086, 0x1589, 0x1137, 0x0000, "Cisco(R) Ethernet Converged NIC X710-T4"},
    {0x8086, 0x1589, 0x1137, 0x020B, "Cisco(R) Ethernet Converged NIC X710-T4"},
    {0x8086, 0x1589, 0x8086, 0x0000, "Intel(R) Ethernet Converged Network Adapter X710-T"},
    {0x8086, 0x1589, 0x8086, 0x0001, "Intel(R) Ethernet Converged Network Adapter X710-T4"},
    {0x8086, 0x1589, 0x8086, 0x0002, "Intel(R) Ethernet Converged Network Adapter X710-T4"},
    {0x8086, 0x1589, 0x8086, 0x0003, "Intel(R) Ethernet Converged Network Adapter X710-T"},
    {0x8086, 0x1589, 0x8086, 0x00A0, "Intel(R) Ethernet Converged Network Adapter X710-T4"},
    {0x8086, 0x1589, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710/X557-AT 10GBASE-T"},
    {0x8086, 0x158A, 0x1590, 0x0000, "HPE 10/25Gb Ethernet Adapter"},
    {0x8086, 0x158A, 0x1590, 0x0286, "HPE Synergy 4610C 10/25Gb Ethernet Adapter"},
    {0x8086, 0x158A, 0x8086, 0x0000, "Intel(R) Ethernet Controller XXV710 for 25GbE backplane"},
    {0x8086, 0x158A, 0x8086, 0x000A, "Intel(R) Ethernet 25G 2P XXV710 Mezz"},
    {0x8086, 0x158A, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XXV710 for 25GbE backplane"},
    {0x8086, 0x158B, 0x1137, 0x0000, "Cisco(R) Ethernet Network Adapter XXV710"},
    {0x8086, 0x158B, 0x1137, 0x0225, "Cisco(R) Ethernet Network Adapter XXV710"},
    {0x8086, 0x158B, 0x1590, 0x0000, "Intel(R) Ethernet Network Adapter XXV710-2"},
    {0x8086, 0x158B, 0x1590, 0x0253, "HPE Ethernet 10/25Gb 2-port 661SFP28 Adapter"},
    {0x8086, 0x158B, 0x8086, 0x0000, "Intel(R) Ethernet Network Adapter XXV710"},
    {0x8086, 0x158B, 0x8086, 0x0001, "Intel(R) Ethernet Network Adapter XXV710-2"},
    {0x8086, 0x158B, 0x8086, 0x0002, "Intel(R) Ethernet Network Adapter XXV710-2"},
    {0x8086, 0x158B, 0x8086, 0x0003, "Intel(R) Ethernet Network Adapter XXV710-1"},
    {0x8086, 0x158B, 0x8086, 0x0004, "Intel(R) Ethernet Network Adapter XXV710-1"},
    {0x8086, 0x158B, 0x8086, 0x0005, "Intel(R) Ethernet Network Adapter OCP XXV710-2"},
    {0x8086, 0x158B, 0x8086, 0x0006, "Intel(R) Ethernet Network Adapter OCP XXV710-2"},
    {0x8086, 0x158B, 0x8086, 0x0007, "Intel(R) Ethernet Network Adapter OCP XXV710-1"},
    {0x8086, 0x158B, 0x8086, 0x0008, "Intel(R) Ethernet Network Adapter OCP XXV710-1"},
    {0x8086, 0x158B, 0x8086, 0x0009, "Intel(R) Ethernet 25G 2P XXV710 Adapter"},
    {0x8086, 0x158B, 0x8086, 0x000A, "Intel(R) Ethernet 25G 2P XXV710 OCP"},
    {0x8086, 0x158B, 0x8086, 0x4001, "Intel(R) Ethernet Network Adapter XXV710-2"},
    {0x8086, 0x158B, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XXV710 for 25GbE SFP28"},
    {0x8086, 0x15FF, 0x8086, 0x0000, "Intel(R) Ethernet Network Adapter X710-TL"},
    {0x8086, 0x15FF, 0x8086, 0x0001, "Intel(R) Ethernet Network Adapter X710-T4L"},
    {0x8086, 0x15FF, 0x8086, 0x0002, "Intel(R) Ethernet Network Adapter X710-T4L"},
    {0x8086, 0x15FF, 0x8086, 0x0003, "Intel(R) Ethernet Network Adapter X710-T2L"},
    {0x8086, 0x15FF, 0x8086, 0x0004, "Intel(R) Ethernet Network Adapter X710-T2L"},
    {0x8086, 0x15FF, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller X710 for 10GBASE-T"},
    {0x8086, 0xFAFA, 0xFFFF, 0xFFFF, "Intel(R) Ethernet Controller XL710 Barleyville"},
    {0x8086, 0xFBFB, 0xFFFF, 0xFFFF, "Intel(R) XL710 X710 Virtual Function Barleyville"},
};
uint16_t Global_supported_devices_size = sizeof(Global_supported_devices)/sizeof(supported_devices_t);

ddp_status_t
validate_output_status(ddp_status_t status)
{
    /* For status equal or higher than 100, set generic return status */
    if(status >= DDP_AQ_COMMAND_FAIL)
    {
        /* TODO: Add debug log to print status value before overwriting */
        status = DDP_INTERNAL_GENERIC_ERROR;
    }

    return status;
}

char*
get_error_message(ddp_status_value_t status)
{
    char* message = "";

    switch(status)
    {
    case DDP_SUCCESS:
        message = "Success";
        break;
    case DDP_BAD_COMMAND_LINE_PARAMETER:
        message = "Bad command line parameter";
        break;
    case DDP_INTERNAL_GENERIC_ERROR:
        message = "An internal error has occurred";
        break;
    case DDP_INSUFFICIENT_PRIVILEGES:
        message = "Insufficient privileges to run the tool";
        break;
    case DDP_NO_SUPPORTED_ADAPTER:
        message = "No supported adapter found";
        break;
    case DDP_NO_BASE_DRIVER:
        message = "No driver available";
        break;
    case DDP_UNSUPPORTED_BASE_DRIVER:
        message = "Unsupported base driver version";
        break;
    case DDP_CANNOT_COMMUNICATE_ADAPTER:
        message = "Cannot communicate with one or more adapters";
        break;
    case DDP_NO_DDP_PROFILE:
        message = "Lack of DDP profiles on all devices";
        break;
    case DDP_CANNOT_READ_DEVICE_DATA:
        message = "Cannot read all information from one or more devices";
        break;
    case DDP_CANNOT_CREATE_OUTPUT_FILE:
        message = "Cannot create output file";
        break;
    case DDP_DEVICE_NOT_FOUND:
        message = "Cannot find specific devices" ;
        break;
    default:
        message = "";
        break;
    }
    return message;
}

ddp_status_t
get_driver_version_from_os(uint16_t* major, uint16_t* minor, uint16_t* build)
{
    char         line[32]          = {'\0'};
    char*        version_file_path = "/sys/module/i40e/version";
    char*        result            = NULL;
    FILE*        file              = NULL;
    int          sscanf_status     = 0;
    ddp_status_t ddp_status        = DDP_SUCCESS;

    do
    {
        file = fopen(version_file_path, "r");
        if(file == NULL)
        {
            ddp_status = DDP_NO_BASE_DRIVER;
            break;
        }

        result = fgets(line, MAX_DRIVER_VERSION_STRING_LENTGH, file);
        if(result == NULL)
        {
            ddp_status = DDP_INCORRECT_FUNCTION_PARAMETERS;
            break;
        }

        sscanf_status = sscanf(line,
                               "%hu%*[.]%hu%*[.]%hu",
                               major,
                               minor,
                               build);
        if(sscanf_status != NUMBER_OF_DRIVER_VERSION_PARTS)
        {
            ddp_status = DDP_UNSUPPORTED_BASE_DRIVER;
        }
    } while(0);

    if(file != NULL)
    {
        fclose(file);
    }

    return ddp_status;
}

void
free_memory(void* pointer)
{
    if(pointer != NULL)
    {
        free(pointer);
    }
}

bool
is_virtual_function(adapter_t* adapter)
{
    if(adapter->device_id == LINUX_40G_VIRTUAL_DEVID ||
       adapter->device_id == BARLEYVILLE_40G_VIRTUAL_DEVID)
    {
        return TRUE;
    }

    return FALSE;
}

/* Function is_supported_device() verifies if a given device is supported
 * by ddptool. This is done by matching device's 4-partID (VendorID, DeviceID,
 * SubvendorID and SubdeviceID) and if that fails - 2-partID (VendorID and
 * DeviceID). If a match is found this function sets adapter->branding_string
 * value for the current adapter. The list of legal devices is stored in
 * Global_supported_devices.This matching assumes that Global_supported_devices
 * uses 0xFFFF values in 4-partID entries as generic values.
 *
 * Parameters:
 * [in,out] adapter      Handle to current adapter
 *
 * Returns: TRUE if device is supported and FALSE if it is not.
 */
bool
is_supported_device(adapter_t* adapter)
{
    uint16_t i            = 0;
    uint16_t a_ven_id     = adapter->vendor_id;
    uint16_t a_dev_id     = adapter->device_id;
    uint16_t a_sub_ven_id = adapter->subvendor_id;
    uint16_t a_sub_dev_id = adapter->subdevice_id;
    uint16_t generic_id   = 0xFFFF;
    bool     is_supported = FALSE;

    do
    {
        for(i = 0; i < Global_supported_devices_size; i++)
        {
            if(a_ven_id     == Global_supported_devices[i].vendorid    &&
               a_dev_id     == Global_supported_devices[i].deviceid    &&
               a_sub_ven_id == Global_supported_devices[i].subvendorid &&
               a_sub_dev_id == Global_supported_devices[i].subdeviceid
              )
            {
                adapter->branding_string = Global_supported_devices[i].branding_string;
                is_supported = TRUE;
                break;
            }
        }
        if((a_sub_dev_id & a_sub_ven_id) == generic_id ||
           is_supported == TRUE)
        {
            /* 2-partID matching failed or we've found a match*/
            break;
        }
        else
        {
            /* 4-partID matching failed - try 2-partID */
            a_sub_ven_id = generic_id;
            a_sub_dev_id = generic_id;
        }
    } while(TRUE);

    return is_supported;
}

ddp_status_t
get_connection_name(adapter_t* adapter)
{
    char           path_to_net_names[300] = {'\0'};
    struct dirent* entry                  = NULL;
    DIR*           dir                    = NULL;
    ddp_status_t   status                 = DDP_UNKNOWN_ETH_NAME;

    /* Set default value */
    strcpy_sec(adapter->connection_name,
               sizeof adapter->connection_name,
               DDP_CONNECTION_NAME_NOT_AVAILABLE);

    snprintf(path_to_net_names,
             sizeof(path_to_net_names),
             "%s/%04x:%02x:%02x.%x/net",
             PATH_TO_SYSFS_PCI,
             adapter->location.segment,
             adapter->location.bus,
             adapter->location.device,
             adapter->location.function);

    dir = opendir(path_to_net_names);
    if(dir == NULL)
        return status;

    while((entry = readdir(dir)) != NULL)
    {
        if(entry->d_name[0] == '.')
            continue;

        strcpy_sec(adapter->connection_name,
                   sizeof adapter->connection_name,
                   entry->d_name);
        status = DDP_SUCCESS;
        break;
    }

    return status;
}

ddp_status_t
send_adminq_command(adapter_t* adapter, adminq_desc_t* descriptor, uint16_t descriptor_size, uint8_t* cmd_buffer)
{
    ioctl_structure_t* ioctl_data  = NULL;
    ddp_status_t       status      = DDP_SUCCESS;
    uint16_t           buffer_size = 0;

    do
    {
        if(descriptor == NULL)
        {
            status = DDP_INCORRECT_FUNCTION_PARAMETERS;
            break;
        }

        /* preparing ioctl buffer - calculate size for data (ioctl + descriptor + adminq) */
        buffer_size = sizeof(ioctl_structure_t) + descriptor_size - 1;
        ioctl_data  = malloc_sec(buffer_size);
        if(ioctl_data == NULL || ioctl_data->data == NULL)
        {
            status = DDP_ALLOCATE_MEMORY_FAIL;
            break;
        }

        /* copy desciptor at end of ioctl_data strcuture */
        status = memcpy_sec(ioctl_data->data, descriptor_size, descriptor, descriptor_size);
        if(status != DDP_SUCCESS)
        {
            break;
        }

        /* set proper value for ioctl */
        if(adapter->is_virtual_function == TRUE && adapter->is_usable == TRUE)
        {
            ioctl_data->config    = adapter->pf_device_id << 16 | IOCTL_EXECUTE_COMMAND;
        }
        else
        {
            ioctl_data->config = adapter->device_id << 16 | IOCTL_EXECUTE_COMMAND;
        }
        ioctl_data->data_size = descriptor_size;

        status = get_data_by_basedriver(adapter, ioctl_data);
        if(status == DDP_SUCCESS)
        {
            status = memcpy_sec(descriptor, descriptor_size, ioctl_data->data, descriptor_size);
            if(status != DDP_SUCCESS)
            {
                break;
            }
        }
    } while(0);

    free(ioctl_data);
    return status;
}

ddp_status_t
get_ddp_profile_list(adapter_t* adapter)
{
    adminq_desc_t*     descriptor       = NULL;
    uint8_t*           data_buffer      = NULL;
    uint16_t           data_buffer_size = DDP_ADMINQ_WRITEBACK_SIZE;
    uint16_t           descriptor_size  = sizeof(adminq_desc_t) + DDP_ADMINQ_WRITEBACK_SIZE;
    ddp_status_t       status           = DDP_SUCCESS;

    do
    {
        descriptor  = malloc_sec(descriptor_size);
        if(descriptor == NULL)
        {
            status = DDP_ALLOCATE_MEMORY_FAIL;
            debug_ddp_print("Cannot allocate buffer\n");
            break;
        }

        memset(descriptor, 0, descriptor_size);

        descriptor->opcode   = ADMINQ_COMMAND_GET_DDP_PROFILE_LIST;
        descriptor->flags    = ADMINQ_FLAG_BUF;
        descriptor->datalen  = data_buffer_size;

        status = send_adminq_command(adapter, descriptor, descriptor_size, data_buffer);
        if(status != DDP_SUCCESS)
        {
            status = DDP_AQ_COMMAND_FAIL;
            debug_ddp_print("send_adminq_command status 0x%X\n", status);
            break;
        }
        else
        {
            status = memcpy_sec(&adapter->profile_info,
                                sizeof(profile_info_t),
                                &descriptor->params.raw[OFFSET_TO_ADMINQ_WRITEBACK],
                                sizeof(profile_info_t));
            if(adapter->profile_info.section_size == 0)
            {
                status = DDP_NO_DDP_PROFILE;
            }
        }
    } while(0);

    free_memory(descriptor);

    return status;
}

ddp_status_t
get_firmware_version(adapter_t* adapter)
{
    adminq_desc_t       descriptor;
    firmware_version_t* firmware_version;
    ddp_status_t        status            = DDP_SUCCESS;
    uint16_t            descriptor_size   = 0;

    MEMINIT(&descriptor);
    MEMINIT(&firmware_version);

    do
    {
        descriptor.opcode  = ADMINQ_COMMAND_GET_FW_VERSION;
        descriptor.flags   = ADMINQ_FLAG_SI;
        descriptor.datalen = 0;      /* in this case we don't need additional buffer */

        descriptor_size = sizeof(adminq_desc_t);

        /* second -  alloc memory for ioctl data and set the proper value data */
        status = send_adminq_command(adapter, &descriptor, descriptor_size, NULL);
        if(status != DDP_SUCCESS)
        {
            status = DDP_AQ_COMMAND_FAIL;
            debug_ddp_print("send_adminq_command status: 0x%x\n", status);
            break;
        }

        /* fill the FW version structure */
        status = memcpy_sec(&adapter->firmware_version,
                                  sizeof(firmware_version_t),
                                  &descriptor.params.raw,
                                  sizeof descriptor.params.raw);
        if(status != DDP_SUCCESS)
        {
            break;
        }
    } while(0);

    return status;
}

ddp_status_t
initialize_tool()
{
    ddp_status_t  ddp_status = DDP_SUCCESS;
    uint16_t      major      = 0;
    uint16_t      minor      = 0;
    uint16_t      build      = 0;

    do
    {
        if(is_root_permission() == FALSE)
        {
            ddp_status = DDP_INSUFFICIENT_PRIVILEGES;
            break;
        }

        if(check_command_parameter(DDP_XML_COMMAND_PARAMETER_BIT))
        {
            ddp_func_print_adapter_list = generate_xml;
        }
        else if (check_command_parameter(DDP_JSON_COMMAND_PARAMETER_BIT))
        {
            ddp_func_print_adapter_list = generate_json;
        }
        else
        {
            ddp_func_print_adapter_list = generate_table;
        }

        ddp_status = get_driver_version_from_os(&major, &minor, &build);
        if(ddp_status != DDP_SUCCESS)
        {
            break;
        }

        if(major < DDP_MIN_BASE_DRIVER_VERSION_MAJOR)
        {
            ddp_status = DDP_UNSUPPORTED_BASE_DRIVER;
            break;
        }

        if(major == DDP_MIN_BASE_DRIVER_VERSION_MAJOR &&
           minor < DDP_MIN_BASE_DRIVER_VERSION_MINOR)
        {
            ddp_status = DDP_UNSUPPORTED_BASE_DRIVER;
            break;
        }

        if(major == DDP_MIN_BASE_DRIVER_VERSION_MAJOR &&
           minor == DDP_MIN_BASE_DRIVER_VERSION_MINOR &&
           build < DDP_MIN_BASE_DRIVER_VERSION_BUILD)
        {
            ddp_status = DDP_UNSUPPORTED_BASE_DRIVER;
            break;
        }
    } while(0);

    if(ddp_status == DDP_NO_BASE_DRIVER)
    {
        debug_ddp_print("Cannot find base driver!\n");
    }
    else if(ddp_status == DDP_UNSUPPORTED_BASE_DRIVER)
    {
        debug_ddp_print("Unsupported driver version: %d.%d.%d\n", major, minor, build);
    }

    return ddp_status;
}

adapter_t*
get_adapter_from_list_node(node_t* node)
{
    return (adapter_t*)node->data;
}

ddp_status_t
check_fw_version(adapter_t* adapter, bool* is_fw_supported)
{
    ddp_status_t status = DDP_SUCCESS;

    do
    {
        status = get_firmware_version(adapter);
        if(status != DDP_SUCCESS)
        {
            break;
        }

        /* Compare firmware version with minimal supported firmware version */
        if(adapter->firmware_version.fw_major > MIN_FW_VERSION_MAJOR)
        {
            *is_fw_supported = TRUE;
        }
        if(adapter->firmware_version.fw_major == MIN_FW_VERSION_MAJOR)
        {
            if(adapter->firmware_version.fw_minor >= MIN_FW_VERSION_MINOR)
            {
                *is_fw_supported = TRUE;
            }
        }
    } while(0);

    return status;
}

ddp_status_t
discovery_device(adapter_t* adapter)
{
    ddp_status_t status          = DDP_SUCCESS;
    bool         is_fw_supported = FALSE;

    do
    {
        if(adapter->is_usable == FALSE)
            break; /* tool cannot read below data and need to copy them from other function */

        status = check_fw_version(adapter, &is_fw_supported);
        if(status != DDP_SUCCESS)
        {
            break;
        }
        if(is_fw_supported == FALSE)
        {
            status = DDP_NO_SUPPORTED_ADAPTER;
            break;
        }

        status = get_ddp_profile_list(adapter);
        if(status != DDP_SUCCESS)
        {
            break;
        }
    } while(0);

    return status;
}

ddp_status_t
discovery_devices(list_t adapter_list)
{
    node_t*         adapter_node      = get_node(&adapter_list);
    ddp_status_t    status            = DDP_INCORRECT_FUNCTION_PARAMETERS;
    ddp_status_t    function_status   = DDP_INCORRECT_FUNCTION_PARAMETERS;
    adapter_t*      adapter           = NULL;
    adapter_t*      previous_adapter  = NULL;
    bool            is_profile_loaded = FALSE;

    while(adapter_node != NULL)
    {
        adapter = get_adapter_from_list_node(adapter_node);
        if(adapter == NULL)
        {
            status = DDP_CANNOT_READ_DEVICE_DATA;
            continue;
        }

        if(previous_adapter != NULL &&
           previous_adapter->is_usable == TRUE &&
           COMPARE_PCI_LOCATION(adapter, previous_adapter) == TRUE)
        {
            memcpy_sec(&adapter->firmware_version,
                       sizeof(firmware_version_t),
                       &previous_adapter->firmware_version,
                       sizeof(firmware_version_t));

            memcpy_sec(&adapter->profile_info,
                       sizeof(profile_info_t),
                       &previous_adapter->profile_info,
                       sizeof(profile_info_t));
        }
        else
        {
            function_status = discovery_device(adapter);
            if(function_status == DDP_SUCCESS && is_profile_loaded == FALSE)
            {
                is_profile_loaded = TRUE;
            }
            else if(function_status == DDP_NO_DDP_PROFILE && is_profile_loaded == TRUE)
            {
                function_status = DDP_SUCCESS;
            }
            else if(function_status != DDP_SUCCESS)
            {
                status = function_status;
            }
        }

        adapter_node = get_next_node(adapter_node);
        previous_adapter = adapter;
    }

    if(function_status == DDP_SUCCESS)
    {
        status = function_status;
    }

    return status;
}

void
print_help(void)
{
    /*      ********* 10 ****** 20 ****** 30 ****** 40 ****** 50 ****** 60 ****** 70 ******/
    printf("Usage: ddptool [parameters] [argument]\n");
    printf("    -a                  Display information about all functions for all\n"
           "                        supported devices\n");
    printf("    -h, --help, -?      Display command line help\n");
    printf("    -i DEVNAME          Display information for the specified network\n"
           "                        interface name\n");
    printf("    -j [FILENAME]       Output in JSON format to a file. If [FILENAME] is not\n"
           "                        specified, output is sent to standard output.\n");
    printf("    -l                  Silent mode\n");
    printf("    -s dddd:bb:ss.f     Display information about the device located at the\n"
           "                        specified PCI location, (where d - domain, b - bus,\n"
           "                        s - slot, f - function, all numbers are in hex)\n");
    printf("    -v                  Prints version of DDP tool\n");
    printf("    -x [FILENAME]       Output in XML format to a file. If [FILENAME] is not\n"
           "                        specified, output is sent to standard output\n");
}

void
print_header()
{
    /* for silent mode tool shouldn't print anything on screen  */
    if(check_command_parameter(DDP_SILENT_MODE_PARAMETER_BIT))
        return;

    printf("Intel(R) Dynamic Device Personalization Tool\n");
    printf("DDPTool version %d.%d.%d.%d\n",
           DDP_MAJOR_VERSION,
           DDP_MINOR_VERSION,
           DDP_BUILD_VERSION,
           DDP_FIX_VERSION);
    printf("Copyright (C) 2019 Intel Corporation.\n\n");
}

ddp_status_t
get_device_identifier(adapter_t* adapter)
{
    ddp_status_t status = DDP_SUCCESS;

    do
    {
        /* try read data from pci config space */
        status = get_data_from_sysfs_config(adapter);
        /* if function returns SUCCESS and deviceId/vendorId are incorrect, tool needs read 4-PartId using other method */
        if(status == DDP_SUCCESS && adapter->vendor_id != 0xFFFF && adapter->device_id != 0xFFFF)
        {
            break;
        }

        get_data_from_sysfs_files(adapter);
    } while(0);

    return status;
}

ddp_status_t
generate_adapter_list(list_t* adapter_list, char* interface_key)
{
    adapter_t       current_device;
    adapter_t       last_physical_device;
    char            location_from_dir[13];
    adapter_t*      adapter             = NULL;
    struct dirent** name_list           = NULL;
    ddp_status_t    status              = DDP_SUCCESS;
    ddp_status_t    function_status     = DDP_SUCCESS;
    int32_t         items               = 0;
    int32_t         i                   = 0;
    int             compare_result      = -1;
    bool            is_vf               = FALSE;

    MEMINIT(&current_device);
    MEMINIT(&last_physical_device);
    MEMINIT(location_from_dir);

    do
    {
        items = scandir(PATH_TO_SYSFS_PCI, &name_list, 0, alphasort);

        for(i = 0; i < items; MEMINIT(&current_device), i++)
        {
            /* Get adapter PCI location */
            sscanf(name_list[i]->d_name,
                   "%04hx:%02hx:%02hx.%hx",
                   &current_device.location.segment,
                   &current_device.location.bus,
                   &current_device.location.device,
                   &current_device.location.function);

            function_status = get_device_identifier(&current_device);
            if(function_status != DDP_SUCCESS)
            {
                status = function_status;
                continue;
            }

            if(is_supported_device(&current_device) == FALSE)
            {
                continue;
            }

            is_vf = is_virtual_function(&current_device);
            if(is_vf == TRUE)
            {
                if(check_command_parameter(DDP_ALL_ADAPTERS_PARAMETER_BIT) == FALSE)
                {
                    /* only with parameter -a tool works with virtual functions*/
                    continue;
                }

                current_device.is_virtual_function = TRUE;
                current_device.is_usable = FALSE; /* virtual function cannot be use for communicate with base driver */

                if(last_physical_device.is_usable == TRUE)
                {
                    strcpy_sec(current_device.pf_connection_name,
                               sizeof(current_device.pf_connection_name),
                               last_physical_device.connection_name); /* need for getting data by base driver */
                    current_device.pf_device_id = last_physical_device.device_id;
                    current_device.is_usable = TRUE; /* it's true if we have a connection name fgrom physical function */

                }
            }

            function_status = get_connection_name(&current_device);
            if(function_status == DDP_SUCCESS)
            {
                current_device.is_usable = TRUE;
            }
            else
            {
                /* if the driver did not write connection name to sysfs
                 * we won't be able to communicate with that function */
                status = DDP_CANNOT_COMMUNICATE_ADAPTER;
            }

            if(is_vf == FALSE && current_device.is_usable == TRUE)
            {
                memcpy_sec(&last_physical_device, sizeof(adapter_t), &current_device, sizeof(adapter_t));
            }

            if(check_command_parameter(DDP_LOCATION_COMMAND_PARAMETER_BIT))
            {
                memcpy_sec(location_from_dir,
                           PCI_LOCATION_STRING_SIZE,
                           name_list[i]->d_name,
                           PCI_LOCATION_STRING_SIZE);
                compare_result = strncmp(interface_key, location_from_dir, PCI_LOCATION_STRING_SIZE);
                if(compare_result != 0)
                {
                    status = DDP_DEVICE_NOT_FOUND;
                    continue;
                }
            }
            else if(check_command_parameter(DDP_INTERFACE_COMMAND_PARAMETER_BIT))
            {
                /* we need PF for VF */
                compare_result = strcmp(interface_key, current_device.connection_name);
                if(compare_result != 0)
                {
                    status = DDP_DEVICE_NOT_FOUND;
                    continue;
                }
            }

            adapter = malloc_sec(sizeof(adapter_t));
            if(adapter == NULL)
            {
                status = DDP_ALLOCATE_MEMORY_FAIL;
                break;
            }

            memcpy_sec(adapter, sizeof(adapter_t), &current_device, sizeof current_device);

            /* Add node to the list */
            function_status = add_node_data(adapter_list, (void*) adapter, sizeof adapter);
            if(function_status != DDP_SUCCESS)
            {
                status = function_status;
                break;
            }
            if(compare_result == 0)
            {
                status = DDP_SUCCESS;
                break; /* the provided device was found - skipping enumarete next devices */
            }
        }

        if(status == DDP_DEVICE_NOT_FOUND)
        {
            /* Break do-while loop to protect this status from overwriting
             * in the next if-statement */
            break;
        }
        if(adapter_list->first_node == NULL)
        {
            status = DDP_NO_SUPPORTED_ADAPTER;
        }
    } while(0);

    return status;
}

int
main(int argc, char** argv)
{
    list_t       adapter_list;
    char*        file_name       = NULL;
    char*        interface_key   = NULL;
    ddp_status_t function_status = DDP_SUCCESS;
    ddp_status_t status          = DDP_SUCCESS;

    MEMINIT(&adapter_list);

    do
    {
        function_status = parse_command_line_parameters(argc, argv, &interface_key, &file_name);

        print_header();

        if(check_command_parameter(DDP_VERSION_COMMAND_PARAMETER_BIT))
        {
            break;
        }

        if(function_status != DDP_SUCCESS)
        {
            /* header shall be print conflict parameter */
            status = function_status;
            break;
        }

        function_status = initialize_tool();
        if(function_status != DDP_SUCCESS)
        {
            status = function_status;
            break;
        }

        if(check_command_parameter(DDP_HELP_COMMAND_PARAMETER_BIT))
        {
            print_help();
            break; /* The 'Help' flag has higher priority */
        }

        function_status = generate_adapter_list(&adapter_list, interface_key);
        if(function_status != DDP_SUCCESS)
        {
            /* Do not break in case of errors - we still want to perform
             * discovery on NICs we can communicate with */
            status = function_status;
        }

        function_status = discovery_devices(adapter_list);
        if(function_status != DDP_SUCCESS)
        {
            /* Do not break in case of error to print all parsed information,
             * just keep status */
            status = function_status;
        }
    } while(0);

    /* if ddp_func_print_adapter_list is NULL set the default output */
    if(ddp_func_print_adapter_list == NULL)
    {
        ddp_func_print_adapter_list = generate_table;
    }

    status = validate_output_status(status);

    function_status = ddp_func_print_adapter_list(&adapter_list, status, file_name);
    if(function_status != DDP_SUCCESS)
    {
        status = function_status;
    }

    free_list(&adapter_list);

    return status;
}
