/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1alpha1

import (
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// +genclient
// +genclient:nonNamespaced
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// RuntimeClass defines a class of container runtime supported in the cluster.
// The RuntimeClass is used to select or configure the container runtime that is
// used to run all containers in a pod. RuntimeClasses are (currently) manually
// defined by a user or cluster provisioner, and referenced in the PodSpec. The
// Kubelet is responsible for resolving the RuntimeClassName reference before
// running the pod.  For more details, see
// https://git.k8s.io/enhancements/keps/sig-node/runtime-class.md
type RuntimeClass struct {
	metav1.TypeMeta `json:",inline"`
	// More info: https://git.k8s.io/community/contributors/devel/api-conventions.md#metadata
	// +optional
	metav1.ObjectMeta `json:"metadata,omitempty" protobuf:"bytes,1,opt,name=metadata"`

	// Specification of the RuntimeClass
	// More info: https://git.k8s.io/community/contributors/devel/api-conventions.md#spec-and-status
	Spec RuntimeClassSpec `json:"spec" protobuf:"bytes,2,name=spec"`
}

// RuntimeClassSpec is a specification of a RuntimeClass. It contains parameters
// that required to describe the RuntimeClass to the Container Runtime Interface
// (CRI) implementation, as well as any other components that need to understand
// how the pod will be run. The RuntimeClassSpec should be treated as immutable.
type RuntimeClassSpec struct {
	// RuntimeHandler specifies the underlying runtime and configuration that the
	// CRI implementation will use to handle pods of this class. The possible
	// values are specific to the node configuration & CRI implementation.  It is
	// assumed that all handlers are available on every node, and handlers of the
	// same name are equivalent on every node.
	// For example, a handler called "runc" might specify that the runc OCI
	// runtime (using native Linux containers) should be used to run the
	// containers in a pod.
	// If this is not specified (or empty), a default will be chosen by the
	// implementation.
	// +optional
	RuntimeHandler *string `json:"runtimeHandler,omitempty" protobuf:"bytes,1,opt,name=runtimeHandler"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object

// RuntimeClassList is a list of RuntimeClass objects.
type RuntimeClassList struct {
	metav1.TypeMeta `json:",inline"`
	// Standard list metadata.
	// More info: https://git.k8s.io/community/contributors/devel/api-conventions.md#metadata
	// +optional
	metav1.ListMeta `json:"metadata,omitempty" protobuf:"bytes,1,opt,name=metadata"`

	// Items is a list of schema objects.
	Items []RuntimeClass `json:"items" protobuf:"bytes,2,rep,name=items"`
}
