package translatortext

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"github.com/Azure/go-autorest/tracing"
	"net/http"
)

// TranslatorClient is the # Introduction
//
// The Microsoft Translator Text API provides a JSON-based Web API. It provides:
//
// * Translation between any supported languages to any other supported language.
// * Translation to multiple languages in one request.
// * Transliteration to convert text from one script to another script of the same language.
// * Language detection, translation, and transliteration in one request.
// * Dictionary to lookup alternative translations of a term, to find back-translations and examples showing terms used
// in context.
// * Rich language detection.
// # Base URLs
//
// The Translator Text API is available in the following clouds:
//
// | Description | Region    | Base URL                                |
// | -------     | --------  | -------                                 |
// | Azure       | Global    | api.cognitive.microsofttranslator.com   |
// | Azure       | Europe    | api-eur.cognitive.microsofttranslator.com |
//
//
// # Authentication
//
// Subscribe to the Translator Text API, part of Azure Cognitive Services, and use your subscription key from the Azure
// portal to authenticate. You can follow the steps in
// https://docs.microsoft.com/en-us/azure/cognitive-services/translator/translator-text-how-to-signup.
//
//
// The simplest way is to pass your Azure secret key to the Translator service using the http request header
// `Ocp-Apim-Subscription-Key`.
//
// If you prefer using a short-lived authentication, you may use your secret key to obtain an authorization token from
// the token service. In that case you pass the authorization token to the Translator service using the `Authorization`
// request header. To obtain an authorization token, make a `POST` request to the following URL:
//
// | Environment | Authentication service URL                                |
// | ----------  | ----------                                                |
// | Azure       | `https://api.cognitive.microsoft.com/sts/v1.0/issueToken` |
//
// Here are example requests to obtain a token with a lifetime of 10 minutes, given a secret key:
//
// ```python
// // Pass secret key using header
// curl --header 'Ocp-Apim-Subscription-Key: <your-key>' --data ""
// 'https://api.cognitive.microsoft.com/sts/v1.0/issueToken'
// // Pass secret key using query string parameter
// curl --data "" 'https://api.cognitive.microsoft.com/sts/v1.0/issueToken?Subscription-Key=<your-key>'
// ```
//
// A successful request returns the encoded access token as plain text in the response body. The valid token is passed
// to the Translator service as a bearer token in the Authorization.
//
// ```
// Authorization: Bearer <Base64-access_token>
// ```
//
// An authentication token is valid for 10 minutes. The token should be re-used when making multiple calls to the
// Translator APIs. If you make requests to the Translator API over an extended period of time,  you  must request a
// new access token at regular intervals before the token expires, for instance every 9 minutes.
//
// To summarize, a client request to the Translator API will include one authorization header taken from the following
// table:
//
// | Headers       | Description  |
// | ----------    | ----------   |
// | Ocp-Apim-Subscription-key    | Use with Cognitive Services subscription if you are passing your secret key.
// The value is the Azure secret key for your subscription to Translator Text API.                         |
// | Authorization                | Use with Cognitive Services subscription if you are passing an authentication
// token. The value is the Bearer token: `Bearer <token>`.       |
//
// ## All-in-one subscription
// The last authentication option is to use a Cognitive Service’s all-in-one subscription. This allows you to use a
// single secret key to authenticate requests for multiple services.
// When you use an all-in-one secret key, you must include two authentication headers with your request. The first
// passes the secret key, the second specifies the region associated with your subscription.
// `Ocp-Api-Subscription-Key` `Ocp-Apim-Subscription-Region`
// If you pass the secret key in the query string with the parameter `Subscription-Key`, then you must specify the
// region with query parameter `Subscription-Region`.
// If you use a bearer token, you must obtain the token from the region endpoint:
// `https://<your-region>.api.cognitive.microsoft.com/sts/v1.0/issueToken`.
//
// Available regions are: `australiaeast`, `brazilsouth`, `canadacentral`, `centralindia`, `centraluseuap`, `eastasia`,
// `eastus`, `eastus2`, `japaneast`, `northeurope`, `southcentralus`, `southeastasia`, `uksouth`, `westcentralus`,
// `westeurope`, `westus`, and `westus2`.
//
// Region is required for the all-in-one Text API subscription.
//
//
// # Errors
//
// A standard error response is a JSON object with name/value pair named `error`. The value is also a JSON object with
// properties:
// * `code`: A server-defined error code.
// * `message`: A string giving a human-readable representation of the error.
//
// For example, a customer with a free trial subscription receives the following error once the free quota is
// exhausted:
//
// ```json
// {
// "error": {
// "code":403000,
// "message":"The subscription has exceeded its free quota."
// }
// }
// ```
// # Enter your subscription keys to try out Microsoft Translator.
// Select the `Authorize` button and enter your Microsoft Translator subscription key, OR your `all in one Cognitive
// Services` subscription key. If you are using the all in one Cognitive Services key you will need to also enter your
// subscription region.
// ## Available regions are:
//
// `australiaeast`, `brazilsouth`, `canadacentral`, `centralindia`, `centraluseuap`, `eastasia`, `eastus`, `eastus2`,
// `japaneast`, `northeurope`, `southcentralus`, `southeastasia`, `uksouth`, `westcentralus`, `westeurope`, `westus`,
// `westus2`.
type TranslatorClient struct {
	BaseClient
}

// NewTranslatorClient creates an instance of the TranslatorClient client.
func NewTranslatorClient(endpoint string) TranslatorClient {
	return TranslatorClient{New(endpoint)}
}

// BreakSentence identifies the position of sentence boundaries in a piece of text.
// Parameters:
// textParameter - # Request Body
// The body of the request is a JSON array. Each array element is a JSON object with a string property named
// Text. Sentence boundaries are computed for the value of the Text property.
//
// The following limitations apply:
// * The array can have at most 100 elements.
// * The text value of an array element cannot exceed 10,000 characters including spaces.
// * The entire text included in the request cannot exceed 50,000 characters including spaces.
// * If the `language` query parameter is specified, then all array elements must be in the same language.
// Otherwise, language auto-detection is applied to each array element independently.
//
// # Response Body
// A successful response is a JSON array with one result for each string in the input array. A result object
// includes the following properties:
// * `sentLen`- An array of integers representing the lengths of the sentences in the text element. The length
// of the array is the number of sentences, and the values are the length of each sentence.
// * `detectedLanguage`- An object describing the detected language through the following properties
// * `language`- Code of the detected language.
// * `score`- A float value indicating the confidence in the result. The score is between zero and one and a
// low score indicates a low confidence.
// * Note that the `detectedLanguage` property is only present in the result object when language
// auto-detection is requested.
// # Response Header
// X-RequestId - Value generated by the service to identify the request. It is used for troubleshooting
// purposes.
// language - language tag of the language of the input text. If not specified, Translator will apply automatic
// language detection.
// script - script identifier of the script used by the input text. If a script is not specified, the default
// script of the language will be assumed.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) BreakSentence(ctx context.Context, textParameter []BreakSentenceTextInput, language string, script string, xClientTraceID string) (result ListBreakSentenceResultItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.BreakSentence")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: textParameter,
			Constraints: []validation.Constraint{{Target: "textParameter", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "BreakSentence", err.Error())
	}

	req, err := client.BreakSentencePreparer(ctx, textParameter, language, script, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "BreakSentence", nil, "Failure preparing request")
		return
	}

	resp, err := client.BreakSentenceSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "BreakSentence", resp, "Failure sending request")
		return
	}

	result, err = client.BreakSentenceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "BreakSentence", resp, "Failure responding to request")
	}

	return
}

// BreakSentencePreparer prepares the BreakSentence request.
func (client TranslatorClient) BreakSentencePreparer(ctx context.Context, textParameter []BreakSentenceTextInput, language string, script string, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(language) > 0 {
		queryParameters["Language"] = autorest.Encode("query", language)
	}
	if len(script) > 0 {
		queryParameters["Script"] = autorest.Encode("query", script)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/BreakSentence"),
		autorest.WithJSON(textParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// BreakSentenceSender sends the BreakSentence request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) BreakSentenceSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// BreakSentenceResponder handles the response to the BreakSentence request. The method always
// closes the http.Response Body.
func (client TranslatorClient) BreakSentenceResponder(resp *http.Response) (result ListBreakSentenceResultItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Detect identifies the language of a string of text.
// Parameters:
// textParameter - # Request Body
// The body of the request is a JSON array. Each array element is a JSON object with a string property named
// Text. Language detection is applied to the value of the Text property.
// The following limitations apply:
// * The array can have at most 100 elements.
// * The text value of an array element cannot exceed 10,000 characters including spaces.
// * The entire text included in the request cannot exceed 50,000 characters including spaces.
//
// # Response Body
// A successful response is a JSON array with one result for each string in the input array. A result object
// includes the following properties:
// * language- Code of the detected language.
// * score- A float value indicating the confidence in the result. The score is between zero and one and a low
// score indicates a low confidence.
// * isTranslationSupported- A boolean value which is true if the detected language is one of the languages
// supported for text translation. Not all detected languages can be translated by the API.
// * isTransliterationSupported- A boolean value which is true if the detected language is one of the languages
// supported for transliteration.
// * alternatives- An array of other possible languages. Each element of the array is another object with the
// same properties listed above- language, score, isTranslationSupported and isTransliterationSupported.
// # Response Header
// X-RequestId - Value generated by the service to identify the request. It is used for troubleshooting
// purposes.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) Detect(ctx context.Context, textParameter []DetectTextInput, xClientTraceID string) (result ListDetectResultItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.Detect")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: textParameter,
			Constraints: []validation.Constraint{{Target: "textParameter", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "Detect", err.Error())
	}

	req, err := client.DetectPreparer(ctx, textParameter, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Detect", nil, "Failure preparing request")
		return
	}

	resp, err := client.DetectSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Detect", resp, "Failure sending request")
		return
	}

	result, err = client.DetectResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Detect", resp, "Failure responding to request")
	}

	return
}

// DetectPreparer prepares the Detect request.
func (client TranslatorClient) DetectPreparer(ctx context.Context, textParameter []DetectTextInput, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/Detect"),
		autorest.WithJSON(textParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DetectSender sends the Detect request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) DetectSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DetectResponder handles the response to the Detect request. The method always
// closes the http.Response Body.
func (client TranslatorClient) DetectResponder(resp *http.Response) (result ListDetectResultItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DictionaryExamples provides examples that show how terms in the dictionary are used in context. This operation is
// used in tandem with `Dictionary lookup`.
// Parameters:
// from - specifies the language of the input text. The source language must be one of the supported languages
// included in the `dictionary` scope.
// toParameter - specifies the language of the output text. The target language must be one of the supported
// languages included in the `dictionary` scope.
// textParameter - # Request body
// The body of the request is a JSON array. Each array element is a JSON object with the following properties:
// * `Text-` A string specifying the term to lookup. This should be the value of a `normalizedText` field from
// the back-translations of a previous Dictionary lookup request. It can also be the value of the
// `normalizedSource` field.
// * `Translation-` A string specifying the translated text previously returned by the Dictionary lookup
// operation. This should be the value from the `normalizedTarget` field in the `translations` list of the
// Dictionary lookup response. The service will return examples for the specific source-target word-pair.
//
// The following limitations apply:
// * The array can have at most 10 elements.
// * The text value of an array element cannot exceed 100 characters including spaces.
//
// # Response body
// A successful response is a JSON array with one result for each string in the input array. A result object
// includes the following properties:
// * `normalizedSource-` A string giving the normalized form of the source term. Generally, this should be
// identical to the value of the `Text` field at the matching list index in the body of the request.
// * `normalizedTarget-` A string giving the normalized form of the target term. Generally, this should be
// identical to the value of the `Translation` field at the matching list index in the body of the request.
// * `examples-` A list of examples for the (source term, target term) pair. Each element of the list is an
// object with the following properties:
// * `sourcePrefix-` The string to concatenate before the value of `sourceTerm` to form a complete example. Do
// not add a space character, since it is already there when it should be. This value may be an empty string.
// * `sourceTerm-` A string equal to the actual term looked up. The string is added with `sourcePrefix` and
// `sourceSuffix` to form the complete example. Its value is separated so it can be marked in a user interface,
// e.g., by bolding it.
// * `sourceSuffix-` The string to concatenate after the value of `sourceTerm` to form a complete example. Do
// not add a space character, since it is already there when it should be. This value may be an empty string.
// * `targetPrefix-` A string similar to `sourcePrefix` but for the target.
// * `targetTerm-` A string similar to `sourceTerm` but for the target.
// * `targetSuffix-` A string similar to `sourceSuffix` but for the target.
//
// # Response Header
// X-RequestId - Value generated by the service to identify the request. It is used for troubleshooting
// purposes.
// NOTE - If there are no examples in the dictionary, the response is 200 (OK) but the `examples` list is an
// empty list.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) DictionaryExamples(ctx context.Context, from string, toParameter string, textParameter []DictionaryExampleTextInput, xClientTraceID string) (result ListDictionaryExampleResultItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.DictionaryExamples")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: textParameter,
			Constraints: []validation.Constraint{{Target: "textParameter", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "DictionaryExamples", err.Error())
	}

	req, err := client.DictionaryExamplesPreparer(ctx, from, toParameter, textParameter, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryExamples", nil, "Failure preparing request")
		return
	}

	resp, err := client.DictionaryExamplesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryExamples", resp, "Failure sending request")
		return
	}

	result, err = client.DictionaryExamplesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryExamples", resp, "Failure responding to request")
	}

	return
}

// DictionaryExamplesPreparer prepares the DictionaryExamples request.
func (client TranslatorClient) DictionaryExamplesPreparer(ctx context.Context, from string, toParameter string, textParameter []DictionaryExampleTextInput, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"from":        autorest.Encode("query", from),
		"to":          autorest.Encode("query", toParameter),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/Dictionary/Examples"),
		autorest.WithJSON(textParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DictionaryExamplesSender sends the DictionaryExamples request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) DictionaryExamplesSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DictionaryExamplesResponder handles the response to the DictionaryExamples request. The method always
// closes the http.Response Body.
func (client TranslatorClient) DictionaryExamplesResponder(resp *http.Response) (result ListDictionaryExampleResultItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DictionaryLookup provides alternative translations for a word and a small number of idiomatic phrases. Each
// translation has a `part-of-speech` and a list of `back-translations`. The back-translations enable a user to
// understand the translation in context. The Dictionary Example operation allows further drill down to see example
// uses of each translation pair.
// Parameters:
// from - specifies the language of the input text. The source language must be one of the supported languages
// included in the `dictionary` scope.
// toParameter - specifies the language of the output text. The target language must be one of the supported
// languages included in the `dictionary` scope of the Languages resource.
// textParameter - # Request Body
// The body of the request is a JSON array. Each array element is a JSON object with a string property named
// `Text`, which represents the term to lookup.
// The following limitations apply:
// * The array can have at most 10 elements.
// * The text value of an array element cannot exceed 100 characters including spaces.
//
// # Response Body
// A successful response is a JSON array with one result for each string in the input array. A result object
// includes the following properties:
// * `normalizedSource`- A string giving the normalized form of the source term. For example, if the request is
// "JOHN", the normalized form will be "john". The content of this field becomes the input to lookup examples.
// * `displaySource`- A string giving the source term in a form best suited for end-user display. For example,
// if the input is "JOHN", the display form will reflect the usual spelling of the name- "John".
// * `translations`- A list of translations for the source term. Each element of the list is an object with the
// following properties:
// * `normalizedTarget`- A string giving the normalized form of this term in the target language. This value
// should be used as input to lookup examples.
// * `displayTarget`- A string giving the term in the target language and in a form best suited for end-user
// display. Generally, this will only differ from the `normalizedTarget` in terms of capitalization. For
// example, a proper noun like "Juan" will have `normalizedTarget = "juan"` and `displayTarget = "Juan"`.
// * `posTag`- A string associating this term with a part-of-speech tag.
//
// | Tag name  | Description       |
// | --------- | ----------------  |
// | ADJ       | Adjectives        |
// | ADV       | Adverbs           |
// | CONJ      | Conjunctions       |
// | DET       | Determiners       |
// | MODAL     | Verbs             |
// | NOUN      | Nouns             |
// | PREP      | Prepositions      |
// | PRON      | Pronouns          |
// | VERB      | Verbs             |
// | OTHER     | Other             |
//
// As an implementation note, these tags were determined by part-of-speech tagging the English side, and then
// taking the most frequent tag for each source/target pair. So if people frequently translate a Spanish word
// to a different part-of-speech tag in English, tags may end up being wrong (with respect to the Spanish
// word).
// * `confidence`- A value between 0.0 and 1.0 which represents the "confidence" (or perhaps more accurately,
// "probability in the training data") of that translation pair. The sum of confidence scores for one source
// word may or may not sum to 1.0.
// * `prefixWord-` A string giving the word to display as a prefix of the translation. Currently, this is the
// gendered determiner of nouns, in languages that have gendered determiners. For example, the prefix of the
// Spanish word "mosca" is "la", since "mosca" is a feminine noun in Spanish. This is only dependent on the
// translation, and not on the source. If there is no prefix, it will be the empty string.
// * `backTranslations-` A list of "back translations" of the target. For example, source words that the target
// can translate to. The list is guaranteed to contain the source word that was requested (e.g., if the source
// word being looked up is "fly", then it is guaranteed that "fly" will be in the `backTranslations` list).
// However, it is not guaranteed to be in the first position, and often will not be. Each element of the
// `backTranslations` list is an object described by the following properties-
// * `normalizedText-` A string giving the normalized form of the source term that is a back-translation of the
// target. This value should be used as input to lookup examples.
// * `displayText-` A string giving the source term that is a back-translation of the target in a form best
// suited for end-user display.
// * `numExamples-` An integer representing the number of examples that are available for this translation
// pair. Actual examples must be retrieved with a separate call to lookup examples. The number is mostly
// intended to facilitate display in a UX. For example, a user interface may add a hyperlink to the
// back-translation if the number of examples is greater than zero and show the back-translation as plain text
// if there are no examples. Note that the actual number of examples returned by a call to lookup examples may
// be less than `numExamples`, because additional filtering may be applied on the fly to remove "bad" examples.
// * `frequencyCount-` An integer representing the frequency of this translation pair in the data. The main
// purpose of this field is to provide a user interface with a means to sort back-translations so the most
// frequent terms are first.
//
// Note - If the term being looked up does not exist in the dictionary, the response is 200 (OK) but the
// `translations` list is an empty list.
//
// # Response Header
// X-RequestId - Value generated by the service to identify the request. It is used for troubleshooting
// purposes.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) DictionaryLookup(ctx context.Context, from string, toParameter string, textParameter []DictionaryLookupTextInput, xClientTraceID string) (result ListDictionaryLookupResultItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.DictionaryLookup")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: textParameter,
			Constraints: []validation.Constraint{{Target: "textParameter", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "DictionaryLookup", err.Error())
	}

	req, err := client.DictionaryLookupPreparer(ctx, from, toParameter, textParameter, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryLookup", nil, "Failure preparing request")
		return
	}

	resp, err := client.DictionaryLookupSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryLookup", resp, "Failure sending request")
		return
	}

	result, err = client.DictionaryLookupResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "DictionaryLookup", resp, "Failure responding to request")
	}

	return
}

// DictionaryLookupPreparer prepares the DictionaryLookup request.
func (client TranslatorClient) DictionaryLookupPreparer(ctx context.Context, from string, toParameter string, textParameter []DictionaryLookupTextInput, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"from":        autorest.Encode("query", from),
		"to":          autorest.Encode("query", toParameter),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/Dictionary/Lookup"),
		autorest.WithJSON(textParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DictionaryLookupSender sends the DictionaryLookup request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) DictionaryLookupSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DictionaryLookupResponder handles the response to the DictionaryLookup request. The method always
// closes the http.Response Body.
func (client TranslatorClient) DictionaryLookupResponder(resp *http.Response) (result ListDictionaryLookupResultItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Languages gets the set of languages currently supported by other operations of the Translator Text API.
// **Authentication is not required to get language resources.**
//
// # Response Body
// A client uses the `scope` query parameter to define which groups of languages it is interested in.
// * `scope=translation` provides languages supported to translate text from one language to another language.
// * `scope=transliteration` provides capabilities for converting text in one language from one script to another
// script.
// * `scope=dictionary` provides language pairs for which `Dictionary` operations return data.
//
// A client may retrieve several groups simultaneously by specifying a comma-separated list of names. For example,
// `scope=translation,transliteration,dictionary` would return supported languages for all groups.
//
// A successful response is a JSON object with one property for each requested group.
// The value for each property is as follows.
//
// * `translation` property
// The value of the `translation` property is a dictionary of (key, value) pairs. Each key is a BCP 47 language tag. A
// key identifies a language for which text can be translated to or translated from. The value associated with the key
// is a JSON object with properties that describe the language
// * `name-` Display name of the language in the locale requested via `Accept-Language` header.
// * `nativeName-` Display name of the language in the locale native for this language.
// * `dir-` Directionality, which is `rtl` for right-to-left languages or `ltr` for left-to-right languages.
// ```json
// {
// "translation": {
// ...
// "fr": {
// "name": "French",
// "nativeName": "Français",
// "dir": "ltr"
// },
// ...
// }
// }
// ```
// * `transliteration` property
// The value of the `transliteration` property is a dictionary of (key, value) pairs. Each key is a BCP 47 language
// tag. A key identifies a language for which text can be converted from one script to another script. The value
// associated with the key is a JSON object with properties that describe the language and its supported scripts
// * `name-` Display name of the language in the locale requested via `Accept-Language` header.
// * `nativeName-` Display name of the language in the locale native for this language.
// * `scripts-` List of scripts to convert from. Each element of the `scripts` list has properties-
// * `code-` Code identifying the script.
// * `name-` Display name of the script in the locale requested via `Accept-Language` header.
// * `nativeName-` Display name of the language in the locale native for the language.
// * `dir-` Directionality, which is `rtl` for right-to-left languages or `ltr` for left-to-right languages.
// * `toScripts-` List of scripts available to convert text to. Each element of the `toScripts` list has properties
// `code`, `name`, `nativeName`, and `dir` as described earlier.
//
// ```json
// {
// "transliteration": {
// ...
// "ja": {
// "name": "Japanese",
// "nativeName": "日本語",
// "scripts": [
// {
// "code": "Jpan",
// "name": "Japanese",
// "nativeName": "日本語",
// "dir": "ltr",
// "toScripts": [
// {
// "code": "Latn",
// "name": "Latin",
// "nativeName": "ラテン語",
// "dir": "ltr"
// }
// ]
// },
// {
// "code": "Latn",
// "name": "Latin",
// "nativeName": "ラテン語",
// "dir": "ltr",
// "toScripts": [
// {
// "code": "Jpan",
// "name": "Japanese",
// "nativeName": "日本語",
// "dir": "ltr"
// }
// ]
// }
// ]
// },
// ...
// }
// }
//
// ```
// * `dictionary` property
// The value of the `dictionary` property is a dictionary of (key, value) pairs. Each key is a BCP 47 language tag. The
// key identifies a language for which alternative translations and back-translations are available. The value is a
// JSON object that describes the source language and the target languages with available translations.
// * `name-` Display name of the source language in the locale requested via `Accept-Language` header.
// * `nativeName-` Display name of the language in the locale native for this language.
// * `dir-` Directionality, which is `rtl` for right-to-left languages or `ltr` for left-to-right languages.
// * `translations-` List of languages with alterative translations and examples for the query expressed in the source
// language. Each element of the `translations` list has properties
// * `name-` Display name of the target language in the locale requested via `Accept-Language` header.
// * `nativeName-` Display name of the target language in the locale native for the target language.
// * `dir-` Directionality, which is `rtl` for right-to-left languages or `ltr` for left-to-right languages.
// * `code-` Language code identifying the target language.
//
// ```json
//
// "es": {
// "name": "Spanish",
// "nativeName": "Español",
// "dir": "ltr",
// "translations": [
// {
// "name": "English",
// "nativeName": "English",
// "dir": "ltr",
// "code": "en"
// }
// ]
// },
//
// ```
//
// The structure of the response object will not change without a change in the version of the API. For the same
// version of the API, the list of available languages may change over time because Microsoft Translator continually
// extends the list of languages supported by its services.
//
// The list of supported languages will not change frequently. To save network bandwidth and improve responsiveness, a
// client application should consider caching language resources and the corresponding entity tag (`ETag`). Then, the
// client application can periodically (for example, once every 24 hours) query the service to fetch the latest set of
// supported languages. Passing the current `ETag` value in an `If-None-Match` header field will allow the service to
// optimize the response. If the resource has not been modified, the service will return status code 304 and an empty
// response body.
//
// # Response Header
// ETag - Current value of the entity tag for the requested groups of supported languages. To make subsequent requests
// more efficient, the client may send the `ETag` value in an `If-None-Match` header field.
//
// X-RequestId - Value generated by the service to identify the request. It is used for troubleshooting purposes.
// Parameters:
// scope - a comma-separated list of names defining the group of languages to return. Allowed group names are-
// `translation`, `transliteration` and `dictionary`. If no scope is given, then all groups are returned, which
// is equivalent to passing `scope=translation,transliteration,dictionary`. To decide which set of supported
// languages is appropriate for your scenario, see the description of the response object.
// acceptLanguage - the language to use for user interface strings. Some of the fields in the response are
// names of languages or names of regions. Use this parameter to define the language in which these names are
// returned. The language is specified by providing a well-formed BCP 47 language tag. For instance, use the
// value `fr` to request names in French or use the value `zh-Hant` to request names in Chinese Traditional.
// Names are provided in the English language when a target language is not specified or when localization is
// not available.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) Languages(ctx context.Context, scope []string, acceptLanguage string, xClientTraceID string) (result LanguagesResult, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.Languages")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: scope,
			Constraints: []validation.Constraint{{Target: "scope", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "scope", Name: validation.MaxItems, Rule: 3, Chain: nil},
					{Target: "scope", Name: validation.MinItems, Rule: 0, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "Languages", err.Error())
	}

	req, err := client.LanguagesPreparer(ctx, scope, acceptLanguage, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Languages", nil, "Failure preparing request")
		return
	}

	resp, err := client.LanguagesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Languages", resp, "Failure sending request")
		return
	}

	result, err = client.LanguagesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Languages", resp, "Failure responding to request")
	}

	return
}

// LanguagesPreparer prepares the Languages request.
func (client TranslatorClient) LanguagesPreparer(ctx context.Context, scope []string, acceptLanguage string, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if scope != nil && len(scope) > 0 {
		queryParameters["scope"] = autorest.Encode("query", scope, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/Languages"),
		autorest.WithQueryParameters(queryParameters))
	if len(acceptLanguage) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("Accept-Language", autorest.String(acceptLanguage)))
	}
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// LanguagesSender sends the Languages request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) LanguagesSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// LanguagesResponder handles the response to the Languages request. The method always
// closes the http.Response Body.
func (client TranslatorClient) LanguagesResponder(resp *http.Response) (result LanguagesResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Translate translates text into one or more languages.
// Parameters:
// toParameter - specifies the language of the output text. Find which languages are available to translate to
// by using the languages method. For example, use `to=de` to translate to German.
// It's possible to translate to multiple languages simultaneously by repeating the `to` parameter in the query
// string. For example, use `to=de&to=it` to translate to German and Italian in the same request.
// textParameter - # Request body
// The body of the request is a JSON array. Each array element is a JSON object with a string property named
// `Text`, which represents the string to translate.
// The following limitations apply:
// * The array can have at most 25 elements.
// * The entire text included in the request cannot exceed 5,000 characters including spaces.
// # Response body
// A successful response is a JSON array with one result for each string in the input array. A result object
// includes the following properties-
// * `detectedLanguage`- An object describing the detected language through the following properties.
// * `language`- A string representing the code of the detected language.
// * `score`- A float value indicating the confidence in the result. The score is between zero and one and a
// low score indicates a low confidence.
// The `detectedLanguage` property is only present in the result object when language auto-detection is
// requested.
// * `translations`- An array of translation results. The size of the array matches the number of target
// languages specified in the `to` query parameter. Each element in the array includes.
// * `to` A string representing the language code of the target language.
// * `text`- A string giving the translated text.
// * `transliteration`- An object giving the translated text in the script specified by the `toScript`
// parameter.
// * `script`- A string specifying the target script.
// * `text`- A string giving the translated text in the target script.
// The `transliteration` object is not included if transliteration does not take place.
// *`alignment`- An object with a single string property named `proj`, which maps input text to translated
// text. The alignment information is only provided when the request parameter `includeAlignment` is `true`.
// Alignment is returned as a string value of the following format-
// `[[SourceTextStartIndex]-[SourceTextEndIndex]–[TgtTextStartIndex]-[TgtTextEndIndex]]`. The colon separates
// start and end index, the dash separates the languages, and space separates the words. One word may align
// with zero, one, or multiple words in the other language, and the aligned words may be non-contiguous. When
// no alignment information is available, the alignment element will be empty. See Obtain alignment information
// for an example and restrictions.
// * `sentLen`- An object returning sentence boundaries in the input and output texts.
// * `srcSentLen`- An integer array representing the lengths of the sentences in the input text. The length of
// the array is the number of sentences, and the values are the length of each sentence.
// * `transSentLen`- An integer array representing the lengths of the sentences in the translated text. The
// length of the array is the number of sentences, and the values are the length of each sentence.
// Sentence boundaries are only included when the request parameter `includeSentenceLength` is `true`.
// * `sourceText`- An object with a single string property named `text`, which gives the input text in the
// default script of the source language. `sourceText` property is present only when the input is expressed in
// a script that's not the usual script for the language. For example, if the input were Arabic written in
// Latin script, then `sourceText.text` would be the same Arabic text converted into Arab script.
// Example of JSON responses are provided in the examples section.
// from - specifies the language of the input text. Find which languages are available to translate from by
// using the languages method. If the `from` parameter is not specified, automatic language detection is
// applied to determine the source language.
// textType - defines whether the text being translated is plain text or HTML text. Any HTML needs to be a
// well-formed, complete HTML element. Possible values are `plain` (default) or `html`
// category - a string specifying the category (domain) of the translation. This parameter retrieves
// translations from a customized system built with Custom Translator. Default value is `general`.
// profanityAction - specifies how profanities should be treated in translations. Possible values are:
// `NoAction` (default), `Marked` or `Deleted`.
// ### Handling Profanity
// Normally the Translator service will retain profanity that is present in the source in the translation. The
// degree of profanity and the context that makes words profane differ between cultures, and as a result the
// degree of profanity in the target language may be amplified or reduced.
//
// If you want to avoid getting profanity in the translation, regardless of the presence of profanity in the
// source text, you can use the profanity filtering option. The option allows you to choose whether you want to
// see profanity deleted, whether you want to mark profanities with appropriate tags (giving you the option to
// add your own post-processing), or you want no action taken. The accepted values of `ProfanityAction` are
// `Deleted`, `Marked` and `NoAction` (default).
//
// | ProfanityAction | Action                                                                    |
// | ----------      | ----------                                                                |
// | `NoAction`      | This is the default behavior. Profanity will pass from source to target.  |
// |                 | Example Source (Japanese)- 彼はジャッカスです。                           |
// |                 | Example Translation (English)- He is a jackass.                           |
// |                 |                                                                           |
// | `Deleted`       | Profane words will be removed from the output without replacement.        |
// |                 | Example Source (Japanese)- 彼はジャッカスです。                           |
// |                 | Example Translation (English)- He is a.                                   |
// | `Marked`        | Profane words are replaced by a marker in the output. The marker depends on the
// `ProfanityMarker` parameter.
// |                 | For `ProfanityMarker=Asterisk`, profane words are replaced with `***`     |
// |                 | Example Source (Japanese)- 彼はジャッカスです。                           |
// |                 | Example Translation (English)- He is a ***.                               |
// |                 | For `ProfanityMarker=Tag`, profane words are surrounded by XML tags <profanity> and
// </profanity>
// |                 | Example Source (Japanese)- 彼はジャッカスです。                           |
// |                 | Example Translation (English)- He is a <profanity>jackass</profanity>.
// profanityMarker - specifies how profanities should be marked in translations. Possible values are-
// `Asterisk` (default) or `Tag`.
// includeAlignment - specifies whether to include alignment projection from source text to translated text.
// Possible values are- `true` or `false` (default).
// includeSentenceLength - specifies whether to include sentence boundaries for the input text and the
// translated text. Possible values are- `true` or `false` (default).
// suggestedFrom - specifies a fallback language if the language of the input text can't be identified.
// Language auto-detection is applied when the `from` parameter is omitted. If detection fails, the
// `suggestedFrom` language will be assumed.
// fromScript - specifies the script of the input text. Supported scripts are available from the languages
// method
// toScript - specifies the script of the translated text. Supported scripts are available from the languages
// method
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) Translate(ctx context.Context, toParameter []string, textParameter []TranslateTextInput, from string, textType string, category string, profanityAction string, profanityMarker string, includeAlignment *bool, includeSentenceLength *bool, suggestedFrom string, fromScript string, toScript []string, xClientTraceID string) (result ListTranslateResultAllItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.Translate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: toParameter,
			Constraints: []validation.Constraint{{Target: "toParameter", Name: validation.Null, Rule: true, Chain: nil}}},
		{TargetValue: textParameter,
			Constraints: []validation.Constraint{{Target: "textParameter", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "Translate", err.Error())
	}

	req, err := client.TranslatePreparer(ctx, toParameter, textParameter, from, textType, category, profanityAction, profanityMarker, includeAlignment, includeSentenceLength, suggestedFrom, fromScript, toScript, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Translate", nil, "Failure preparing request")
		return
	}

	resp, err := client.TranslateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Translate", resp, "Failure sending request")
		return
	}

	result, err = client.TranslateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Translate", resp, "Failure responding to request")
	}

	return
}

// TranslatePreparer prepares the Translate request.
func (client TranslatorClient) TranslatePreparer(ctx context.Context, toParameter []string, textParameter []TranslateTextInput, from string, textType string, category string, profanityAction string, profanityMarker string, includeAlignment *bool, includeSentenceLength *bool, suggestedFrom string, fromScript string, toScript []string, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"to":          autorest.Encode("query", toParameter, ","),
	}
	if len(from) > 0 {
		queryParameters["from"] = autorest.Encode("query", from)
	}
	if len(string(textType)) > 0 {
		queryParameters["textType"] = autorest.Encode("query", textType)
	}
	if len(category) > 0 {
		queryParameters["category"] = autorest.Encode("query", category)
	}
	if len(string(profanityAction)) > 0 {
		queryParameters["profanityAction"] = autorest.Encode("query", profanityAction)
	}
	if len(profanityMarker) > 0 {
		queryParameters["profanityMarker"] = autorest.Encode("query", profanityMarker)
	}
	if includeAlignment != nil {
		queryParameters["includeAlignment"] = autorest.Encode("query", *includeAlignment)
	}
	if includeSentenceLength != nil {
		queryParameters["includeSentenceLength"] = autorest.Encode("query", *includeSentenceLength)
	}
	if len(suggestedFrom) > 0 {
		queryParameters["suggestedFrom"] = autorest.Encode("query", suggestedFrom)
	}
	if len(fromScript) > 0 {
		queryParameters["fromScript"] = autorest.Encode("query", fromScript)
	}
	if toScript != nil && len(toScript) > 0 {
		queryParameters["toScript"] = autorest.Encode("query", toScript, ",")
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/translate"),
		autorest.WithJSON(textParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// TranslateSender sends the Translate request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) TranslateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// TranslateResponder handles the response to the Translate request. The method always
// closes the http.Response Body.
func (client TranslatorClient) TranslateResponder(resp *http.Response) (result ListTranslateResultAllItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Transliterate converts the text of a language in one script into another type of script. Example-
// Japanese script "こんにちは"
// Same word in Latin script "konnichiha"
// Parameters:
// language - specifies the language of the text to convert from one script to another. Possible languages are
// listed in the `transliteration` scope obtained by querying the service for its supported languages.
// fromScript - specifies the script used by the input text. Lookup supported languages using the
// `transliteration` scope, to find input scripts available for the selected language.
// toScript - specifies the output script. Lookup supported languages using the `transliteration` scope, to
// find output scripts available for the selected combination of input language and input script.
// texts - # Request body
// The body of the request is a JSON array. Each array element is a JSON object with a string property named
// `Text`, which represents the string to convert.
// The following limitations apply:
// * The array can have at most 10 elements.
// * The text value of an array element cannot exceed 1,000 characters including spaces.
// * The entire text included in the request cannot exceed 5,000 characters including spaces.
//
// # Response body
// A successful response is a JSON array with one result for each element in the input array. A result object
// includes the following properties:
// * `text`- A string which is the result of converting the input string to the output script.
// * `script`- A string specifying the script used in the output.
// xClientTraceID - a client-generated GUID to uniquely identify the request. Note that you can omit this
// header if you include the trace ID in the query string using a query parameter named ClientTraceId.
func (client TranslatorClient) Transliterate(ctx context.Context, language string, fromScript string, toScript string, texts []TransliterateTextInput, xClientTraceID string) (result ListTransliterateResultItem, err error) {
	if tracing.IsEnabled() {
		ctx = tracing.StartSpan(ctx, fqdn+"/TranslatorClient.Transliterate")
		defer func() {
			sc := -1
			if result.Response.Response != nil {
				sc = result.Response.Response.StatusCode
			}
			tracing.EndSpan(ctx, sc, err)
		}()
	}
	if err := validation.Validate([]validation.Validation{
		{TargetValue: texts,
			Constraints: []validation.Constraint{{Target: "texts", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("translatortext.TranslatorClient", "Transliterate", err.Error())
	}

	req, err := client.TransliteratePreparer(ctx, language, fromScript, toScript, texts, xClientTraceID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Transliterate", nil, "Failure preparing request")
		return
	}

	resp, err := client.TransliterateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Transliterate", resp, "Failure sending request")
		return
	}

	result, err = client.TransliterateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "translatortext.TranslatorClient", "Transliterate", resp, "Failure responding to request")
	}

	return
}

// TransliteratePreparer prepares the Transliterate request.
func (client TranslatorClient) TransliteratePreparer(ctx context.Context, language string, fromScript string, toScript string, texts []TransliterateTextInput, xClientTraceID string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"Endpoint": client.Endpoint,
	}

	const APIVersion = "3.0"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"fromScript":  autorest.Encode("query", fromScript),
		"language":    autorest.Encode("query", language),
		"toScript":    autorest.Encode("query", toScript),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{Endpoint}", urlParameters),
		autorest.WithPath("/transliterate"),
		autorest.WithJSON(texts),
		autorest.WithQueryParameters(queryParameters))
	if len(xClientTraceID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("X-ClientTraceId", autorest.String(xClientTraceID)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// TransliterateSender sends the Transliterate request. The method will close the
// http.Response Body if it receives an error.
func (client TranslatorClient) TransliterateSender(req *http.Request) (*http.Response, error) {
	return client.Send(req, autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// TransliterateResponder handles the response to the Transliterate request. The method always
// closes the http.Response Body.
func (client TranslatorClient) TransliterateResponder(resp *http.Response) (result ListTransliterateResultItem, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result.Value),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}
