// Code generated by "esc -pkg frontend -o examples/hotrod/services/frontend/gen_assets.go -prefix examples/hotrod/services/frontend/web_assets examples/hotrod/services/frontend/web_assets"; DO NOT EDIT.

package frontend

import (
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"sync"
	"time"
)

type _escLocalFS struct{}

var _escLocal _escLocalFS

type _escStaticFS struct{}

var _escStatic _escStaticFS

type _escDirectory struct {
	fs   http.FileSystem
	name string
}

type _escFile struct {
	compressed string
	size       int64
	modtime    int64
	local      string
	isDir      bool

	once sync.Once
	data []byte
	name string
}

func (_escLocalFS) Open(name string) (http.File, error) {
	f, present := _escData[path.Clean(name)]
	if !present {
		return nil, os.ErrNotExist
	}
	return os.Open(f.local)
}

func (_escStaticFS) prepare(name string) (*_escFile, error) {
	f, present := _escData[path.Clean(name)]
	if !present {
		return nil, os.ErrNotExist
	}
	var err error
	f.once.Do(func() {
		f.name = path.Base(name)
		if f.size == 0 {
			return
		}
		var gr *gzip.Reader
		b64 := base64.NewDecoder(base64.StdEncoding, bytes.NewBufferString(f.compressed))
		gr, err = gzip.NewReader(b64)
		if err != nil {
			return
		}
		f.data, err = ioutil.ReadAll(gr)
	})
	if err != nil {
		return nil, err
	}
	return f, nil
}

func (fs _escStaticFS) Open(name string) (http.File, error) {
	f, err := fs.prepare(name)
	if err != nil {
		return nil, err
	}
	return f.File()
}

func (dir _escDirectory) Open(name string) (http.File, error) {
	return dir.fs.Open(dir.name + name)
}

func (f *_escFile) File() (http.File, error) {
	type httpFile struct {
		*bytes.Reader
		*_escFile
	}
	return &httpFile{
		Reader:   bytes.NewReader(f.data),
		_escFile: f,
	}, nil
}

func (f *_escFile) Close() error {
	return nil
}

func (f *_escFile) Readdir(count int) ([]os.FileInfo, error) {
	if !f.isDir {
		return nil, fmt.Errorf(" escFile.Readdir: '%s' is not directory", f.name)
	}

	fis, ok := _escDirs[f.local]
	if !ok {
		return nil, fmt.Errorf(" escFile.Readdir: '%s' is directory, but we have no info about content of this dir, local=%s", f.name, f.local)
	}
	limit := count
	if count <= 0 || limit > len(fis) {
		limit = len(fis)
	}

	if len(fis) == 0 && count > 0 {
		return nil, io.EOF
	}

	return fis[0:limit], nil
}

func (f *_escFile) Stat() (os.FileInfo, error) {
	return f, nil
}

func (f *_escFile) Name() string {
	return f.name
}

func (f *_escFile) Size() int64 {
	return f.size
}

func (f *_escFile) Mode() os.FileMode {
	return 0
}

func (f *_escFile) ModTime() time.Time {
	return time.Unix(f.modtime, 0)
}

func (f *_escFile) IsDir() bool {
	return f.isDir
}

func (f *_escFile) Sys() interface{} {
	return f
}

// FS returns a http.Filesystem for the embedded assets. If useLocal is true,
// the filesystem's contents are instead used.
func FS(useLocal bool) http.FileSystem {
	if useLocal {
		return _escLocal
	}
	return _escStatic
}

// Dir returns a http.Filesystem for the embedded assets on a given prefix dir.
// If useLocal is true, the filesystem's contents are instead used.
func Dir(useLocal bool, name string) http.FileSystem {
	if useLocal {
		return _escDirectory{fs: _escLocal, name: name}
	}
	return _escDirectory{fs: _escStatic, name: name}
}

// FSByte returns the named file from the embedded assets. If useLocal is
// true, the filesystem's contents are instead used.
func FSByte(useLocal bool, name string) ([]byte, error) {
	if useLocal {
		f, err := _escLocal.Open(name)
		if err != nil {
			return nil, err
		}
		b, err := ioutil.ReadAll(f)
		_ = f.Close()
		return b, err
	}
	f, err := _escStatic.prepare(name)
	if err != nil {
		return nil, err
	}
	return f.data, nil
}

// FSMustByte is the same as FSByte, but panics if name is not present.
func FSMustByte(useLocal bool, name string) []byte {
	b, err := FSByte(useLocal, name)
	if err != nil {
		panic(err)
	}
	return b
}

// FSString is the string version of FSByte.
func FSString(useLocal bool, name string) (string, error) {
	b, err := FSByte(useLocal, name)
	return string(b), err
}

// FSMustString is the string version of FSMustByte.
func FSMustString(useLocal bool, name string) string {
	return string(FSMustByte(useLocal, name))
}

var _escData = map[string]*_escFile{

	"/index.html": {
		name:    "index.html",
		local:   "examples/hotrod/services/frontend/web_assets/index.html",
		size:    3530,
		modtime: 1597936165,
		compressed: `
H4sIAAAAAAAC/9RXX1PbSBJ/96fom82d5DOSMIZAjOUtDmcJ2cuSM5Ct3FYeRqO2NEaaUWZGtlmK7341
+mNkSKru3i5+gJnunu7+9V97kpo8m/YAJjkaCiylSqMJyeX1lXdycvTGG5InrqA5hmTFcV1IZQgwKQwK
E5I1j00axrjiDL3qsgdccMNp5mlGMwyH/v4e5HTD8zLvkkqNqrrTKMNwvzaWIo3tAWBiuMlw+k6a+dUM
PJjzGDVcCZhhTkU8CWp+LauZ4oUBrVhI/GD5tUR17438oT/0cy78pSbTSVALNS8yLu5AYRYSbe4z1Cmi
IZAqXIQkNabQ4yDI6YbFwo+kNNooWtgLk3mwJQQjf+QfB0zrJ1plkGlNgAuDieLmPiQ6paOTQ+8fnz5z
fn35C/46jC/y9/Ozu3tWvjt7N09GB1f5LVuvj6UYzT/HyeEnOviYX9/oP4NfX5+sovjtMj0sCTAltZaK
J1yEhAop7nNZavKNMPyvIJbPMSy/CeGGHV3+i0f7B8dfV/fL6w+Ld8urD/Sfd4vy90+bf29uP4rz92fH
2UF+/vtvl8XFm/zifHayvvjtkn2cHd9s6PchPCWoAWPzMu35ZcljeICcqoQLz8hiDMOjYnMKjz0/lUbJ
2ItKY6SAByhoHHORjOFg30qwUmmpxlBIC0Sd7irZ/5aScSpXqODh5dsFzwyqMUSKJ6kRqLV7cvTXvlXx
U6Mik8l3PP3J8OI7rAps0KC1PRC0TTCJZHzfpDbmK2AZ1ToktvcoF6iatO9yq3DRDJWp/3pcLKSNbsxX
W3mGFlN7tX03tJ0Gc//Kn/mTIB12eYfTCebTFw2I+XQSpIcdyY4bSq7JE+clhMzLY28E9qBz7/Uz2boA
CipeUO2nURIZAZERFcDqEGWS3cFOOsk3FcTUUI+V2sgcVUiGByMynVOWYuZo+CWTimYwQ80ToSeBdeMZ
km4s/9/Bjd4ckOmNkjmcp5LJjBqO6odHdTwakul7WlCBGm2uNCrz4yfr6PUxmZ7l9E8uEjiXiwUizCXV
BtV/A+751eLkcUgML8j0POPsDqSA1ly11IFGcoVgJEgVowIKjCr/e4qe5hxpsWdI4+fjJejOly1rEtTz
rLfdVNNeb1EKZrgUsJAqp2ZWKmqvbtwc+vDQA1hRBTGE0FIhAHe4X33g7zCs/73e7582sqXgRkMITs6F
Y4kKTakEfKAm9ZUsRezGfRjUcqe9x17PvmIZR2Fuby9nEHZF6yMVsczdfmPP2rJvMqrNHL+WqE31bP+0
13vlkmprkb5vv2O55LMsFawxaiw4Gng8thtOSZFMCQy6pgdA7DqoWf1G3W4p9X1mk+m2wXNxZepA7bgz
GLTxUB0Pd0151vrOq/bNQqFOz6mCEF65r1zSWXKk7xcKCxSx63SbqXriMapItTFmXBfUsNQWc11Xvv+H
wq9jcAZbjwbOl2aV2DJx+j5LeRYrFG7/j/0v24xuizYEXBnfUJWg8W37aDR+y22l7QZFZSvgoalJZ0kx
QeVFNElogs4YHI1acylC53n8nb02YBVv62oP4NFaYFJomaGfycRtLG39jHAhFUIIM2rQF3Lt2hQCBAHc
arTfKhQKA7fzS6AaIqqxoCa11Q90STetMd2os8yPChd8AyGsuYjl2s8kq5rAt0zbw9b2jmDn8pcQSEDg
5y5tDI5TOfXKtzbdDmsAThA3aft5O5eqCLUJGIDzNyGFxoq80xt7TbiboIzbw15FzdGkMh6Dc/H2xqlJ
umQMtR7DtpJtRvfA4MZcG2pK3d9m0IaDLkxVAt3gNpO4kxOrY8uoRkc7NsIXY4Ya6r+9OduKt1Vfd67T
/PyYRFMLtpKeKb6qwzAJoilQpfjKVjgXUMm0tgbgQFPt3SKq6yujBgW7r3luBcurS8eOJSfXX5zGpUcb
qcf+ae+xLqTO1+RJUP+E+08AAAD//1QoAETKDQAA
`,
	},

	"/jquery-3.1.1.min.js": {
		name:    "jquery-3.1.1.min.js",
		local:   "examples/hotrod/services/frontend/web_assets/jquery-3.1.1.min.js",
		size:    86709,
		modtime: 1597936165,
		compressed: `
H4sIAAAAAAAC/8y9fZebONIo/v/9FG02D4PaMm1nZvbewa1wMslkN7vztpPMzO5iskeAwLgxuAGnO2PY
z/47KkkgME5mn+fec34nJ21A71KpVFWql5vr2dXub0dWfrh6/7m9sldXzZUVIvXtVXHMI1qnRX7VXO3u
+Te7KJObLA1ZXrGr65v/NYuPechzWBQH6GQcK3ZV1WUa1sbaKIIdC2uDkPrDgRXx1b6IjhkzzQsJNns8
FGVducNXQu2oCI97ltduYFE8WyKnbxWd0tia9VlQvS2Lh6ucPVx9U5ZFaRlyMCW7P6Ylq67o1UOaR8XD
1UNab6/olSppoHXJ6mOZXwUWRa0Dfy3jmEcsTnMWGTPVXVHeFT9OvU0r/JFpeE/Lq5B4Po60kWBGfoBJ
sBNW/1gWdcHr/iHGMQntik8wTkhoh0Ue0hpvSWgfjtUWpyS00zxijz/EeEdOLb4jO7su3tRlmic4Izt7
S6sfHvIfy+LAyvoD3pOsT8/J3g5pllmiaYQLcmrXqutXB9H5gARNE8luB3ZYMlqzbzLGu20ZVVimBz5X
oV2zx5pQHNhbRiObHg4sj15s0yyyQmQfaMny+vsiYnbJ9sV7plJaXvE9MQDeDFyS4dTJJeDrV9pxbqd5
WkNKiyty887bVJvjq29evdo8Pl/682b0/uQmwTW5ebfYV4sbfCQ3C8uji998dJPg99MtBXZd/Hw4sPIF
rZiF2jVvlpT2QS0KOQnQd+5xWORVXR7DuiidEmcsT+qts8R18bws6YceKrvKYzHfHERQixNWDyBXjfWY
ZYRQV8/s0Nuly588Ouc/tmjMd8Q3v8UcHt7UNLwbVMknNyClvWdlwqAqW+u0hTDtodw+lOy9AAUCQBy0
mNFwO9XH0uYpUCGmqMV7epjKBg12PbNKe08P1nCDBDjsslMx4gCHfIERX2MO+hMTOao45uCWfZD9KRPY
UxWvIE7Lqr5UAbu3lqjFGf1olsUKtZjdT8yrthI4JHM6t/gyBc6ym9RRP8NnZGma4W3gerBwoe87ns+r
z6OLo+xWpWnOF1AsvLPFVVHWTmjzH1wdYNpCWzy0uLTZY83yiMAeks9ae3w4FPN5jzDDMU5IN4ne0m+a
U4u3ZIXT/rMa9o7MVuuY49WgKDJG8x6LJ6Zp7UgyqGwrK5vPET5D+0nTlHZavVL9SlDTWAk5tQhvCSGp
aVqJgMztYoHW29t0zStKY4tvmRmx6KAlhHi/gqs0v6IoJIkXAM7lP8mMkIh3zzT5D2/1x4ymuZhnK+IN
M8I/w062IoSQazHXYmS24jjZNPvEELkhX0Wn+67XBamnFvPmiVoHa4djHCHkvC/S6GopewNZItQBT9Iv
nHVijweaR4Ujzy9jbt3Pv6P11i75572FkF2yQ0ZDZt1sXt4k2DAQTqufGI0+OLMlZvz0G8Dw+GTk+zgv
ioMOiC3u12Nigxvqk0EIKW2+jlCNnBoH/qqJwmn1qzgiLyC9GaGmSQkh1BZnKa/o++OelWk4idVke3K+
LCM/7gNW8r4ETWNUcMrBGzLNWVp9T7+36OJAy4q9ygpaWxRBX7XlOm8Gh7L6mTWjTWN4AmqvRH7fmBFy
J9AWRcg0rZkVEN4pDkIhyRRGM7RdayDTDPRtjPV5lJshNE15OoeIEJKLrn6zP9QfLnV1rcG77PNKdX7Z
Yl7xx84bOjcM52xT8jGf9466O68btt80qpij0lucZEVAs2/e02zQ6AEAJKR7lvETdqpDtIPkGhv7amH0
oH3E7wFKI/Y93euFtROc2irdNPtnuy6+LR7Usc6BYvjl7KzjNQLRgyOyXKex9cBX9cTnOCRU4r91dBuu
I4GEAjkfXuTjCPMf3sxshYKS0buWZRW74qUjsUKfLKHQAG1xXab7jy2dYTgWX75+oiq+//m5fMdGxEg/
ML5LPH893oHWgyQI+XhdRTuEuN9QHRB41HcocrYSUDFFCIctTvPzNrVjXvQ6cBcrJ1X7g+KQd5c3Neoq
nzLR3XmgTp2ILDE7XwTqsfncJ4EXdaNSeQjDHCZLdjjrlWogwoyT5TFZ8iNLNbUls3Ad3ybreD5HEZkF
FvViH8cIRzNCtqbJ4HyHrx3mZmOKSLQkW4EmtsTzO6gCsOjHE99G0BwjqjUcIiwWiJnmVrTI0LqDqVjA
1CcLqJNF0kuej7ecED2mkbPCh7J4/DC9AzDjfT2DgMA0+WbgR2pAKKYkRHhwhFMkMdFVRCQ92x3R+CnC
jJxTPVR2LhD0Do4k22ONK0AcJzKb955Q/YfTEfx3PseMY4sH5yWtmZ0XD7g6HjgX6RQtmsK6bz7sgyID
kiDOPfFmpzUraV2UPgnPPvHxAilsfC3on6vv4RC6EjzWlZqJK9gRV7wfVz+x5JvHgzxBxOErGzaAYqst
48pAIx5y53VHjzEP5oZv+GcoDPXM20N/Ts5m1DQNAVoGBxOOFjuqdXyK9pMyI5yemfH1FOe2RfkBZ1A+
En6shk2zlGdtd/L2gBE8W5pmsFgJyAQ275FMHK+K5MRbnOIdvsMZ3uMcF/iA73GJK1zjIzGq9LffMmbM
V9ecYOHTiN/r/PMDWeJHssQfyJZaCP8mfp6Ln6+nWT1OanAIzshsifCyxS/IqR1zzC85TviGvLQPxQG/
4r+c8f6TevgzeSn589fkEt5aYm1vh7fROhQHBvVCH6gT2R9FaSxWLf4LMcItC+9Y1FQsY2HNooZWH/Kw
oce6iIvwWMHTIaMfmrDI67LIqiZiMSubKK1okLGo2aZRxPImrfb00GRFcWj2x6xODxlrigPLm5LRqMiz
D40UhkRNFRYHFhn4r8TwNpvHp8vNpt5sys0m32xi38DfEsNync1ms7Ebb7N5WPiN926zXGw2j3Tpo7mB
vyPGZuMZ87/OjWvLmH87N5DlOvLdu373pJn923cJkl9c5zOrr/Ed//3MR9fos2ZjjBM2Bk/ZGI2sFzWy
ls3GN/D3xHD6Bjcby7L+86pRM06xkLfZ+H5jzL+bG+gaNfY12mx40/gHwmFR7Gbrr3NjbmAjMRD+Uf9u
vIM+zqHid7JSH6lW0LUYw/yJLPy3icLXWPwYCP80lWx5z+b/5l3869xAXdY3g6xEZX232fifbQz/2tVn
D9p+q5f4HuGfx419OzeeGAj/Qk6vXzqDtD/IqTcQfvHt8zdvhqmbjd2nv33+p2GqSGq8a58nP3/79idn
1O53CP/45pufX/4wTvge4Rd/fv3tqDOOBVANvH/Dufsmr7f8/4K/oIUVbtMsaop4wdGVBBc5P+w9y5si
ihrL8uYLv0HWZhNdo7zpIVYmyPfNJpqjBnWTCaBhpAbCnCMejZTvhL/MDfREZskZi6oXRV6zx9qZWFmx
sE7fK3bfJHWTiRH1AxyOwXKdxWYTIRe6rnXMcon3buE3T2QXW/wrueG9SvPDsZaYpuGdoSWjTXCs6yJH
T25S/Hdy8267ifjjP8jNO+/dyZ9vTpvqeuPltE7fs6vNww3+p6jtD5bHUcMcNdbmYY6aja0+oCc3+Am5
8eb/9m/wvwbgBZvN22wiuoj90wr/sYWOu40YFWps6DSHU0rJJGlFjOWjMQ8Wf/zyy8//qGgdTqZFTRO6
gRPdLl1xLttxWexfbGn5ooiYFc2hBHImE589Wy2bL798+tUf8Wr59HMzar784+dPl6jFASU3lsfR3+Mq
3jz+79hv3i3cTYSad4snEjHKlMXm+OrVq1d8Fm4SHNILwkfX2Cz50UpdY3OM4zgyHCrOF2uJFys0NzYb
Y07tUPbueW2pk2WxQp1o11r9Ec2NK8MR2VscUZ3Q2gt2h9TUmmS/1AHCmZElP/LjotwD6dA0RkYDlhni
UMenKC0doxfuGjjnkGxkLGF5ZLRoXZcfTn+SFN1L8mdBwr23YQfyEhXCw7eXnv6uBJ3Ax739cGBtSOtw
a8UUnf5ETlCv81LmcoeT+ko2S7FsNkConWSEqMZYrB+2acb44Sx5ifncR+uOjwgXq7ZtOzoroQCBEWai
rhhvJRFTcOKF0xeBXTzkrHzZkyqBG3Tjcb7ilHXEQdTzOx5rprHfM9o0qxkhD6b5lfhZwWtHWfMKZsw0
LYtXPGisaQLnPZoRkpvm3goQBml+jg8ILkpWsl4rI/+02SMLOcHOSZOYZN7KhzxfEd4WXKtYOxLYCaul
9P/rD68jK0Zo0JGdnXK4ibuPgvfYIRwJFjiNrQokg9VEVaZZWwHeIdP8VD28Q5n31FfpCsYirHex+vrD
W5pw/p+PDEMPYXCf+8g0w2HOFxmtKiE3CC6kfLK1LicfDY7aNLZC+76ipjl77lG+J33TtGb3TTO7t2tW
AZMt1gLWtCIBLgldq6lSgpUZIcEFWQY6WXdiXZ7XdZkGx5pZRhoZCLl35K6TCAQUhxQ5gV2NM+I7ckS4
IIlFEd6SQpGqYiNsFwtUeFufGH8w5nd8BPOKWvwLWpeksHdFmlsGNhCuyBM1JNO8p1ag3fqgpgn4ZJSI
I4SzWaxsuFN5AwdQUT7PMquE+RO7/RGd2jjNaZZ9ON0RQo58hcQ90mjMbdvKylOrlyP9iI0nK34iw0bt
dy/nDoTwm3Pj3efACvl+7vAhQF7Ih46eRXZIwy37FqbINCOWsZpdBR61q20a1xbyceBBXp8w1ZegbzKl
Oqr1jj6ZLTHt03e059zy8W1bnLIsqlhtCLwqZXszagVIzlQnZZmtuhnT14FP3IW7OMAO+THLNPR2R4do
UrKnjcFZ93AIKGyxQJFN67r8M82jjHmhx3yfaGPPBrUFpklxxHnMFch8FUYU74H2Tu2qOJYhe51H7HER
6G+APAcIKESiOyEJbX4YvUmDLM0TjtVCjdtarDoRkbtyFqu+l3t9hfp7qq7bF7ah4p+BlAL2mM813BgS
QrU5zf9H9VtaA01jCBIN3tCF9opL7SmGXx7u/PAYAopOB6zc/uQP9GetzKACvTR1gvFrWr3UPjSN/mVG
yIyaJuNwPVVaa33UTX3cB33cqUbqaPQWmQdYTwpxJNaB4ZhQy/OxAnIcIJyQeAjyyWKBQo+R2Et8jtc5
xJOZFfEf/oxQy/91XbofbH7TnNImoJPHl2nSNiQJtaUAi5xaHPP3tPr7d9+ey1VAkEvHxABFncREtqDu
B2aBaRp/fvvdt8OTpsV7aJTVqpIJEQ7DCaHueWPO+07eKCgQTkok2r5Oxr1xrZwkuCD5OAEfyCy2coTf
i5osxvOwmB6z+peUPSDT5NvlMCOEE0LMplH0zXuW19+mVc1yVrrnnyzjmGcFjQwcUTxbIYdx5EXDLeTi
FWqvllHkfXaEcAiYDk6eiuwuENKhogaIkRp4RkeHdJdsoJbXOLXwF+vWFSzUQfGi2IuDwkBINndOCBnX
BpJgfN5qR7+Qf4izPL9ECYmSnHC70MVi0EWKOEF3xLNRhbyuppn6ah3H3eSNuVZkx2lWs9J+/XIK7rtD
/1+Y9toNk1N4TjLxA6JtMW8ij4YN8IOLE2QTW3ZMFJvmocfmIyK371LoeqHveH7bIud/PijR3EWE0o1T
4Pbzb2L8Xd84cfyeZkf2f3tGhLh3cl443wE1hiS+0D886Ji65vRif81IMAFCFGksXUyY4Ob+e01IYg4W
TM3H2+d/ItP71p1i8D81VVrxi1yMA/yEG5xTzFTpE7QXrjE9HzOyHM3+oHK45bkGCQQ6KToqhms1xKmy
UMPekiMLO6CJFLkbd/ACskBymc/6j4FpWPyAOuWlC1wYRS0u+cDv+R/Bi/WobTyFcIM/wmbnaCzPWcmP
SmLc0qs0Ip8Z8+Pc+OzZ7Q19ditkaP3nxabcbD672lc0y4qHkB7qY8nIZ589uy0OQBMo8T58uxEfn93e
iM/PDEzPF9rwhtW9I5995ndI3TTvxcoY3vW7Jz7pJe2fNRtjAzLYyUpVT/qqmkZV1cv0XQf2RiMEmZfq
SqN/EzH+qdr+TS6Uc+Slx0SZPmmyJP0DNDe/nihq/8Gee/N/+3DMjpaXjtZzW7KYfPbZVUdUfqaehgs8
mS5W70ZbvvUFTk4Q8Wg9Zsc5yBvYEDc3MFQd+gI0yp5z4gEbLy8tA08n0RRsQEkhY+4uZQyEn844Kzax
MCyHQU7U1CVhw1FzYSB8tm+6GZstLzfTV/B725mq5ho7jwbCqiS2rx2+9ogjgD3nk1ml8itkUJFCJTVN
YT+w4C6tvxvm5Qn74reJr8VUzmr0kWOXEfSFfFbCIs9h40F+Uslrbgy3OP2bV834PoeRlXJkM2Lg7zlU
35P7bsI0qfq9FM80nCAsSTmVp9TzBGo+Cjss9pybUwT9j0WV8m4jXJOgabRseU3TvELulGT1qwFr79Ix
Ye9QHAkhac85rrX72ahpZtYsEhLQqKuIfw27pt3+0YqQQy913TRXfzQvpoKW3/joTGMrkAKFgAwEWjxF
IxBmy3UneMFfk8A9q4fqd70Z3wXLtbi2mF3s02IWXErqTl03cqyITDF7hBDrXCCM3MtTECBnhVcmn3Wh
S/qScRaIRUJjbboQNBS5fHx504z6QQh5b5q19R5T5C5WTiByBZdyBchdOXfua+sOU7TgPwFyls4XZsRL
r6YW6NLEhp2eUL9sQPxorwnxqI+3xAt8IUZvmlknc4YRdZ12Vw7jL/FUB3lhpsurpaxrHRK67gVSGvwk
9jEXIsOQ5wqmc231XFLo4EU+IWTrRT6K5vMeDjIKaRhSHJntPe/yVj2vnGWLc+TkLU6ownfTd1Fws5Ef
s0z8CZBepMOeZ4sxBYfq/oHC/UPHy7zBBvnsyYqTLvgML5vmAWTmQSczL5tmVgqsEwhVS02KHiAEkmWx
oTqkGQBFGzXNBKLloKqwkbxT6T90mKaTLUoBK0Ontp+hAOdiejzqqxPr2RJmSmGkydn9xCwpE4eEgpRh
VMXHCwPoMzKQxI4UhXwcE2aaL8Qs6TnxKCdyGdxvzQ6Kt1Dg1mkux27s6MKQppkd3BFvHSDHiskE0wkL
GdvVgYVpnLLIjQXn5YAcmo+fVSE9MHLOv48UHsX1hihSlgPIPFd1Nt58yGv6eAU58dUxL1lYJHn6G4uu
2OOhZFWVFrlzZcxllcc8vT+yN0U5Jf/SGCvAAxmZhXbEahbWL4+HLA1pzSp8RyRKfVNz0gWk2uJil9Mw
PMH6GuFMsV0BoV7M2S44ZLzYB6GX5LlihDShO5V6/iB3xCuk4PMOpPmYtpiRBCb/LXucGkBIDANwZayd
1T07zhm/uGm+Ej8reBXM2plGKBgBgT5DXndYVP8IOsqUUBt0F4A2XNM1/6DL6sM56FB3F4afi6a/0BGs
6OkvHFpEvn7egMmHOoJeotHiSIgyBX6pyEm7yHG+XGJBof9YsWNUOCnFgJCcX3C/O5xTizlby39LloHe
g3MynhnO+T24MDyZLVtsXE2kt9iYd59L9j4tjpUc/aDsvy9lalt8KNkrEBo5J1CamZJxeSuf8D9DARKm
3uc+sfjfpqHeF/D3S79p9B0lcnJeDSDwKYdAKGfwfeF97sOFGO71E75ArdTG+WhPBggGG3m9FQ2s/K6m
z5ErO6e2s0W9pc/7/YVP5hb/cXmP+eMf/aZZIefptWWw9ywXlX0OSulRpN4QL/ulKPu//Tn1/s9ZBof/
mOa4xVapHk3tmxlv3jT57ChA+8WGOZA3obwOl29DBwbk8pxkOONOaJpvRfaQH2sBSawQz5ZIvHQWfpaB
jO46YhGghXpGsDBLXu+yn8OAj/ipr+wH4Yu+Wp8j1HJwFgD09vmfnE/KHacvvIS4yD1T5J0N5FD/sXp+
20pNsvNufZA36Z3pWtNYwUCZyHrXacPRuSE0iJonyOBz+sGieKJfgViCCZwW9gIn7aVpPi1sHQtapcjf
QLDTWtTi0cYd6Ml3n9WdFJH0gBVpdntCm565nBfk8+bM+HSwOcfphvjkcso0dFQOl83g9Z18DU1zSQhh
HaCFyDGu+0Q94dli5RhP9DQBTz0wiqb+LbNYHFmwDox+4NgQVJPQuNJGlIBjDu42WQeqqu75CmqfGwsD
gHeMbJQFm1TFIYBbgKzrgR4nxMhoVevfF18gvCWG1AeEbqjZ5YddJOfHPYea2UxnLjR45z1JRT8GOs0k
nhGSuIZ22hkTJ8D9kEspyZYzYNObBVdklprmbItrMlvxY/seTudYkRIHdNp3/MWe7L0DiMC37v7y9isd
PvL9mAyerdYFORCjyDNQAaemOStMczCcttv+aWwVxEvce+20d+5tPv3w7OPENCveu3uckb139JvG4j9g
83dHMm8vKa/XL3nS4B3y7MidR31QncrJjuNA0GfaeSsf15wi3nlPfbznT/eaVpmX+910zOeccDZNPi1N
Y9UkJ0vUNIV9KA4WKEMNZ8I05/PaNPfAdJ5488R7wDmu/bUw9BmoOe1J8P9maAjXwlbo94/jP1xxOVAY
xn9vCGpyah9hMV9Dy6Z6QRgMI2qa+r8iQsjSNOub6BlZtu3EudvfZQAlDJRaBZMU2RWrBTFUeXTE7WhU
hHHM5QU6i65EBYLK7yx3vKPvAsPCFDu3cq2QeBRTbBg48LHe1shawKJj9klXLqC9DVBMgK+6oFIQkdf8
jPISoHgin8yskP/AlxZNnaq8uiUOeSrjpKGYG+eUF7WTDmWO4gz1fCy9EGzPFaX6Kx0+HcMxcMzWKTsm
JFJSAoY9n6NReq4+ZsUk8ba+oFO2fDgB/4nRcDCY4bg/joGgwRFnsXn1YNEEH+F1Fgq4bluEt7Qaj/Gi
totU2NR49RZhxapfqIWO6SB8Xq8V6FwOqLPkOSs5w9U0wOd25x3l5x1vNqN5cqHJnyX5CHTCJfiF8gC9
mH6cVMNnakbrqLiCa9CDG9hQ0ViD8HGfOTyBtz9OE987uxUSjpoLhU7PEm4Mu3HDuS3BYixNHSt9oV6a
2iJc03LgLEFX3C1CKmS7/TPfldvBVbY451fC7jONWlwWxaTzBUoIKVoMBjaX0nObhpz1kyJs07Rm0OQr
sMpp+meLk5uzGccKILOm9rZkcdP8m9o1DUB/DWz/4SrDKag1WyGsrjbgfYmwvPaapM5/pyZawHtBbWVQ
1BjinkpLUld/LVZP06S7rh+mv3UVwKBwXyGokLeY7Q/1h0GVv0sOkMZWL5C4/eOUVbPow0RvZ90JYUPr
cBW8ZTRi5dTY/i53XDenqMUwgVOZf53ILDTx/ofLpOnzKaDRPgUtBkORc8vvcVWX2jRNg9fQ12+almAg
rICMeRUghhHnVVSZsXRQ+dc4aEhMTZK39AHHjZI1IagXLFY8D7sf5+iZIC+8XbrhPHBCyPme5ee1aZZ2
6/A2WIdz8hTRsXYCbREuouhjxVefKJ7VUx3t7IdJ19f1YsGJmLWqJhpUk/zuaubz6DaYrgV0chSA5/WW
aOB+33kCOJU0SgtnthRoJCge+XOcZoz/HmhVPRRlxJ/TPU34xxb1lFXgkz21AtRXVx2DfVrz/CWrWH2e
Pxf5lcJjSS10akuquc9Rmk5V3+MBSQV8fElxwrndurhjefobI5NEoG42Sn5TQoE0tu46vRB36dx1ctb1
llCccqpnxxtX4jNFqaBTaJozi5G/CVuILThx4FtkS7ayEuYtfcXpNs0W4VRKZInnI37wzVbYYuSnrgaw
kGZKORzHIvtJSLlD4ZQBKtVIsCtQVOgb7XhrsRLJVZpfqWlEvMO/eImv9XnnJb4YCX+ymHRG8YleJFhe
wTjsUutpbM1C6c+gm+GtTHe2bi87Q85vFsUp6uZecyxVUXUACAS5xAMLHMNYB7fhOpjPUTQH43JxJ9Cr
GXU11XRo/xXYUVpixukI9lhzKrtpIpyQ0DR14SshJMZb8thfngXiHHIH/DvrZfGBF/mKFQy0W+ikk0bL
Ej3Z8jFZAGeKtgCp6XkbE42YJpW1dLfNQ5n35f7BfUQgmLdAZ96CEfMWjJg3ZpqMfMTqRDTaNMFameNY
nGWNOZk/ZFcJIVs1T3vvqU84Uw7b1It9ssfw7XyAPQHYrfdxoD/dM2iDqe6kZfTcPgGcw3nMlxknCAqH
8oOra/E9PcfPS4CwzvcEA98Tgq/wIh+H2p1Db49PdfGUqoszUJ6Pt2SJ034L7Ijw3BB0Lo6smFDJQHEU
FVox1NM0VqIuhPAOlCj4yxbpPoS6LjxqXdAYrcg0Z5zNM00rIo/UihCs/YyJb4x/4/nRQFFeYt4RVPs4
538KkqixHEjcNO+pFTSNcW3gba8K4m19Zws84z2Z0aaZxaYZuAfngVoHvMcUqsclCV3WNFbsUqdomgi5
nu8kzj1YephmaN3jUuSM0GlHHqhV4hzhyNphPrE84Y7shoBwx1nSjOy8O5jR0su9O59zpffyKUOo7W6/
ON0uHngDno/vOjWaYX2lqG8nFuHeu+MVrRnQV0IHcYdT1H6iuLUjzH1txThDzp5/erZYmaYVezvew4T/
8O6J7V/CgAkhiVuqS8ACq/qRUyLMXNmDBJc4RY6yuUpwOTBN+DBEyhiO1x4mExLZ6srLg3sGfmZw0E2a
RkviJzBOSeKunCW+u2RhKmjZLeb8TXYh02srwIJVlhn3xNNFKf0+nyUcdpsmnBGy45vCCkiIeki7k9md
TD70Pu+EMIG1/jq9jdep8MsQDseayrGiPfFqah2ptUc4RH6H98KOptFyy2kWN7D8qzxcEQ69oy9mmpH5
PF2z23jNRMuDdplqdyAze6RW+mxlmqIb8MjP007CnS5WSDlMkee7cSUuudLFU1Gla1wbjmG0ms8wZSEX
4vSWmeaHvsoUM44RbmPxtROad1/hOEftXlHMijKAHvYA9tvQ+KsTwGiOfJ4tcUzGKAbfiTIZLvA9LskS
V8RYGrgmsWl6Pj7ynfWe7PAjRzWgl6xUsy2Ocu4Q/kAe5kSwOe/dlTNwm9Y09gr/Rh473ztFad0Jz3VC
Uylpmju0rmaE/Gaa0tFcRh69ykfraj4XqME0M3QqyBInTZOdqTblTWPtrYwTVbODkoDcE+oVUhv93sp4
yRxvETpJijJDUp7Me/NAPqA2NE3LysjsnrdmmuVigWPTrFV2wFnlnFQ4NE3e3xK61DUXiOburRof+dRq
Cvfls6WUI1eLBaq9ym+aI/y1+A/5RmiKpAitjxzlHFGr8EiKjwjfmSZH38duHU2z7Lw1cRAdqFBYaa+i
IMaGd+Q9wnXbm0bwswY5scq3JYlQWkuzaV5A6mCA/6bnGicwi9FJXA8mYAHcQ57St1osUEw+UCvwQh/h
2Dv6bqdl4TD1tI7Jc4vi3/hZyE/IuFMkIFRTe097FQNySWbaebkhU47VTJPinMyYaSYWJfuuGX4KwaFH
QrgTWIEgSiJ6voYpyUE6qt0Bc+5ErsFT0zRev+R4wNqR1Fv6SLL/X41sLQ98A3VYKIULe8BCoApJrM4Q
xNoplKZzL0I+GiDeR+QtfTzT3Nvs4UJ7IPfDPUJJFXsiyH01tjYmv9i6nwxlY+wunXS4lvFiAd3kI4x9
rI0D/OJqyJRTqxkRY5HnLiPZx4YkTZtTdfRNmzhLW+5UHcYxXvEhsk7vtqJWivCMjg3JOXzgUN0gSYiy
9k2ztSjOEbIYaGHhEM+Cpvm4nTX4YNN1jchRGe4aSGkaSaVfMHk64nN9JTKbZXhvITzUA71g67X6iFrt
R6yYJ/TxO+J+Si3/D0IH38DGH4QIq5cejmRXPD/nopvmjgpJVgOy1y1Lk23dPKRRvTXwWAYj+Ntp+7AA
G93t8JAVclfOU2HA12vBnWl6T44L5HU3YOShjWSo4g+bwQB3nsYnBi2ydqOWJScHaZqfFhb2E6EsLUHZ
6tKSSWeEoz71mvqyW3/Bk75b1sO59wIfRMfuaK4d8JM3rUcYaXqEka5HiHBCW7C8gw1PHsGv6qEkj706
mPzkGY4h/LQeyk48Ver6f+qFPGpfcSncXz8qLTtwg/f37759WYTkUTzistcJfeweebug4tip1D7KD6DI
+eGCpxs47TodzFDdqlCQlyDhKaQXnSvRgPZFki0lWDRVVqiuZuXZR7sjOmrxbxddm3n+hNR+bM5PZyDb
DlXNmkreczXVgHcHaB5/TW7e3Qp/3d67zc1m+cwBj2T1ptzkm9i/Rt7wfXPjPrNc53Zzs1k9a8Bt0Qty
88723jl/2HgbG/vXT256MdTLkbrNwHFhgNzSTkp2GCgLcZ6jM5hWOsYRBvXbsEVOf5BOlR7yP7LImceX
4KNFe4eZnDWSdbxQOtBuKRkR6VHTsQKif0L4P6t77JqhRahV9V0AzMBb+trVm0WJ4eRFbYEiFjIQFiIq
dR6Cfo0+a0BdjHTBQd/J9SLf8XxnmMWiakjB1JDOu481F9jWCbQ5J50SDpx7M3jTXWD2Cl/ogq9zvrHk
zGt3IyBJJ8t1cBuBhJMT373xCvMCH4PD914MJoW+IRk14PkI6zWJebEohko0q8xnK1dHYFaInLBT/Pu0
0/aX0qs6UJ2zFQL/u/V/XHApHBhPXdTNZM5z1TvTfN5ReyBU7nqhLMNboe/+DX4lXI9tqmvr1ts8bH71
58+Q9+6Zf938Qfc+9ifSxRGYhGCGYziKBuvaEd7fTHRSoFJi3Aoue+mbpvFMPPe+ufze6eYz8rnrCbkA
qDb4zivlewmD/QnzeHZFN3N6L7BFzAHXCpomRHKlkXPmDz7o0sAgRflwCkhwleZVTfOQd7l0+S51AqwH
BsClDa6xOWkCJXGgdM5ga054lopgWfHXYo2gtXMH6IFwb8uu0vwqQAMkCz7wmY9c+WAF/E2MCvQcGYYv
ujf9jtki0diEnHlPOQdnmqLipU9irO1iskJYr0EzS+sK0GEBsRedkUtbtz937ZLR6IMrfwFCrZIX6Bwv
W6JK1K7/pF2CcRjE35DSigT8/llAr6Djq+ZQsveW6/yc12nWgF3yDX5NTqDAVrIc7vOEpknFn8Hn2myJ
eTFnthQxKzoct6XVlPd01TGdJdY38jTuEs6Slmt6G67pGf4SYSM86uv4q8VhVlRMj7EwdK4t0atCtjEn
bhJyjmkFtQK7s0MLAFudEL5HlF7kr0PTDDnpsR7ZVIFPrk6rYLUyTStxE6GnIrVRx7buF84l8HiNTnF3
WTy4GjsLUtHdVwzwcYycGJQNIvY4qXXhTnjelmd1CdZCAniR8qYt0K5CGRwjORJXAHKSD7rfHrngZcW5
bw5Fz7PMUjjWWaxaTKNo2tv6WYgPbWSDuCMJqy2ESygMnptpFH09jleiV0qjyFIOzkeRMJzRuwJWilCr
e0H+i+jqmDheDYnj/lZd+nI+TSiVKF2OczPVQBqP6bhS2i/J/Tw1RI4Y9DtJ1OWGXX9JJ3xcDNym52Ol
EJmZD3+gmotagSMu5R3rIcu6n2fZxSFMVP+x7Bda+PSY9XZg0Lym3zFVY81qXrQSL5Pr8ptl6YvcNKcW
aRw3RLvp8PBkeZ1B55kVop72ECRS+1NV7RnPx1T3i9m2I4fg4KGc+mTCdxXRAu0EPSlowHQNFN+/RHDt
FyIcmeYZmolALNYxEBHcBGio+hlHna9BAXew7xnCf+4lVMwu2XtWgkQBj9AFQ4qC+yu58d4N+Ln5TdLv
5G/7PXjqxMRyu0qrVOuvIHXEQ33bwAt9wo8hHLSl/YJmWUDDu2pgBUfJBIblLTp9/JUWS9PK7gZVHFNw
fbtY4ZSMTkpGqF3kIcMRCchsuVZXomueHZ1Ckih5p2Te5/PtrTojUOxt1RVS6C19HHLiChS+wXCwLg4/
5K9oVgntlLj35z5boZbae7Yvyg+g9TFbcT5htsI8a0xC1/MdiA2xI6cBVu+1ZuU1czCoe7HCiTrneoXS
qwh0YMVCBOOwTjrdFCJXXQKYJgQms0LUNN3Z6YCipuIL+9NfeagPkWlCwDDUaiEAsOxoqoCrxcKn4cS4
FLh0AQgm6BElSAlJacswFnxYOEScLkCxEuyGeIVweEu2prldLFrV9pjW6o7wvjaKYyAxYs1ceKQeOVgJ
0CyS1Uth2kRORhJx+QG2nIPcE0s8i1ucFfrZO64obJqgaSxRn2qeF5msbsZAE5D9mtaDECo9MmYAi+Lq
ArTppU6uq3RzkRP6PYDhoGm0JeV1T/R1Z6tGx5G/tHKT/Y3aDonsNM2k7/Q1668uv+9tiLWvPwzll3X5
4UQF+9PBfERA822fVgy5kYqXY0dFzoBVo2nGIX+qWL1leV8GGnKkyEmwH/wkEibp4EdklNL2mOsli1k5
mIfuFtbzjLyo0/iDwU/KIilZVRlYQ5OWITAJuByZ+vrUx55RsqrI3jMDG3xkowo4DryarmWYtMSqosgQ
tYLnW2zwafrvVrrCsh5eqY8jYhxYHgElwMipqmk9BVhRi2n2QD9UkzH8YPl6WBPLeAZ7BiyOMWmsAasr
dQM4KZMeBr0QTFZXY3/cqaUc+rIUaC3E05SAzrp6kfeFzylg+bSOvchb+b511jgD/4dTQfXWY2jVQLx7
BDZCwJPF6WIOYwrubbnKcuL4O4SfdAIv8pb+3OAb2vBFu0zEGOpab1GLMBVualHfWov5jOoqd5oZ4LLf
4InV6WCNTURE7q0IMKdFusO780h5O2BCreA2RtIDSSQnCzSOKHCPXee0GKScO5AuCN5uheL/VcWyeAFz
coS7YLTegffO3xuKiy8nANQw9M4OuczdKfyRWDEO8XeYIfn4PSfmHCuez/HHM3VfQ7mOfHkQLxvNCPkO
aASJd1LCWX9ssaYJ1SpDbpgWTsLeEebudFDnWHNnaZisB3KbPYYMDBX+XBR3HOimUyyK7+yKk5NvSxoy
hIP56hnhRyfv4PcTHQwlyMHJIbu2Dtw7CzmW1krCaqBSRfOW3gi5kM0SJtasfpvuWXGsrTte90f2L8fd
3tL3PveB302sJabDdWTIZc53mA5mH+jBjxQKkBs430G+px/LFyE3cr5HaLiX5ONH4vS53fFCMRMmZ/E5
YR5OUFgJCbynnGYOvC/9NfMCjn9IwjuIt6YJDzo6isi2xaH3+YL63lNfORHDMGlPfZuTJAiLUgEfKD/y
EY69wFv6/kR0KZEyQDKEkFjK9JwLdMSgEEk6uqNFmHUzFyMhMoXdFOMY4bjFD1s2aRQyDqQZkgBHREWU
xOwsXBbCiQZ1FsJb8jGLtxCdIs4eCizKn8ZNPlu5Z204IV4sgqZJ9O0LjF/bcpwX3BLO83HSJxHoHNgB
lRsnCpnj7pwFn79w0FpoFOCTeaFvmvyvIHe6G+dEHJBI09H5AXJjIA67Vvp4Zj30Sn7yR3LzzvrmPc2a
13nNypxmzU80T1jzE59BloesEU5gGtCU/vmn1wgQ85Ob9SVEQ4bcDTDvVSFcuchH+4GWuWkGpvmjvJOz
c7pnaJxFhaHuWrrqWnKujHlg71lV0YThQCAdEFuUQsr8jcpJdCpggHV0BAtnT9CiFqblbwMYEoJiqHUK
mP4mloEiTxIz/vDef9Qfi6KOEerJzy4G6QpD9l9pWjsrvC2ySCQMOHFXVspzzeeOfLNmSz7+s+zg+mu2
dBcLrRiI66Fq1DRW9wJ+9mcioskg/7Nl0/xtBO9e6YMoUc05TAQR89FTEz9x/CR96I9cS7/84TtpC/pt
QSMWGfgnhP8Xnc4s3Er/1LVmodYIi/0hYzXom0Ti8xu+iZoGskveWE8xzVk0dq9nR8WbsCyyzB3Ah2wH
/A6fu8We7DudyKj6LXbcG3LuiQHHOFGE1YSOOyHhWgtyQTRmH50YmS3h6nN7leZXIXoD1W5x6G19PFtC
1Z2t/SCQLi85PucglPBsCXryVuJa/fW9CnuAHGtHAhyQC/LEnSZyD8GbpLza6tT0AzBvxiFO3MhRrBv/
ssUqSVPNZy51dq7qB3JSNwA3LpzTi1v8lnzyPlv4ZNLfZ3PttrsH1J8BEaSVLaMIC60L/jT/2YZAiS38
khX+WY90Dh6Spk4vT69t5Ifl1OK3IlSg7vhxWIIEjpTcCw8qvQE91vNhqcEc4LDI4zQ5liD+gFtxhIMW
V6y+FGVT3CnBCJRj47PwlYh5pd3ForUC5JNwPQzUOs4ToWGAUTaO467djQigBD8og844o9ng/Njgw7hX
LaZhyKrqkqi7b6hpgglBbmCaXZbQ7a5jeF8dcTtTiVccItzfbrqhE6Bzgdrg9m4MCoO9CMxR9xoIiaAU
qiE3IAHIqbXBSo0VffBYXF5fRa4X+E4wkPVe0OeVQVkiL/BC328tfXp4D7QoylaEPgGpkjCUdY5SOU+z
pdVLWtPfv036CZHhNfX+BJzS4sV/AUPIn/Gv8vfvUsHhJLQbrjdts/HUs4+e3OB/kBvv+eKfvi43/6cW
rLsuj3CcAB9Js4rBVQDnNDn6E16cwFUVP1XndG4Y7pw6f+80MP7y5ofvhY4AwLCm2//kTJWvB7exDhHY
ORAjojVdcEpHqdn+AxuLJ6aBzlwJjDQKI3SugxEKRjIk/7RCpDlq/FWDa3FUhGToxjDUBGVTC9kZfMtE
i6Km+UV7a3E0LDPYmL/aYuOqPsi99HJcBJ1+laSB9Pz4r4/V+su41n9drPaXQbVA0GiqAdF5Ad1Jv7wp
xgnnpmNNu3WwwkIBRrsWghP4V8AwsdD3irWr9NkvIgUbchL50lYGQuJO5GwzJ8AkWBHhD0BN4yXQRJ1b
NAFK4iZrgKqlaPlLhPATYTLnRT4/gX8BuBh3AXMqs0PrctOcCWEE+gT+ViOzBaQJ7hG1yHkjns/9bqSx
FWvYOJABD+R0Yarj315zHvI8uZguYPuTHeMggFrpzxWf84MiRdLaEwA1vJ8/b0mCWjcLAt4UsN0f2ZE5
F5V/O4WrgIC1YPxooLkBhQwckV+6EwuHpgkuofvDJEQuz6HtCl0DJ0TIifqrK4i01+KInXcInSA+Hm9b
eBAmpQ25RMNRF/IKol911tSktP8F2TiXWHWOfTTZgy1bE9twbaR5J24nMmSNVmG0WIANpsU7QqTyrPID
rJdFWB5LMVwJYtZJ8sBgcxaZJuxcuF8CoYXFsUXf1+mw8HLaIYfR+zUUCxACAuymOsQncXt1WQg/Fulo
OMkLuiUOfREsaYyjplZJ9POputee0BSyRDxyPoGYY5EzaL8NXbW2Cs118TM6is2ZBPQhZHRbaz2EArkL
sFxDKjTu1crx7QvaOD1o9NtmAjY/vvUmKwkzRsu/fbQeCZMC4rHnT8r+dGpvhdlQCCUOCZwMNLi2Ouwv
FlHTsAF/HWMv9vk2uLxwYiHAPEjuRy3ilkQGsZf4mA6AVUZvAYjk58F8juUbAOG25762li66Czq50b/I
jTdf+C4ntaLrjd2gTTS3XMdj3/iQsInmDbqRMegwpWQc4RfCAyPSIMuY/2tuIFBh/y//ugv6G1DiGW+L
g4GNn9JkWxvY+Lqo62JvYONbFteGfzFALOXoiWIjL3JBytlV/SGDWGsQCbwxJr5yIOtU9Ea+ozFn1Lrq
SjuETW3IkgYaho6VfH2nKYsTcmqBR48Fr5R4sa+a92If948kEFF6wi4oK+BhpJfWcvOKeharp2cZPesE
WeGEPF3iLYncias8OzyWlu42XtMHCOVpAfoQKeEwseMUawj+Yy3IIOL5e4HvGoZjHB4NhO/IKKlpeMKM
kJ1pzlNkmpQKld6uCeE8484077zPfbBTPu3Irmn4K1YX43dknjbNah0VVzGJm8awvzTw3Q2JcSlmBjp7
N98pwWg8I8SKeb9vUqnXFpvmYpF0dAw/K+/I/K5peNVLfm55K9+9m1v8d75C16H31Hfm/C+O+Jaxj3la
kx2O7KqmZU3ucGSzPCJg7wth/GMKq66HwO094o7hK9LMizAjMdV5Z5c54OE2KKIPg3Ah4chmDJwSSPAM
NPDEH4tm2YG1OF+NICvCOwNh6ANh40igQ9MWTi0KjYolTnpXEvFtso7ncxSBT26YpA8ZAycv0XDX4cC1
uh7wZQC4Fpgr0obQNMIG3Ys5yI0qIRDJzRBCwNGmDqkVIVVvAo4bkCNanGktii9YkLxaw0IWBbuPLLtx
ifsdJvyN8yGO+sNTBgqT2kldbYuHiW22peJUArpym0bsch7U4rpIkmzqxDKCosgY1W8+XUmM84YtqTfO
G1DP42MylK24pfhVBdWrKNt2Z0FKBc+tfCc14E0JDIt2lNwMjZOGumvoJsV3vPiTZnNjuc6OvqcNC/cU
VWGZHuqbFGeUnIQjNsdbYUMFFtofszo9ZIx8pp4+e2Zgow8p5ON6y2gkCoF9p0iXjz4Oi8zxnnaJt2GR
JWVxPIhs3ZtWoi4HBWq+H2Wl8KhnjRzv83HW27qU2ctnE2X+Je0HHW+JDQMbht+uM2oXhxp6QsRzWuQ4
ozaU5p/quChq/qB6DM9UXH1kcOVNIyixhddoPQjh2qt0dW4Kf3+4TfdC7EThxgQ50zWNAxG5E7GJVA2e
j0dSu7LDk9B5t9MEpSASDocRZEeOzwDPShwV3kbrcD5HYstTL/SxkWRFQLNv3tPMAAtigYeCcRpCgOEL
Dt2N+Qd38zBfayZzh0tebJQXLpyRQKJudQK8KmkCOBwpDzFLXPSdzW+LdS4MCWJCvdzHsXAcGaPJm4EY
ITUxexxr7mRi34lRFze7oEJsFSN0Sgh4ZRgE0hrbJEfpe45pt8TayaM7Rk3jCWhF527rU5JRb+s3TUZt
Bd440Yx7U368lva23mc/lkwZU6B5yg/aOzDB1/y/JCTp3SKv+wEmg9D4Ccl0VeREdzlKDENIAaQfDjXC
t+wRDkeIGZGNSvDF6IIj7r1cuqOIQMNH6ivGOALVR9AIUt4Q1DTviEZaxqOjP+Y93lNrOPUxwobAgQZc
xuTUShAO0elO60vi3fG+3KlVVL4QDdTZkcYdjZO1szP9pegyFAZkGFQtugALIYkuRW0Lp6O2wfEAwWyG
ySpyHTb6GHbhdCS3GmiaASmEcCFcd77Iipzx/cV/YU1nSzR664BIefvEwcDinK8/LRl99nh70z0buLDz
Aqp/IUoRsG+9UPPAGtwSR+U9n/Jx0N6SH4B37MMNruRJui+OFWsORZrXrGxCYem7Z/mxiUqaNFFZHFAT
Zml4d4NrKOO9s/1rxLkx27LnqEEaOjpSPdiA7rNLj4KuueASYuKOMxi4ce1VkNrLPrNAgwVvh5eW3W3S
6YyhDUEoGTVNiJXkGfV3mgGC+rdQf+BtfRxrtFUaS/OXSPidIYS5FiOcDe/qcmSCaZ7fboU8d4SjLq98
1brCiWpwIs7Ie9pt6lkX9IjqbvBj07QSAkT8uTihtJBdxLFFQRlnSm+wxcxOjmlEEvgB32XwXsLPfA6q
U+eSDfae5TVw79KegeEI7l9bmUZO4gBzTudmSoN4APoxpfzx43slWUTSc35obyHYSynU5XFIYvUFtHKU
xwCQ0F2wC7unWARIFgMNRwPFVkruRecr1DTaG3jesxJyL5sU19bqjUxEb58gREreLzFtdZkmCSshpLoI
Du+qJE7Rg9G94sc1LSAZRpbjSiBYDKRfuxmGj3fje7fdYoG2pJaHZ+DtfCR91x/I1lv5uCDW1nsqY8xI
dyS28keCcA5e5lXnwJEDzbzcb5pTi3NiMTezI5axhJ9pHw7MyewgzSO4UWqaHF8se0d6ETT4ucxxUaYJ
1HEQlzcRlgvshJgvkSMWDKuldhjWPRM4sPAX3BZIW1iEOUKvDjRkTiF9rdgGavmhaO1J6uV8fuQTJ4v2
3eheFMe8Jkuc8RPieDBN+dD7GyhwgmZ83zbNuVqGaU6oauQ4QQhnPIHPM/9V1d0hfKcAXAHs8AMR08Gx
hbtXBhCj/s7neInvkCOpjzsQ/MNyiM3JR8kZwMnb5N+3S/s7N9Mc7FnT1PcTOv1nQJvG1v8IbtHpY2Ab
fQJsBQBAi1uyhSBLozg2m42NjLmCoc3GtlzHvt5s7AYZaG5Y/OkJMuDGgezPHSLdkb0X+3jGTPMwI+TO
VtDfNCAr5UsL38Xab01zthVAfGd3MIyaJjJNka/qIvRZxvU1aCE1zaz/DnA9cICklxnBzWKBMym34YAu
nnrQROvENGf7/mKRU7C0jIqHnGdXz6pAgTvEKfdHqWtcWRTnfQ51dwJ7se20PnJ+MqdILWl3VZHPOXAA
qM6WaD2+t08BKrvchmjkSgClAe6aJcKdMsaWjcXpI2gXKdUpvhNAvixNh0Z3GAjviCW2gnCgoFoDq0VP
IHxfihbHMNqlSgku50hIgIUX7HFLwE+mXqhpkXohKHsEPXiDk35xHzC7sw8leylH3DSDV81KORDrhE7b
rn8S81R6Lrzj5LhiD6yYbL2Q8wemOQvstPqxLA40Adf/b+ricGCRxdGAHR7LkuW17Fhss4zttXjwVtIf
7ZUIGicrfL3fsyilNZusObDLbmNAgf5VbJxksHGsQDbyQ7AjCQ5sfuaQBH4wI5Y1Xpmk26K+MD8VxZsm
Ub1F8tCWQ0q1S2FOCoKJxTGrCVMmg1YA1tIsr18K4p3zQXBnqI3QQqh3zGcfiqpWK2aaw/fBCmLVHOjC
iNm8rFTAgZqfdykJhpgA7wi1RaAHcIxsmjtda8AygC3QPddTW7jGf0ZWUhFvNyPCU8uO7AZGvCDfk06Q
9GpVrbPe3/6uC1QNyqJIyDmkteepxcLLfAo7IiIgQWEk6jDc3LgyeslO4jGfk80e80k0oBPc0mLCAYN0
PLADq0DpxUUkydCiuy60KMKiwrhzCB93qHErfXpziHB22kq03aLupHHLbfDJUoFyygeeyIGw5pAyWtdp
Hb7SBmzbKxJiik8sP+6ZUt8bq/OBCt3YgEHjApR+Cd8XaU4zqL/TF5hKG9z3fLT4eYpH/ZFm4cWhysvf
T4zuoUxr9SwVGkHI3OI4nXb54HUKmr5LHfCIb8sjDLVYIgrnlBU0ck558fUxkHqRMnLJSRL+E7MwI4Rz
wsIPhA3ZB1MRi8gleLZq8YByMSAlzY0WB9mx/FgbZNAGzz1ogn+43EJxrI0Ww7681IahJPIcHUCVAiMI
LUteciBOFYejFN4MegJ5ZVc6KfUUe6vJZiWSwgaFYz1gcVGyYy7WQseSw4NeIWgqsSXHXwOwAzXEwRdb
tA6Clq4cattW2AJ0dM2ZxvKknjlvcEr9XCqm2edVDZQFBg6ERGbXomB71rnQGXZfedMBXWKRTXxIK3kK
/SjOJBaRzsdg96lpeg2zs0Qxln5y+CnnHqnzXjUpjny1Vqb5uTg34E0PRy+/9GeF062vAJABCUGH7yIL
OBllUZdl8C5vosQU4EAwjsCMqtNTpKd79qam+wMRM6peOQGbFw+WOOCFpkqPGuB4GmIHUB48Q8DkpDmJ
cmQyPl8GPn9T5JT4/hGqSGR4k+6PMHZntsJDeuPcvPUcXtaXgONIMTXNmUzuWoEQRCOqpsUjouY/afh8
XB9r+ox6Em1PzdJ/0omPzPKnejNV1JIQdt5bMNIBdzc0q//KPvDTKYCDBJxKhXyvZ92RtqV5wqK3xREi
hPAvdZnJUhGraZrxJ1iMH7e0gkJ7VlOZ5UAT9nf18A/+ANpsMvV9yh74rxFuaWnI9soXfDvOlvhOZLpj
H9QXGWupexIdylKW13/vH6GZIo4rJr6KR/gqJdCvI+0FjiDesbBkLP97/wglBFLQxl8XUnAsXrrvD9t0
ksdT1Op65C4V8ptmSVVMKfAu7EpDS1tNhNs/OtTu5qIrryuVV6O6VmbgrpynZuB+7nxhBu5TZ+nIggIK
lIiVAwjqwAKE9XwHlo4Bz8V7VhoYHjNG3zP1+VgbahJldvkmCsgXWUQlwSk/dmkzZISoT04DAiHASnji
BJJWPZtp5bsRszEuhngGHSvWKaHIiLOgUNrdZYGvm6aRU0jijiXT5cHTgm4sy4Af5jN1xkGIMKXJ1Pv2
V9qE8L3FRc5+d3a84gXi+JJBDCjZnuNM/qWbFEUcRfpM4dKiIy5fCPujns91o26G5oZtzLUkp0/CPZ+E
o46ZxZ1zzTNla+lK5yrNrygSODOOwRc2veQYMJgJlrc3le9dq4ITkgAHnXphqPjjkLyXntQuXkEMtJtx
KJSppabIB1AHcWe0ZLQJyiYssobtAxY127JJ90kDxGeTpfldw7Fic6Al3SPrsgbJtXBeiTY3z26SFP/G
GxDXps0tqOQ0t7y2mxQ/p+RGXvZtqmvLdbx3xG/IprpWd4A2uknx15TcvKvLI9vcWPY1usEv+IdNdX07
s1xn4714+fzt843XLBao4R/8jc+fn22q6ye6RclLOiANB1oLBrA5BhpQ3+fO0QL9FtuoSwNd0rcwQBvE
QN7SbxrqaN5XvhmGAoIdZwmkeSmQ3dy4MeaSDtVqetXrr5GvpRxYYs7OsM1Ve9pb+Y4io89a0Gv9E50U
feBUOI8YRbvkTM1Qwm3FmkI7wgkR2hwBjhHekbiXdkspppIN4aS/Slp3m2cnXSMucUR2HvPHqiL6NVuA
GeZ5vNBH7a/DTm3Jr1qnUqL7x9oi/KvsI3hU6Kbiz6PIHxfCIfYOyUPTTEdHWKBgmXRxLB1ZQCi7Geo2
G95B6qbfU5OhE3NN8++1plEaEHV56fk4EFubL1yv3bLE+96INSf7xQrfg89jXA19rdzD3UTVNPtnqwmL
vHvTnOnX+qb5XI74XvO7PkRFnUMTarN7i6F1ZZoWhMS91wRyDMegdWIhhPBrasXqMIEO7eG0O1CLY9Cl
P9LYmK0wGMTGhOlbdAXR4XtNFN3OJ0Y4bppIism20vXcnlpMU/P4BoBFBYpbZ7f7dTafox1hmO/XHOKt
lELhwNrh2RIC/qQcjQhNmC3eU2vX14gQDpV9bebjHc6QiKYG4VvI1tv2nndHYxQd3OJXFPH1XGe3qezM
llelVE52usrJrNuKu6G61EC1+g7vIICTXZWhW9r/Yu9p9nOZ8TzqWSQi58Dr71VxOiO4FxQc/d/1glgN
pfzlPBYZZiQYOP1GDsUxWa5loJpIKmzG8zkKm2Y1071tA8GTMZrD/t6L0F/RIMarFQ5GGI2VexHo7/Ci
2toM6hgo5Wqhl6hueKcrSU37SVQT9IFi4/bJ6tntzZOnzwzhWzZnl0geKW6mQ1UWwFsfUYgXzn3GKjFi
9nS/9qvhB2GQJHz+WxQJkTQoQG35jtpzFCrd3cbj6HF/plbsRT5OvMiH9gPwUouE6DluGlEaNNlEhRcq
+tOgIrjUgpNo2/vKgB5pmzPRovbk1ErwLDVN0KDUlnQr7EnyM4swPWJYBDraAxoegLHjTawQ1KYFPKax
9RYM/cFBdej90lvVwid1xPWm2OoL8yLfHZFlEOBreNsHWjXqtm+tNyDpvzb0ftWMsC39VZGI5xR8BKFQ
Jn2odgS5ZkP30YxoIqKvzDO2IKTn1t7UFWEnhPqy1HmGWMfojIpdyauRMfyOvn01/tQ0QrKnKw7S3oyQ
nhlWoRYL9bUJNe/XdMQsDYYnyaJR86uJb1+NPyn67WU3seuhFh0V8vZDyf5/0bU0r1hZfw2iY46zBi5j
eUeFVPk/7qdwr6oh8NGHccNC6Z4vWVwPpez/z5obhArhTZ85v+z8jEN8A3mSCZGZF/gIoh2MY4xYo6OM
QqgCzooPFF6HHOPE4aFZWkkJjGlS6SCEkMClTiBGwYmISZMioGEG9rb8ePudO1yAijRIBB0SQbNr1nVD
2+uzuPldQNtO63PS9QU1zdlvtHfSO8uo16k+04+pPvvoRMlYsZki8H8Jl7CKp5DjCMU4VqM4Y+MFC8SC
6dqqfKeQpWbO3wYSwhSSEzsc7sQuYyNJOgx9kyo5rNcZI30E0jveZQTe615LWm7t2+X5yEQ8D2GYKPsi
ldBlqI8WHHb2cjcxqreFY4gnQ6Et/kk+GljfWo4h8IX6+hx2swGb2lAT8DzLHEObDOOTHqXpwB5KxBwC
9zdxF9FsscIJWa6TWxKvE05kQtjEWLOeFRuCk12lxbzER17gWyHCSvEwwqHwDT/Ym5qP6Kj3EU3Jzbs9
LZM0v8Hfji0vpbWlOzs8ImFyOVcWl99Rcr6Yo4AWHX+YsgctgE1oFweWsxJERxSJzr4o9odjzaI3YJQX
oHata6B3BHsgbitTdEqlGVVYVW/ZY02MoHhcVOlvaZ44QVFGrFwExeP6IGOmOSpw3VraXTlgu7YWY3fo
sS7WopizOjyuDzSKeE38uS4Ozuq/1hDgzPly+V8GTnUFcAPfD7Xft9KrtRCaDAeWonVIjNV/GUJ7tDjg
hBhPD8Im3Rad+ZbFNWbE+EJ9hpaxGrDIBGauxIDuxFpeLZX3S2cVtpxMBzef7SDaO5lW18fphYR12pno
qT4FNLxLyuKYRy+y9EAM6YOdrwCfraHa+3QRAxewyUvgEGC2hvUQQqabw9tzUICFXMo1+z+Hx7WIUucs
YTWX64zFtbP46quvvtIWeymhYQErftCAhwbgS5QZYo91S51q3hgKfDqkjyxTUfomjv7AgkCCQfH4BgD1
J5alF7xP86ysxVDjd/2aXsgZc5ws6vqug6ELeZO2Rai1tJgO39NLTB7Ma2/tFTbNd8A1haC4DhtXKW6A
EMgKUNOEXuBjg0N4MrgAOGcLQRVbM8RFCM8KezxmC5nmt/JgTZBp/pWq4FxgCrCV+4ORrb1P81/hJeYv
9FG89N+1r6ocSTAfyYPMKb5FehmGtVIx0jTDEjeZG4ajRQT/YSDKPQ38WPWEkCUD3khBo/Ie5XRhPUiA
Lqj8i737Ixhy5EXOGpAPW+5sEXqM+sieoxv8N0pOHewaPfC+T6s0SLO0/uAY2zSKWG5ghQ6lLW+Lf6Lk
lLG6ZuWbAw35vjCWBo6LvP5V7CHji+XSaPEbSjzjVxbcpbWBje+K3wxs7CvDx2/pBcQhwan3oUYlI0A5
L/qW9jYSytnTklNKPx8OilKaqzALK4RD8oae+7YB18FvqBf68wCP6+3X6ZchyBNlXR70Qb5cCGq8p4/W
EkfeU39hhU2zRGhuRWDLDobrTtDX+Sud0LwmS2kQHBJCrMiVuMlwFG4zkPuFY4jomeAlY+Us1/HtF+t4
Tp4iQ2AkZfCczDuHAuE8AHvp2RJiSESu1dWoMi967wMSxRnDQqr22XkB2U2Rf278KqJ7inLI0TsyWXf/
dXbW84/XPRFm/+9j/ERmSxyT76QAx+iP+4HHhQ7NGni2wjEIgcSRDHfZP7GwrrqQPIBKIPVrfq6kedLn
sjiBhXB0S5bCtFz5YYuIRJ0xwlZ0q6XK6oTzBV5YIbCok0VHa0YS07QK++w8sFDTRL3XCdE6AT9hr7KC
1uB8cdmFk5wryOP4dArCwNJqzgFWExGGVSUc1pwKvs/rD87p3O0eyM0kdQdjNWTmju8DS/rQNVaGE4Ie
VefFwTnRPN2DbsTrmpXwAHqpQqEvO+771zjNsh9kN/hrxh7/VBYP6vnNtkzzO3jr8dBsibM0Z3/u3oq+
AkEAwMNhS4U2w0MaFQ/w9NtriAzFn4piD0p+YQW6KpVzMmI+xXz6qgpm22gxLMPEtbG4//18JMX8P6N3
uYyajw+8HbvkS8enbSAAGToFZrHW4J38TK0tapqtcGKsFhM8Z2ivW80mOnQT0zQSVhtpfpVo+g0WI4l0
RTRb4QghlzmpF/iOFXeWb5qHODBbYx3KDBEyTRFgzwqJ9CbCOEdj5AAHBsKC8ee0v8RNKknWFs7BPf65
j5Bt7yMEnROITWPIuylQ0+j9l/UEogHWYF7gEyPNt6xMQT5rmkY1mgcCstREurfjq9uVDLszHwn4dj7i
vuVsaf+vL+hwFfXFW+K+q52TDomiIgTuO8o9zaQDD/AB+ROFTD9BHC/hFiNsmtC1Yh3fMHmvv2yatHqV
5mnNrBi5cdMsHeGsvOO1PUOQ2gaWh5p/xhNrYyNjnKO2lfLENvtRIs5zRzqoaT6Czi/ickHiuX9X0+JE
/OlvWvzWjlDr8rTozDHqwF9OZJrqNIpMU6LjCP/ek0kzUk3O95Z0hGOxnuwAxVZ1NoBFpeYWBwNxE+JE
SdrVfGv8JfmBWoV9zjPgCfQvI6dr4CBOgr62fi0mJ5wzWwuY5VNfxlm2EzP+8UpaJE8xTaNK8O+GgRUX
ZxhY8n+SsmgvAyCdcxAUFxSO7ui9uw6EO6FTy7HZuTVwqKzXrgzkeKG/jm6/gDsjxmumXuTz+mMv8psm
9qLFU/hdam6YWvzXTmSoMELfNY6MyC/gA3xwXzJGQZflnyO2jo+Ek6QQY7L3GyvGG0mKivU2fcktAxlU
7AVe4vs9lHmJL84KNZS4HbtiDV3dxVKInB5EW1CwOveWOAw1+I8BPa10/NjD1T9or4cLYWe1fG1pv31g
LCf/oFjPN9TX/QfFvJwz5V5b+XXO2F6pWx/K4kBCpctUpXlC4G5SPPfeKoQ6FbhTqUigtERpWav7iwei
VKCVEinLIxKJR/ASFY8OwLA/AFscHstzeasY5UFATdfdDsaoRIVCH0jea+lluu736S0uj/lUEOdPNaZP
gB0dBc0nQ00WFQGzPJgzT5tLX11jjAteU7zEq+k0eTcnalXXIMUDsdSsLvrZR9fBvH8b1lfV7CBF4Pqn
XvtD2IOp+lWAV9MEbxNu2PkCvTSpXbpw49/iM+jVgFRPw3p95NTZUoxOTLk2nVsBoD15n7UbNKX5yj97
olZfeUCQmcVhItPcYVbpQZpvXjEb4jvoVQSmadBjXYBA0w2c5dlJCVLwR5jVrv6zLxZFzsWujzqpkU2y
tA9en3vUOTkKIMkfnA4pDYYCaXOIpgfmIIPJtytw+P+2OJCJz3CcnsZjHg2Eb0L4MFAFmeqg0iIHLHPi
DA6dVuBocfXAz7vzNPvLBcguwqKy6DU8/vga3TzVDHIMKGtA7PZHcgaSuFsgchK+s/9J8ROK/yWdjgiX
Yk21LR6abRox9OQG04Dc9H4jn2hORYLAQqcnFAZcsvsjq+rnijN8VYITqACJJmswHtJ0ycJAJw6mA3L8
Uzm3bBH+JyXSvqOvIwrOwyOfpECYCpW+QApexAFOni4CFBI4wTHzlJhkHvqEeb28I/QJ1ULWWsyWHChh
Uo7IT9O+HyyY0jaytoFd8yOLlYLa93xkh0Ue0nqQZFwbPpLO89jYeR4II5gX+wJxhfx47SUNfRfiYFLU
2ynjSZo9hcgDipLnb3gvVMBzTkIUilnFhwGjGwL90Pn/wEb8yEEE3PsAaEiB79C1L/glxmKjJ/Yxh8TI
NK3+hSzxliSa012sv+i+UfsyTbO1OOXUf5nP8d4WUeh08Jn61hdaLHDvsRi6Kue+aZKhF2DUSu9/MsoB
xLAPOAj9iyofFkJsJMS+kCS8Uor9BHwZ+I+xDq7B95XhGGIChY0ePM8IJz5m95o1170X+Sgs8jrNj2x9
ILNlm3uRT+5N8x6Iz54Si1CbxlZKJrzk47Rpzj7nCJ2yscN5zrHbxXtWxlnxQLyie8b949+153/4eAed
6Tw5itXegZZiBy09ZwfG9WN+r/c+ecfLuXdk51hbCt7cIAzJue/W3XRFohBCCFtGmnMcC7U2jXxbCFm4
aEkcnrtJp65CVMTZsRRcNED8Jg2Mis7L465F2qDvSJeCd6QfF6f17kDS0XmrHPQIlDTVrILssFuFTqA/
Bc5avr68t/T19dJTVnrKP/SUp36LEE7JbNWDeo746O9d1YU0v7o3TetA7m3xBTn3ui9xhRbwSV0+8NmJ
TdNSBcjsgPDBNLXVPZ/cQ9PIhcS6m4gO52j90+Af852BxCBYYB1cvkOcJY7wHmHIfs8Hw7dPKgnGA9x0
ckpdfZG/ZDnQyk6CCQV16EYozC0AS+vSIQhLJvEAeBXsubIYCWVgvhYikXDWEeFQhuQBl6vx/0fcuza5
bSxpwt/nV5D16kWgxGo2W3ZszIAuI3Rk2dI5kqVjybPepTAOFFAgwSYJimTf3OB/36jMuoJg2zMTG/tB
aqIAFIC6ZGVmZT4PsywaRdbxHJXoKkLrFrxF9LFSUhQK4oq6e8vMvWZF8W3bNtZPLTIGnM9SY3oIAHv1
kF5PFxVYpRYAkYDxLB6StY923TNWDYYI6EZHGtJAK5lqPEM4AnGr6H/nbVvA9ir3tm2W2E1KZRgtrelw
gbjzl66kbSes4lcXpTKOWc2XuOrunRlcgxlsymfzbLy72cTObbPwuATjnM2WrGKqQ6rvrqKoToskXgR4
3TlgELDhFbBGLyxqNqIG5GwLvmg/F0BQ1mwPXtlwwh51SOhr0BaTxyNDvTE5MU2PrKDMJEfqLdta7hNh
Cz+g9ZMUzLZZYlvVNFRS2DZj2BTJLGMaFlEdJyGhG7pQtEke52ypjCxgUNc/x8EXgPqjT+AH2Aa2XaJh
GliJuaE9FLwTVnKRdvowAaeH9FPVMZeDDyc2vsr2b4H9e+Wlq8SnXXzF1Hw87VgmMkqThU+JaYq1EagW
uCVo/fgWcxFf97YIDTW8YEZZbW/JcmZut0pfSNYFyaoUIql85WuJRpB+MpygcCnfqbe7f9B3gve4dFyU
63wbXzMJKDIhRar+BjC7o8g/NIhCS6vqr+UutsO5Zmbs55t6nSwZ4vX7r6aE9tKx4OpT5liT4epS9dt8
lyF+0/S4ulT9tvJHl+ERPe7G1kBxCTkLwR6NNp48kuckmfVxKKCHx02JGJkMtMcvjwtteDK356wmZ3HM
jkxX33XuBdzANNWA/mgRJDl3bP1TP7hsAvmanWj2goNR4xsWRcbDQ8TOCoosV4bAkF89BJNZJTJ33GX+
SMMVwNSR0yQ8AXPaROztt1KWvM+FKYHy4ZQ+xheTOU0eTX8nRdsOiygSHZJM4Pd1Mi03UhMvNRRSHnBK
FAmPiLW6HzdV1balVlZSaYUinyRmZ8umSLqzYCKaA7XGopWtvnfvV+IVz1xxlvRfYmW82WaTOFuiSP8Y
4sZNrA852lty3KxKLu3kYO5nyLPis4uqe6iquFmVzkWmKtOP7NBm6HJ6ZPIYurCrvJSfm/OZuAicgqGw
RU5B4bZ7z2xigMaVVqa0CJitMrZb2eLIcp3wq8+d27PjXYsnR/oZaFzktu5Qz6DGsUCQHBYOPlbRaSwN
GDSipFT1pt4vCFXDCKQpcG86olU8z+dMtm2l+wjQN1zu7Fy7O7Fp9UVsTrtrYBDT0pcwry6fGrUPRbuI
C/s2vfQhmN/rU4iIKMp1TnA/6Qk7y/Ii+FDJJRcJHpCNQMvDwrCH/EzrkcbFG6Ga1P/w6lFUAg4UdQSD
kJo5P70wF8YCdzfh8m/zfeTFxZRW6hYln3XShvWdRhGcMhNKacdQoEYXdmpBmeDDK1YZoD7JriidDkUU
FUoA9fDJYM/3evd088aWreWpNmWF31ZKuZ2ZdiUZEFSEzZyF7VympdOSwA4wYxJIPwNSHtW5IFqwXfGv
lyupTDXhGlVgowpsVJ3IodpSZHakqzEGBa4tVS22HQW0I3rpJlPx3RzSFcqZyKJI/a9fNjjwhJMZ7uaj
jjTYrdauF4bOFYZul4z1rewQzS2yqf7rr1HBFhW6rNu2j1+h6I/rw+ltxFgpYgEtiRX7kez7VV3KH5q7
TVIK7RCiDAp/3UIRvL8u+ozMD6pYfyZlSvC+3bigH6zjCOUfbg7eCagJT+iK3Dld3fHPA95Ppbr5SmFE
NHwejkY+y5wjuCt0IXsF9rrxYhgTzuE7Fd8VdtyNRjTnEImaxzomFadtYYfVxQW7otPCuvO0z7vZxuBG
1i5lT1kNUZr1SxjlRT3H7Gvku0NME3eFqlJXBEgdt/mKX33D3NX+lz5TsuVZzs/4ytMnfOgJeMjf6kfE
tiHDR1PzLqDoe0/Ox4hQc/LAvuL4Wa6eBxE59onPcsqeYZ6PeQboKPxxv2rukv8xmbAq3x+SF5OJ24v4
djLRykEpV/lDhxPci56p7lNfNVLGokgEcv3AyuMtRh5RvrfWBzsIhdLLy75WUN9krpIaVu4cHH4XyZ6J
01OI0EGQW/VJoHzcgySUTjU8gsNlM5j1HzacILoiAN+xRpkuiJMtS15oOBBZsifeUd/LyYEYbBUNtc+a
Mfz4d3Oe2ycZYPpasKXgGq45Pxx2byAfdBrod6r8yRgFvPVsIABzgBD/Cc42d9M5ysST9wrdVs7HrTSO
b5D96V/xzwv1h56AhPOA98QiWKRgQW9N8EOs8d6DhGallHPdhhgQ1UkMg2AADxZbLSgm+D2tRWJB5//F
C7vA9YcXGGMeNArQE0sX/Sb9KEBlDZjoN0FpWiZxHlIqCFaMiBrGWMncVqJRQqAKQyoJFZRc47nn5vHa
C16mhgUYsrVMCyQIKP6474tEHfpjM4r0iEU2xxA0xQAT6oVbD+Cpv494wjWhaTDN1FDqHOIC9g3E7nai
ULq9tcKVsnpC0qtj1LlEXNxTyJMChmot+GMfB7eVg2AadHq10KLfqwx9ChaR7HQYId2efunLL3ejyznt
VXuWQgch2o6cQlFoowcUdp1xbGPqldas6ptnDP6H7S4dHKprSefAwKDPV5RJk6t2LXDzGYF/UMy1BqOk
RSgwYJJa6QvzVp1RRaFs2p7CrnZlEwYFPC2bOuitT8smL9VjhtX/WN/P8qxt86wro07e778mowZnRI7g
7gVgDQWMARfWI7KAsfyvi4scghH/omhQFwM35tZM/FxgWHZ/gEtHkJBDLiDcl3g+WghRfLs5KJNhQpNr
YRBvjGygbbs6LQS8rp2s0klycaWmvG6d5JFUzY4kZHFYr35sdoSRYpXv9yTBv+pmorouWIJBZntBImZl
PvNZeZD9aoMJRBApEnDymRqhtRjqWvs/r1sAZnV48199Ssdk0j1FGNnJvPywWT0QRtb5/TuYIqqZ5Gql
s4f00UcdNMHIrrn7tM03qrxZ6V83e/k+3xJGKqVa/g0TGJhJYHhdItivb5ap1d4MYkRTCBZOsDP9YMK1
8JvFZ7y3ja75AAbECx3ZiAAzxV/gdVydT0oEg4O2LSEhjV9elq/Uub6QugCY3kZmer5Kel6sKMGlifbM
I+I8QHnfCJ0srcGK+rLWc/p40iZmsQJ5VWsSK8k3Ii4oK7mSLIUXDkAGZLQWsaQj1XyPc493Sszm6naP
XHxARhVcB6nd5Yjj0XTB1yIuKZNDzhc6wO6kddmCHo8B4oBOcf2/3L7eU/6kiYcnCfOB8avEl/kWQv7f
dgoe9nXN9xdXtOSlBQmyZ9h/p6vQa9DtKm/rRH/+9Bw/pfBxv3iRIqe0N/i1IyXoLZp0N1B6urpwXe29
ZdDVhe1qgOinR5o84ZKDYRf2Li/oI6qL+lGwmPd0seAVKolyvMj3+CaCpjL4LqXJuy8Xem/egWJHkW0/
g9/G3QdE0b87Gnvy++92Qfv9d2IxlPeBuDspsh0t2jZHvZSQxHd9h/WCZKQI4KE/q3/G8slUcDXkcjXk
wnFf6nHfHe2xHu4wG3DI23EtYDwbWrOpJTLTimUj+OWX3eU81BJv89U5eWLgRKaQAHZmupe8M+xY/7AD
rXl6gnsD0R9lGg5AM0Rv81VMaZIzw2ImOSGJS6oyW0+pHKkTLqpEYlQJ7uDKPigk7b1UPZmPCDlSypTu
dZuvvKBvzXPSLe6HIcwgSFirkMLH1XUjUDICZheka0FdaIVJI1UdDIv/MrLnTeSTrzE/fY244ALGrHsJ
mhaJMtW0i6U39cOIxgbIrjVtXKHarTCJN95QsqmWsOX411Rc/TI+vPEQgqzXItYwVTmlx6Nm1TpTqx6y
hhB2D0InVAPnXLuoLiwJOkSgL/gcvAkJsIvM02p0lbjQU0gZqb6bpHUyTysInq1hj7mu0MzNWOzcUZjR
WtEoGhaWDySK4mHha53mBEQiWovev4QRQ3JLqEE628WFnhNsbsfJVBNxCAv8tzhJ4vKDtYIWUhPkWpr8
GD/i1pHml1wqQzYunY7v0GvcgNS1Iks1ZRXIIthWA9Ijja0AwMhBr/CLK8qq4zFQvbWTznkFOypxMB8z
3u9HCVJ/rG5r0EC9b4hz06wgQY9H54MEKyd8GgAXnJUoJxCyenSnpNmQxLgYqRbJW23Ca4IM+NMaLgtg
v3h2ObUTTAPXsVNuCzX6fZpJHegMJGh8Jtu2zNiWr1DICqZRZ1NkckoE++qds/DLcIEjKHIMZskMCJwW
vOaSq8qZss5lJzXYPx5utTm6HZ0QDKpBsnW62Ri0siiKv/Kt90y25V/HGFdB2VfNoUbZNfduTVDXVi09
2jLBhU+GIgK6gy07ZeGMIqAkqvef8dV4lb5IvmFeG/CvjpCP+exN3Lso7aVh+/qnNGzoD7IMTCi3mdCk
EoBLq8kpIE3VCOKZyBJ/GhcMPBubE9quLcJpVW073JjWb1v7U2/YSVZoZi30RVZRNNyMDVGMjhn5n/Wm
bO4gzluJxyXfBFR1bbtlpsOXo61a6xZ84Qk3Ol1MOyUNSjEA71lMa855LEMglbZVQ0VfV/ugS21b66rw
xdo2p0dnB8QL3qAh8BTrF84FPv/+Kl0mG8u1p75lbbjSFmeI0gyt4MKwtxHK1lG01m26YAVla34dRYvZ
deadiaLP8YL6vFvBLbYUN8lPmbicTwhffqt6V/QQYcRU9bTZiooi91s/r8HdO9P5bTv8HEvattfoV3Oh
MbMtNGMwCoBGUapPq5XaNbvOYHQ6Jkc71/mWqRrivlN6wNsqahoQhO01Z8WZKBArJL1ZzgqGfvYOxcjk
SKed58cl4sAJeoLaeSJsu/w2Z3HfTd3qRS1eoi58o3lDz4XWKbXbyw8f9NVZAFqoWzeJWjEGsHwM9Ioy
MIvJYCf39R9ygMlcA6AvGpRihT+AjaFs7jb462aLf5UVNrAEDgPD2TBw/A4Dx+kwQKaPAa7ug/2NWNeH
wbV8gHqv5cN2J/d79eNmO/AInImXXdy3zR564N0WwYnLeoIGLyyHGtSvMCw6utV6unehPu2JznXfqtQE
97mxADEDblHd2Jw0G0N1VW8GuTuDWbTFIn5Eli3LiIWEWD59Vd+ACDfktVjX4zkWzBIPhWyQ9Dg9oW5E
Xelm282v1UCRHYGr2TBsTkPJBJ2qBf+UJzbH0cj8a1ks23ZCR1cAVouEm//1B19cTWUaVC9pEpe9zFTu
bXTChKoAzBXkHBc8H6+aAiO8d8KEVrC9MpFTpXKBj/239+9O4f/A+SXa9iSIzEAIgCYIqJYFB3GUj3/4
8P6jqnBHseIfd836E9wO2pi8P1zer1eEasjKkj4apu2jAxQcFmeoDqDKndztmp2XtrUbQ0lM3m5u81Vd
Dn57/y4ZkBGwi0AzHNTXzr5kzy7ZDfgG0i+byzm71UopzmC949TW63wu253cy0Nb1SsJW1B3T+5VXcuH
udzQy9r5p+9FJy7xJEdfTxThgWgySR+Ltj2YLQ2alnGuel/VNiIzMjolALFbMjIVCVGKVkaYNHj2hrC8
aFtz55DzHayi6iVKjGYO4+0Etc+TUJ2YyQxrhPGSr3n/zinAX/ZHWoXEhSKmiZiWMxOnlnG5KZpS/vrL
21fNettskMdvRDgZ9ZwJbfOwZXOArlh+vZG7B71+f1zl9cZGg5qGDwBCMB4OdFvm3BW2Bb2cn3sRFwxy
e1R32Z1QrfVGhHYCY/dyV+er+o8+UEDdmrH2xOkL8TsoJF77RT0V9OHsYjiLjlbQpLd6KjnfQ576qrR2
sx6pidDtJ+NSgyZABlDNpdpYe7HqfUwSY/ETGkV3eiAHziSlVd0Kh6uLZwsDIT+0vBaUHmn4cf6I8j1n
IWEURkx4TrKCpuggK3ocZI/qIxI0azT1pMPSvxGMfNl92RC1/iU9lxb9lyJcq6W+Efzy/38xuZyzPwS/
/P/Gz59dspeCX8azNMro73z2H1H2/JL9DWTM+HlKk9ngyyF7Hs/+Q9WYPafPLudr9spsg4vm5tDm2636
d7E/NDslsMajCxhy+7rZgNxSIqy9q0ug2Ht2yX7Qt//0+nP75vXLH+izS/ZalX25/HJ5yX4U/PHIfoL/
3whOnl8Sk7VMnhPK3vbEQeWETt8K2GvlX/Gvk4B/97fawhiw0xXFhiWrZ01NKvWEVd3Qg9C93t39KajZ
A+HVTCpLiIwI57ycTbI0Lnlp4QbbljwnDJPucsipnWXUZjQUlCbdc2COFX5+4D+6Mh6BZXLO+U/CNcQ8
XuD52maNzBYZxuKiJAL4qFkQpKpvWfKFjlk/F9G9bNuqbeVsmaVVOoxrvqQ6jCiJkUhZWXcuWWNJ2Vz9
N7yiR8pqs+zO/Ytnk4y27VBCRkoUzWEEuO9+182L5LtxvszvP8nDod7M9+NqlR90/pklzy5wZXHO4lmR
RVEcy1mRpXlStm1c8scjpbMig5NOtnqUkMMJ0q940I/vxVlKC9xu3gMYm/04bSyT54BAqz61tv4Wm4Fd
asC/db3Wlj64lH+R+22z2cs3Mi/lLiYaNv3iM9InqfFYUruOLoA5H0LX1f8WMPCxtr0h6VTsZH4NSdyz
SVZvBgWt4LVg5XHkXgW6KWrkeVLfdit3kDYmR2RARupERh8rLnWNc0jOl/QIKelz9QQbSVIN8dOjyL1K
RVkxqzLXsD+LXjg0VrOlGujXfqPqeQUtcD27ypDZA1ZM71XpcjbvbuwHnzLPphW/Nt1htjcBBFQZyND2
P9ZyVe5nFdJR9JRnXFBg6ADWR/WKP8KqFkWx4H4BWBTmEyCBuGLe49VjcZRUqkusLkWA0b+OohqCc1S3
zPlyVkM3VFnbLmfkOfxkwzn1CKbAlSh9K3Axu8o0ZJ6rYqF60tYCR5Q+zgGvLZ0DOVWi/oM8IKDBUtew
a9uTqlbqDaw50nSrX1GUz8hhsWvu9iSjgs9jo/0pNR6PtW6+sivk/pAfZBJo3wz+JPN0lZCfmwF24R4Q
KXbNWg3HERkcGtUKR7NpbOrZ34BlQ5hq+kQcPQzLvDjUtzKZsFW+P7xvyrqqZQl5sIcc8mF9MZM83uxW
iV53GPg/yE+vPxNW7981Rb5KXmkt46tAOJSiWVGG3ERAarlr1HsAX4uSK/uHTaG5qdWkRr7lfLtd1Wg/
Xd5f3N3dXVTNbn1xs1uhXlpOB8VCNcyB//r5x4t/JUzZbduDzih8I5C9BE2frVJECbIdYIn6Sdi9Og6e
tF6xgbWW2HIPuL/eBapEX7HMb3PNQnM0775PHlWdl1/E/Xr1RVziIy+/CPX3Euu7/CLU3y/i8sjCOYQ3
E1P42/t3RH+FKfos7w/mtUzZ3z99+BnfQM9m1QLwiiRBExANwAF8M3CZwqGqhSTqbjQZdbH68MSZp0fm
LSrY86ar7g/JcHK0Y+PmTCygSN+JGJaucLWiTNDknYjDUiCeUQWOdenvIv5RUCj8vMs3+22zO6jCn3Rh
J1e6z+9+wq1YwPrYka7smq30loZ9qZtt/AhJ31vejPVnt23DvrpD2F5wSEtbbQTRdBdvaWL2VPZBvj47
8N34Vb5aiby43sek2RRysJbrZvdAKLvhzVhN2pv9q6aU4FW/VaL/Tv13zwnmEciSsAf+uJN5+fAJpjiw
1YdrZQ/ClVoqcFFb0McFfzzaaIm/CcyknVO6mIlTfg0uZi+yo+CLWd45cwwsAYGWgDiq93m5WoWvtO8x
qK51uCrsLP6CeRknH+ANKnzONSR23Z28TU9R2+bsdpbDGoVBNM2t3O3qUr7XqkZydsvtGhBCjErCc1OD
66L+Vgagimv6YJKj89mD7lY/1U6gfXszExmf3UCmzUxkWUgHkws16HsiE9v23qOLlWO4MBaUvYwnJv7k
CKb63oIjPFDWjG92Kx7Hom3hZ9tqeU5HhFBrXL0WzBPiI3J5SdS9sBdQjNfysGjKti00IV1jS/AS1jhF
hcfuAGwAet6+ICTTARnNuNg1+/0PzTqvN/Rx2W8KqSV0idYQfAzTB/iHBZXwt53vGSlDqtkfhggk4J1Y
Qrlelf+gj2E9SvTVlf4q2DZyy5qLrrLmgrlO38CNCwIPVZvu8hKA5vMVpewfSuaxhhXsgbJroz4+TFdm
1009EZdUtoqiCYDp4Co+GoHeHjjzCUizQ747uP7DPyEuPGsgUEqTEg1/0As5XkqZbl87PP6AkI3gpvRM
gwAucDP21ngIYnJ7q3+64AM8kG49/OOo9wQjI6Jsx41+Q1SLTfokZbb5qxGP9/qzKpqSiCQkJXSk+0GH
l+MRICXnxUIaKt6KV/aZLwUjz64IZZv+CsnvnIx2YjQajDZmvlX4s66MigVAEr7OBer1w7grBGPytrow
11x8qjeFJOzkTtinOeTzpyr5udnIi/dqSBN3NaXMG8iuM9WR12N6567wy2j/kwIDjQW1UNZ3w0vQ3ogv
MsBOasZar5uFZ7L07JmRthTC4pSwARm9ESMyHXzlk/HkiiSE0MRVAzAQYDOvlUhuxgtcrmjP+66ZPT1b
Iy9iM0amo09yUyIcvT3E0Icte2AN5diI1zZi5EHLbKjknhM4IgwZbxsHusEeEJijGWtFXpUAEEeDjnjK
JP+HUou03IA1/mHs9AN+pWTF145UAMqm2QNrskDSNGNQzFUPHDCF7/sJ7MOeQw8030H05YQemb2XopS+
5sMrJsd7ZXTcspfUCVdYKcFAGvwxfRlfXLE/6BFDMeFIWTtW+yNeQPpL9NNYxw0ocLfsjt3zYnrdtvE1
H8KmbycfsVbNpbeE53yhpBELGkt8P0m/TSZsycX3/MVkEkXiu28mk7b9ZvIt51ww1cu3/L2IG/YAwC63
/Gd1cMse2JKyZRp3pvodf+hzZrzL9wc7uQlld31Sgd9RduZ+NYntbXpG8ztK2Qt80bYlb16//EEZ1CjI
03tONo1hNUj092DpYW1eJInv+S3oK5Kt+S3Kxw2/xcHGlny4QYl7z4Yiiu7bVo1ebaYKiEkXfEIpzHfQ
eriwP4FfKC7a9l4pHGyZ7gP0ny2brdk9e8hosvfhf7ZqoN6zTeYqVepXfKN0Zt2ZwRBfpjjItembwNFr
fEc15tkyXSequgOgD3oPyaiqKe7Mlld6NtoZc3Fh1l3YEOtbdRuIsDMa8gMoxcrsOpeatjP5PaxgBKw0
Cvd8AmOzXx8295igHkeQGoS9zeWBMLJt9odTOPnuPniQi9/x9kLwKwRmlYCabgwrNJscGBHYiznTQWBG
HCcSXRAF05IsKY/sdMMoiiAqU9VqSIr7wuL0M/1HoTvCPs60BYPF3PM5XFmvxBUz3hmI3ujsJd3t8u3L
1eo8WrAOqIAZ2AHIiXPuxR7PJhkGAmus5hOiazqWX+MJ9bjpzGVhhlFAX2lqZqJ/UwpQutDIy5HPU+vP
SOup3vCk1LEhWzJDTReIFolqk7ebTT8r8l9IFPByQmxVYUYIfTozwAb/F1xYl3PsgAjSYqz7Lc5pIjxG
Ropv3xs4HLz39HwQThG+vnqKCIPMaYKPutmcPCxgE4RujXM63jSHmIimfCCnHLkufcaSJJr9O0N9To8O
FU1nqW738qZs9gYM8vQVhp0LgdhKjRKYAX2n+ioZxvm4qaq9PABRQNuaQ6SUeYJeAh4TbGHcL3Y+iIBS
GjzE+nz82/t3bw6HrdbGtP4g6OMRvTkfBH+cACzC1YsX3yQvJt8e2UfBTx8S06nSrXZ7Phx+FFFE7urD
4tVOlnJzqPPVntSbwUfBGriRfxRwmX5Zq4fEJ7EbrERVUNXctqrioQhsWOOSDah+/chaLvTr1VW8AIbH
GIPvmADbVqBepo72EgJEIftwv79rdqWSAPeLHboT3V6AX7iYzTPuFczm2VRY50YULcZdx0hfWexuUc/0
vrBt5Yz8dqE7SJYXQOiZtW3cW85J2KMabxReW9LFqeY9Z0DVOS361gJv5BTg9Sv5YtxsVk2OP0A7gV+g
q8IvUPlAzcH4MggtZFoNByrrhdHQE63eQOkJQNlCayRpFU+YvpImqhexnC085Ued+CBmpihr297L0CkL
tCULu+OirWc4Q3vihLwr5f0hfRT1Jt89JK74mDyC+za88MgW415/XUwhO8K2ZBFT5rdnEZuvdfnOto1T
19pl0tveXqcpRXThKeHA+dRrbRRRVAJWNCvU87G30MhYoH0hAkMWd1kRhsqYHbVmyEGzoz6euNngMUVs
0lACt3QcotR7MwDAD8yCNEbdg+O2b+BWthsWeInemXDbCmwQ7DycKZfFurf8/sKdCTYo9NMuv4g4TVSt
rbqQYjHsSgRbCbApoKvpX+pRh3p9m68Avqe3sawKFqy5lisa3SzYcOBwGV5RdtqoCLCilDvXmE4Y9z4D
tjADJ6JO5pmel8RKtSDfYW3fE4oRPY96uynJdZe+wmO23xVJrkTzkY6bTUzUFBloMygUUsIEC5rwepZH
URV7QgXNs28n38IShofqU0twNgSQN0IpfE+NWB0E80/BZxn7RfDLmNMvaZzyqH1G2y8phiF641GZGtuE
FHpXAveZtmaT4pQw5Z8CY7rB8MEsiBH5HZ1egWKcQ9BF3jsu1DNgK2pLWJBt0uECE2O4yMC5/aI9brAm
0pTc7FYkOUkvE9qnBc5H8d91PrpngnMwioj6i6nHi7Yl+BXAHhxEdBj3vPkG06Kd6LzwJE07BbHSYYMS
tkjFbJFx9Z91S/6CbsmRtJebJoOm8ryfpumMw1JfjrF/sKi7+AA9tXDPMOOn+zdzFw4qR2Rwl+8Hm+Yw
UGMJfBrz2SQ7srBhOBq3gMotM6b+CxgIuA3APrKyB+Daio8qhdBphyYSS5pgdRBCCR8fNn3RaUo1lm/2
i1hSYLULeqaiapbOwcCa88pxZBiBAyHauC/x5jMAhMBr8x6sq3q9RRsLxlrPTbEal8oIcCA7Hkf2d2pY
fv/dJf7xDwh7gYLU2gNaxT5CLgRs50IdXaeDl1nt0Cr1mJ1l097ccb2dOryibgvVZoq0bdzXGmks/ur3
A3NWZ59H5HsJaKuws1Pa4Grc4hGn8lFSmgheUlbxv+GuJtDKDYsommWsSmfdR1Szq4wCV+MW0fIFm8Ng
mFsuvF08p06G78ZruZvLWFXn22HGewC60ln4GTZHIpAFz33cAIeEjklqJV+L2LD1LihlyljHowlTx90Q
33CrOxE9SL/AscLJxw+fPqupadMb1Lg/8aUoUdlxp2DwhI4ioR0+gZw+Vm7msvlYXR2XqVpSy/pWrafa
EvdGZZxTCjm7cYlms5nuRRSFbqr5iWkcIEFWbeuFJoEGLVieAamIdlxYZ5jbE2POU8c6bj7fYRi4Epnz
nmd/mlUSmvywwuW0x1DXkI5lv59rvpPb2OKNBp4ULStg8UG8fzP9naf8k+h4aHR+VU7TPPk33uVa9fLf
jjtt0ENyxwf4mTwxsk04hcdtrFm0CWUrrqQ1W/PH45QoO6AuDBeJ4R0xV3Oyk6v8UN+qqb/gK/0WMSIL
66rBv8pqV7ACJkW25LGj7Na0JFV9D0hz/JrCXmDt6wA3hwZSQ9kyjUu+sm8RAx3l+NBsmVSSR1YHmsRz
n9ezom07YdIvqlXRyfQEp7gwEO5FgHq8oNRAT4sxoMLGa/UXjy4W6v/R3F2iXgSuUT/08cUC/qgFnNzs
lVBX5n8qxnBgnrumyQpaa40sWZ6rs+l2Ln0Km8Fp8ODiS874+XS/je3gMSi4wuQpW4AZPzfNhU2e8x+p
jqrOUV2yEtmjAGEcPFEp4BSH7PkF/yRipe9wOS51oV4S2OOh2SbQ86NivM3n8n/h61+IcQEP+txsmWrv
BIeFvuq3zlXvgHIzKWkCFU7wlsnxyCy5fEjNYfy5Bq2V2ZQ9VvKwDhOebEc2ToPCn3Kp4MW5RkpinWSA
XfQRvZGqObxSWOwcLqB6D1wEKEQP5/Yyat4PG01PSbgeuUQ/N1tLU06Dpju9VjWbu/hIsTtwMkD9F/Zb
kVjss5Lftl49IbD67qXAtorZjUfmf/p/KtnjpOGMkx0CQIxk6xGC4HIPbjQSvG2/5scQqdiy4yXEG12E
WTY9LNdj80yGX3CJWiamJxDD5RP0p46Kn8MyMu0KAJlWaTUTWZLPygwC4eMqrSzjX1yklT83EsmKVCaV
P6so3Msl0JqWJ4iBXbhptVaDqH+aGBkIerf1vVx91M3PwmxPPwlWk7MLxzNfUDQu3GIwExmQ5ybFkXrd
pOnTLWszjNxE88CdwjvjTZZrF3R8MsqNnyYRjJCENDcHKD52UHih70qv75wHmXfbTanrHoa241NQS2oB
0KlIbFBhpLNh6UsMAb7t7O6wAFAIPSymp3qFoCmE9LslFj+HpmJmPzhLhBW7Xfk7Iyg/4bJ/A93GKChp
XHFxIq8tZZEAEwrM1ma1UvezKjgyF+AcNBeER+7h1KR2wEDH+aw+fkEtDSYCZCzoUZkNqUxMZAGgHncy
gkXtu5zCzV+jIOY6XZwVsIV0ckvnhqqyd9AjM1gJf0ILgXqopne42Zy5y94DOKydwZV6jyfPnxO9YacK
BMvbVpWxQk9d0I7+/unDz9yPOjZPc/YJIgJHEf4d5+vS/I4JxtYSNst6WK93Wpv4LHg+Xv5TXcl+Vb+f
ueG5aV41m2pVFwfep0GPnymJDbroM/6rQI4GXZc9ow8/C8p2RyYAc0aXqdt2qphO/+X/BAAA//9+meTv
tVIBAA==
`,
	},

	"/": {
		name:  "/",
		local: `examples/hotrod/services/frontend/web_assets`,
		isDir: true,
	},
}

var _escDirs = map[string][]os.FileInfo{

	"examples/hotrod/services/frontend/web_assets": {
		_escData["/index.html"],
		_escData["/jquery-3.1.1.min.js"],
	},
}
