import ResizeObserver from 'resize-observer-polyfill';
import useLayoutEffect from '@react-hook/passive-layout-effect';
import useLatest from '@react-hook/latest';
/**
 * A React hook that fires a callback whenever ResizeObserver detects a change to its size
 *
 * @param target A React ref created by `useRef()` or an HTML element
 * @param callback Invoked with a single `ResizeObserverEntry` any time
 *   the `target` resizes
 */

const useResizeObserver = (target, callback) => {
  const resizeObserver = getResizeObserver();
  const storedCallback = useLatest(callback);
  useLayoutEffect(() => {
    let didUnsubscribe = false;

    const callback = (entries, observer) => {
      if (didUnsubscribe) return;
      const targetEl = target && 'current' in target ? target.current : target;

      for (let i = 0; i < entries.length; i++) {
        const entry = entries[i];

        if (entry.target === targetEl) {
          storedCallback.current(entry, observer);
        }
      }
    };

    resizeObserver.subscribe(callback);
    return () => {
      didUnsubscribe = true;
      resizeObserver.unsubscribe(callback);
    };
  }, [target, resizeObserver, storedCallback]);
  useLayoutEffect(() => {
    const targetEl = target && 'current' in target ? target.current : target;
    if (!targetEl) return;
    resizeObserver.observer.observe(targetEl);
    return () => resizeObserver.observer.unobserve(targetEl);
  }, [target, resizeObserver.observer]);
  return resizeObserver.observer;
};

const createResizeObserver = () => {
  const callbacks = new Set();
  return {
    observer: new ResizeObserver((entries, observer) => {
      for (const callback of callbacks) callback(entries, observer);
    }),
    subscribe: callback => callbacks.add(callback),
    unsubscribe: callback => callbacks.delete(callback)
  };
};

let _resizeObserver;

const getResizeObserver = () => !_resizeObserver ? _resizeObserver = createResizeObserver() : _resizeObserver;

export default useResizeObserver;