"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMaxLegendTextSize = exports.getPieLegendY = exports.getPieLegendX = exports.getChartLegendY = exports.getChartLegendX = exports.getBulletLegendY = exports.getLegendY = exports.getLegendX = exports.getLegendItemsPerRow = exports.doesLegendFit = exports.getLegendDimensions = exports.getComputedLegend = void 0;
const tslib_1 = require("tslib");
const defaults_1 = tslib_1.__importDefault(require("lodash/defaults"));
const victory_core_1 = require("victory-core");
const victory_legend_1 = require("victory-legend");
const ChartTheme_1 = require("../ChartTheme");
const chart_origin_1 = require("./chart-origin");
const React = tslib_1.__importStar(require("react"));
// Returns a legend which has been positioned per the given chart properties
exports.getComputedLegend = ({ allowWrap = true, chartType = 'chart', dx = 0, dy = 0, height, legendComponent, padding, position = ChartTheme_1.ChartCommonStyles.legend.position, theme, width, 
// destructure last
orientation = theme.legend.orientation }) => {
    // Get the number of legend items per row
    const legendItemsProps = legendComponent.props ? legendComponent.props : {};
    const legendItemsPerRow = allowWrap
        ? exports.getLegendItemsPerRow({
            chartType,
            dx,
            height,
            legendData: legendItemsProps.data,
            legendOrientation: legendItemsProps.legendOrientation ? legendItemsProps.legendOrientation : orientation,
            legendPosition: position,
            legendProps: legendItemsProps,
            padding,
            theme,
            width
        })
        : undefined;
    // Include new itemsPerRow prop when determining x and y position
    const legendPositionProps = defaults_1.default({}, legendComponent.props, {
        itemsPerRow: legendItemsPerRow
    });
    const legendX = exports.getLegendX({
        chartType,
        dx,
        height,
        legendData: legendPositionProps.data,
        legendOrientation: legendPositionProps.legendOrientation ? legendPositionProps.legendOrientation : orientation,
        legendPosition: position,
        legendProps: legendPositionProps,
        padding,
        theme,
        width
    });
    const legendY = exports.getLegendY({
        chartType,
        dy,
        height,
        legendData: legendPositionProps.data,
        legendOrientation: legendPositionProps.legendOrientation ? legendPositionProps.legendOrientation : orientation,
        legendProps: legendPositionProps,
        legendPosition: position,
        padding,
        theme,
        width
    });
    // Clone legend with updated props
    const legendProps = defaults_1.default({}, legendComponent.props, {
        itemsPerRow: legendItemsPerRow,
        orientation,
        standalone: false,
        theme,
        x: legendX > 0 ? legendX : 0,
        y: legendY > 0 ? legendY : 0
    });
    return React.cloneElement(legendComponent, legendProps);
};
// Returns legend dimensions
exports.getLegendDimensions = ({ legendData, legendOrientation, legendProps, theme }) => {
    if (legendData || legendProps.data) {
        return victory_legend_1.VictoryLegend.getDimensions(Object.assign({ data: legendData, orientation: legendOrientation, theme }, legendProps // override above
        ));
    }
    return {};
};
// Returns true if the legend is smaller than its container
exports.doesLegendFit = ({ dx = 0, height, legendPosition, legendData, legendOrientation, legendProps, padding, theme, width }) => {
    const { left, right } = victory_core_1.Helpers.getPadding({ padding });
    const chartSize = {
        height,
        width: width - left - right
    };
    const legendDimensions = exports.getLegendDimensions({
        legendData,
        legendOrientation,
        legendProps,
        theme
    });
    let occupiedWidth = 0;
    switch (legendPosition) {
        case 'bottom-left':
            occupiedWidth = left + dx;
            break;
        case 'right':
            occupiedWidth = chartSize.width + ChartTheme_1.ChartCommonStyles.legend.margin + left + dx;
            break;
        default:
            occupiedWidth = dx;
            break;
    }
    return width - occupiedWidth > legendDimensions.width;
};
// Returns the number of legend items per row
exports.getLegendItemsPerRow = ({ dx, height, legendPosition, legendData, legendOrientation, legendProps, padding, theme, width }) => {
    let itemsPerRow = legendData ? legendData.length : 0;
    for (let i = itemsPerRow; i > 0; i--) {
        const fits = exports.doesLegendFit({
            dx,
            height,
            legendPosition,
            legendData,
            legendOrientation,
            legendProps: Object.assign(Object.assign({}, legendProps), { itemsPerRow: i }),
            padding,
            theme,
            width
        });
        if (fits) {
            itemsPerRow = i;
            break;
        }
    }
    return itemsPerRow;
};
// Returns x coordinate for legend
exports.getLegendX = (_a) => {
    var { chartType } = _a, rest = tslib_1.__rest(_a, ["chartType"]);
    return chartType === 'pie' ? exports.getPieLegendX(rest) : exports.getChartLegendX(rest);
};
// Returns y coordinate for legend
exports.getLegendY = (_a) => {
    var { chartType } = _a, rest = tslib_1.__rest(_a, ["chartType"]);
    switch (chartType) {
        case 'pie':
            return exports.getPieLegendY(rest);
        case 'bullet':
            return exports.getBulletLegendY(rest);
        default:
            return exports.getChartLegendY(rest);
    }
};
// Returns y coordinate for bullet legends
exports.getBulletLegendY = ({ dy = 0, height, legendPosition, legendData, legendOrientation, legendProps, padding, theme, width }) => {
    const { left, right } = victory_core_1.Helpers.getPadding({ padding });
    const chartSize = {
        height,
        width: width - left - right
    };
    switch (legendPosition) {
        case 'bottom':
        case 'bottom-left':
            return chartSize.height + ChartTheme_1.ChartCommonStyles.legend.margin + dy;
        case 'right': {
            // Legend height with padding
            const legendDimensions = exports.getLegendDimensions({
                legendData,
                legendOrientation,
                legendProps,
                theme
            });
            const legendPadding = (legendDataArr) => (legendDataArr && legendDataArr.length > 0 ? 17 : 0);
            return (chartSize.height - legendDimensions.height) / 2 + legendPadding(legendData);
        }
        default:
            return dy;
    }
};
// Returns x coordinate for chart legends
exports.getChartLegendX = ({ dx = 0, height, legendData, legendOrientation, legendPosition, legendProps, padding, theme, width }) => {
    const { top, bottom, left, right } = victory_core_1.Helpers.getPadding({ padding });
    const chartSize = {
        height: Math.abs(height - (bottom + top)),
        width: Math.abs(width - (left + right))
    };
    const legendDimensions = exports.getLegendDimensions({
        legendData,
        legendOrientation,
        legendProps,
        theme
    });
    switch (legendPosition) {
        case 'bottom':
            return width > legendDimensions.width ? Math.round((width - legendDimensions.width) / 2) + dx : dx;
        case 'bottom-left':
            return left + dx;
        case 'right':
            return chartSize.width + ChartTheme_1.ChartCommonStyles.legend.margin + left + dx;
        default:
            return dx;
    }
};
// Returns y coordinate for chart legends
exports.getChartLegendY = ({ dy = 0, height, legendPosition, legendData, legendOrientation, legendProps, padding, theme, width }) => {
    const { top, bottom, left, right } = victory_core_1.Helpers.getPadding({ padding });
    const chartSize = {
        height: Math.abs(height - (bottom + top)),
        width: Math.abs(width - (left + right))
    };
    switch (legendPosition) {
        case 'bottom':
        case 'bottom-left':
            return chartSize.height + ChartTheme_1.ChartCommonStyles.legend.margin * 2 + top + dy;
        case 'right': {
            // Legend height with padding
            const legendDimensions = exports.getLegendDimensions({
                legendData,
                legendOrientation,
                legendProps,
                theme
            });
            const originX = chartSize.height / 2 + top;
            const legendPadding = (legendDataArr) => (legendDataArr && legendDataArr.length > 0 ? 2 : 0);
            return originX - legendDimensions.height / 2 + legendPadding(legendData);
        }
        default:
            return dy;
    }
};
// Returns x coordinate for pie legends
exports.getPieLegendX = ({ dx = 0, height, legendData, legendOrientation, legendPosition, legendProps, padding, theme, width }) => {
    const origin = chart_origin_1.getPieOrigin({ height, padding, width });
    const radius = victory_core_1.Helpers.getRadius({ height, width, padding });
    const legendDimensions = exports.getLegendDimensions({
        legendData,
        legendOrientation,
        legendProps,
        theme
    });
    switch (legendPosition) {
        case 'bottom':
            return width > legendDimensions.width ? Math.round((width - legendDimensions.width) / 2) + dx : dx;
        case 'right':
            return origin.x + ChartTheme_1.ChartCommonStyles.label.margin + dx + radius;
        default:
            return dx;
    }
};
// Returns y coordinate for pie legends
exports.getPieLegendY = ({ dy = 0, height, legendPosition, legendData, legendOrientation, legendProps, padding, theme, width }) => {
    const origin = chart_origin_1.getPieOrigin({ height, padding, width });
    const radius = victory_core_1.Helpers.getRadius({ height, width, padding });
    switch (legendPosition) {
        case 'bottom':
            return origin.y + ChartTheme_1.ChartCommonStyles.legend.margin + radius + dy;
        case 'right': {
            // Legend height with padding
            const legendDimensions = exports.getLegendDimensions({
                legendData,
                legendOrientation,
                legendProps,
                theme
            });
            const legendPadding = (legendDataArr) => (legendDataArr && legendDataArr.length > 0 ? 2 : 0);
            return origin.y - legendDimensions.height / 2 + legendPadding(legendData);
        }
        default:
            return dy;
    }
};
// Returns an approximation of longest text width based on legend styles
exports.getMaxLegendTextSize = ({ legendData, theme }) => {
    const style = theme && theme.legend && theme.legend.style ? theme.legend.style.labels : undefined;
    if (!(legendData && legendData.length)) {
        return 0;
    }
    let result = '';
    legendData.forEach(data => {
        if (data.name && data.name.length > result.length) {
            result = data.name;
        }
    });
    // The approximateTextSize function returns height and width, but Victory incorrectly typed it as number
    const adjustedTextSize = victory_core_1.TextSize.approximateTextSize(result, Object.assign({}, style));
    return adjustedTextSize.width;
};
//# sourceMappingURL=chart-legend.js.map