'use strict';

var path = require('path');
var schemaUtils = require('schema-utils');
var fs = require('fs');
var fsExtra = require('fs-extra');
var cpy = require('cpy');
var del = require('del');
var archiver = require('archiver');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var path__default = /*#__PURE__*/_interopDefaultLegacy(path);
var fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
var fsExtra__default = /*#__PURE__*/_interopDefaultLegacy(fsExtra);
var cpy__default = /*#__PURE__*/_interopDefaultLegacy(cpy);
var del__default = /*#__PURE__*/_interopDefaultLegacy(del);
var archiver__default = /*#__PURE__*/_interopDefaultLegacy(archiver);

var optionsSchema = {
  title: 'FileManagerPluginOptions',
  type: 'object',
  additionalProperties: false,
  definitions: {
    Copy: {
      description: 'Copy individual files or entire directories from a source folder to a destination folder',
      type: 'array',
      minItems: 1,
      additionalItems: true,
      itmes: [{
        type: 'object',
        additionalProperties: false,
        properties: {
          source: {
            description: 'Copy source. A file or directory or a glob',
            type: 'string',
            minLength: 1
          },
          destination: {
            description: 'Copy destination',
            type: 'string',
            minLength: 1
          }
        }
      }]
    },
    Delete: {
      description: 'Delete individual files or entire directories',
      type: 'array',
      minItems: 1,
      additionalItems: true,
      items: {
        anyOf: [{
          type: 'object',
          additionalProperties: false,
          properties: {
            source: {
              type: 'string',
              minLength: 1
            },
            options: {
              additionalProperties: true,
              type: 'object',
              description: 'Options to forward to del'
            }
          }
        }, {
          type: 'string',
          minLength: 1
        }]
      }
    },
    Move: {
      description: 'Move individual files or entire directories from a source folder to a destination folder',
      type: 'array',
      additionalItems: true,
      items: [{
        type: 'object',
        additionalProperties: false,
        properties: {
          source: {
            description: 'Move source. A file or directory or a glob',
            type: 'string',
            minLength: 1
          },
          destination: {
            description: 'Move destination',
            type: 'string',
            minLength: 1
          }
        }
      }]
    },
    Mkdir: {
      description: 'Create Directories',
      type: 'array',
      minItems: 1,
      additionalItems: true,
      items: {
        type: 'string'
      }
    },
    Archive: {
      description: 'Archive individual files or entire directories.',
      type: 'array',
      additionalItems: true,
      items: [{
        type: 'object',
        additionalProperties: false,
        properties: {
          source: {
            description: 'Source. A file or directory or a glob',
            type: 'string',
            minLength: 1
          },
          destination: {
            description: 'Archive destination',
            type: 'string',
            minLength: 1
          },
          format: {
            type: 'string',
            enum: ['zip', 'tar']
          },
          options: {
            additionalProperties: true,
            type: 'object',
            description: 'Options to forward to archiver'
          }
        }
      }]
    },
    Actions: {
      type: 'object',
      additionalProperties: false,
      properties: {
        copy: {
          $ref: '#/definitions/Copy'
        },
        delete: {
          $ref: '#/definitions/Delete'
        },
        move: {
          $ref: '#/definitions/Move'
        },
        mkdir: {
          $ref: '#/definitions/Mkdir'
        },
        archive: {
          $ref: '#/definitions/Archive'
        }
      }
    }
  },
  properties: {
    events: {
      type: 'object',
      additionalProperties: false,
      properties: {
        onStart: {
          oneOf: [{
            $ref: '#/definitions/Actions'
          }, {
            type: 'array',
            items: {
              $ref: '#/definitions/Actions'
            }
          }]
        },
        onEnd: {
          oneOf: [{
            $ref: '#/definitions/Actions'
          }, {
            type: 'array',
            items: {
              $ref: '#/definitions/Actions'
            }
          }]
        }
      }
    },
    runTasksInSeries: {
      type: 'boolean',
      default: false,
      description: 'Run tasks in an action in series'
    },
    context: {
      type: 'string',
      description: 'The directory, an absolute path, for resolving files. Defaults to webpack context'
    }
  }
};

const defaultTask = async () => {};

const pExec = async (series = false, arr = [], task = defaultTask) => {
  if (series) {
    await arr.reduce(async (p, spec) => {
      await p;
      return task(spec);
    }, Promise.resolve(null));
    return;
  }

  const pMap = arr.map(async spec => await task(spec));
  await Promise.all(pMap);
};

/*!
 * is-extglob <https://github.com/jonschlinkert/is-extglob>
 *
 * Copyright (c) 2014-2016, Jon Schlinkert.
 * Licensed under the MIT License.
 */
var isExtglob = function isExtglob(str) {
  if (typeof str !== 'string' || str === '') {
    return false;
  }

  var match;

  while (match = /(\\).|([@?!+*]\(.*\))/g.exec(str)) {
    if (match[2]) return true;
    str = str.slice(match.index + match[0].length);
  }

  return false;
};

/*!
 * is-glob <https://github.com/jonschlinkert/is-glob>
 *
 * Copyright (c) 2014-2017, Jon Schlinkert.
 * Released under the MIT License.
 */
var chars = {
  '{': '}',
  '(': ')',
  '[': ']'
};
var strictRegex = /\\(.)|(^!|\*|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
var relaxedRegex = /\\(.)|(^!|[*?{}()[\]]|\(\?)/;

var isGlob = function isGlob(str, options) {
  if (typeof str !== 'string' || str === '') {
    return false;
  }

  if (isExtglob(str)) {
    return true;
  }

  var regex = strictRegex;
  var match; // optionally relax regex

  if (options && options.strict === false) {
    regex = relaxedRegex;
  }

  while (match = regex.exec(str)) {
    if (match[2]) return true;
    var idx = match.index + match[0].length; // if an open bracket/brace/paren is escaped,
    // set the index to the next closing character

    var open = match[1];
    var close = open ? chars[open] : null;

    if (open && close) {
      var n = str.indexOf(close, idx);

      if (n !== -1) {
        idx = n + 1;
      }
    }

    str = str.slice(idx);
  }

  return false;
};

const fsExtraDefaultOptions = {
  preserveTimestamps: true
};

const copy = async (task, {
  logger
}) => {
  const {
    source,
    absoluteSource,
    destination,
    absoluteDestination,
    context,
    toType
  } = task;
  logger.log(`copying from ${source} to ${destination}`);

  if (isGlob(source)) {
    const src = path__default['default'].isAbsolute(source) ? source : path__default['default'].posix.join(context, source);
    await cpy__default['default'](src, absoluteDestination);
  } else {
    const isSourceFile = fs__default['default'].lstatSync(absoluteSource).isFile(); // if source is a file and target is a directory
    // create the directory and copy the file into that directory

    if (isSourceFile && toType === 'dir') {
      await fsExtra__default['default'].ensureDir(absoluteDestination);
      const sourceFileName = path__default['default'].basename(absoluteSource);
      const filePath = path__default['default'].resolve(absoluteDestination, sourceFileName);
      await fsExtra__default['default'].copy(absoluteSource, filePath, fsExtraDefaultOptions);
      return;
    }

    await fsExtra__default['default'].copy(absoluteSource, absoluteDestination, fsExtraDefaultOptions);
  }

  logger.info(`copied "${source}" to "${destination}`);
};

const copyAction = async (tasks, options) => {
  const {
    runTasksInSeries,
    logger
  } = options;
  const taskOptions = {
    logger
  };
  logger.debug(`processing copy tasks. tasks: ${tasks}`);
  await pExec(runTasksInSeries, tasks, async task => {
    try {
      await copy(task, taskOptions);
    } catch (err) {
      logger.error(`error while copying. task ${err}`);
    }
  });
  logger.debug(`copy tasks complete. tasks: ${tasks}`);
};

const moveAction = async (tasks, options) => {
  const {
    runTasksInSeries,
    logger
  } = options;
  logger.debug(`processing move tasks. tasks: ${tasks}`);
  await pExec(runTasksInSeries, tasks, async task => {
    try {
      await fsExtra__default['default'].move(task.absoluteSource, task.absoluteDestination);
      logger.info(`moved ${task.source} to ${task.destination}`);
    } catch (err) {
      logger.error(`unable to move ${task.source}, ${err}`);
    }
  });
  logger.debug(`move tasks complete. tasks: ${tasks}`);
};

const deleteAction = async (tasks, taskOptions) => {
  const {
    runTasksInSeries,
    logger
  } = taskOptions;
  logger.debug(`processing delete tasks. tasks: ${tasks}`);
  await pExec(runTasksInSeries, tasks, async task => {
    const {
      options = {}
    } = task;

    try {
      await del__default['default'](task.absoluteSource, options);
      logger.info(`deleted ${task.source}`);
    } catch (err) {
      logger.error(`unable to delete ${task.source}. ${err}`);
    }
  });
  logger.debug(`delete tasks complete. tasks: ${tasks}`);
};

const mkdirAction = async (tasks, options) => {
  const {
    runTasksInSeries,
    logger
  } = options;
  logger.debug(`processing mkdir tasks. tasks: ${tasks}`);
  await pExec(runTasksInSeries, tasks, async task => {
    try {
      await fs__default['default'].promises.mkdir(task.absoluteSource, {
        recursive: true
      });
      logger.info(`created directory. ${task.source}`);
    } catch (err) {
      logger.error(`unable to create direcotry: ${task.source}. ${err}`);
    }
  });
  logger.debug(`mkdir tasks complete. tasks: ${tasks}`);
};

const archive = async (task, {
  logger
}) => {
  const {
    source,
    absoluteSource,
    absoluteDestination,
    options = {},
    context
  } = task;
  const format = task.format || path__default['default'].extname(absoluteDestination).replace('.', ''); // Exclude destination file from archive

  const destFile = path__default['default'].basename(absoluteDestination);
  const destDir = path__default['default'].dirname(absoluteDestination);
  const ignore = (Array.isArray(options.ignore) && options.ignore || []).concat(destFile);
  const archiverOptions = {
    ignore,
    ...(options.globOptions || {})
  };
  await fsExtra__default['default'].ensureDir(destDir);
  const output = fs__default['default'].createWriteStream(absoluteDestination);
  const archive = archiver__default['default'](format, options);
  archive.pipe(output);
  logger.log(`archiving src ${source}`);

  if (isGlob(source)) {
    const opts = { ...archiverOptions,
      cwd: context
    };
    await archive.glob(source, opts).finalize();
  } else {
    const sStat = fs__default['default'].lstatSync(absoluteSource);

    if (sStat.isDirectory()) {
      const opts = { ...archiverOptions,
        cwd: absoluteSource
      };
      await archive.glob('**/*', opts).finalize();
    }

    if (sStat.isFile()) {
      const opts = {
        name: path__default['default'].basename(source)
      };
      await archive.file(absoluteSource, opts).finalize();
    }
  }

  logger.info(`archive created at "${absoluteDestination}"`);
};

const archiveAction = async (tasks, options) => {
  const {
    runTasksInSeries,
    logger
  } = options;
  const taskOptions = {
    logger
  };
  logger.debug(`processing archive tasks. tasks: ${tasks}`);
  await pExec(runTasksInSeries, tasks, async task => {
    try {
      await archive(task, taskOptions);
    } catch (err) {
      logger.error(`error while creating archive. task ${task}`);
    }
  });
  logger.debug(`archive tasks complete. tasks: ${tasks}`);
};

const PLUGIN_NAME = 'FileManagerPlugin';
const defaultOptions = {
  events: {
    onStart: [],
    onEnd: []
  },
  runTasksInSeries: false,
  context: null
};

const resolvePaths = (action, context) => {
  return action.map(task => {
    if (typeof task === 'string') {
      const source = task;
      return {
        source,
        absoluteSource: path__default['default'].isAbsolute(source) ? source : path__default['default'].join(context, source)
      };
    }

    const {
      source,
      destination
    } = task;

    if (!destination) {
      return { ...task,
        source,
        absoluteSource: path__default['default'].isAbsolute(source) ? source : path__default['default'].join(context, source)
      };
    }

    const toType = /(?:\\|\/)$/.test(destination) ? 'dir' : 'file';
    return { ...task,
      source,
      absoluteSource: path__default['default'].isAbsolute(source) ? source : path__default['default'].join(context, source),
      destination,
      absoluteDestination: path__default['default'].isAbsolute(destination) ? destination : path__default['default'].join(context, destination),
      toType,
      context
    };
  });
};

class FileManagerPlugin {
  constructor(options) {
    schemaUtils.validate(optionsSchema, options, {
      name: PLUGIN_NAME,
      baseDataPath: 'actions'
    });
    this.options = { ...defaultOptions,
      ...options
    };
  }

  async applyAction(action, actionParams) {
    const opts = {
      runTasksInSeries: this.options.runTasksInSeries,
      logger: this.logger
    };
    await action(resolvePaths(actionParams, this.context), opts);
  }

  async run(event) {
    for (const actionType in event) {
      const action = event[actionType];

      switch (actionType) {
        case 'delete':
          await this.applyAction(deleteAction, action);
          break;

        case 'mkdir':
          await this.applyAction(mkdirAction, action);
          break;

        case 'copy':
          await this.applyAction(copyAction, action);
          break;

        case 'move':
          await this.applyAction(moveAction, action);
          break;

        case 'archive':
          await this.applyAction(archiveAction, action);
          break;

        default:
          throw Error('Unknown action');
      }
    }
  }

  async execute(eventName) {
    const {
      events
    } = this.options;

    if (Array.isArray(events[eventName])) {
      const eventsArr = events[eventName];
      await pExec(true, eventsArr, async event => await this.run(event));
      return;
    }

    const event = events[eventName];
    await this.run(event);
  }

  apply(compiler) {
    this.context = this.options.context || compiler.options.context;
    this.logger = compiler.getInfrastructureLogger(PLUGIN_NAME);

    const onStart = async () => {
      await this.execute('onStart');
    };

    const onEnd = async () => {
      await this.execute('onEnd');
    };

    compiler.hooks.beforeRun.tapPromise(PLUGIN_NAME, onStart);
    compiler.hooks.watchRun.tapPromise(PLUGIN_NAME, onStart);
    compiler.hooks.afterEmit.tapPromise(PLUGIN_NAME, onEnd);
  }

}

module.exports = FileManagerPlugin;
//# sourceMappingURL=index.js.map
