"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TsJestTransformer = void 0;
var util_1 = require("util");
var config_set_1 = require("./config/config-set");
var constants_1 = require("./constants");
var json_1 = require("./util/json");
var jsonable_value_1 = require("./util/jsonable-value");
var logger_1 = require("./util/logger");
var messages_1 = require("./util/messages");
var sha1_1 = require("./util/sha1");
var INSPECT_CUSTOM = util_1.inspect.custom || 'inspect';
var TsJestTransformer = /** @class */ (function () {
    function TsJestTransformer(baseOptions) {
        if (baseOptions === void 0) { baseOptions = {}; }
        this.options = __assign({}, baseOptions);
        this.id = TsJestTransformer._nextTransformerId;
        this.logger = logger_1.rootLogger.child({
            transformerId: this.id,
            namespace: 'jest-transformer',
        });
        this.logger.debug({ baseOptions: baseOptions }, 'created new transformer');
    }
    Object.defineProperty(TsJestTransformer, "_nextTransformerId", {
        /**
         * @internal
         */
        get: function () {
            return ++TsJestTransformer._lastTransformerId;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * @internal
     */
    /* istanbul ignore next */
    TsJestTransformer.prototype[INSPECT_CUSTOM] = function () {
        return "[object TsJestTransformer<#" + this.id + ">]";
    };
    /**
     * Use by e2e, don't mark as internal
     */
    TsJestTransformer.prototype.configsFor = function (jestConfig) {
        var csi = TsJestTransformer._configSetsIndex.find(function (cs) { return cs.jestConfig.value === jestConfig; });
        if (csi)
            return csi.configSet;
        // try to look-it up by stringified version
        var serialized = json_1.stringify(jestConfig);
        csi = TsJestTransformer._configSetsIndex.find(function (cs) { return cs.jestConfig.serialized === serialized; });
        if (csi) {
            // update the object so that we can find it later
            // this happens because jest first calls getCacheKey with stringified version of
            // the config, and then it calls the transformer with the proper object
            csi.jestConfig.value = jestConfig;
            return csi.configSet;
        }
        var jestConfigObj = jestConfig;
        // create the new record in the index
        this.logger.info('no matching config-set found, creating a new one');
        var configSet = new config_set_1.ConfigSet(jestConfigObj, this.options, this.logger);
        TsJestTransformer._configSetsIndex.push({
            jestConfig: new jsonable_value_1.JsonableValue(jestConfigObj),
            configSet: configSet,
        });
        return configSet;
    };
    TsJestTransformer.prototype.process = function (input, filePath, jestConfig, transformOptions) {
        this.logger.debug({ fileName: filePath, transformOptions: transformOptions }, 'processing', filePath);
        var result;
        var source = input;
        var configs = this.configsFor(jestConfig);
        var hooks = configs.hooks;
        var shouldStringifyContent = configs.shouldStringifyContent(filePath);
        var babelJest = shouldStringifyContent ? undefined : configs.babelJestTransformer;
        var isDefinitionFile = filePath.endsWith('.d.ts');
        var isJsFile = constants_1.JS_JSX_REGEX.test(filePath);
        var isTsFile = !isDefinitionFile && constants_1.TS_TSX_REGEX.test(filePath);
        if (shouldStringifyContent) {
            // handles here what we should simply stringify
            result = "module.exports=" + json_1.stringify(source);
        }
        else if (isDefinitionFile) {
            // do not try to compile declaration files
            result = '';
        }
        else if (!configs.parsedTsConfig.options.allowJs && isJsFile) {
            // we've got a '.js' but the compiler option `allowJs` is not set or set to false
            this.logger.warn({ fileName: filePath }, messages_1.interpolate("Got a `.js` file to compile while `allowJs` option is not set to `true` (file: {{path}}). To fix this:\n  - if you want TypeScript to process JS files, set `allowJs` to `true` in your TypeScript config (usually tsconfig.json)\n  - if you do not want TypeScript to process your `.js` files, in your Jest config change the `transform` key which value is `ts-jest` so that it does not match `.js` files anymore" /* GotJsFileButAllowJsFalse */, { path: filePath }));
            result = source;
        }
        else if (isJsFile || isTsFile) {
            // transpile TS code (source maps are included)
            /* istanbul ignore if */
            result = configs.tsCompiler.compile(source, filePath);
        }
        else {
            // we should not get called for files with other extension than js[x], ts[x] and d.ts,
            // TypeScript will bail if we try to compile, and if it was to call babel, users can
            // define the transform value with `babel-jest` for this extension instead
            var message = babelJest ? "Got a unknown file type to compile (file: {{path}}). To fix this, in your Jest config change the `transform` key which value is `ts-jest` so that it does not match this kind of files anymore. If you still want Babel to process it, add another entry to the `transform` option with value `babel-jest` which key matches this type of files." /* GotUnknownFileTypeWithBabel */ : "Got a unknown file type to compile (file: {{path}}). To fix this, in your Jest config change the `transform` key which value is `ts-jest` so that it does not match this kind of files anymore." /* GotUnknownFileTypeWithoutBabel */;
            this.logger.warn({ fileName: filePath }, messages_1.interpolate(message, { path: filePath }));
            result = source;
        }
        // calling babel-jest transformer
        if (babelJest) {
            this.logger.debug({ fileName: filePath }, 'calling babel-jest processor');
            // do not instrument here, jest will do it anyway afterwards
            result = babelJest.process(result, filePath, jestConfig, __assign(__assign({}, transformOptions), { instrument: false }));
        }
        // allows hooks (useful for testing)
        if (hooks.afterProcess) {
            this.logger.debug({ fileName: filePath, hookName: 'afterProcess' }, 'calling afterProcess hook');
            var newResult = hooks.afterProcess([input, filePath, jestConfig, transformOptions], result);
            if (newResult !== undefined) {
                return newResult;
            }
        }
        return result;
    };
    /**
     * Jest uses this to cache the compiled version of a file
     *
     * @see https://github.com/facebook/jest/blob/v23.5.0/packages/jest-runtime/src/script_transformer.js#L61-L90
     * @param fileContent The content of the file
     * @param filePath The full path to the file
     * @param _jestConfigStr The JSON-encoded version of jest config
     * @param transformOptions
     * @param transformOptions.instrument Whether the content will be instrumented by our transformer (always false)
     * @param transformOptions.rootDir Jest current rootDir
     */
    TsJestTransformer.prototype.getCacheKey = function (fileContent, filePath, _jestConfigStr, transformOptions) {
        this.logger.debug({ fileName: filePath, transformOptions: transformOptions }, 'computing cache key for', filePath);
        var configs = this.configsFor(transformOptions.config);
        // we do not instrument, ensure it is false all the time
        var _a = transformOptions.instrument, instrument = _a === void 0 ? false : _a, _b = transformOptions.rootDir, rootDir = _b === void 0 ? configs.rootDir : _b;
        return sha1_1.sha1(configs.cacheKey, '\x00', rootDir, '\x00', "instrument:" + (instrument ? 'on' : 'off'), '\x00', fileContent, '\x00', filePath);
    };
    /**
     * @internal
     */
    TsJestTransformer._configSetsIndex = [];
    /**
     * @internal
     */
    TsJestTransformer._lastTransformerId = 0;
    return TsJestTransformer;
}());
exports.TsJestTransformer = TsJestTransformer;
