/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/

var inspectClient = require('./lib/inspect-client');
var cookieUtil = require('./lib/cookie-util');
var pdpClient = require('./lib/pdp-client');	
var log4js = require('log4js');
var logger = log4js.getLogger();
var url = require('url');
var auditClient = require('./lib/audit-client');
var regClient = require('./lib/registration-client');
var tokenClient = require('./lib/token-client');

if (process.env.LOG_LEVEL) {
  logger.level = process.env.LOG_LEVEL_MW;
} else {
  logger.level = "info";	
}	

function redirectToLogin(req, res) {
  if (process.env.NODE_ENV === 'development') {
    res.redirect(process.env.cfcRouterUrl+'/v1/auth/authorize?client_id='+process.env.WLP_CLIENT_ID +
	    '&response_type=code&scope=email+profile+openid&redirect_uri='+process.env.WLP_REDIRECT_URL);
  } else if (process.env.ICP_MASTER_IP && process.env.ICP_DEFAULT_PORT) {
    var proxyUrl = 'https://'+process.env.ICP_MASTER_IP+ ':'+process.env.ICP_DEFAULT_PORT+'/idauth/oidc/endpoint/OP/authorize?client_id='+process.env.CLIENT_ID +
		  '&response_type=code&scope=email+profile+openid&redirect_uri='+process.env.ICP_CALLBACK_URL;
    if (isNaN(process.env.ICP_DEFAULT_PORT)) return res.status(401).send('Unable to verify the host.');
    urlParse = url.parse(process.env.ICP_CALLBACK_URL);
    validateUrl(urlParse.host, function(err, resp) {
      if (err) {
        logger.info('Could not verify host');
        res.status(401).send('Unable to verify the host.');
      } else if(resp && resp === true) {
        logger.debug('proxy url:'+ proxyUrl);
        res.redirect(encodeURIComponent(proxyUrl));
      } else {
        res.status(401).send('Unable to verify the host.');
      }
    })    
  } else {
    var loginUrl = 'https://'+req.headers['host']+'/login';
    validateUrl(req.headers['host'], function(err, resp) {
      if (err) {
        logger.info('Could not verify host');
        res.status(401).send('Unable to verify the host.');
      } else if(resp && resp === true) {
        logger.debug('login url:'+ loginUrl);
        res.redirect(encodeURIComponent(loginUrl));
      } else {
        res.status(401).send('Unable to verify the host.');
      }
    });
  }
}

function validateUrl(url, cb) {
  if (url && url === 'platform-ui:3000') return cb(null, true);
  regClient.regDetails('', (err, body) => {
    if (err) {
      return cb(err, null)
    } else if(body) {
      var prefixes = body.trusted_uri_prefixes
      if (url) {
        if (prefixes.find(prefix => (prefix.indexOf(url) > -1))) {
          return cb(null,true);
        } else {
          return cb(null, false);
        } 
      } 
    }       
  })  
}

module.exports.version = '2.0.95';
module.exports.ui = (req, res, next) => {
  var token;
  if (process.env.AUTHN_DISABLED === "true") {
    logger.info('Skipping authentication check');	  
    return next();                                                                                  
  }
  if (req.path === '/logout' || req.path === '/logout/') {
    return next();
  }
  if (!cookieUtil.isLoggedIn(req)) {	  
    return redirectToLogin(req, res);
  }	
  if (cookieUtil.getAccessToken(req)) {
    token = cookieUtil.getAccessToken(req);
  }

  inspectClient.inspect(req, token, (err, response, body) => {
    if (err)
      return res.status(500).send(err.details);

    if (body && body.sub && response.statusCode === 200) {
      req.user = body.sub || null;
      return next();
    }
    redirectToLogin(req, res);
  });
};

module.exports.app = (req, res, next) => {
  var token;
  if (process.env.AUTHN_DISABLED === "true") {
    logger.info('Skipping authentication check');
    return next();                                                                                                                                                                   
  }
  if (req.url.trim() === '/identity/api/v1/') {
    return res.status(204).send();
  }
  var host = req.headers['host'];
  logger.debug('host:' + host);
  //if (host.indexOf('8443') > -1) {
  //  return next();	  
  //}	  
  if (req.headers['authorization'] || req.headers['Authorization']) {
    token = req.headers['authorization']? req.headers['authorization']:req.headers['Authorization'];
    var words = token.trim().split(/[\s,]+/);
    if (!(words[0].toLowerCase().match('bearer'))) {
      res.status(403).send('No bearer in value');
    }
    if (words[1] && words[1].length > 1) {
      token = words[1]
    }
  } else if (req.get('Authorization')) {
    token = req.get('Authorization');
  }// else {
  //  token = cookieUtil.getAccessToken(req)	  
  //}
  if (!token) {
    return res.status(401).send('The request is unauthorized, no token provided.');
  }
  inspectClient.inspect(req, token, (err, response, body) => {
    if (err)
      return res.status(500).send(err.details);
    if (body && (body.sub || body.active ==='true' || body.active ===true) && response.statusCode === 200) {
      req.user = body.sub || null;
      if (host.indexOf('8443') > -1 || host.indexOf('icp-management-ingress') > -1) {
	return next();      
      } else {
	if (process.env.AUTHZ_DISABLED === true) {
	  logger.info('Skipping authorization check');
	  return next();
        } else {		
          pdpClient.pdp_check(req, token, (err, response) => {
            if (err)
              return res.status(400).send(err.details);
            if (response && response.decision === 'Permit') {
	      logger.debug('Middleware check passed');	
              return next();
            } else {
              return res.status(400).send('You are not authorized to perform this operation.');
            }		
          });
        }
      }	      
    } else {
      return res.status(401).send('The token provided is not valid');	    
    }
  });
};

module.exports.token = tokenClient.token;
module.exports.regClient = regClient;
module.exports.audit = auditClient.audit;
