/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
'use strict'

var request = require('request'),
    log4js = require('log4js'),
    logger = log4js.getLogger('registration-client'),
    path = require('path');

exports.regDetails = (req, cb) => {
  if (!process.env.OAUTH2_CLIENT_REGISTRATION_SECRET || !process.env.PLATFORM_AUTH_SERVICE_URL) {
    return cb('Please specify the auth service url and registration secret')	  
  }
  var id = 'oauthadmin'
  var secret = process.env.OAUTH2_CLIENT_REGISTRATION_SECRET
  var auth = 'Basic ' + new Buffer(id + ':' + secret).toString('base64')
  var clientId = process.env.CLIENT_ID || process.env.WLP_CLIENT_ID 	
  var options = {
    url: process.env.PLATFORM_AUTH_SERVICE_URL+'/oidc/endpoint/OP/registration/'+clientId,
    headers: {'Authorization': auth},
    rejectUnauthorized: false,
    json: true
  }
  request(options, (err, res) => {
    if (err) {
      logger.error('Unable to get registration details'+ err)
      return cb(err, null)
    } else {
      cb(null, res.body || {})
    }
  })
}

exports.regUpdate = (req, cb) => {
  if (!process.env.OAUTH2_CLIENT_REGISTRATION_SECRET || !process.env.CLIENT_ID || !process.env.CLIENT_SECRET) {
    return cb('Please specify the client id, client secret and registration secret')	  
  }
  if (!process.env.PROXY_REDIRECT_URL || !process.env.PLATFORM_AUTH_SERVICE_URL) {
    return cb('Please specify the auth service url and redirect urls to register')
  }
  var redirects = []
  if (process.env.PROXY_REDIRECT_URL) {
    process.env.PROXY_REDIRECT_URL.split(",").forEach(function (final) {
      redirects.push(final);
    })
  } else {
    redirects.push("https://"+process.env.ICP_PROXY_IP+":"+process.env.ICP_PROXY_PORT+"/auth/liberty/callback")
  }
  var clientId = process.env.CLIENT_ID
  var id = 'oauthadmin'
  var secret = process.env.OAUTH2_CLIENT_REGISTRATION_SECRET
  var auth = 'Basic ' + new Buffer(id + ':' + secret).toString('base64')
  var reqBody = {
    "token_endpoint_auth_method": "client_secret_basic",
    "client_id": process.env.CLIENT_ID,
    "client_secret": process.env.CLIENT_SECRET,
    "scope": "openid profile email",
    "grant_types": ["authorization_code", "client_credentials", "implicit",    
            "refresh_token", "urn:ietf:params:oauth:grant-type:jwt-bearer"],
    "response_types": ["code", "token", "id_token token"],
    "application_type": "web",
    "subject_type": "public",
    "post_logout_redirect_uris": ["https://"+process.env.ICP_PROXY_IP+ ":"+ process.env.ICP_PROXY_PORT],
    "preauthorized_scope": "openid profile email general",
    "introspect_tokens": true,
    "trusted_uri_prefixes": ["https://"+process.env.ICP_MASTER_IP+":"+process.env.ICP_DEFAULT_PORT, "https://"+process.env.ICP_PROXY_IP+":"+process.env.ICP_PROXY_PORT],
    "redirect_uris": redirects  
  }
  var options = {
    url: process.env.PLATFORM_AUTH_SERVICE_URL+'/oidc/endpoint/OP/registration/'+clientId,
    headers: {'Authorization': auth, 'Content-Type': 'application/json'},
    body: reqBody,
    rejectUnauthorized: false,
    json: true,
    method: 'PUT'	  
  }
  request(options, (err, res) => {
    if (err) {
      logger.error('Unable to update client registration.')
      return cb(err, null)
    } else if(res && res.statusCode === 200) {
      logger.info('Client registration update successful.')
      cb(null, res)
    } else {
      cb(null, res || {})
    }
  })
}

exports.regNewClient = (req, cb) => {
  if (!process.env.OAUTH2_CLIENT_REGISTRATION_SECRET || !process.env.CLIENT_ID || !process.env.CLIENT_SECRET) {
    return cb('Please specify the client id, client secret and registration secret')	  
  }
  if (!process.env.ICP_DEFAULT_PORT) process.env.ICP_DEFAULT_PORT=8443
  if (!process.env.ICP_MASTER_IP || !process.env.ICP_DEFAULT_PORT || !process.env.ICP_PROXY_IP || !process.env.ICP_PROXY_PORT){
    return cb('Please specify proxy ip and port details')
  }
  var redirects = []
  if (process.env.PROXY_REDIRECT_URL) {
    process.env.PROXY_REDIRECT_URL.split(",").forEach(function (final) {
      redirects.push(final);
    })
  } else {
    redirects.push("https://"+process.env.ICP_PROXY_IP+":"+process.env.ICP_PROXY_PORT+"/auth/liberty/callback")
  }
  var id = 'oauthadmin'
  var secret = process.env.OAUTH2_CLIENT_REGISTRATION_SECRET
  var auth = 'Basic ' + new Buffer(id + ':' + secret).toString('base64')
  var reqBody = {
    "token_endpoint_auth_method": "client_secret_basic",
    "client_id": process.env.CLIENT_ID,
    "client_secret": process.env.CLIENT_SECRET,
    "scope": "openid profile email",
    "grant_types": ["authorization_code", "client_credentials", "implicit",    
            "refresh_token", "urn:ietf:params:oauth:grant-type:jwt-bearer"],
    "response_types": ["code", "token", "id_token token"],
    "application_type": "web",
    "subject_type": "public",
    "post_logout_redirect_uris": ["https://"+process.env.ICP_PROXY_IP+ ":"+ process.env.ICP_PROXY_PORT],
    "preauthorized_scope": "openid profile email general",
    "introspect_tokens": true,
    "trusted_uri_prefixes": ["https://"+process.env.ICP_MASTER_IP+":"+process.env.ICP_DEFAULT_PORT, "https://"+process.env.ICP_PROXY_IP+":"+process.env.ICP_PROXY_PORT],
    "redirect_uris": redirects  
  }	  
  var options = {
    url: process.env.PLATFORM_AUTH_SERVICE_URL+ '/oidc/endpoint/OP/registration',
    headers: {'Authorization': auth, 'Content-Type': 'application/json'},
    json: true,
    body: reqBody,
    rejectUnauthorized: false,
    method: 'POST'	  
  }
  request(options, (err, res) => {
    if (err) {
      logger.error('Unable to generate new client id'+err)
      return cb(err, null)
    } else if (res && res.statusCode === 201) {
      logger.info('New client created')
      cb(null, res)
    } else {
      cb(null, res || {})
    }
  })
}

exports.regDelete = (req, cb) => {
  if (!process.env.OAUTH2_CLIENT_REGISTRATION_SECRET || !process.env.PLATFORM_AUTH_SERVICE_URL) {
    return cb('Please specify the auth service url and registration secret')
  }
  var id = 'oauthadmin'
  var secret = process.env.OAUTH2_CLIENT_REGISTRATION_SECRET
  var auth = 'Basic ' + new Buffer(id + ':' + secret).toString('base64')
  var clientId = process.env.CLIENT_ID
  var options = {
    url: process.env.PLATFORM_AUTH_SERVICE_URL+'/oidc/endpoint/OP/registration/'+clientId,
    headers: {'Authorization': auth},
    rejectUnauthorized: false,
    json: true,
    method: 'DELETE'
  }
  request(options, (err, res) => {
    if (err) {
      logger.error('Unable to delete client registration details'+ err)
      return cb(err, null)
    } else {
      cb(null, res || {})
    }
  })
}

