import * as React from 'react';
import { css } from '@patternfly/react-styles';
import styles from '@patternfly/react-styles/css/components/CodeEditor/code-editor';
import { Button, EmptyState, EmptyStateBody, EmptyStateIcon, EmptyStateSecondaryActions, EmptyStateVariant, Title, Tooltip } from '@patternfly/react-core';
import MonacoEditor from 'react-monaco-editor';
import CopyIcon from '@patternfly/react-icons/dist/esm/icons/copy-icon';
import UploadIcon from '@patternfly/react-icons/dist/esm/icons/upload-icon';
import DownloadIcon from '@patternfly/react-icons/dist/esm/icons/download-icon';
import CodeIcon from '@patternfly/react-icons/dist/esm/icons/code-icon';
import Dropzone from 'react-dropzone';
import { CodeEditorContext } from './CodeEditorUtils';
export var Language;
(function (Language) {
    Language["abap"] = "abap";
    Language["aes"] = "aes";
    Language["apex"] = "apex";
    Language["azcli"] = "azcli";
    Language["bat"] = "bat";
    Language["bicep"] = "bicep";
    Language["c"] = "c";
    Language["cameligo"] = "cameligo";
    Language["clojure"] = "clojure";
    Language["coffeescript"] = "coffeescript";
    Language["cpp"] = "cpp";
    Language["csharp"] = "csharp";
    Language["csp"] = "csp";
    Language["css"] = "css";
    Language["dart"] = "dart";
    Language["dockerfile"] = "dockerfile";
    Language["ecl"] = "ecl";
    Language["elixir"] = "elixir";
    Language["fsharp"] = "fsharp";
    Language["go"] = "go";
    Language["graphql"] = "graphql";
    Language["handlebars"] = "handlebars";
    Language["hcl"] = "hcl";
    Language["html"] = "html";
    Language["ini"] = "ini";
    Language["java"] = "java";
    Language["javascript"] = "javascript";
    Language["json"] = "json";
    Language["julia"] = "julia";
    Language["kotlin"] = "kotlin";
    Language["less"] = "less";
    Language["lexon"] = "lexon";
    Language["liquid"] = "liquid";
    Language["lua"] = "lua";
    Language["m3"] = "m3";
    Language["markdown"] = "markdown";
    Language["mips"] = "mips";
    Language["msdax"] = "msdax";
    Language["mysql"] = "mysql";
    Language["objective-c"] = "objective-c";
    Language["pascal"] = "pascal";
    Language["pascaligo"] = "pascaligo";
    Language["perl"] = "perl";
    Language["pgsql"] = "pgsql";
    Language["php"] = "php";
    Language["plaintext"] = "plaintext";
    Language["postiats"] = "postiats";
    Language["powerquery"] = "powerquery";
    Language["powershell"] = "powershell";
    Language["pug"] = "pug";
    Language["python"] = "python";
    Language["r"] = "r";
    Language["razor"] = "razor";
    Language["redis"] = "redis";
    Language["redshift"] = "redshift";
    Language["restructuredtext"] = "restructuredtext";
    Language["ruby"] = "ruby";
    Language["rust"] = "rust";
    Language["sb"] = "sb";
    Language["scala"] = "scala";
    Language["scheme"] = "scheme";
    Language["scss"] = "scss";
    Language["shell"] = "shell";
    Language["sol"] = "sol";
    Language["sql"] = "sql";
    Language["st"] = "st";
    Language["swift"] = "swift";
    Language["systemverilog"] = "systemverilog";
    Language["tcl"] = "tcl";
    Language["twig"] = "twig";
    Language["typescript"] = "typescript";
    Language["vb"] = "vb";
    Language["verilog"] = "verilog";
    Language["xml"] = "xml";
    Language["yaml"] = "yaml";
})(Language || (Language = {}));
export class CodeEditor extends React.Component {
    constructor(props) {
        super(props);
        this.editor = null;
        this.wrapperRef = React.createRef();
        this.timer = null;
        this.onChange = (value, event) => {
            if (this.props.onChange) {
                this.props.onChange(value, event);
            }
            this.setState({ value });
        };
        this.handleGlobalKeys = (event) => {
            if (this.wrapperRef.current === document.activeElement && (event.key === 'ArrowDown' || event.key === ' ')) {
                this.editor.focus();
                event.preventDefault();
            }
        };
        this.editorDidMount = (editor, monaco) => {
            // eslint-disable-next-line no-bitwise
            editor.addCommand(monaco.KeyMod.Shift | monaco.KeyCode.Tab, () => this.wrapperRef.current.focus());
            Array.from(document.getElementsByClassName('monaco-editor')).forEach(editorElement => editorElement.removeAttribute('role'));
            this.props.onEditorDidMount(editor, monaco);
            this.editor = editor;
        };
        this.handleFileChange = (value, filename) => {
            this.setState({
                value,
                filename
            });
        };
        this.handleFileReadStarted = () => this.setState({ isLoading: true });
        this.handleFileReadFinished = () => this.setState({ isLoading: false });
        this.onDropAccepted = (acceptedFiles) => {
            if (acceptedFiles.length > 0) {
                const fileHandle = acceptedFiles[0];
                this.handleFileChange('', fileHandle.name); // Show the filename while reading
                this.handleFileReadStarted();
                this.readFile(fileHandle)
                    .then(data => {
                    this.handleFileReadFinished();
                    this.handleFileChange(data, fileHandle.name);
                })
                    .catch(error => {
                    // eslint-disable-next-line no-console
                    console.error('error', error);
                    this.handleFileReadFinished();
                    this.handleFileChange('', ''); // Clear the filename field on a failure
                });
            }
        };
        this.onDropRejected = (rejectedFiles) => {
            if (rejectedFiles.length > 0) {
                // eslint-disable-next-line no-console
                console.error('There was an error accepting that dropped file'); // TODO
            }
        };
        this.copyCode = () => {
            if (this.timer) {
                window.clearTimeout(this.timer);
                this.setState({ copied: false });
            }
            this.editor.focus();
            document.execCommand('copy');
            this.setState({ copied: true }, () => {
                this.timer = window.setTimeout(() => {
                    this.setState({ copied: false });
                    this.timer = null;
                }, 2500);
            });
        };
        this.download = () => {
            const { value } = this.state;
            const element = document.createElement('a');
            const file = new Blob([value], { type: 'text' });
            element.href = URL.createObjectURL(file);
            element.download = `${this.props.downloadFileName}.${CodeEditor.getExtensionFromLanguage(this.props.language)}`;
            document.body.appendChild(element); // Required for this to work in FireFox
            element.click();
        };
        this.toggleEmptyState = () => {
            this.setState({ showEmptyState: false });
        };
        this.state = {
            value: this.props.code,
            filename: '',
            isLoading: false,
            showEmptyState: true,
            copied: false
        };
    }
    static getExtensionFromLanguage(language) {
        switch (language) {
            case Language.shell:
                return 'sh';
            case Language.ruby:
                return 'rb';
            case Language.perl:
                return 'pl';
            case Language.python:
                return 'py';
            case Language.mysql:
                return 'sql';
            case Language.javascript:
                return 'js';
            case Language.typescript:
                return 'ts';
            case Language.markdown:
                return 'md';
            case Language.plaintext:
                return 'txt';
            default:
                return language.toString();
        }
    }
    componentDidUpdate(prevProps) {
        const { code } = this.props;
        if (prevProps.code !== code) {
            this.setState({ value: code });
        }
    }
    componentDidMount() {
        document.addEventListener('keydown', this.handleGlobalKeys);
    }
    componentWillUnmount() {
        document.removeEventListener('keydown', this.handleGlobalKeys);
    }
    readFile(fileHandle) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => resolve(reader.result);
            reader.onerror = () => reject(reader.error);
            reader.readAsText(fileHandle);
        });
    }
    render() {
        const { value, isLoading, showEmptyState, copied } = this.state;
        const { isDarkTheme, height, width, className, isCopyEnabled, copyButtonSuccessTooltipText, isReadOnly, isUploadEnabled, isLanguageLabelVisible, copyButtonAriaLabel, copyButtonToolTipText, uploadButtonAriaLabel, uploadButtonToolTipText, downloadButtonAriaLabel, downloadButtonToolTipText, toolTipDelay, toolTipCopyExitDelay, toolTipMaxWidth, toolTipPosition, isLineNumbersVisible, isDownloadEnabled, language, emptyState: providedEmptyState, customControls, isMinimapVisible, showEditor, options: optionsProp, overrideServices } = this.props;
        const options = Object.assign({ readOnly: isReadOnly, cursorStyle: 'line', lineNumbers: (isLineNumbersVisible ? 'on' : 'off'), tabIndex: -1, minimap: {
                enabled: isMinimapVisible
            } }, optionsProp);
        return (React.createElement(Dropzone, { multiple: false, onDropAccepted: this.onDropAccepted, onDropRejected: this.onDropRejected }, ({ getRootProps, getInputProps, isDragActive, open }) => {
            const emptyState = providedEmptyState ||
                (isUploadEnabled ? (React.createElement(EmptyState, { variant: EmptyStateVariant.small },
                    React.createElement(EmptyStateIcon, { icon: CodeIcon }),
                    React.createElement(Title, { headingLevel: "h4", size: "lg" }, "Start editing"),
                    React.createElement(EmptyStateBody, null, "Drag a file here or browse to upload"),
                    React.createElement(Button, { variant: "primary", onClick: open }, "Browse"),
                    React.createElement(EmptyStateSecondaryActions, null,
                        React.createElement(Button, { variant: "link", onClick: this.toggleEmptyState }, "Start from scratch")))) : (React.createElement(EmptyState, { variant: EmptyStateVariant.small },
                    React.createElement(EmptyStateIcon, { icon: CodeIcon }),
                    React.createElement(Title, { headingLevel: "h4", size: "lg" }, "Start editing"),
                    React.createElement(Button, { variant: "primary", onClick: this.toggleEmptyState }, "Start from scratch"))));
            const editorHeader = (React.createElement("div", { className: css(styles.codeEditorHeader) },
                (isCopyEnabled || isDownloadEnabled || isUploadEnabled || customControls) && (React.createElement("div", { className: css(styles.codeEditorControls) },
                    isCopyEnabled && (!showEmptyState || !!value) && (React.createElement(Tooltip, { trigger: "mouseenter", content: React.createElement("div", null, copied ? copyButtonSuccessTooltipText : copyButtonToolTipText), exitDelay: copied ? toolTipCopyExitDelay : toolTipDelay, entryDelay: toolTipDelay, maxWidth: toolTipMaxWidth, position: toolTipPosition },
                        React.createElement(Button, { onClick: this.copyCode, variant: "control", "aria-label": copyButtonAriaLabel },
                            React.createElement(CopyIcon, null)))),
                    isUploadEnabled && (React.createElement(Tooltip, { trigger: "mouseenter focus click", content: React.createElement("div", null, uploadButtonToolTipText), entryDelay: toolTipDelay, exitDelay: toolTipDelay, maxWidth: toolTipMaxWidth, position: toolTipPosition },
                        React.createElement(Button, { onClick: open, variant: "control", "aria-label": uploadButtonAriaLabel },
                            React.createElement(UploadIcon, null)))),
                    isDownloadEnabled && (!showEmptyState || !!value) && (React.createElement(Tooltip, { trigger: "mouseenter focus click", content: React.createElement("div", null, downloadButtonToolTipText), entryDelay: toolTipDelay, exitDelay: toolTipDelay, maxWidth: toolTipMaxWidth, position: toolTipPosition },
                        React.createElement(Button, { onClick: this.download, variant: "control", "aria-label": downloadButtonAriaLabel },
                            React.createElement(DownloadIcon, null)))),
                    customControls && (React.createElement(CodeEditorContext.Provider, { value: { code: value } }, customControls)))),
                isLanguageLabelVisible && (React.createElement("div", { className: css(styles.codeEditorTab) },
                    React.createElement("span", { className: css(styles.codeEditorTabIcon) },
                        React.createElement(CodeIcon, null)),
                    React.createElement("span", { className: css(styles.codeEditorTabText) }, language.toUpperCase())))));
            const editor = (React.createElement("div", { className: css(styles.codeEditorCode), ref: this.wrapperRef, tabIndex: 0 },
                React.createElement(MonacoEditor, { height: height, width: width, language: language, value: value, options: options, overrideServices: overrideServices, onChange: this.onChange, editorDidMount: this.editorDidMount, theme: isDarkTheme ? 'vs-dark' : 'vs-light' })));
            return (React.createElement("div", { className: css(styles.codeEditor, isReadOnly && styles.modifiers.readOnly, className) }, isUploadEnabled || providedEmptyState ? (React.createElement("div", Object.assign({}, getRootProps({
                onClick: event => event.preventDefault() // Prevents clicking TextArea from opening file dialog
            }), { className: `pf-c-file-upload ${isDragActive && 'pf-m-drag-hover'} ${isLoading && 'pf-m-loading'}` }),
                editorHeader,
                React.createElement("div", { className: css(styles.codeEditorMain) },
                    React.createElement("input", Object.assign({}, getInputProps())),
                    (showEmptyState || providedEmptyState) && !value ? emptyState : editor))) : (React.createElement(React.Fragment, null,
                editorHeader,
                showEditor && React.createElement("div", { className: css(styles.codeEditorMain) }, editor)))));
        }));
    }
}
CodeEditor.displayName = 'CodeEditor';
CodeEditor.defaultProps = {
    className: '',
    code: '',
    onEditorDidMount: () => { },
    language: Language.plaintext,
    isDarkTheme: false,
    height: '',
    width: '',
    isLineNumbersVisible: true,
    isReadOnly: false,
    isLanguageLabelVisible: false,
    loading: '',
    emptyState: '',
    downloadFileName: Date.now().toString(),
    isUploadEnabled: false,
    isDownloadEnabled: false,
    isCopyEnabled: false,
    copyButtonAriaLabel: 'Copy code to clipboard',
    uploadButtonAriaLabel: 'Upload code',
    downloadButtonAriaLabel: 'Download code',
    copyButtonToolTipText: 'Copy to clipboard',
    uploadButtonToolTipText: 'Upload',
    downloadButtonToolTipText: 'Download',
    copyButtonSuccessTooltipText: 'Content added to clipboard',
    toolTipCopyExitDelay: 1600,
    toolTipDelay: 300,
    toolTipMaxWidth: '100px',
    toolTipPosition: 'top',
    customControls: null,
    isMinimapVisible: false,
    showEditor: true,
    options: {},
    overrideServices: {}
};
//# sourceMappingURL=CodeEditor.js.map