/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode = require("vscode");
const client_1 = require("./client");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
var protocol2code;
(function (protocol2code) {
    function asSemanticTokens(value) {
        if (value === undefined || value === null) {
            return undefined;
        }
        return new vscode.SemanticTokens(new Uint32Array(value.data), value.resultId);
    }
    protocol2code.asSemanticTokens = asSemanticTokens;
    function asSemanticTokensEdit(value) {
        return new vscode.SemanticTokensEdit(value.start, value.deleteCount, value.data !== undefined ? new Uint32Array(value.data) : undefined);
    }
    protocol2code.asSemanticTokensEdit = asSemanticTokensEdit;
    function asSemanticTokensEdits(value) {
        if (value === undefined || value === null) {
            return undefined;
        }
        return new vscode.SemanticTokensEdits(value.edits.map(asSemanticTokensEdit), value.resultId);
    }
    protocol2code.asSemanticTokensEdits = asSemanticTokensEdits;
    function asLegend(value) {
        return value;
    }
    protocol2code.asLegend = asLegend;
})(protocol2code || (protocol2code = {}));
class SemanticTokensFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.SemanticTokensRequest.type);
    }
    fillClientCapabilities(cap) {
        const capabilites = cap;
        let capability = ensure(ensure(capabilites, 'textDocument'), 'semanticTokens');
        capability.dynamicRegistration = true;
        capability.tokenTypes = [
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.comment,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.keyword,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.number,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.regexp,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.operator,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.namespace,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.type,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.struct,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.class,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.interface,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.enum,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.typeParameter,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.function,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.member,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.macro,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.variable,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.parameter,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.property,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenTypes.label
        ];
        capability.tokenModifiers = [
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.declaration,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.documentation,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.static,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.abstract,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.deprecated,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.async,
            vscode_languageserver_protocol_1.Proposed.SemanticTokenModifiers.readonly
        ];
    }
    initialize(cap, documentSelector) {
        const capabilities = cap;
        let [id, options] = this.getRegistration(documentSelector, capabilities.semanticTokensProvider);
        if (!id || !options) {
            return;
        }
        this.register(this.messages, { id: id, registerOptions: options });
    }
    registerLanguageProvider(options) {
        const hasEditProvider = options.documentProvider !== undefined && typeof options.documentProvider !== 'boolean' && options.documentProvider.edits === true;
        const documentProvider = {
            provideDocumentSemanticTokens: (document, token) => {
                const client = this._client;
                const middleware = client.clientOptions.middleware;
                const provideDocumentSemanticTokens = (document, token) => {
                    const params = {
                        textDocument: client.code2ProtocolConverter.asTextDocumentIdentifier(document)
                    };
                    return client.sendRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensRequest.type, params, token).then((result) => {
                        return protocol2code.asSemanticTokens(result);
                    }, (error) => {
                        client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensRequest.type, error);
                        return undefined;
                    });
                };
                return middleware.provideDocumentSemanticTokens
                    ? middleware.provideDocumentSemanticTokens(document, token, provideDocumentSemanticTokens)
                    : provideDocumentSemanticTokens(document, token);
            },
            provideDocumentSemanticTokensEdits: hasEditProvider
                ? (document, previousResultId, token) => {
                    const client = this._client;
                    const middleware = client.clientOptions.middleware;
                    const provideDocumentSemanticTokensEdits = (document, previousResultId, token) => {
                        const params = {
                            textDocument: client.code2ProtocolConverter.asTextDocumentIdentifier(document),
                            previousResultId
                        };
                        return client.sendRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensEditsRequest.type, params, token).then((result) => {
                            if (vscode_languageserver_protocol_1.Proposed.SemanticTokens.is(result)) {
                                return protocol2code.asSemanticTokens(result);
                            }
                            else {
                                return protocol2code.asSemanticTokensEdits(result);
                            }
                        }, (error) => {
                            client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensEditsRequest.type, error);
                            return undefined;
                        });
                    };
                    return middleware.provideDocumentSemanticTokensEdits
                        ? middleware.provideDocumentSemanticTokensEdits(document, previousResultId, token, provideDocumentSemanticTokensEdits)
                        : provideDocumentSemanticTokensEdits(document, previousResultId, token);
                }
                : undefined
        };
        const hasRangeProvider = options.rangeProvider === true;
        const rangeProvider = hasRangeProvider
            ? {
                provideDocumentRangeSemanticTokens: (document, range, token) => {
                    const client = this._client;
                    const middleware = client.clientOptions.middleware;
                    const provideDocumentRangeSemanticTokens = (document, range, token) => {
                        const params = {
                            textDocument: client.code2ProtocolConverter.asTextDocumentIdentifier(document),
                            range: client.code2ProtocolConverter.asRange(range)
                        };
                        return client.sendRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensRangeRequest.type, params, token).then((result) => {
                            return protocol2code.asSemanticTokens(result);
                        }, (error) => {
                            client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.SemanticTokensRangeRequest.type, error);
                            return undefined;
                        });
                    };
                    return middleware.provideDocumentRangeSemanticTokens
                        ? middleware.provideDocumentRangeSemanticTokens(document, range, token, provideDocumentRangeSemanticTokens)
                        : provideDocumentRangeSemanticTokens(document, range, token);
                }
            }
            : undefined;
        const disposables = [];
        const legend = protocol2code.asLegend(options.legend);
        disposables.push(vscode.languages.registerDocumentSemanticTokensProvider(options.documentSelector, documentProvider, legend));
        if (rangeProvider !== undefined) {
            disposables.push(vscode.languages.registerDocumentRangeSemanticTokensProvider(options.documentSelector, rangeProvider, legend));
        }
        return [new vscode.Disposable(() => disposables.forEach(item => item.dispose())), { document: documentProvider, range: rangeProvider }];
    }
}
exports.SemanticTokensFeature = SemanticTokensFeature;
