/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_1 = require("vscode");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const client_1 = require("./client");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
var protocol2code;
(function (protocol2code) {
    function asCallHierarchyItem(converter, item) {
        if (item === null) {
            return undefined;
        }
        let result = new vscode_1.CallHierarchyItem(converter.asSymbolKind(item.kind), item.name, item.detail || '', converter.asUri(item.uri), converter.asRange(item.range), converter.asRange(item.selectionRange));
        if (item.tags !== undefined) {
            result.tags = converter.asSymbolTags(item.tags);
        }
        return result;
    }
    protocol2code.asCallHierarchyItem = asCallHierarchyItem;
    function asCallHierarchyItems(converter, items) {
        if (items === null) {
            return undefined;
        }
        let result = items.map(item => asCallHierarchyItem(converter, item));
        return result[0];
    }
    protocol2code.asCallHierarchyItems = asCallHierarchyItems;
    function asCallHierarchyIncomingCall(converter, item) {
        return new vscode_1.CallHierarchyIncomingCall(asCallHierarchyItem(converter, item.from), converter.asRanges(item.fromRanges));
    }
    protocol2code.asCallHierarchyIncomingCall = asCallHierarchyIncomingCall;
    function asCallHierarchyIncomingCalls(converter, items) {
        if (items === null) {
            return undefined;
        }
        return items.map(item => asCallHierarchyIncomingCall(converter, item));
    }
    protocol2code.asCallHierarchyIncomingCalls = asCallHierarchyIncomingCalls;
    function asCallHierarchyOutgoingCall(converter, item) {
        return new vscode_1.CallHierarchyOutgoingCall(asCallHierarchyItem(converter, item.to), converter.asRanges(item.fromRanges));
    }
    protocol2code.asCallHierarchyOutgoingCall = asCallHierarchyOutgoingCall;
    function asCallHierarchyOutgoingCalls(converter, items) {
        if (items === null) {
            return undefined;
        }
        return items.map(item => asCallHierarchyOutgoingCall(converter, item));
    }
    protocol2code.asCallHierarchyOutgoingCalls = asCallHierarchyOutgoingCalls;
})(protocol2code || (protocol2code = {}));
var code2protocol;
(function (code2protocol) {
    function asCallHierarchyItem(converter, value) {
        const result = {
            name: value.name,
            kind: converter.asSymbolKind(value.kind),
            uri: converter.asUri(value.uri),
            range: converter.asRange(value.range),
            selectionRange: converter.asRange(value.selectionRange)
        };
        if (value.detail !== undefined && value.detail.length > 0) {
            result.detail = value.detail;
        }
        if (value.tags !== undefined) {
            result.tags = converter.asSymbolTags(value.tags);
        }
        return result;
    }
    code2protocol.asCallHierarchyItem = asCallHierarchyItem;
})(code2protocol || (code2protocol = {}));
class CallHierarchyProvider {
    constructor(client) {
        this.client = client;
        this.middleware = client.clientOptions.middleware;
    }
    prepareCallHierarchy(document, position, token) {
        const client = this.client;
        const middleware = this.middleware;
        const prepareCallHierarchy = (document, position, token) => {
            const params = client.code2ProtocolConverter.asTextDocumentPositionParams(document, position);
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyPrepareRequest.type, params, token).then((result) => {
                return protocol2code.asCallHierarchyItems(this.client.protocol2CodeConverter, result);
            }, (error) => {
                client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyPrepareRequest.type, error);
                return Promise.resolve(null);
            });
        };
        return middleware.prepareCallHierarchy
            ? middleware.prepareCallHierarchy(document, position, token, prepareCallHierarchy)
            : prepareCallHierarchy(document, position, token);
    }
    provideCallHierarchyIncomingCalls(item, token) {
        const client = this.client;
        const middleware = this.middleware;
        const provideCallHierarchyIncomingCalls = (item, token) => {
            const params = {
                item: code2protocol.asCallHierarchyItem(client.code2ProtocolConverter, item)
            };
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyIncomingCallsRequest.type, params, token).then((result) => {
                return protocol2code.asCallHierarchyIncomingCalls(client.protocol2CodeConverter, result);
            }, (error) => {
                client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyIncomingCallsRequest.type, error);
                return Promise.resolve(null);
            });
        };
        return middleware.provideCallHierarchyIncomingCalls
            ? middleware.provideCallHierarchyIncomingCalls(item, token, provideCallHierarchyIncomingCalls)
            : provideCallHierarchyIncomingCalls(item, token);
    }
    provideCallHierarchyOutgoingCalls(item, token) {
        const client = this.client;
        const middleware = this.middleware;
        const provideCallHierarchyOutgoingCalls = (item, token) => {
            const params = {
                item: code2protocol.asCallHierarchyItem(client.code2ProtocolConverter, item)
            };
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyOutgoingCallsRequest.type, params, token).then((result) => {
                return protocol2code.asCallHierarchyOutgoingCalls(client.protocol2CodeConverter, result);
            }, (error) => {
                client.logFailedRequest(vscode_languageserver_protocol_1.Proposed.CallHierarchyOutgoingCallsRequest.type, error);
                return Promise.resolve(null);
            });
        };
        return middleware.provideCallHierarchyOutgingCalls
            ? middleware.provideCallHierarchyOutgingCalls(item, token, provideCallHierarchyOutgoingCalls)
            : provideCallHierarchyOutgoingCalls(item, token);
    }
}
class CallHierarchyFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.CallHierarchyPrepareRequest.type);
    }
    fillClientCapabilities(cap) {
        const capabilites = cap;
        const capability = ensure(ensure(capabilites, 'textDocument'), 'callHierarchy');
        capability.dynamicRegistration = true;
    }
    initialize(cap, documentSelector) {
        const capabilities = cap;
        const [id, options] = this.getRegistration(documentSelector, capabilities.callHierarchyProvider);
        if (!id || !options) {
            return;
        }
        this.register(this.messages, { id: id, registerOptions: options });
    }
    registerLanguageProvider(options) {
        const client = this._client;
        const provider = new CallHierarchyProvider(client);
        return [vscode_1.languages.registerCallHierarchyProvider(options.documentSelector, provider), provider];
    }
}
exports.CallHierarchyFeature = CallHierarchyFeature;
