package fake

import (
	"bytes"
	"compress/gzip"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"sync"
	"time"
)

type localFS struct{}

var local localFS

type staticFS struct{}

var static staticFS

type file struct {
	compressed string
	size       int64
	local      string
	isDir      bool

	data []byte
	once sync.Once
	name string
}

func (fs localFS) Open(name string) (http.File, error) {
	f, present := data[name]
	if !present {
		return nil, os.ErrNotExist
	}
	return os.Open(f.local)
}

func (fs staticFS) Open(name string) (http.File, error) {
	f, present := data[path.Clean(name)]
	if !present {
		return nil, os.ErrNotExist
	}
	var err error
	f.once.Do(func() {
		f.name = path.Base(name)
		if f.size == 0 {
			return
		}
		var gr *gzip.Reader
		gr, err = gzip.NewReader(bytes.NewBufferString(f.compressed))
		if err != nil {
			return
		}
		f.data, err = ioutil.ReadAll(gr)
	})
	if err != nil {
		return nil, err
	}
	return f.File()
}

func (f *file) File() (http.File, error) {
	return &httpFile{
		Reader: bytes.NewReader(f.data),
		file:   f,
	}, nil
}

type httpFile struct {
	*bytes.Reader
	*file
}

func (f *file) Close() error {
	return nil
}

func (f *file) Readdir(count int) ([]os.FileInfo, error) {
	return nil, nil
}

func (f *file) Stat() (os.FileInfo, error) {
	return f, nil
}

func (f *file) Name() string {
	return f.name
}

func (f *file) Size() int64 {
	return f.size
}

func (f *file) Mode() os.FileMode {
	return 0
}

func (f *file) ModTime() time.Time {
	return time.Time{}
}

func (f *file) IsDir() bool {
	return f.isDir
}

func (f *file) Sys() interface{} {
	return f
}

// FS returns a http.Filesystem for the embedded assets. If useLocal is true,
// the filesystem's contents are instead used.
func FS(useLocal bool) http.FileSystem {
	if useLocal {
		return local
	}
	return static
}

var data = map[string]*file{

	"/data/en/adjectives": {
		local:      "data/en/adjectives",
		size:       119,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x14\x8b\xc1\n\xc20\x10D\xef\xf3W\x85H\xf5d\xb0\xe2=\xa6cYH\xbb\xb2n\xf0\xf7\xbb\xb9\xcc\xf0\x98y\x93\x18f6L\xdd\x15Y\xff4<\xcb\x16\xb8ʻ\x11ׄ9/1\x98\x97\xc1I~\x15\x97\xc6\xea&\x15\x99\xf6Q\xdb\xcbQ\x89EV\xe2\x15\xa1\xb8\x1d\xdf\xee\xb8w\x1f\x95\xc4\xe2\x8d\aw\xf5\xe1o\xe2\xa5\xe1\f\x00\x00\xff\xff\xd8\x18m\xadw\x00\x00\x00",
	},

	"/data/en/characters": {
		local:      "data/en/characters",
		size:       72,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x04\xc0\xc3\x01\xc4\x00\x00\x00\xb0\u007f\xa69c\x9c\xdav\xa7o\x02\xa1H,\x91\xca\xe4\n\xa5J\xad\xd1\xea\xf4\x06\xa3\xc9l\xb1\xda\xec\x0e\xa7\x9b\xbb\x87\xa7\x97\xb7\x8f\xaf\x9f\xbf+\x00\x00\xff\xffq\xa7L\x96H\x00\x00\x00",
	},

	"/data/en/cities": {
		local:      "data/en/cities",
		size:       4837,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\\X[\x9a\xe3(\xd2}\xd7*\xb4\x81\xfe\xf7\x90wgwf\xa5\u007f\u06dd\xf9\xd5cX\xc2\x16m\x04\x1a@v\xb9v5k\x98\x8d\xcd9\x01rWϋ\xe3\x04B\b\xe2r\"\xf0]o\x9c\xf8\x1c\x9a\xbbc\x98\xa3\xb4+\xeb\\j\ue70c\xa6\x17Ƚ\xf8+\xc4 \xe3>R\xb7)\xb4\x9f\xd6\xfc\x85\x17\\\xc6\v\x98;J\x1fb\xfb`3&\x8e&\xdaN|\xfb\x80ׂo\xee\xbc\fƎ\x90\xbd\x89I\a\x8e\xc6%<\x1e'\xe0lC74w\xd3\xe4L\xfb)\xce\x19\xac\x10;魨\xcc\x141\\C\xfb\x12\x05+@\xcb&\xe9ó\xc5ZY\x12&\x9b\x88\xbd\xe4\xc1\xc4\xcc\xf5\xf3E\xb2\x89\xcdݼ\x9f#Գ8\x8e\x9e\x8d\x17\xd7\xdc\xfd\x9c\x934\xf7r\xc2^\x0eָ\x1e\xd8\xf5\x17\xeb۵\xc4\x13\x14\xef\xad?BƔå\xb972\x8f\xc1g\x00\xe7\xf4\xa7}\x91\xd8\x1b\x9fT9\xb8p\xc1\x97\x00\x97Ig\x83\xcd\x18 o;\xec\xf2\xdeē\xe1\x99\xee\xcd\xd9Dw\xad\xb6\xbd\xb7\xc7\x16K\xc7\xf6\r\x1b\xa1v\xe4X\x1a\xc2\xd4ܻ\xd9\xd4awő\x9a\xfb(=\x0er%\xb8\xe8R\xd1\b\u007f|\xbe\x84\x80\xfdG\x9b\xe0!L\x9c\xb1#\x1a\x00\xc0K=\xcf\x1c\xf1L\xa5ù\xe0\xd1\xe6A\x9c\xec%\xc1k@\xe6\x87\xed\x02\x81=\x84\xe8\xad\x14\x17R\x1f\r\xecD0I\x8e\xb6@\x98\xe4H4J\xc41\xf8\x1e=\\v\xfb \x93\xcd\xc1\xf1qt\xd8PO0\x1a\xf7\xdb\xfe\xfa\x1b\x8e\xf1\xdb\xd6\xe8\xa3\xc9z\xf8F\xd7\xd3`x\x10<죸\xfae\x18/\xf17b1\x80A\xb77X_\u007f\xab\xfd\x1e\x86p\xe9\x06@\xac3\xccN\x10\x8e\t\x91\x82%\xe2\x9cڕ\xb1\xc7!c\x96\x93h\xd41@WZ\xe6\xc1\xc1\xe8N\xb7\xeb\x02\x1c\xd2\xc3\x02\n-f\a\xe9\x06\xa3K\x06Qc\x01\xb8\x83\xfc\xa0\x18U\xd1%\x82c\x04q\xb7m8\xb4\x0faD\xbcwX%\x8cSy\xee\xbb\x10q\xfa\x10!\x85\x03QC\n2x\xa9\xa2\x0f\x04ٴ\xef\f^\x8eb\xff\xed\xbb\xda<d\xc9\x16\x02\xf1\r\r\xf6\xe8\xe0\xeaj\x9e\xb9\x97\x81\x02q\x16\x97\xa1\ta\xaf\x06\xbaN\x98\x9c\x9aGA\x9c\xe9\xb3Ga\x1c\x04\x98\x9c\xf0\f{\x19\x00\x1e\xf6\xd18|:\xaaܘk\xfb!'\x1d\x14\xac\xf3h\x12VlW!\xb7\xdb)b\xebxb\x05v\xec\xdb{\xbeb\xfd\xbc\x17\x88\x1f8\xedc\x80\xa7\xf0<$ě\vD\x17\x8f\x18}\x9c\x05\xa7\x83\xd8Ò\x10>\x8d\xb3\x8d͓\xa4\xac\x13[0Gh\x9e\xe0u\xf9\v\xcb\x10\xe0\x90\xb1\f\x15\xef\x13\xbe\xc3y\x86`k\x8e\xb3\xef9v\x02\x11\xc0u\xcd\x13\xcc~-Gz\xf2\x9d\xf5\x16L\xd0<\x81\f\x98\xec\x90خ\xe5\v\x19F|\x9a\xa39A\xfc0\xe4\x85g\xb1\x91nU\xa9\x14\xf0\xccH\x8d\xa9\xac\xf6l\xa2\u05f8x\x86:\x86H\x10\xcd^\xe6\xe3\xd0<\a\x97\xc2\b\xe13L\v\t;!-\x8f\xc7\x02\u007f\x0f\x1e\xd1K8\xebӔ\x17/=\x87\x19\xaf\x80e*\xc5=\a\xa42\xb6R\xe3\x132\xc1\xf2\xcf3\x1e*\x87\xbd\x88\xcbMa\x9azޢH\xf3b\x1d\xe8\xb0yq\xa6\xecR\x01\x02\xady\t\xced\n\xff\x13\xe3\xa9Q\xbelw0\xa6 <\xa1\xa5\xb4|\xfd%\x1a\xe3\xcb\xd1_\xa2\xed\xcb\x10\xf3\x81\xa4\x046~\x99\x05k#\xb0\x80\x12b\xcb4+q\ax\x03\xc9~/Wh\xfe\xc0\x18_\xc9E\xac\x05\xd5/\x9c\x88\x81< \xde\xf9\xc2\xf5\"\x9cb@\xaf\t\xfcu\x04\x1cM\xc6o\xecf\xee\x0f`\fI\xea'W&M\xca\v+\xdb\xf3\xd0%\xcfWHd\xc4$\x96\xa1\xbaG\xea\xd0\v\xab\xe0\xc8E\xb0\x1fP.^[\xe1\x01\xf9d\x05;#d3wZ\x16\xfe{@\x19q5#\xff\x9a\xd7Q\xbf\xe6n\xa0\xce~E\xd4\xe0<_\x86_\xa7\x12n\x99\x0e\r\xc6\x00\x0f\xbf\xfa\xa33J\xbd\xaf\xf0w\xf3\xcaBD\x81\"\xa2\x1eyMp\x04\xbe\xf1\xbbt'n\xe9\x0f\x1c\x14<\xf0\avQ\x82\x81\xa8\x98\xe4\r\x84+\xff\xf97\xac\xdd>#Q@\xb4\xfd\xd1pt%,\xb5\v\xb81\xda[\xe5\bJ\x8b\xba\xa0\b\xf94\x16\x00\xde\xcf\xfa\xfa\xff\xcfXL\xc7>\r\xbd\xf1&\a\xb9\x9a\xac\x0f\x91H\x8bѫRl]\x95o\xf6\x88:\xb2h_8&\x1f\x9dL\xfb\xe4\x12X&\x9a\xa2!\xc8Mʊ'Ļ\x02\xb5ɛ\xf8N\xd49o0w\x9af\x82<D\x94\xb67\xb9\x14\v\xbd1\xeak\\\x13\xeb\xaa\x16\xbc\xe9<e\x9f\x10co\xf6lHK\n\xe2X\xa6\x9ci9\xda\xf6-\xf4\x16?\xa3\xb4\x9c/\x846\xab\x98B\a\xe1\x8f\xcb\x11\x03\x9e\xe0\x04 (64ZU\x8a\xf2\x0f\xb4\u0600:\xdb\x13S0ځ:\xebE\xca\xfc\xabh\x15x\xbbz=m\xa5\xefw\x14\x8c\xfd\f1\x8e!\x0fZ\x10\x134?H\x06Qԭ@Ϧ\xe3d\xf4Ga*\xc0\xab`\x8e\xfd$\xb8V\x12zG\xfa\xe8\xfa\xdd3\v\x162\xe0\x9di\x8e#B\x82>5\x94\x01\xed\xc1`2\v\x10f\xe0\xcd%É\x118\x86`Rj|\xb7)YX\xbd4n\uf847\xfb(=\xbc\x80\xb4#\xd3\xc2\xe5FA\xe7@\x8d\x8a\f\x98\x17\x9dF(\xcaް\xea+\x8c\xfcH\x05u7p\xe3T\x00\x18\x91+\xf2\xc5ۆB<JIk\xe2\x18\x94H\xdeI\x8b\xedv\x10V\xef\xf7\x1bGZsi\xdegT\x01\x12\xda7\x99\xf8\x83>\xd1/]\xc27cz\xba\xe8\x9b93u\xea\u0605\x1f\x87`\xb2ALJ\xcej\xf9o\xac\xc5\xfc\xbd\x88Ôp\x863\x1bĖ\x06#\xa4\x1a\x98\x12\x1b\xfd\xe8\x8c\xf8d\xd1i~\xfc%\xb6\xf9\xc0\x96\"X\xe0\x03l\x8a\xd4,\x02U\xffL\xac\xa1\xf7Q\xfc\xf3A;\xd2u\x1f?<\x89o-\x9d=خFyդa\xb2\xb6\xa5\xc2\x16\xbc\x94X*\xba\x9f\xbfˣVTf0|\xd5>\xc1D\xe8|1\x88\xbc\xb7\xc9(\x18i2\"g\x91pktuZ#\x00B\xbb\t{\xa7\xf3\x91\xf6\xdaj\xad\x8d\xd6\xea5\x8c\xeaf0\xc6\x1aQ\xd8n\x98]Ħ\xd7J\xb4F\x8es\x136\x8d\xa1Zoms.t\xb5v\xc2^\x04\xe1\xa2(\x96\x03\xaf\xd1T%\xc4vq\xee\xa2\xf1\x83\xee\x8a\rf\xac\xc0\x16\xa4\xbd\x8b\xba\xb70\x06\x15l1@X(\n\xaa\xdcV\x03f\x1bY\xe5\x12?\xebpA\xbcl@.C\x80\xf9c\x0f'\xdeԹc{r\xbc\r\x90\x1bᬪ\xfdj\xc6el\x8b\x1d\n;\xa0#\xbc\x8b \xdb\x1847n>\x1c\x88z\xb6k\x90\xf4k\"@/\xb1\x18\x03\x1a\x13\xb3\xc4\xdcF\xe3\x10Ғ\x15 \xba\x81m\x12\x00h\xbcS\x82\xdc\xd8\x00w\xc3.\x04\xa5K\xdd\xd8\t\xc6Q\xc3ko\xaeH#n\x13\xba\x13\xfb\xa5M\x18<[0ͫ\r\xab\x1d\xf8\xac0\xd4?\xb4[PlB\x82\x1d\xd1s\x13\x14;\x02\xa5f+\x1d\x82\xc4\xf0n\xb7eO+\x1c\xf2\xa0\xb8\xc4\x1b\x8bb\xdcP\x18\xafl U\x8dsE\xec\xe3C\x99\x8f\xb6y\xd4\xd2B\xe5њc\xa8h\xac\v\xb2g\x12\xf6g\xaa G:\x8b\x0eL\xb5\x17\x140t\x1a\x8aQ\x11m\xd9\vp\x90\u007f\xa1JU\x9c\xcaڿ\xcf܂\xccY\ru\x1b\xe1\xdd\x02\xa5W\xea\xceސ\xa1}\xacx\xb6\xa9\xfd\xd8\xdb4\x15\x1d.\xed\uab95^\x97\xd1l\nZ\xcb\xde\x15\xb4AM\xac\xfb\xda0|\x88\xaa_\xb7L\xf2ؔ \xc1\xfd\xb5\"t\xbf{$]\xd5x\xbb\xf8\x15\xdb|\xd3\xe2\xfc\xb3\xc2gsK\xf0[\xc8\xd9e\x1e\xf8\x93\x94P\x94\xb5\xe0\x1eS1|W\xa0\xa1Y\xa2\xe8\xb5k+\xb8w86\xc6\xdb. #\x97\xcc\xc0\xc5j\xe9^\x005\x8e\xb6hW\xc1\xeeS\xc0\xf1\f\x9b\x030\xec\x81%\xb90m\xb9\xb0m-[C^@\"\x95#\xd9U3xkG{[\x1c)xk\x17\xa0\x98\x1eA\x06y\x86\x85 }\xe0\xe2\x10\xb4\x04S\xbd\xbd5\xedEE\xd9\\\xa0\xb6\r;\x19\xc22p\xa3\xab\xa2\xfeO\xe8\xe4\xffC\xd7\xe3\xf4q.t\xb2\xcdH\x8f\x02f4f\xd5U\xf0\xff\xec\x17\xec\xfd\xf5L\x0e\xdd\xc2T\xf5\xbe\xb3\x9d\xb3v\xe1hyO\xcd\x0efhvf\xc0i\x10R\x8ap\x80\x1d\xa2\xbb\xa3\xf5\x01\xf87\x84.\xb6\x1b\x02\x17\xe9\xcb\xd5h\x87\xf2ξq\x87҅-\x9af\x87\xae\x1a\x93\xe0ZK\x93\xec\xe2ܝ\xe0\xad\xdd\xcc;'\x85\xd1k\xe6\x0e\xf7n\xec\x1a\x92}t\xb3\xbb \x8d\xae\xb8\x0e\x1a\xed\xdaR\xf3\xe7\xc9\xca\xd0\xfc\xe9Y\x99\xf5\xb3\u007fNZM>\xa5\x93\xb2\u007fu\x04*\xf6'ޜaf6t\xd8ȧ\xedr\xa8\x94\xf9ɋp!\v0\fbD\x9a\xc24_\xe2\xfc\x9c\xab\xa8&\xf8\x12\x9a\xf7\x8b\xff\x90h\x0f\x0f\x84\xe2P\x16\xfa\x02\x9f\xe1\aw\xb4z\xfbT\xfc\v\x89P\xe7\xb9Z\xfd\xe4\xd1\xe8\xc0h\xbd\xf6{\x8aC\xa9\x86_\x83\x91\\\x11\xca\a\xab\xd3\x17^\xc152\x15\x90\x8b\f\x17J\xd0l\xe0\x04\xb6\x13\xd0\xc1\xafj\xbc\x02\xb8\x06\xbbP\x06\xf6\xf7\x80\f\xac-\xdfw־\xb2\xef\xefz\xbd\xfb\x8eki1\xe2w\x14\x04;q\xa4\xeb\x96\n\xf2\xdf\x00\x00\x00\xff\xffPpU\a\xe5\x12\x00\x00",
	},

	"/data/en/colors": {
		local:      "data/en/colors",
		size:       128,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\f\xcaA\xaa\xc2P\f\x05\xd0\xf9\xdd\xd5\xff\x82ED,R\x04\x87\xa1\xefRC\xd3ĦF\xb7oGgrnl\xf8\xb7\"\xba$\x1d\x0f\x9a\xc5\x17\xd7\x14\x9f\x88\xbe\xf2e\xc4]\xc3\xf8\xc6ɛN\x81\x81b\xe8\xd5g\x1ck\x1bUЅ5zF\xc3Eꃿ\xb5d\x91T'\x86ʵB7\u243al\xe1\xfb\xc8\xd8\xe9k$\xceO\x99\xf5\x17\x00\x00\xff\xff\xfe\x16\b\xa6\x80\x00\x00\x00",
	},

	"/data/en/companies": {
		local:      "data/en/companies",
		size:       3157,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffTV\xdbR\xe3<\x13\xbc\xd7[q\b\xd4G\b\xb0@\xb1\u007fr'\xc7J\xa2\x8dm\xb9|\b\xb1\x9f\xfe\xef\xee\x91\xd9ڋ-6\xb6<\x9a\xe9\xe9\xee\x99\xf5w\xac\xe2pr+?%\xf7\x95R\xfeQ\xa4\xab{\x8cG\xfd݄p\t\xc9mS]\x8cn\x15\xbb\xe0^\xa7Ԕ\xa1s\xf7S\xe3\xe7yr\xf7\xa1\x8e|\xf9;\x9e\xe3<\xbb\xff\x9a&\xed\xdc[\xbc\xa6f\xba\xba\x8fƷ\x83?\xf6n\xeb\xeb\xe8\xdd\xcd\xc5\u05f5\xbb\v\xcd\x10\xf5\xb3\xf0\x83w\x0f!\x94u\xbc\xba\a_\xe2\xa6\xcfS\x1a\x8f\xa7\xa1\xe8by\f\xcb/\xbe\xbe9\xc6\xcb\xe8\x9e<\xc3ގEQ\x05>\xbd\x8f\x17\xdf\x06\xf7\x1c\x18\x13\xcf\xe7\xb9\b\xbe\xe1\xe1\xba\xf08}\xed[\xfc\xfc8O\xf8\xf6\xa1\x8am?\xa4\xaev\xb7]\xfa\xeeCQ\xf9\u00adpM;V}pw\xe9\x18Y\xf2\xb3\xff\x93\xdc{\xf0Ւ\xc3wD\xd2\xc0\xaa\x19\x97|,\n\xe3\xa5\x16u\xdc\xfb\xa2?\xa5\xa1w/\xad\a\x82\xa8\xf2\x12\x82\xbb\xad\xd2\xf1\u007f\x1f\b\ap\x91\x80\xd2)\x8a\xd0\xf5\xed)t?\xa5)\x89\x8fslP\xcc\x1b\x82\xa0\t\xfd \x14\xff\xf8Z/\b6@G\x95\x19\xe2[\xcf\xea\xf5\xe1\xfbi\x9ac\"\xba\x05n$\x18MJe\x15\xf2\x99>\f\xee\x13)\xee\xbf=nܦ2%\xf4\xebҦ\xd8\f\xaa\\A\xf0\xa4?\xf1\xc0g7\xf2\x04\x81\xe0_\x16\xd0\xfa\x92\xccP\x12\x9b\xa9\x19\x02\xda4\xcc\xee\xd1\x17\xc3\xd8\x04<*\x00\xb9[\xfb\x89,Y\xfbyt_\x91\xa7T\n+\x00\x81p~\x97\x12\x0f\xe8i1\x1eY\x0f\xf0 \xc8\xfb\xb1@\xfb\xe2%\x1cb\u007fB\x04\xa4\xf1D\xc4n\xbbHt\xd8L\xbe\x15\xb0@1\x97\x87\x94\xe3a\"\xf0\xbcc\xed\xed\x8azN\r?\xaf\x98\xbc\xb5\x18w}\x9c\xc6á\ne\x870L\x87\xa4}\xc0\xffs{QJߦ\xc1\xbd\xe3\xfbZ/\xf6'?\xa8x0\x97a\xf3\x9d\xeb\x05\xdb;p\xb9$\xe8e\xa8\x04\xaaal\xa4\xb0\x1e\x0fS\xcbD\x86.\x928\xe4\xf5\u007f\xecӗ\xee\x00+U\x17\x9a\x9b\xf9\xb8\x89\xd0\x01%\xb7\xafF\xb1\x81\xea\xc8\x19\xf4@\x14\xe5Qn\x17P(\f\xdfT\xe1\x06TM\xee\xd7(\x9a\xaf\"\xc9\xffJ\x86\x8du\vαc\xe0S曮\xf8\x87\xb9\x9b\xe9;u\xe7\x9e0\x12\xa1\xe7\u061c3\x1a@\xf4\x00=\x12'Q\n\xb9 \xecV\xdd[I\xef?=$\u007f\x94\xaf\x1c\xc1\xe4\xc5\xe7\xd4>\x93\xdb1\xdfO\u007f\x14Q\xd8\x1d\x8ak\x15\t\x1a\x10h\xfc\x10SCq\rC\xe8,\x9dߩ+\xdbPF\xf3\x84|\x02\x19\xed\xdcf$ʾ\xf2MX>Y\xd4ٙ\x871\xe1Zw\xb7?G$\xa5W2\x8d\xcc\x0e$\x85\x8f\xcd\x05\x90\x04\xa1m\xa2\x85\x86rc\xc9\xc7\x03\xd1\xddҗd\x01\xd4\a07\x87\xd8\xfa\xb3\x84\x81\xa2\xa8\v\x02\xf0m\xc7\xf7\xa13\xbd\xe4\xa4e\x01ɽ\x92\x89π\r\xe0\xe0\xb6\xecE\xb1\xd5% \xdc5\x1b\r5\xb4\x19\xf1K\xd4\a\x9b\x8c\xac/$\xa0\xb8\xa5<P\xa2\t\xb8\x96\vTc\x98\x81*\xb3Q\xcb֞\\\xd8L{\xb0\x97l!ܦ\xa22\x1d\xd1\xc2Q\xd73i\x86}WC\x044\x9b\x86\xd8p=ei\xee\x80\xf3\u007f\x05r\x89\xc0\xfd\xd7H\x99\x81r\x9dI\x80\x9f\xdfğ!\xa1\x83R\xb3IO\x12\xc8\xdaㅛH羏\xc78\xf0\x10\x1dNG4\rH\n\x9a73}\xb2\xb1\x02/\x83g[,\x83\x83\xac\x1e\x98\a\x9d\x18\xd9\\9\x97F90\x99CC\x00\xb8̄l\xdb{4\x9f`\xcb\xc2Է\x9cM\x0fj\x88\xf3\x15\xfe\xf5?\xc8X\\\xc1]\xa5\xb1A\xe5\x04G\xb6\xf4\x9a\x00\xb8\x12\x85\xc11\xa6J\xf8\x8a\xad]Ŭ9%\nk\xf6\xe4^r\x80\x8f̗p\x16\x15\xd7\xd2\xf5By\xc9X\x96\xbah\xd3\x06\x03\xf8(:ъ\x8c1\x966<c\xd0 \xa9\xb2\xbd\xe2\xe5^p\x80\xf8$1\xe7\xed\xd9\x17\xa2\x1c:\xb5fF\x00\x83\xba\x94\x06\xe9\xa3;\xf7\x18\n\xc3K\xcf4\xf5=\xd9\xc0\x16ܨG\xbb\xec\xf7\x1c7h\x81܁\f帞5?\xa4\x06h\x8a\xa7_\x01\tA\xc8%o&)\xf5(\x16\x86\x8b\xba`>$n\x00\x11ɕ\xb47\xa2\xa1\x91u\xceHS\x8a\xa3\x85\xbc@[\xcc\xfd䙨\x8fL\xfb\xa4\x96\xde\xe3\xe1`6aSN\f\x90\xc1\x10!\xf3'\xfa+\x1e\xb1\x19\xc0\x9b\xa2\xea\xc6\xd6:*L\x91&\xf4\xf9\xa2\x1bОC\x85s\xe4\x1bc\xab3\x1c\x82\xcaT\x84\x90\xd0\xe0Z\xe6S,\x8f3B\xde\xe39\xfd\xd8\xfa\xad \\\xa9\x06\x98\x8b>\xb9\xcf\xc5\xdc\xe18zn\x15\xbfU\xdc0\x10Ĩ\x1bI\x12f)\xd0wr\x05bn\x02:k\x1ba\xc5y\xb5\x90\xe1C\x97\xa4\xfb&\xda\b\xe2l\xd9L\x80)\x8f}\x99<\x00\x94/\xf0\x02b\xc7]b\b恙j\x1c\xcbc\xf6&\xac\n7Q\x06\xbb\xf3\x8c\xa7s\xb9\xa9\xb6o1\x88\x99\x80\xfd\x1e\xfe\x8a=\x17;ϸ\xdcB\x9bv:3\xcd?\x1a}\xd2\u007f6\x93\x0fˉ\x9a\x97\x1c\x81\x89\xc8\xfa\xb3\x12,\x0e\xf6\xcf\xceD룾1q\xd8\xd2e ?i\xb9\\\xc6&\xf3\x16WLUֲmb\xda0h3ASƍ\x82mu\x0f/\xe6B\xa9\x8c\xb4\r\xbd\x12\xe9ܸl\x85\\@\xcaŔ\xc8\xf0\xac\xe5\xec\xfa\x0fKBˈ\"\x1f?\xad0\xab\xe3g\x9f\x19\x96\xb5';\tF\xe5\x97\f\xe4-\x95\x92\x8e/d\x19Xl\xafZ\xac\xa2Qd`\xd6<\aAp\xb8\xa8w\xf4\x9f\xec\xc1\xcbVɛ\x1f\xf5\xe4\xcbx\xb5\x12FZo4\xb3\x93\r2\nD\xeb\x13ǉm\x9c^\xb4\aČ\u007f\xa3\xdd^\x9a\x12\x99\xd8Mn5\xb6\xf1\xd8\xfc\x04r\xe6\xa4+nD\xff\x0f\x00\x00\xff\xff\x16.\x1a<U\f\x00\x00",
	},

	"/data/en/continents": {
		local:      "data/en/continents",
		size:       68,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffr,\xceL\xe4\xf2\xcb/*\xc9Pp\xccM-\xcaLN\xe4\n\xce/E\xe29\x96\x16\x97\x14%\xe6\x00U9\xa6\x81\x05\\K\x8b\xf2\vR\xb9\x1c\xf3J\x12\x8b\x92K@\"\x80\x00\x00\x00\xff\xff\xffM\x18\xc9D\x00\x00\x00",
	},

	"/data/en/countries": {
		local:      "data/en/countries",
		size:       2774,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\\VM\x96\xdb8\x0e\xde\xf3\x14\xdaM\xf2^\xfa\x10\xb1\xeb\xdfv\xc5SrU\xbf\xce\x0e\x92\x10\t1E:\x14iG\xde\xcf\xc9\xe6b\xfd\x81\x92]\xf5\xb2!@\x80\"\xf1\xf3\x01\xd0\xd7\x1fmGN\x86H\xce\xfc\xff\u007f\x96\\c\xbe\xda\n\x12\x02m9(\xedAjrEI\xbd\xc7\xd65>\x04\xa5\xad\xb7\x99$\xb1\x99\x89\x14ꈓ\xcaJ\x9b\xa8\xc0młB\x95\x1a\xc8B\xcb\x10;\xe5z\xce\xf7\x87Ta\x1djv\x83xW<\x0e\xd3\xf3i\x88\x81\xac\x1ePN\xe9\x99CE\xf2\x13&.\xa8\xa3\x9e\x06\xa5\x81D\xf7\xae\xb5\xd4\xf0\xd0\x19}\x88\x1a\x0f\x1d[\n)\xd3VR\xafT\xce\f\xe2\xf4\x03\x0e\xbd\x9a\xb3蒺\xbc\xf0V\x8exb\xe1\a\x98\x94\r~\xe0p\xe6\xd6\x1f\xd5ԅ\x8fÉ2\x93\x8e\x1c/&.\x02\x9dłH\x94\xa1+\xde$\xb4r\xb1\u007f\xb8\x8a\x1f]#\x88ڷ\x9a\xb1\xee8@\xec\xc3\bur,ō\xda8\x90%\x18\x98lK\xea\xe8\"\x85=\x9e-\xeeh\xf0\xbaI\xb8\xc1,\xa9\xaf<nR\x86\x83\xf7\x0e\x8c\xa3F\xf7\a.\xde84\fv\xec\xe9݂%\"\x8d\x10\x16_\u007fL\x89{\xe1C\xaa\xac\xd4f\xd9Q\x83E,\xeb\n\xbf\x96]@\xee\x11ыkK_\xfb\xa1\xf8\xb4b\x04͵\x9f\xdf\xc5\xd6\xf7\x95\x1a\xe1{\x1f\x10\xe4\xa5G\xfa\xbf\\o.\xfc\x8f\x8b\xe8\x86{_\a\x02\x10\xfe\xd0\xfa\xfd\xbb\x81\x1e\x80+^\x14+K\x1f\xb9h\xfe\xf3x\xf4\x12`U\xf0\xf8\x12R\x85\xc6r<h\x1e\x97g\xae\xbbw\x1fn\xd8\xf5\x14\xf6\xe6\xe6\xa7T>E17\xbe\x17\xa7W]\x98\x0f\x0e\xdf\xd6\t\x98\b\xe6\xb6\x1d\x0f\xd1\xdcZ`\xd8\x1e'ɯDȇ L\xf7I\x1c\x93\xb9E\x82\x82\xd2!z\xc5\xe7m\xec\xc4\x1f\xc0ܑݫ\xdd\x17\xf3\x8bO\x1b܂\xf0\r\x9f\xa1\v\x9e\xaf~\xdd\xc9O\xc1\xe2r\xcc\xee\x02\xb9\x9aA\xd8\xc1~<\xa2@\x9aw[oGǃ\\\x05%\\\xe98\xbc\x03Ex0\xf7T!\xdd\xf7\x94#\u007f\xcf\x1e0S\x1a\x90\xeb\xd1\xdcwz\u07fdT\xc84*\xcf\xdc\af<\xa7dz\x1f\\\xc6\xc9=B\xc0֧\x03+\xdb\xeb\x12\xb9\a\xf2\xc0\xe1Łq\xd7\x14\x81\x89\xfc\xb5\x10\x003a7\xea\v\x0f\x048\x9b\a\xa6\xd0\xe4\xfa\xd8\xd47ޑ\xbdF\xc3<xפ@\x99i\x8b\x15\x16\xf3\x90\x1c\x10=\x9aǚ\xb3)\xb9\x14t\xf5\x93ӏ\b\x8d.\xbf\xb0\xcc'\x06ˀI\xb1Q\xc5\x10\x88\xady\x8cdG\xf3\x84z\xd7\xe4>\xd1\x01\xaa'\x0ej\xef\x93\x0f\rv+:Ӿ˽k\xc5n$\xb3\x92 \x15\x00dV\x1e\x99\xfcR<\xfb\x10\xbb\xcb&\x87ج\xd2\t\x0e\x99\xd5\x18\xda\xf1\x9c?]\x13\U0003c9a8}`\xcd\xe8}\x88\xf9\x9a\a\x1f;o\xd6R\xe5\x1e\b:\xea\n$F\xb4\xab\xc8h%k\x89]ʝr\x9d~3\x8a4\x85\xd6l\xa8F\xe8\xb0r\x93A\xb4A\x06Z\x1aj$\b\xa0\xa1\x93d2\x0eYe\x1b9\"\xcd`\xb28\xaa,\f\x1dY{\r.\x04h\x9a\xf2+1\xd8\x04d\xd0t\xab\xb2\x92T?\xfa\x18\xa1\xe4\xdfR{\xb3\x91:\xcc1\xdex\xdb\xf8\xa3RG\xaa\xd1\xea\xb4Y\xee\xe0\x02\xb7!\xcb\xe2\x00\xbcQ\x04\x1b|\x9d\x8f\x9d\x15n\xf9=\xa4\x1f\x95f\x9e\xa9\x17\x05\xe03\x1eM\xe6\x99\x0fd\xb1*Z'\x13?\xf0\x85v~kY\x85\xa7bIv\x8e\x82\xee\xbe3\xe5L?#\x9b\x810\x1e\xc0\xe1\x95g\xc1\x94\x99V=\xe8\xc3\x0fo/\x8d\xc2\xe4\xfciUl\xb4?\xa23^\xc2\x02ŉF\xf3\r\x95`\xb6\xb4\x9f\xe6\xd7\x16\x0f$]yИ}\xec\xba_І\xebt\x10n\xa0w\x80\x14\xc8\x01\x13J\r\x9b\xc1\xbf\x9d\xac\x1a͖\xe1\xe6\x16MR\x0e\ah\x06\xb3\x95X\x93\x84\xeb\x84B\xf1N$\xc4\xd4\"\x18[\x94Q\xf4\xdaμ\xf9/i-\xbepr\x18i\xe6\xc5\xf79_/h\xf5J0KP\x90%&W\xd4\xc1\b\xdf,\xf7\xe3,x\xc0\xc6]\xb4+\x89q\xc8\xf5\xf6\xccG\x19f\xe9:\xd5r91!c\xdel\x05\x9e\xf2T\x9f\x9a<\xeb/\x9a7A\x13\x02U\x15\xde+\xa6\xae\x90\xfd\x9a\xe6yISx\x9d\a닝\xef\xa7{\xb6\x10Ղ\xa6QRB\xf5\x16_\x03)\nJ\xc5\x0e\x9c.1\x93\xf3v\xacᅦ\xbcT\x1b\xa8X3\x00\x88\r:\xc0\x01\x85gJ\v\x14\xee\xf5(\x98<\xf8K\x1d&\xfe}\\\x95\x88\x92\xcdr\x14\xe8<\xb4\xe6\xcd\xdc\xf4\xae\xd6OR\x98ܜ\x04]\xf3z\xc3A\u007f\x06\xca Ś\xdc\x1e\x1f'm\x0f%\x8a\x04\xb9\x86\tG\xc2?\xcdܿ\x9e\xb2\xbf#\u007fx\xff\xa4\xe3\\SZ\x9e\xb8a|x\x92x\x9e m\xcaQq\xb9#A\xee@~\xca\f\xb6\x1d\xb9s\xce\xed\xae\xa3\xe9\xe3\x9d`4\xfe\x85\u0381r\xdc\xf9\xd6cٳ\"r\x87\xd2\xc39\x18#\rMF\xec|Ez\x020Q`\xec0\xf8\xd1֔\xe8\xbf\xd1\xf4\x006\x13\x02\x96h\x80~\xb8\x9a\xbbKp'\x99\xd76\x83\xe9u\xaf?Bl^\x9dDnr\x96\x8a\xdb^P\xd3H\xc9,\\!\x19\x8d\xef/\xdb2\xaaR\xbb\xed\xfck\xf7\x87\xe2\x8f\xff\x99אr]\xbc\x96\xc0\x96\xf3\xa1\xf8\x96\xa2\x1dq\xe5\xfb\x89\xb2|1\xaf\xe7\x8a\xe7ؼ\x91ÌI\xa01\x8f\xe3\xa5\xc4q\xba\xbc\xf8\xf4\x80\xc9W\x94̟\xcd\x1b\x90t\x06Tɼ\tG$\xca\xfc\x8d\xce'\x93\xcfw)\xa6\x0f\xf5\xfe\xb7F5\xe8/h\x8725\xff0\xc2d\xfeI\xad\x87^\xfb\xf6\xf7iF~\x97\xbe\xa2\xea\xc4\xff\x06\x00\x00\xff\xff\xf40\xec\xfc\xd6\n\x00\x00",
	},

	"/data/en/currencies": {
		local:      "data/en/currencies",
		size:       1800,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfftUKv\xeb6\f\x9ds\x15\x1c\xc6\xcb\xf0\xf7%\xb1\x9d\xb8\x96۞\x97\x19,\xa1\x12k\x8at!2~\xca\xea{)\x8bJzN:\xb1\t\x10\x04..>\x9a\xffU7\xe4L\x17\xc8\xe9|Vs{\xc6?\xe9\x1d_\x18B\xcd\x02ae\x1c\t\xeeZH%\xe9\x87ߝ\t\\\xe9\"P\xe0n\xa6W\xde\xda\xe1^jv\x01\xb6\xfa\xc0\x9d\x87\x1c\xbb d\x93\x83l\x914\x90\x8b\xb21\xd6\x1aWwjA\r\xb5\xd4M&\x90\x85\x8c\xcb1\x17\xe4jK\x15w\x8d>х ˙*\xffŞmmb\xab7B\xaeL\xa2\xb4\xb1\xfa\x8c\xb8\x10\xfa0V\x1f\x99\x90\xdc\"ښd\xc8\xee\x9dԒ\x1c}\xb1\\n\xe6z\xb1\\\xcf_\xb3\xabA\xb1\x9e/'\x19\x98yL\rg\xa4\xf93\x82\xba#\xbbָ\xb3Q\xc7\xfdB?|s1SKo}{6\x99\x97\xe5\xe60\xf9\xf4\xa0_\x1f\x13\xab\xc9\xc81T\xe2)\xc0v\x1b\x1d0\xf6W\x89\x9d>\xf8\xe8*\\}p\xd9\xc0\xef5\x9e\xad)\xf5\xd6Kt\xbdZ!\x0e\xc9Eo\x05\xef\x05b\f]ٰ~\xf8\x01.\xc8\xf53\xbd\xc7u\xa7V\x1ep\x10\xc9}z\xb8\xe3Y\xc5\x00\xb7\x0f/\x1c\x1a\x16K\x884\xd3?\xa2\xb1\x15\x83\x965u\x81\xc5\xe9%\x88;\x9f\x19\xaf3a뺿\x86\fm\xdd\x05\x9f\xfa\x06 \xf0\xaf\xd6Q\xe0xc\xfe6\x93\xf9Ƹ\xe4{\x00s!R\x03\x03\x9c\x89\x18\xb1\xea\t\xfd\x1d\xf3\x0fo+\xfd\x1aa\x88\xb30\xe3\xc1J\xa8lZb\xf5\x98\x1c\xc3\xe1\xff\x00\u007f\xf4\xae\x06E\xf8\xc9\b\x1e\xa3C\xf9{\xbd\xf1b\\PO%\x0f\xef\x13oQ\xd4\xd3~\xa3\x8b+\x97\x86l\x8aqCw\xa20u\x03CW!\xb3c\xbc2P@He\xba+\f5\xd0H\xa2\x14φ\xe3?\xb9s\x9f\xe4\xee~$\xe8\xa9\x13b\xab_\xf8\xa6\x8b\x06Õ\xac\x03\xd9^\uf330z\xc6\f\xa4&\xc8P\x9f\xe9\n\xa7?٩g/\x15M\xe3\xb0e\xd7c|\xa6\xe9A\v0\x06\xb2\xf0143\xfd\xa7wj\x1bodB\xb6\xdf1\xc6ٻ\fb\x17\u007fq{\xf6Q\xea\xcc\xfb\x9e,\xf5C6\xf0W\x9b0hВ@ep\x8cb\x82A\x03\x8e\xc9\xef\xf9\x97)\xfd\xd86{/\xbe\x84\xb42\x82\x11\xeeԗ2|V!\xe5\xfb\xc640\x91\x93{\xf1r\xa3>\xf7\xebk;\xd1w\xa0\xcb}\x1d\x8d\xe1\x0e\x84\aU\x9a\xee\xb1\x05\x0e,Q\xbfD~\xc7\x02(\xbcM\x1aPa\xaeW\x83\x92\x8c\xb0\x0e\x16\xe3㾼\xf1\xd6`{\xbcY\x1fz\b\x12b\x8d\n\xaf\xbb2b\x8d\xa8\xdf(\x90 |\x9f\xe2\x1f};,\xbf\x04z\xc7f\x92\x87s\xec\xeeE?\xa7\xb0\x05\xc5\xca\xe8\xb9\xd0y\xa0nx]\x18\xfbΒ\xc3\x16\xe0\x93\xae(ϔva\xfd;\x12\xa4<\xb7I\xe6\xe4\xfe\xe4\xef\xf7\xa9\x88\xd8\xc4Î=\x82\xb0Qs\xafq\xaamqM\x9b\x11ir \xd8\u007f\u05ec\x9d~@#\xcfT!F\xef\xc8]\xa6\xbe-bj\xa3\xb1\x0f\x8a\x1bW\xec\x86\nx\x81d\xc2ǽr\xb9-Ndn0ODd\xf8\xa7\x86\xcc`\x82\r\x1d\xd4\t#d*\xaatҜ\xfc\x99j\xffi\x19\xf1%\xf9\xfcl\x9c\xa2\\x\xe8sR\xe3\x97#\x11\xa7\xd7-4\xf8\x80L\r4^n\x91L\xe5ی\xf5?\x1f\x9b)\xc6\x1f\xec\xf8#b\xc2\xf4\x02\xf5}'\xe1ouz\x13YB\xba3\x1c\x1c\xb5x\xefj\xf5F\xc36\xdeްIH\xfd\x1b\x00\x00\xff\xff\xbc\xbdΙ\b\a\x00\x00",
	},

	"/data/en/currency_codes": {
		local:      "data/en/currency_codes",
		size:       519,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff$\x91Kz\x80 \f\x84\xf7s+0\xa0\xc8C\x1a\xb0E\xef\u007f\x90\x0e\xb8\x98/Bb\x92\xf9q\xb7\xe2n\x82\xddVlF`n\xc1Y\x1f\x84\xa2(\xaf`;<^\xa3\x98u\xc6\x17\x98\x94 \xbc\x9f\xff\x18m\xab\xde\xf4\x06˳=(鰖\xd1y\xd8̨\tv/\xab\xf7\xb8<\x86\xf1\xd8\x12g\x95\xe7\xd3U1*\xeftá\x11\xdb\xc3\xdc\x1b!\x91r\x19\xc2|I;\xc6&p{\x85sq\xed\xe2O\x81\x0f\x19^\xfd\xaa\x1b\xe6Ʈ\xb2j\x8f(8n\x8f\xd0\"\x86\xe8\xf2\x12fT\xeaG\x10\\EH\r\xa1'\x9c\xf9\xf3{^\x82\xe8\x1a\xa2\xfe!\xfe\t\x12y$\xf6ȏ\"\xb3.sf\x1e\x05\xd9|3&\x9brE\\YQ\xa3҃\xa0\xba\x82zL?\x1d5\xf1\xbb;\xfc\x90\x9d^\x85J\xd0ۢ\xf1<̎\xb6\v\x1a=\xb6\xd0\x17\xdf9\u05f5\xba\xf6M\xecׄ5\xf4:\xf9w\xee\xd3\x0f\x8b\xde\x19\v\xa5\x0f\x8c\xfb\xdel\xbeï\xb3\x94\xc7/so\xfe\xf8L\u038b7\xd9.v\xf3Ntq\x9a\xf3W\xe4\x9eso\xe6\xfe\x03\x00\x00\xff\xff\xf7O\x92\xa7\a\x02\x00\x00",
	},

	"/data/en/domain_zones": {
		local:      "data/en/domain_zones",
		size:       753,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\f\xcbA\xb6\xa5 \f\x00\xd1y\xed2 F\x84 \x8f\x80\xb6\xae\xbe\xff\xf4\x9e*\x89Ȇ$dG\x14\xc9HE\fiȅ\xfc\x90\x8182\x91\x85<\xc8?\xe4#\b!\x106B\"\xec\x04%\x1c\x84L8\t\x95`\x84F\xb8\b\x83\xe0\x84I\xb8\t\x0f\xe1%|D!F\xe2F܉J<\x88\x99X\x88\x95h\xc4F\xbc\x88\x83\xb8\x887\xf1\x1f\xf1%~l\x89\xedd+l\xc6v\xb1}\xa4HJ$%\x1d\xa4Ar\xd2$-\xf6\xcc~\xb2\x17vc\xbf\xd8\a*h@74\xa1;\xaa\xe8\x81f\xb4\xa2\x866\xb4\xa3?\xf4\xaftt\xa2\v}З\xa3p\x18G\xe3\x18\x1c\x93c\x917r\"W\xb2\x91\x1b\xf9\"\xffȃ\xec\xe4ə8\x8d\xf3\xe2\xec\x94DQ\xcaA\xc9\x14\xa34\xca\x1f\x0e\xcaCy)\x1fU\xa8\x81\x1a\xa9\x99Z\xa8\x83\xea\xd4I]ԛ\xfab\x82El\xc3\x12\xa6\u0601\x15\xacb\x865\xec\xc2:\xf6\xc3\x06\xe6\xd8\xc4\x16vc\x0f\xf6\x0f\xfb\xdb?\x9a\xd0\"-\xd1v\x9a\xd22\xad\xd2.Z\xa7\rڢ}\\F\x17z\xa2\xeft\xa5\x1f\xf4B\xaf\xf4?o\xf4Aw\xfa\xa4?\xf4\x97\x9f0\x12\xe3b8c1\x1e\\\xf0\x80G|\xc3\x13\xae\xf8\x81g\xfc\xc4\v^q\xc3\x1b~\xe1\x03\x9f\xf8\xc2o\xfc\xc5?fdn̝\xa9̃y2\v\xb32\x8d٘\x17\xb33\as2o\xe6\xc3\xfcX\xc2RVa\x19\xcbY/\xeb\xe3\x16\xeeȝ\xb8\x95;s7\xeeų\xf38o❼\x8bO\xf8\x8c\xef\xe1\u007f\x00\x00\x00\xff\xff\x88\xcdet\xf1\x02\x00\x00",
	},

	"/data/en/female_first_names": {
		local:      "data/en/female_first_names",
		size:       686,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff<R]\xaf\x9b0\f}\xf7\xbf*\xed\xdd\xee(\xb7\xaaJ\xb5=\x1b\xf0\x1aK!ٜp\xa5\xec\xd7\xef\x18\xa6\xbd\x99\xd8\xc7\xe7\xc3|\xb05\xbas5\x9d\x95iд0ul\x13\x1b\xd3[\xd4?<I\r\xd4KJ\xfaS\x8c0\x8e\xb1q+\x9c\xbc~\xb1I\xa5K\xb6\\C\x03\xba0\xdd8͍\xaeh$\xea\xa4\xd6F\xef\x12Q\x8f\x9c\x16\xec\xbc䔘\xcel9\xd2c\xc3\xea1\xa0\xc62\x9d\x83\xc4(4\xf0\x86\xb1\x11\xfc\x81\xae\xbaNb\xb1\xd1E\xa6\xec\x0f\xbd\x94\xa23\xdaA-J\xa3sK5@\xd0)\xbd$2}H\xd4\x02\r\x1d\xc8\xe1\xe3\xb46t@\xf7\x90If\xc0\xbe\xab\xbd4a\xfe\xca5D\x81\xaa;\xaf;\x90\xad\x06v\x1as\x18;z\xac\xf2+pR9ԶD\xe7`Z\xaa&\xd9c\x10\xea9\xc1\xfd\x19\xbb\xc4\xfc\xf5\x8b\xc1\xbb\x14\xe7\xa4>\xb7Y\xe8\xa2\x18\xa1ST\xd4\xfd\x16\x81y\x97}\x9c\x9ebR<\x0e\xac\xa4\xaf1{\xaco\x9f\xe24\xbd\xb0s\x89\xb5\x88X\xe2b\xb2\xec\x82\x0f\x92>\xa3{*\xc1\xed\xf7ۢ\x88\xf0\x91\xcb.\xc6Y\xae\b\xb1\xd1M\xe7\x1c\x9dri\xffU\x1f\xae\x1b=\xc3A\xdd\xc9\xe7\xbfl\x93\x02\xff\xe4\x15q}Cr\x871\x1a\xa0\x89\x1e\xecg\xd9\xfd\xba\xb4\x13n(\xc7\"\xc3\xe7\x907\x87\xfa\xb1ܴ\x03\xe7\xdf\x1b\x8e\x80\xf2\xc7\x1eb\x87s\xeb\xe1\x1dg\xd8&\xfc$\x19\x86\x9f.\xe7\x1eZĽ\xe8\x96m\xc5\x17o8\x84\a\xb6\xef\x02hP\xb4a\xf6m\x057LN\x9a\xfe\x06\x00\x00\xff\xff\xff\u0091\b\xae\x02\x00\x00",
	},

	"/data/en/female_last_names": {
		local:      "data/en/female_last_names",
		size:       1764,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffDU\xddr\xe2:\x13\xbc\xd7[\x05\xb2@%\x90P\x86\xfa\xa8\xefr\xb0'X\x85-qF\x12\x8ey\xfa\xd3-\xb297\x9b\xb5\xfe\xa6\xa7\xbb\xa79\x8c>\xf7\xee-\xf6!\xc5\xe0N~\x18\xbc\x8c\t\vA\x93[X\x9c\x82{\x95\xbbOn\x87-5\x9e\xe0\xc1]\x8c\xa6\xee(\xf3\x10ͽ\x84N\x8d\xab\xc7>\x8e\x82\xcb\xd2^\xebk\xbd\xcf\xea6b\xc6\xfbb\xd9?\x8fܸ\xb9\x16k\xbd\xfc,\xeb\xc35\xf1\xec+\x86\xe5 v\xc5gg\xfeR\xb0\xb1\xd5\t\u05f7\xaa\xee$\xc3\x15\x1062\f\xee\x05h\x82\xfb\u007f,\xe1\xe26jA\x80\xe1\xe1\xde=>O\xb8\xd8g\xb7\x8d7\xacl\x00\xdb\x1dژ\xb3[\x9b\xe2\xcaK\xc7\xfe\x16\u0097\xd61<d\xc0\xa9\x0f\xad]-\x01\x06\xcb;\x9f\xdb^qo\xaf\xf6D\xa6\x96\x93;\x16\v\xd8\xdd\xf7$\xe9\x96pz\xbc\x9d\xeb1\x00\xc6Ɵ\xbb\x84\xe4\xfet\x93X\x87݈S\xf8>d\xc5Bv\a\tx\xf4\x01\xe6*\x1dM\xbc\x803רv8\x1a\xaf\\\xbfHp\v>\xb8+v\xebg\x80\xf4\x83ή\xf1w5\xe1\xa9\x1b\xaa4\xbe\xedY\x80p\xe3\xb7\xdbD\x96\x035\xf8爧\xa1\xda^\xf3S\x8e\xb5\tn\xcb\xe8\xd9ś\x8c\xd8;I\xe6\x0e\x84\x8d\xd7\xe4\xdeQl&2\xea\xe7\xf6\xe6[\x05\x80\x10\x14l\x9db\xec\x80\xc0\xe8\x83&\xa6\x04\x96\xffʌ\xd6t\x04\xd67\rW\xb6\b\x9alv\xfb8\x11\xfb6B\x82\xbd\xe4\x1f\f\x9br\xe9\xf1\xc2\n>\xa9\xe5S\x0f\x892!\x94L?\x1d\xfc8F\xbc\xb1\x8a)\xff\xa7\b\xad7KȐY\xbf+<ה\x94\xf8\xfe\xda\xfc\xd7\x17\x8c\xf4\xea\x05\xeaʌ\xb3\xbb\x99\xf0W\x11\x14l\xd0퐟\xbd\xf72\xbaC\x81\nЅ\xde\x19\x04\xed\xb1\xad*\x0e\xfe\xab)\xc3\xe8\xd6a\xfbs\xd2@5\xe6\x10\a\xec\xaf|\xea\xa9(.\xa7\x1f\x03\xf3M\u007f\xae\xdeo\xbb\x18d\x80nV\x1e4p\xea\xe9\xc8O\x18\xf9\x81\x06F\x8a\\\x8cܯ\xe08\x12u\x02p4\xaf\xe73\xfb\xad\xf6\x87-\xee,y,-\xbd\xb3\x8f\xd5y\x9b\x12\xea\x1f\x8f\x01\xc2\xf32}զ4\x80\xdfE\x9c;T{N\x9fU\x92ީU7S}SR\xf2\x8dM\b\x1ek+d\x11\x96M\xc0d\x1dK\xf62\xc1/\x03}\xd0P\xea\x1fcW\x95\xc0\xf5\x02\x04]1\xed\xc1c\x1a !\x0eZ\x1d\xfb\xe4>`\xba\x88\xbf`\x15\a\xdfC\x1d\xb0\x95ڥ\xf06\xec\xa1h\b\x91\x01\xae\xa6j\x89\xd7\x12\x02}Q?6\xa5\xda\xf5p\xd3\xd0Rcx\x95\x83\x04\nn=9\xd8ˌ\xab{\xafV\xfdG3\xed`\xa0^\xa7\xe4^\x8c\x8a\xa0\xc3Kx\xa6\x0f\x15i\x84=\xe7\xfa\xf8\xe7\xdf\xc95\f\x1c\xeb\xb6 \xfc\x10\xe6\xae&\x05\x06o\t(\xf0\x1cݰ0\xe98Q[A=\xa4\x96\xb1BS<\x8dd\x1c\xae\x15&\xaa\xa9C\xf7bc\xcaF//\xb9Ue9\xa9܉\x9f9\xa2xd26T]\xc2|Y\xf6\xd8~Pa\xc0.\x89i\xf7\x1c\xc5:gJ;H\xb8\"\x14x\x95\xa0W \x1af[\x97\x8c\xd6k\xd2\xcc\xc4\xde\xf6\xa3\xefrm\xc9\xfdO\xd2?\f\xc1\xa5\xe0\xbda\x88\xc8TU\xce\r\x8a\xddh\xadρ\xe9\x00G\x84|\x81\xf5l\xfe\r\x88\xdf0G\xa5\x9a\xefu\xea\x80\x00\x03\x03s\x98[\xc0\xe4\xf1\x06;\xb7qf\x8eh\x8dջ\xd8oJ\xd1\x19\xc84Xk\xf5\x1b\xb1P\xef!\xe4\xebN\"}\xc6\x14\xd3f\xb9zZ\x02\xfb\xfc\xb8\x94\x19w֊HS\xfe\x1a\xc43\r\xe9;\xe4\xf3\xe8V\xa5\xfe\x90lg\xe4\xa1{U\xe1\\\r4\"\x9f6FO\xc3F\"\xa7\x06'\xb6Rs\x04\xdc=<A\x17dmE\x15\xf9\xcb\x04\xa7 \x96\"\x10\x01\xb1\x86\x88)\x99H\xcbN;\x1f\x04zN\xa8\xf5o\x00\x00\x00\xff\xffl@\xa6y\xe4\x06\x00\x00",
	},

	"/data/en/female_name_prefixes": {
		local:      "data/en/female_name_prefixes",
		size:       14,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2-*\xd6S\xf0\x05\xe1\xcc\xe2b.@\x00\x00\x00\xff\xff \xc7\"\xf4\x0e\x00\x00\x00",
	},

	"/data/en/female_name_suffixes": {
		local:      "data/en/female_name_suffixes",
		size:       29,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2T\xf0\x04! \x0eS\bS\xf0uQpq\tV\b\xc8\x00\xd2a\xbe\\\x80\x00\x00\x00\xff\xff\x93~\xf7\xc6\x1d\x00\x00\x00",
	},

	"/data/en/female_patronymics": {
		local:      "data/en/female_patronymics",
		size:       666,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff4\x92͎*;\f\x84\xf7~+\xfe\x06\xd4\xc3\\!@wֆ\xf6trH\xdbȝ\x80\xf2\xf6\xa7\x92\xd1\xd9 \x91vU\xea+g\xe0Y\x16\x1a,(\x9d\xed&\x9e\xe9+\xde\x03K\xa2\xef\x98R䙶\xfc\x8a#\x9d۩\x8f\xb4\xc1o\xea\x8aE\x9e\x81\xae\xc1f^p\xeaq\xc9\xf6\f\xe2\x98\xd7\b\xf9\x89K\xa2/\xf6\amM9\x8d\xb4\x17\xf3I\xe8ST%\a\xbady\x89\xd2n|7۵Gn\t\xfa\xe4Js0\xad\x18}E\xa5\x81\x17S8\xe5\x1c\xe4M{\xf6J\xd78[\x0e\xb5\x87\xa0#;\x8e\x06\xf9\xf9q\xa9\xf4\xe1\xac\x0f\xba\xdc-g\xday\xbc\xb7{\x10Ka::\xf4g\xae\xb3)ҸL\xe6\xdd\"\x14\x86\xbc\x99l\x91-.\xf4\xcd)\x03\xe4\xc4\x19\x06\x0f:I\xfbw`7d\xdbZ\x99\x12\x88\x0f\xa2\x10lP\x06\xad<\x87\xe2t\xae\x9d`\xc2\xec`BC\xe1\x96\x1d\xfaU\xea\xbd\x0e\x80ˡ\x1d\x96%\x83\xebگ܋7\xe4O\x89\xad\x13\x9e˿\xe6\x05Q\x13\x1a>\xf2\xdbE\xefB\xffa\x03֮>[\xa5\xb5\xe8\x1f\x9e\xe1\xb2\xf6\x82okP\x8fhi5ba\x87^ȇ\xcb\b\x92\xaa\xf8\n\xbf\xfa[8\x1d\xad\x80\x10\xb82WZ\x81\t\x99\xa1\xadt\xb0\xbe\x88]\x99\x04\x92\x06f\xb8\xaa,\x8b\xa4Dk\xbb\xdd*\xfd\x1f\xefټ-\xb5\xe5߹ʒ\xe9\x14Z\xda']m\x1c\xe1\x8by\xda8\xc7\t\xd8`\xbd\x04~+m\x12\xff2\\\x04gM\x01A\u007f3\xfd\xe5aջV\xe4\x10\xe7\x16J\xb3i4\xfa\x1b\x00\x00\xff\xffa\x8c\xf8f\x9a\x02\x00\x00",
	},

	"/data/en/genders": {
		local:      "data/en/genders",
		size:       11,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2M\xccI\xe5rK\xcd\x05R\x80\x00\x00\x00\xff\xff\x92'H2\v\x00\x00\x00",
	},

	"/data/en/industries": {
		local:      "data/en/industries",
		size:       4922,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfftX\xcdV\xe3<\x12\xdd\xfb)\xbcb՜\xef\x19 \x01\x9a9\t\xe4\xc4\x19X+v\xc5\xd1`[\x1eY\x0e\x9dy\xfa\xb9%Y?\x0e\xeeM7\xae\x92JUW\xf5s\x95G1\xc82\xdf\nCZ\x8af\xc8\x1ej-˱1\xa3\x16M\xbe:S+K+n\xc6Vvc\x9b\x05Q~\x8f]\xffQ:_\xcb\v\xe9A\x9e$U\xd9J\xf5=\xe9\xecE5U\xf6\xdaU\xd4\x13\xfe\xe9L\xfe.\x9b\xfc.\u007f\x11\x03K\xc7\xc1\xf0Y\xf9\x96\f۹˷\xb2#ͧ8\x83\xaf\x9d\xa1Zã*\xd9\xf7\xa6\xba\x96\xd77\xec\xad[\xcf\xffˮ\xce¢|\xad%\x16t5>\x9f\xfe\xf4\x8d\x82\x11\xa9\xbad\xc1\xd3\u007fGٷ\xec\xd1]^\x90\xbeȒ\x86D\xbd\x93=a\xffL\xb6\xa7\x93=\x86\xfd\x14\xfa\x8b\f\x1fY\xc8\x06AgEO%\"1\xd7\x04\xa9\xc2\x10\xf1\xdeW\x8d\x93\x8bkg\xce\xd8R\x0e\x80\xa6\xab\x1b\xd5\x12\a\xb6\xf05\x8c\xf8\xc8_\x94\xaa\x00v\xdf7Rt\xec\xdb\xc3h\x14\xce\xedƓ(q'\x00\xda\xe3\xeeT;\xa1͐=\x12\xbc\x115\xb1\xf2Q\xd37\x96\xcdd\x85:\x19\x06\xa7\xfb\x9a\xcb?\x19HI|\ak9\x18\x80gw\x8e\x03c\x90\xa0\x95\xadd-4\x19\xebkC\xc2\xe2\xb1Ӫ\x1aKc\xbd?Q\xc9H\xf3\ued50\xfa\x1a\x95O\rT\x80\x02\xd7\x16\xcd=\v\xdd\xc6%\xcf\bz1\x9b~\x03\xa0\xfcyԝ\x1c\xce8\x91\xdd|\x96\u007f\x18\x86\x01\xbaq\xa0o8\xc5҇\x12X\r\x8aCɶ$L\xb4\xfd~:\xe1\x8e\xf3bdH\xa1݉\xf2K\xd4\xee:\xe1\xb7\x11\xd2:\xbd\x13H[\x88\xdc\xffa\xf7\x0e:\xd5!Ӣ䬌Br\xf6\xe74 Φp\x82V\xec\f\x92\x97\xcd\xf1i\xf8\xd3\xdd\xeb\x9eJM6%aҊ~\xe5\xef\xc8\x0f=\xd7|\x10\x8c70\xb5\x1f\x8fG\xe7V#\x06\x9bDE\xaf4'\xe0d\xf0@\u007fpi\x04\xf0\x902\xc0\x02\x15\xdb(\xc3X-\xa8\x00\xb3\xf9&\xa1o\xf0:\xa8\xa3(K\x15B\xf4.\x1d\xd4u\xb0%\xd0\xf2\"=\x96_\xfciu\xd1\xc3g\xd9!OQ\x02\x19L\xca\xcaA\xf1\x9bP\x12g\x94\xf1\x80&\x82,\xce\x1e\x00\x87\xe1,\x06\x16.\x9d\x1a\x05\x80\ue7fa\n\xd7\xdb\xf1\xe5\xaf\xe9\xb8$f\x84\xcduA\xf1\xac4ɺ\xcbV\x9a*ib-'\xe9\x03\a.4\x18>\x90s̮ϷHRT+d\b\xe3QpAx՞j\x87\xbf\x13\a\xf7QQꋓ$ڛD\b\a\xae\xbcM\xf7\xf67\xbd\xb7\x9bm\xe4\x89\x12T\x16\\\xd9\xe2vk\xde\x15Mq>!'\xd1b\x90\xb0b\x18m\xb7\x89F\x826Aw\xff\xf4z`H\x93J\x9aD\xeefJ\xa4C\xfe,J\xd9H\xc3\x19\xe0\x95\xcaP\xf3ϖ\xff\xf5\xa2ج\xbd\xc4Փ\xff\xda\xd3`/=Y\xb0G\x93\x96\xf8\xc2I\xf9\xd3`\xd0\xdep\xb7\x17j\x94\xab\xfc\x002\xea]V\xf7\x0f\xa6\x11\xd8^N\xf1ϵ߀\xe0\xa7\xe2\r\x18\x9dI,\xa9Pn\bw\xc1X\xa1ƿ\xed\xb1\xaa\xe4\xa4B\\\xa6V\xb3Q\xa2\xc37\xba\x8d\x05\xff M\x93^_\xc42{\x94\x00\xad<w\xaaQ\xf5\x15)(\xeaNq\xbd\xa2)\x1c\x01\x81m\xe4k=\xd6@\xa2\xe1K\xb9\xba\xaf嶾\xa8r%iU{j\xech\f-\x89\xa5\xbc慸\x99\x97\x93cH\x16\xdc2\x1aG7\xb8>\x9a\x88!\x18z\xc9\xc37\xdb`\x0e\xdd\x1f\b\xedxu\x93\x15[\x14\x16\x06Z\x1e\x87\x11O\xd5\xd0½\x1a\x80\x18=\xda\"K\xbb\xa0Wođ\xa7\xb0\x9a\xa6\f\x12\x06\x1d\xa8<\a\xfdN#J\xe9g\xc74O\xe5\xff\x10\xdf\xe4o\xa8\xec\x848LS\x924\xc2\x10%\xfd\xb3\xa6\x13u\x03-ΐ\x9f\xab<p)\x05X\xb9\xcaI\xbb\xc7V\x94g\x1e\x0eW7\xad\xec\xb8\xe0P\xed\xa0K\xb4\x16v\xee\x1a\xa3l*n͑Ky\x15\x0f\x1a\x8eS!D\x84u\xb9\xcel\xa5\x91\xb9ii\x91\x89P\xa7\xfad\xe2\xacT\xdb\x036\xeen\x9b\xb1Š\xf8\x95\u007f\xf2$\x9d\xe2c˓\x97\xf9A\xa9\xe6ǜ\x8c)\x06\xb0\x15\xcf\xfc:\xb3\x94\fYpd\x1f\xac\x89\x9dj\x9aц\x8c\x12\xe0\x19eO\xb7\x11\xc1f\x96t\x80yPE\v\xae\xb6x\xb0\x9fLIs\xf9DiR\xda\xc2\x1e*\\\x84\x91\xecS\xfeP\x13F\fs!\xa9C\x051\x15\xe0\xb6}Nz?\xeb\xfd\x87\x9fb~\xf8\x15\x00\xdfө52\v\xd7|\x96\xfd\x90\x90\xa8\xd9\x12'\xf9<+\f9,\xcenXr\xaa\xd1JT%\x8fg\xb8\x8a\xf6'*\xa9n\x85\x87\x8f\xecgv\xa46<\xe1\x8a\xf9\xf8p\xf8ȋ+Pi\xf1!p)\xca\x11P\xd0\xd2w]a`X\xfe\xc3ܫ\xedG\xc3M\"\x9a\vl2NE\x02\n\xee\xe6\xa6(A\xf7J5v\xe6W\xfe!p+\xe8p^\xc1=&\xf9;5\xfc\x84\xbc\x12!\x17\x84\xa3\xc5\xe1\x90\xc8\xf4\x02\x96\xa9(1\xc3\x13\x8f\x99\x97uh>\xa8,\x17\x8cK\xc1>l\n \xa7.\xae-\xf9\x9a\x9a\x19\xc9^\x98t\xe9\x10\xc4\rq\x9c\x89_\xdb^\xab\v\xa5`,\x16Xt\xe2_ \xd4M4\xbeQUmk\xc5\xf6\x1a$\x9d{1\xc4썐\x84\aC\"\x9a\x1a\xdf\xd2A[u\x91\x94\xd4\xef\xaf\xfc\x80\xa9ŷ\x9bmGN\xc0\xbb\xfc\x03զ\xbc'\x81\x9e\x06\xeb\xbb\xf1\xd8L1\xe3%\xa0\xd4\u05cd荾\xd1\f{KyS9,IU\xb9\a\xcc\x1eI\xc6\xf9\x9b\x8c\xc9\x10\xe4\x1e\x0e\v~\xddl0Mgq\xf9\xce>\x93(m\xdb\xec\x8a\xd7*+1\x82\xc7'ZVA娥\x9d\xac\bظ\xf7C\xdc\\\xa0:{\xfb\xd2\nO\xac'[94$\"G6|\xad\ar\x9c\xa4˜0{\xe0\n#\xc0c\xec#\xe0}4\x83\x1au9\v\xe6\xc0\xd3\xdc^S\x14)v\xf5\xb7:\x1eC\"XR\xcc>\x86\x1b\f\xdc9\xb2\x01;>\xa7\xaa\xe1gط%\f\xa1\xe0'\xc9l\x16\xa9\xb6\x1d;\xbf'y\x83M\xa5\x0e\xa2¯\x8b\xd0\x1d\xbc\x98\xef\xb0?\xbb\x87\xf4\x1a-ú\xc9d\x12\xfc\xeb\a7\x9e\x1f\xb2H\x9f\x83a\u007fR\xaaL*;aB\xe8\xe9'\xa5\xdb\x1b\xa3\xa9\xf0.v\xf0E}D.\xd5Ë\x0e/\x88t!\x92\xe6\x82ZЉzڰ\xa8Z\xc0\x99\xa9\x8f}\xf6Z\x92\x0f\xfa\x83䲕\xd6\x18٢J\x05\xbf~\xdcK/^T\xf6F\xe6[\xe9/\xff\x82LQ\xf48\x87\xb2\f\xcd\x194]vL\xdaVR\x97xѠ4\x85\xae\xc2[\xd1\xe6\x9e\x03ٕ\x83#v\x05\xa6\x1e&\xeb\xc9\x16~LɄq\xc5*Z\n\xb0\xe0\x9f$T\xc7\xc6Ш\xf8\x97\x01Tu\xbeA\xe6\xfdP%?\xb9L.\xfeܞ\x90\xb3\x1b]\xcaJ\"\xfb\x99\xad\x015\xa3Vi\xfe\xa1\x84\xc7n\x8c\xe6n\x8a<\"| \xe4\x96jC\x1c<!и\x99P/\xe9\xfcK\xf0Sb\xcesM\xcd.e\x9e\xb5\xff6\x9e\xddz\xa2\x95\x88\xfc;0J\xf8\xe7\x9f\xf8\xf5ɑ%\xdf\xff\x0f\x00\x00\xff\xff$w\xa6\f:\x13\x00\x00",
	},

	"/data/en/jobs": {
		local:      "data/en/jobs",
		size:       2246,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffdU]r\xe36\f~\xd7)8\xd3\x03\xf4\f\xa9㵷\xb3\x89\xd5\xc8\xd3<3\x14$q#\x91\x1a\xfex\xe3\xe9\xf4\xee\v\x88\xa4H\xc9/\x1e\xf3\x03 \x02\x1f>\x80OBh\xaf\x1c;hmZ\xa9\xb8ӦJ\xd8\xf1\v\x84w\xf2\x06+\xf2\x06\xb3\x01\v\xcaq\x82\xd9\x1f\xff\xbd\xfe\x9fl\x1c\xcd\xe5Y\xaa\x9e=Y+miq\x9e\x9b{\xf5\xd4NR\xa1\xc1\x84\xaf콶\xd6K\xd7I\x01\x98\x94\xe2\xa3\xee\xd9A\x1a\xe1\xa5c\xcf`e\xaf\xd8\xc4\x15\xef\xa3\xf5n\x1d\xab\x8d\xee\r\x9f&\x82\xd6Ͼ$\xa7\x8c@+9\xabG\xae\xd4\x06\xc7\xf0\x0e\xac\xd5\v\xa6\x85\xe4\x0eJ\xcc;=aVZ\xb1f\x06\xb4\x8e\x18\x16\x92\xfeKjK\xa4X'\x11W\x11\xf4m\x0f\xeeϒ\x90\x98f4[\xa9\xf0ˬA\b&\x8b5\xdd`\xd4\xf3\x84\xfc&\xcf\xea0\xc0$\x05\x1f\xd9Q\xf5\xe8\x8d\xd9\x1e\x06\t]\xaa?\xa3\xf2&K\xa7\x119\xa4\xb0\x9chu\xd0\xd3\xe4\x95twv\xf1\xce\x00\x17\xc3\xce:\x83\xb2\xa1\xbc\xf5v\x04\xbd\x03\xb3\xa6\xb8)\xe0\x805\xb3\xdc\xfe\xea\x99;\xbe\x11\x12\x01\x1f\xdcb\x8bsO\xb5\t\xc1Ϥ\xa2\x91\x9d\xef\xbd\x042U\xfb\x8a\xf0\xfc\xe9\xf4\xcc\x1a?\xcf\xda8v\x051\xa8\x85\xdd*\x12\x05\xe9SҀ\xa0\x0f\xeb\x8e5|\x04[\x1d[I\xf7\x1fG\xc4͖\xbe\xf4'\x84\x1e\xd5M\x1a\xad\xa6\x90LA\xc7\xd6@wW\xeb8\xb0\x06\x84\x01GR\xfe\x86\x95*\x8a\xc1\x12o\x92DR \x91\xc4oZ\xb7l\xe9#\x1eNߛT\t\xb9\x181H\x87IV'@%\"\x92\xb4z\x02\x8drߦ\x1e\xb1D\xff\xc9\xf0y\x90\"J\x01\xedg\xe0\xa3\x1b\xb0\x03\xd4\xda\xc5\xe5\f\xe3L\xf6OvA\xb6\x96\x9ed\xa8 \xf4\xecq\x90p\xba\xad\xf6F\x80ݏ\xe4ޜ\x92\xfc\xae:m\xd2DD\x85d\x1b\xeaFQ\x8d>4\xe3o\xd4\x1e\xf6(/\x95\x1fГ9]U\xfd\x90\x1f\x86\x1bJ煛O\xd8n\x90\x02K7\xbc`\xfe<\xaa<^\xbeR\x15\x06<z\x86\"^\xbd\x18\x81\x1bV\xeb_\b\xad\x9e\xaf\xdeX\b\xbf8\xea\\\xd0\x00kb\xf3\"\x84\x9f\x97\xdaH\x00\x03\xd27S\x03\xc3>\xda3\xb4\xd2[s\xec\"\x15\x86\xff\xeea\x92۟\u07ba\xe5o9\x1b\xf5\xc0\x91:A\x9f\xac\x87\xbb]\xca\b\xb7܋\xaaq\xf9\xb4^\xb8\x9cn\xdeFyՅ\x14\x8a\xf3fJ\x11\xff\x89\n[i\xfbǣ\xc2q\t\x1c\xb4rFoD\x9fL\xebmo(tܶ\xf9_\xc9\xff\x1b\x90\x16\xc1@\xcb\x02\x89(\x10$\x18\xb5\xb7#\xa7\xc4\xc3R\xcdP\x88lx\axoVd\b\\\x86\xb9\x88\n\xe7\xed#T5\xb0\xe8j\xbf\x8c\"\xbc\xae\x8a\x04\xc4\xd5\x10O\x8f\xd3\x1a\r\x0fTD\xfc!%\xbdD\xbfkT':\xe9\xce\xfd\xe2\x06\x88\\\xebǐH¶\x9bg\x85\xaf`\xdd\xde6\x032\xc1j\xee\x868\xf1U\xe3xױ\xfd[\x1b\xd0F\xe0\nu\xd1k\xf7\x065Π~\xbcY+\x94Ŕ\x14ƌ\xa5M\xff\xb8\xb2\xaf\xfc\xabd\xf8JO\bF\xe4u\xf6n\x16\xad\xfc[\xb3\xfc\xe6\xd1i\x9d]:$I\a\x19\xd1\\\x10\xbaS%Aa\x95\xbf\xc3Ǻ\xe1B\x1a\x01ټ\x00\xbf\x03\x00\x00\xff\xff\xb11\x04X\xc6\b\x00\x00",
	},

	"/data/en/languages": {
		local:      "data/en/languages",
		size:       821,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff,RQv\x1a;\f\xfd\xd7*f+@\x12\xc8\x03\xf2h\u171e\xe6O0\xaa-\xb0e\x8el\x87\xc2_\x17\xd4U\xb4\v\xab\xec\xc9ǌ\xaeeIֽ\xd2\xec\x87\xf2\x05Q2\xcc\xc2\x11\x85Q`\x16=*\x9f`\xa6x\xec&\xd2\xe4\xcf\x19#e\x82\xd9=\xa2\"\xcc\x1e\xa4\fs\n\xa85\xb7\x809\x89\xc3`.\xce\x01#\xc2<\xe5\x9e8\xaf\xc1YŎ\xb4WX`\xc1`\xe7\x85g\xe9gMXZ\xc0\xe2A'\x0fO\xd6Hn\xc6\xea?y\xfe o\xb6\x96v\xf3H\xe2.\x1e\xe1Y\\h1Ϲ\xa4\xfe\xc8\v\x9f'\x13\xf8ʒ\fH/\xf2\xa2$\x96\xf8\x8dr\x19^\x94{\xa3KtX\x1f\xb0\xa4\xa4\xae\x9fIc3Jt\x81e5r\xf2緁\xb3\xa1°Bn\xcd\r\v\xa5\x14\bVtT\xba\xc1\x8ae\xb4;V\x1e\xb6\xa9TXU\xf9d\xf9z2M\xec\xf2\x04\xaf2&#\u061d\xf6\xb6\x1f\x96H\xa1\xf9\x8d~s\xfe\x87W\xec\x02\xacQ\x04G4\x9b}\xb4\x92\x06\x1ex\xf1\xb0\xf6\x91\x14\xd6Iɢ\xd7U\xc7Fi}Ww\u007f\xc0\x06\x93}\xe5\xa3\x15\xdap\xf1u\x1aߦ\xfe\xa4xLըY\xec\x16O4N\nmMr\x87\xf9\xde\xc1\xe7\u07fe\xd8PiMl\xff\xfeJ\xf6\xf4\xb6\xb16ŷ)\x8c\xe9\xa3%\x9a\xe6\xa9\xf7k\xa8\x90\x90S\x16x\x1b\xe9H&\xc7\x1b]\xdb\xd0\xdf\xe86\xbcS\x9b\xea8\xec\xd9ɰAq\x15\x9d\x05$-\x9eT\x86}*>\xb5㍺\xec\xff+\xdf\x11vxe\xdb+)\xc9`\xf6͐v\xc9v\xa9\x8fxg\xf9\xd5\xd5\xd6\xe1\xae\xcaٖ\x12\xbeT[\x93\x8a\xf05ŉ\xf4ހ51\xbd\xb0\xbfN\xfb\xb3\xbf\xa1\xe7潵)\xeeo\xd4\xd5;\xe0\x99/\xf6\x8f\x1c\xe0@\xc1*\x9b)5\xc2\xc1#\xc3!]\x86\x9dm\x89\xc0!\x1bm4sCA\xf8\xcec\xcf~\xaf\xa1¿\x00\x00\x00\xff\xffq\xe9\xea75\x03\x00\x00",
	},

	"/data/en/male_first_names": {
		local:      "data/en/male_first_names",
		size:       665,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff4\x92͎*;\f\x84\xf7~+\xfe\x06\xd4\xc3\\!@wֆ\xf6trH\xdbȝ\x80\xf2\xf6\xa7\x92\xd1\xd9 \x91vU\xea+g\xe0Y\x16\x1a,(\x9d\xed&\x9e\xe9+\xde\x03K\xa2\xef\x98R䙶\xfc\x8a#\x9d۩\x8f\xb4\xc1o\xea\x8aE\x9e\x81\xae\xc1f^p\xeaq\xc9\xf6\f\xe2\x98\xd7\b\xf9\x89K\xa2/\xf6\amM9\x8d\xb4\x17\xf3I\xe8ST%\a\xbady\x89\xd2n|7۵Gn\t\xfa\xe4Js0\xad\x18}E\xa5\x81\x17S8\xe5\x1c\xe4M{\xf6J\xd78[\x0e\xb5\x87\xa0#;\x8e\x06\xf9\xf9q\xa9\xf4\xe1\xac\x0f\xba\xdc-g\xday\xbc\xb7{\x10Ka::\xf4g\xae\xb3)ҸL\xe6\xdd\"\x14\x86\xbc\x99l\x91-.\xf4\xcd)\x03\xe4\xc4\x19\x06\x0f:I\xfbw`7d\xdbZ\x99\x12\x88\x0f\xa2\x10lP\x06\xad<\x87\xe2t\xae\x9d`\xc2\xec`BC\xe1\x96\x1d\xfaU\xea\xbd\x0e\x80ˡ\x1d\x96%\x83\xebگ܋7\xe4O\x89\xad\x13\x9e˿\xe6\x05Q\x13\x1a>\xf2\xdbE\xefB\xffa\x03֮>[\xa5\xb5\xe8\x1f\x9e\xe1\xb2\xf6\x82okP\x8fhi5ba\x87^ȇ\xcb\b\x92\xaa\xf8\n\xbf\xfa[8\x1d\xad\x80\x10\xb82WZ\x81\t\x99\xa1\xadt\xb0\xbe\x88]\x99\x04\x92\x06f\xb8\xaa,\x8b\xa4Dk\xbb\xdd*\xfd\x1f\xefټ-\xb5\xe5߹ʒ\xe9\x14Z\xda']m\x1c\xe1\x8by\xda8\xc7\t\xd8`\xbd\x04~+m\x12\xff2\\\x04gM\x01A\u007f3\xfd\xe5aջV\xe4\x10\xe7\x16J\xb3i\xb4\xbf\x01\x00\x00\xff\xff\xf6\xdb\xdd\xf1\x99\x02\x00\x00",
	},

	"/data/en/male_last_names": {
		local:      "data/en/male_last_names",
		size:       1764,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffDU\xddr\xe2:\x13\xbc\xd7[\x05\xb2@%\x90P\x86\xfa\xa8\xefr\xb0'X\x85-qF\x12\x8ey\xfa\xd3-\xb297\x9b\xb5\xfe\xa6\xa7\xbb\xa79\x8c>\xf7\xee-\xf6!\xc5\xe0N~\x18\xbc\x8c\t\vA\x93[X\x9c\x82{\x95\xbbOn\x87-5\x9e\xe0\xc1]\x8c\xa6\xee(\xf3\x10ͽ\x84N\x8d\xab\xc7>\x8e\x82\xcb\xd2^\xebk\xbd\xcf\xea6b\xc6\xfbb\xd9?\x8fܸ\xb9\x16k\xbd\xfc,\xeb\xc35\xf1\xec+\x86\xe5 v\xc5gg\xfeR\xb0\xb1\xd5\t\u05f7\xaa\xee$\xc3\x15\x1062\f\xee\x05h\x82\xfb\u007f,\xe1\xe26jA\x80\xe1\xe1\xde=>O\xb8\xd8g\xb7\x8d7\xacl\x00\xdb\x1dژ\xb3[\x9b\xe2\xcaK\xc7\xfe\x16\u0097\xd61<d\xc0\xa9\x0f\xad]-\x01\x06\xcb;\x9f\xdb^qo\xaf\xf6D\xa6\x96\x93;\x16\v\xd8\xdd\xf7$\xe9\x96pz\xbc\x9d\xeb1\x00\xc6Ɵ\xbb\x84\xe4\xfet\x93X\x87݈S\xf8>d\xc5Bv\a\tx\xf4\x01\xe6*\x1dM\xbc\x803רv8\x1a\xaf\\\xbfHp\v>\xb8+v\xebg\x80\xf4\x83ή\xf1w5\xe1\xa9\x1b\xaa4\xbe\xedY\x80p\xe3\xb7\xdbD\x96\x035\xf8爧\xa1\xda^\xf3S\x8e\xb5\tn\xcb\xe8\xd9ś\x8c\xd8;I\xe6\x0e\x84\x8d\xd7\xe4\xdeQl&2\xea\xe7\xf6\xe6[\x05\x80\x10\x14l\x9db\xec\x80\xc0\xe8\x83&\xa6\x04\x96\xffʌ\xd6t\x04\xd67\rW\xb6\b\x9alv\xfb8\x11\xfb6B\x82\xbd\xe4\x1f\f\x9br\xe9\xf1\xc2\n>\xa9\xe5S\x0f\x892!\x94L?\x1d\xfc8F\xbc\xb1\x8a)\xff\xa7\b\xad7KȐY\xbf+<ה\x94\xf8\xfe\xda\xfc\xd7\x17\x8c\xf4\xea\x05\xeaʌ\xb3\xbb\x99\xf0W\x11\x14l\xd0퐟\xbd\xf72\xbaC\x81\nЅ\xde\x19\x04\xed\xb1\xad*\x0e\xfe\xab)\xc3\xe8\xd6a\xfbs\xd2@5\xe6\x10\a\xec\xaf|\xea\xa9(.\xa7\x1f\x03\xf3M\u007f\xae\xdeo\xbb\x18d\x80nV\x1e4p\xea\xe9\xc8O\x18\xf9\x81\x06F\x8a\\\x8cܯ\xe08\x12u\x02p4\xaf\xe73\xfb\xad\xf6\x87-\xee,y,-\xbd\xb3\x8f\xd5y\x9b\x12\xea\x1f\x8f\x01\xc2\xf32}զ4\x80\xdfE\x9c;T{N\x9fU\x92ީU7S}SR\xf2\x8dM\b\x1ek+d\x11\x96M\xc0d\x1dK\xf62\xc1/\x03}\xd0P\xea\x1fcW\x95\xc0\xf5\x02\x04]1\xed\xc1c\x1a !\x0eZ\x1d\xfb\xe4>`\xba\x88\xbf`\x15\a\xdfC\x1d\xb0\x95ڥ\xf06\xec\xa1h\b\x91\x01\xae\xa6j\x89\xd7\x12\x02}Q?6\xa5\xda\xf5p\xd3\xd0Rcx\x95\x83\x04\nn=9\xd8ˌ\xab{\xafV\xfdG3\xed`\xa0^\xa7\xe4^\x8c\x8a\xa0\xc3Kx\xa6\x0f\x15i\x84=\xe7\xfa\xf8\xe7\xdf\xc95\f\x1c\xeb\xb6 \xfc\x10\xe6\xae&\x05\x06o\t(\xf0\x1cݰ0\xe98Q[A=\xa4\x96\xb1BS<\x8dd\x1c\xae\x15&\xaa\xa9C\xf7bc\xcaF//\xb9Ue9\xa9܉\x9f9\xa2xd26T]\xc2|Y\xf6\xd8~Pa\xc0.\x89i\xf7\x1c\xc5:gJ;H\xb8\"\x14x\x95\xa0W \x1af[\x97\x8c\xd6k\xd2\xcc\xc4\xde\xf6\xa3\xefrm\xc9\xfdO\xd2?\f\xc1\xa5\xe0\xbda\x88\xc8TU\xce\r\x8a\xddh\xadρ\xe9\x00G\x84|\x81\xf5l\xfe\r\x88\xdf0G\xa5\x9a\xefu\xea\x80\x00\x03\x03s\x98[\xc0\xe4\xf1\x06;\xb7qf\x8eh\x8dջ\xd8oJ\xd1\x19\xc84Xk\xf5\x1b\xb1P\xef!\xe4\xebN\"}\xc6\x14\xd3f\xb9zZ\x02\xfb\xfc\xb8\x94\x19w֊HS\xfe\x1a\xc43\r\xe9;\xe4\xf3\xe8V\xa5\xfe\x90lg\xe4\xa1{U\xe1\\\r4\"\x9f6FO\xc3F\"\xa7\x06'\xb6Rs\x04\xdc=<A\x17dmE\x15\xf9\xcb\x04\xa7 \x96\"\x10\x01\xb1\x86\x88)\x99H\xcbN;\x1f\x04zN\xa8\xf5o\x00\x00\x00\xff\xffl@\xa6y\xe4\x06\x00\x00",
	},

	"/data/en/male_name_prefixes": {
		local:      "data/en/male_name_prefixes",
		size:       8,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2-\xd2Sp)\xd2\xe3\x02\x04\x00\x00\xff\xffs\x11\xfb\xf8\b\x00\x00\x00",
	},

	"/data/en/male_name_suffixes": {
		local:      "data/en/male_name_suffixes",
		size:       37,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2*\xd2S\b\x06bO\x05O\x10\x02\xe20\x850\x05_\x17\x05\x17\x97`\x85\x80\f \x1d\xe6\xcb\x05\b\x00\x00\xff\xff\x93:h\xda%\x00\x00\x00",
	},

	"/data/en/male_patronymics": {
		local:      "data/en/male_patronymics",
		size:       666,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff4\x92͎*;\f\x84\xf7~+\xfe\x06\xd4\xc3\\!@wֆ\xf6trH\xdbȝ\x80\xf2\xf6\xa7\x92\xd1\xd9 \x91vU\xea+g\xe0Y\x16\x1a,(\x9d\xed&\x9e\xe9+\xde\x03K\xa2\xef\x98R䙶\xfc\x8a#\x9d۩\x8f\xb4\xc1o\xea\x8aE\x9e\x81\xae\xc1f^p\xeaq\xc9\xf6\f\xe2\x98\xd7\b\xf9\x89K\xa2/\xf6\amM9\x8d\xb4\x17\xf3I\xe8ST%\a\xbady\x89\xd2n|7۵Gn\t\xfa\xe4Js0\xad\x18}E\xa5\x81\x17S8\xe5\x1c\xe4M{\xf6J\xd78[\x0e\xb5\x87\xa0#;\x8e\x06\xf9\xf9q\xa9\xf4\xe1\xac\x0f\xba\xdc-g\xday\xbc\xb7{\x10Ka::\xf4g\xae\xb3)ҸL\xe6\xdd\"\x14\x86\xbc\x99l\x91-.\xf4\xcd)\x03\xe4\xc4\x19\x06\x0f:I\xfbw`7d\xdbZ\x99\x12\x88\x0f\xa2\x10lP\x06\xad<\x87\xe2t\xae\x9d`\xc2\xec`BC\xe1\x96\x1d\xfaU\xea\xbd\x0e\x80ˡ\x1d\x96%\x83\xebگ܋7\xe4O\x89\xad\x13\x9e˿\xe6\x05Q\x13\x1a>\xf2\xdbE\xefB\xffa\x03֮>[\xa5\xb5\xe8\x1f\x9e\xe1\xb2\xf6\x82okP\x8fhi5ba\x87^ȇ\xcb\b\x92\xaa\xf8\n\xbf\xfa[8\x1d\xad\x80\x10\xb82WZ\x81\t\x99\xa1\xadt\xb0\xbe\x88]\x99\x04\x92\x06f\xb8\xaa,\x8b\xa4Dk\xbb\xdd*\xfd\x1f\xefټ-\xb5\xe5߹ʒ\xe9\x14Z\xda']m\x1c\xe1\x8by\xda8\xc7\t\xd8`\xbd\x04~+m\x12\xff2\\\x04gM\x01A\u007f3\xfd\xe5aջV\xe4\x10\xe7\x16J\xb3i4\xfa\x1b\x00\x00\xff\xffa\x8c\xf8f\x9a\x02\x00\x00",
	},

	"/data/en/months": {
		local:      "data/en/months",
		size:       86,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2J\xcc+M,\xaa\xe4rKM*\x023|\x13\x8b\x923\xb8\x1c\v\x8a2s\x80\xecJ.\xafҼT \x91S\xc9\xe5X\x9a^Z\\\xc2\x15\x9cZP\x92\x9a\x9b\x94Z\xc4\xe5\x9f\\\x92\x0f\xa2\xfd\xf2\xcb \x02.\xa9\xc9\x10\x06 \x00\x00\xff\xff\xa4\x95F\xf2V\x00\x00\x00",
	},

	"/data/en/months_short": {
		local:      "data/en/months_short",
		size:       48,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xf2J\xcc\xe3rKM\xe2\xf2M,\xe2r,(\x02ҕ\\^\xa5y@\x9c\xc3\xe5X\x9a\xce\x15\x9cZ\xc0\xe5\x9f\\\xc2\xe5\x97_\xc6咚\xcc\x05\b\x00\x00\xff\xff\xc5\xed\xe490\x00\x00\x00",
	},

	"/data/en/nouns": {
		local:      "data/en/nouns",
		size:       128,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff,\x8d\xc1\n\xc2@\f\x05\xef\xf9\xabZ\xf0\"^\xb4x_ڧ\x04w\x93%I\x15\xffެx\xca@\x867G\xae\x01\xa3Y[7\xb8\xab\xd1\xf5\xe3\x81F7\xc6;\x1fg\xdd%h.\x0e\x9a\xb6҇;\xb5^\xf9\xceI\a\xe3\xed\x81<e}\"\xe8\xc4A\x8b\x15\xf1\xc61\xc4\vV\xf0+a\xd9\xe5א0\xad\xf5\x9fSA.\u007f\x03\x00\x00\xff\xffh\xac\xda\x1b\x80\x00\x00\x00",
	},

	"/data/en/phones_format": {
		local:      "data/en/phones_format",
		size:       26,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffR\xd6UV\x86a \xe2B\xb0\x00\x01\x00\x00\xff\xff\xb5\xe6\x87^\x1a\x00\x00\x00",
	},

	"/data/en/state_abbrevs": {
		local:      "data/en/state_abbrevs",
		size:       149,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\f\xc6\xc1\r\x02 \f@\xd1\xfbߪ\x01\x94\n-\x06\x10\xc4\xfd\a\x91\xe4\x1d\x9eT\xa4 ?\xa4\x13\x84\xd0\b\x93\x98xT\x9eBV4\xa2\x15uT(\x83r\xa8\x82%,b7\x8a96\xb0\x86M<\xe1\v\xcf\xf8\v7\xfc\xe0\x01\x8f\xb4L+\xb4\xce[\xe8\xca\b\x8c\xc8t\xe6\x97\xcfd]¾\x16[\xd9\xe7\x1f\x00\x00\xff\xff\xc9\xcb@ƕ\x00\x00\x00",
	},

	"/data/en/states": {
		local:      "data/en/states",
		size:       471,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffL\x90oo\xc2@\b\xc6\xdf\xf3)\xfaU\x16\xcdf\xe7\xd4e\u007f4{\x89-\xeb\x91^\xc1\x1cW\xb5\xfb\xf4\xe3\xaaɖ4\xe5\xc7A\x1exx\x88x\xc4\x01\xc1\xa3\xf5\x1e\x12\xff\xa8\x94أ\x18\x1a,0\xf2\xb7&a\x84\x85FMت\x83\b5\x99\x9b1Ò\"^0\x11<z\x91[\x84'\xd2\xd4y\xf7ʟ\x99\xa1n1(\xd41\xb2(\x1b\xd4\xd22\xba~\xad\x17\x84\xf5mĚ$\x8fM?\xc1\x8b\x8elsy\x83,\xe4\xff4E\x94\xd6\xc1\f\x9b0\x1a\xe5l\xb0\xe1&p\x87\xe2\xe0{\x98f\xefg\xb3\xf2\x9dN<\xb3\x8e\xc9A%\x17\xb1-\x1d\xd3lnKglK\xb8T+\x1cN\x16\xd8\xf7.\xd93%\xa3i\xc6\r]\xb9\xd1\x19\xbf4\xf5\xb0ՔC\xb5\xc0\xa4\xee\x00\xef\xe9\x12\xfb2u\x17Xa\xd7Gw\xe8\x17\xdc%\xeaT\xe0\x95Dl\x8ag,'{\v\xdaRU\xdb\xec\xe2]\xc7\xffR\xb7\xf4.\xf5AŊ\x119]\xfd&\x9f\x19\x03\xec)\r\xee\x01\xf6\xec\x17-r\a\xf4\x9d\xa5\xcb>\xe6@\x96\xab\xbf\x02[\xa3b\xec\xef\x93\x0e\xde\xf2\x1b\x00\x00\xff\xff\x9e\xdfr&\xd7\x01\x00\x00",
	},

	"/data/en/street_suffixes": {
		local:      "data/en/street_suffixes",
		size:       132,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x1c\xc9\xc1\n\xc2@\f\x84\xe1\xfb\xbcU\xa9\a\x11\x0fE\v\x9e\xc3:H0d!\x9b\xad\xacOo\xeb\xe5\x1b\x86\u007f2\xe3\xc0\xb4\xd1;1ӓ\x81Y\xa3\xd8\xfej\x8f\xc4\x1c\xb55\xf5\x17N\xa1\x1bqV3\\\xba\x97\xd4긊\x13\x8b\xc4\xfb\xcfGƾ\xada1)<\xfc\n\x96\xaa\x9e\xb8Uy\xe2\x9eA&VF\x1c}\rQ\xc3C\xc6/\x00\x00\xff\xff\xb0\xa40\r\x84\x00\x00\x00",
	},

	"/data/en/streets": {
		local:      "data/en/streets",
		size:       4162,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffDW\xc9b\xea:\x12\xdd\xfb+\xfc\v=\xf7\xf6\x86$p_\xe0\x86\x86\xbcd-\xec\xc2ֳ,\xd1\x1ap\xc8\xd7\xf79%\xd2oA\xd5\x11\x1ak.\xff%\xe5毾o\xfe\x16\xfb\xe6\xefyl\xfe\x81\xdf?\xf1\xfb\x17~\xff\xc6\xef\xc7I\xe2\xd0\xfc\xe8&\x89\xcd\x0f7\x84ـ\xcd\x01\xe4b\xbd4?f\x89\xb63\xfe\xff\xa0\xfd\x91\xb0i\x0e\xdc\xea{\x89)`Ώ\xc6e0of\x13'\x02{Ŕ\x897\xe0<J\x02\xfb\xb2~\xe0\x1d\xd1\\\xcc\x18pr\xb4_\xc1\xe3\xb68\x19\x9f\f\x96\xc4٦l;NŰ\x84\xd0\x03d\x9bxy\xaeÒ\xcb\xecۭ\x98s\xf3`z1\x05\xcc\xf78\x18<\x9en\xa4^r&\xcf\xe2*\xb7\xce\xe1\x89\x0f&\x8d\xe7\x00\x1d\x00X\a\xcaŷd{i\x1eD\x85\u007f\x10\xeb\xce%%\x00\xe7\x86\x18\xae\x98\xb1\xb1\x1b\xf5\xe6\ag\xba\xe9d\xa3\"\xea\xe5\xc1\x15i\x1fpv\xbb\xa7\xc4\x1c\xfe\xc13\xc1S\x96\xb9y\b\xa7e\xb4\x19\v\x83\xf7<=,3\x04y\b7\x1c\x11\xc5W\x81\x1e\xa25\xb1]E\xec!\xee&h\xf3~ 4\xe0+U\xf1\n\x1eE\xeaM\xfb\x0e-\x19\xe0\xd1\xf1\xdc\xe2\xb2\x1e\\<\x84h7V\x97\xf9\\7Enn?\xf4\xaaB\xa5B\xba\x92\xb3ĳ\x15\xd7߱/\xb9Y\x19g\x8d\x1f\x14\xdc.)\x80ϧh\xfbA\x80<\xed\xb8\x82\x82%V\x00\x8d\x1bG :\xb4\xa13\xe4\x17\x88\x85'\xadL\xea`\x9cfu\x1f\x8eX\x81W\xacb\x80rW\xa3\xc40\tfG\xebC\x98ȡ\xe9\x95\xc3as\xf0\xb9\"\xdf\xc3d+\xc7\u007f\xb0\xc5\x05]q\xeb\xa5]\x1b\xe7\xcc0\xf2\xd8\xd0\xe3\xf2\x805\x90\x1e<D\xd3\a\x822\x9f\n6\xc1\x91}7\n\x01<\xb7\xb3|p\xa0^\xf0*\xc5\xc1U\xf1\x02\xe4\xe2\t\xb1\x17耀B\x86\x98\xba\x80ׄ\x9c\x83W[\xd1J\x1d6\xb7\xaffJ:\xca햮\x00\xc1\x16GM\xd0X\x10\xb6\xcfͣ\x81q@\xa7\x00K=\x1aD\x12h\\\x94]m\x0f\xba\x10\xde`\xc8H\x8e;\x9aG9EC\x9a@:h\xc1d\xab\u007f\xbbv\xc7U\xe0\xc7\xe0\xc8\xcd\x02\x15\x118\x88\x8c\x98\x03D\xc0q[\x94\t\xf4\x93j\u007f\xb4\xed6D\xc9\x01\xe8j\x93\x1ee?I\x83T[\xd0E\x1e\x11\x81p\x84\xc7\x10\xf5\t\x90\xd6\xe2d\xe8;\x1bD\xc9c\x94\x05\xcax\x8cP=f˄\xb9\xe2\xd5!\x1f\x17;\x8c\xb9y2\x03\xd4\x0f\xea\xe4\xee\xc6O\xe6N\x1ce$P\xed=\r\x03\xac\x8apl\x9e\xdc\x00q\x9e\x9c\rē\x01\x995\x0e\x9f`(\xfc\x15\xf1\x82\xa7ԍ \f\xcb'$\x12\x864\xf9\x10\x05\xefx\xfa\xbc\xe0\xa6\xd4<\x1b\x1b\xab1\x88\xaaG\x13]\xad,\x00\xce\xddA\x9c\xbb@\x16\xcd\x00O\aX\x18K\xcf\\\x9f \xb5\x00\xa6\xa9y\x86\a\x89G\xf2y\xa6'\xccd\x14\xa7\xb2\xf6\xd18\xf9Ƈ\xe2\t\x13\xb5\xfc|\xf7\xda\xe7\xe2\x18\x8d`\xd4㚫\u05f8\xd1&\x1deY\x90\x1b\xd6\x12\xa9\xeb5\x82\xa7Y#\x9f\xd8\xefx];\xaa\x86[\x82\x83\xa2k~\x03\xae\xa4]\x85\x12\x93T\xfcN\x91\xd6P\xa8C\xdek\xd6\xd1t\xf2\x8d\x04\xfeK\xee\x91\a\xc1R\x9a\xcc|!\xba\x8df\x99\x00\xa8;\xa5\xedA=\u007f\x1d\xed\f\x02c\xc7f]\\\x12\xa4\xf4\x8dA\"ިQ7\xd0 \x13\xc6\xc6x]\x02Ni*\xe3\xdf\xf1\x04\xaf!\xbf\xe8l\x99@\xae\x88\x83\x8d\xb9\t\xe7\xbf\xc4u\xaaÍ\x18\x1e-\xe73\x92\rO\x16\bl\xc0\x90\xf0\xc9\xe1I\u007fz\xe4\xc6\xfa\xfc%\xcd& \x90@\xe5B'\u0604\xf3\x99q\xbe\t|\xd3\xfd\xfd\x18\xccZ\xbc\x00n\xf7\xf4\xb2\t\xb3\xa8\xbfm\x90[t\xe3\x15\x9e\xb5)\xbd>\xbd|2N\u007f\xe2vi~\xa2x!a\xf5j\xf3\x9f\xccT^\x03\x0e\xb9\xe1gXL\xf3\x9b\xd1l\f\xee9\xb85\xbf\x89\xb7\x10\x80\x9c\xff\x84ѷ\x1fP\x91\"]X\x90\x90^\f\xfd\x02\xeay1\x8c\xa4\x17\u9fd4\xa1\x06p\x163\x90\x91.\x02\xa8\xef|\x81\x03\xf4\x82c_`\x8c\x17L\xa6J\xb5V\x01%\x8a\xfdb/\x8e\xbayA\xaa\xe6U/1\\\xce\xcdִ\xcf\xd0\a\x82C\x80'\xa9\xceA\xa4.A\xa0W|\x03$Ox6\u07b65Lq[S\"<f\xcb0\xddJ쑲\xc8\x03f,<e\xcb\xf8\x1eCI(s\x86\xff9V\bp\xdfS\xe9\xa3\"]\x973\x82\xff\xd9I\x89\xcd\x16\xf6B\x19\xc3d8\xe74@\x06\xbcv\x1b\xe8N\xdb\xc0\xa4\xdek\t\x01\x1e48\xb7!\xa0惡G\xb8\xe2\x06\bR\xfa\xbb\x86\xb6e\x9ax~\xe1&0\x8d\xb7퍾\xb73\xdde\xac\xad\xc7\xce\f\xcc\xdb0\xe8\x0eu\x19\x04I\x11\x9a#\x0f\xb8i\aMD\xc8O`sX,\x9a\x17@\xba\x01X9\x9b.C\v\x91G^\x9cTZ\xeb%rn\xc7E\xa8]\x98G\x10^9\x1d\xff[T\xdf@7\xd5\xf2\x0e\xce\x1eIk\xfe\x01P\x9d\xef:VO\xac\xeb\xba\x00/\xef&\xa0\xa1X\xa4\xed\x9d i\u007f\xbb\xf0}\xf0\xce\xe0\xff\x13\xeb\xebĝ\x94\xb2́\xc1\xbb\xb9\x049\rB\"\xe7D\x16\xb4\x1d\v%*\fx@\x1a\xb2\\\x81\x8e\xc5 !\x01D\xbdV\x15\xbeC\xbe\x84\x14\xdd8۞sl v\xb0\x93\xa5ev\xf6|v<\xd6j\x0e\x03[\x10\xcc(\xd1;\x9bQA\xe1\xe2\xbb\xd0M\xb0\x8a\xf6?\xbbP\xe5\xc6u\xa8\xfa`\x99\x01B.l\a\x00\xca,|@\xd0\x17\amA\xbe\xb9\xd6;`TJ0\xc4\x02o\t5i\ue81efw\x8b\xf0\xa6旙e2\x03\xfe\xfd\x85*~k~\x89\xa3\xb1\u007f\xc9\xd5\xf4\x06liW\xac0X\x18N\xf0\xe1_HDc\xa5\xd5\f\n\xf5\x9d\x8a.LT\x8ajK\xa3P\x1dP\xd1\u0096-z\x0e\x90\xa7\xdb\xea\n\xbf\xc2ՐP\x89(2\xfc}\xdb\x06\xb0\x9e\xf3:\xdaм:F\x16|\x82\xe08\xa2F4\xaf^\x98\xd7_}\x1a茯\xf0!\x124+\xcd\xeb\xa7F\xf6ް\xe7L\xe0\x03j\xde\xde\xf8\t\xe5\xfe\xaba\xeb\xd7VGP\xacuqo\x16\x8d\xfe\xbd\x18u\xf9=\xb2I\xba\xb9+<\xda`pA\xfa\xad.\xbb\xa7kF\\\xb2GiAi\x8b\xe73\xa0\xbf'\x86\xbd\x13\x93\xe0\xec\xcd\x1ea\b\x123\xeeV\x0e\xa3\xef#\n\xa7\x95\xf6@\xa3\xdd\a\xaa\xa0={U\xf6\xe4\xff)\x96\x968\x989\t\x99\xefI\xaf\x88Ѓ\xf4-z\xb4\xd2\x13-|\xe9A\x90\x1b\x98O\b\x10|\xea\xec,\xd5\x15\xe9\xc1\x8aX\x16\x0fV\x18\xe0\a;x\x1e\xac\x1f\x0e\x14\xfb\x00\xa7\x93\xb3\xa8O\x1e\xb4Mi\x0fB\u007f=\xf0\xdb〞\x8b\xf6=\x84\xcfS\x81\x87\x1f\nj8\tT\r\x96趇\x02\x17\xa1\x9d\x0e7l;\x9anLJ\xf3\x1f\xc2!&\x8e\xc8\x19\x19\rwq\x80\x05\xe9\x0e\x1a;\",\xf4R\x00g\xe7\x81k\x118x$S1 :G\xb2\x12\xbf\xc0Bu8\x00\xf4(\xa05\x8b\x1d\x05]\b\x94v\x1c\rC\xed\xc8(\x82p\xe0p\x1c\x9e\x02\xa0\xa7\x8d\xe12\x05\xb24\xb2\x139\xba\xc0\u007f\xb5e\xc5[\x03\x9eTi\xf5\xb7\xe3\xc5h\xd7uD\xe1J|\xe1\x05\xb5\a4jݨ\\\x95{DP\x0e\xa4Q=\xe7\x98Q\xf5\x93\xea\v\x10\xa9\x13\x9b\xd8\xf9 \xabE~\x9c`Ppn\xe0\xdf\xd1\x14\x86\xd41\x176k\xc7\xc2f\xed\x88\xee\xc6^\xf9\xb2\u0097ݓW\x1d\xd4\xfb\x00-W\xfbSdS\x0f\xd0\xf3\xfb\x05\xfc\xae\x1f\xb4\x8d\xf5k\xebX\xe0\xaf6\xf0\xd0Ȅq,I\x0fF\x05\x88\xf5\x85\xfaQRoX\xb4\ai\x8e\xb7\xce\xcc\f\xac7|\xa6hE\x04\x98\r\x97\xbc\xb1\xb4\xc2\xdc\x15\xdd\x18\x1fo\xf2\x89&\xe6m\xd4\x10\x83\x83\xbd\x8d\b\aH\xf96\x86\xf9\xa2\v\u0089G\x84Y[\xfc\xb7\xe0\xb1\x06\xaf\xc5\x11p|\x97\xe8\x94@>i\xdaxcE|\x8b\xc5|6op\xee\x96\x01\x95\x9a\xdf=\x1b\xea\xdf/\xa3\x99\x9b߳\x19\x9bwv\xfc\xbe\xa9\x19\xa2}\xe2\xd3\xde\xc5\xc2|\xefx\x03\x89\xb6\x89\xe0\x1e\xfb\xdem\x97\x99\xc1\x01z\x1d\xf2\xa3F\xbey[[54\x17\xb4\xc8\a\xfa,z\xe3\x87QS}\xb0\xab\f\xe4i\xbc\x17\xca\x0fZ,\x19\xf0O\x8d\xbe\x0fs\xab\xea\xfb\xb8\x17\xa3\x0f\x91\v\xbf\xc0\xf4s\x16#\xa7\x14\x8d\x19\tJH\xe5\xf7\x0e\x9aX\x05'\xd0s\xfe\x17\x00\x00\xff\xffT\xdd\xfd\xb3B\x10\x00\x00",
	},

	"/data/en/top_level_domains": {
		local:      "data/en/top_level_domains",
		size:       37,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffJʬ\xe2J\xce\xcf\xe5\xca\xccK\xcb\xe7\xcaK\xccM\xe5\xcaK-\xe1\xca/J\xe7J\xcf/\xe3JM)\xe5\xca\xcd\xcc\x01\x04\x00\x00\xff\xff\x04\x1a\xcc\xeb%\x00\x00\x00",
	},

	"/data/en/weekdays": {
		local:      "data/en/weekdays",
		size:       57,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\n.\xcdKI\xac\xe4\xf2\xcd\aS!\xa5\xa9\xc5 :<5%\x0f\xc2\n\xc9(-\x023܊2ATpbIi\x11\x88\x01\b\x00\x00\xff\xff\u007f&+w9\x00\x00\x00",
	},

	"/data/en/weekdays_short": {
		local:      "data/en/weekdays_short",
		size:       28,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\n.\xcd\xe3\xf2\xcd\xcf\xe3\n)M\xe5\nOM\xe1\n\xc9(\xe5r+\xca\xe4\nN,\xe1\x02\x04\x00\x00\xff\xffܹ\x9c\x9e\x1c\x00\x00\x00",
	},

	"/data/en/words": {
		local:      "data/en/words",
		size:       1685,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffLU\xdbr\xeb0\b|\xe7/\xb1D\x1bft;B\xca\xf4\xf3ς\x9c\x8c_:\xb1\xcdewY(\x17e\xa3ԛɿ\xcdkO\xe2\xbdh\xc8\xfc\x91)-\xab\x91\xe9\xa2w/{,^R\x89S\xda\xc6m鮔{\xe9S\xea\xbf-\xc4HQ\xae$\xecO:\x8c\t\xf5\xf0\xfe\"-\xa5\x93\xb6\xb7\xb4\x85hz#\x10\xa5PY\x96ǘ\x12\xcf\xf4\xd2%iu\xba\x84\x17\xd2\xde\xea\u007f\xb3\xa6\xc5d\xbb-\x92\xbfQ4\xf1Չ\r\xad\xda\x17j\xcf\xc0\xe7?~\xf6/~\x99d\x14U\xfePB.\x02+\xff6\xbd\xe1\xa2o7\x8f\xa1\t\x14\xbdɓ\x9d\xa7(5\xb1\xa4\u07b5\x89\xb3\xb9i:+\x90\x8e\xea\xf1*\xa4\xe1\n\x1a\xd1,-\tPY\x87\"\x1fD\xcb\x13P\xeb\x8dچX\xae\x92n\xa3\n脦\xa3O\x86>I\xb3\xb7\x04\x93\x02\x9a\x10\nuO瀏,\xcc*\xb2]X`\u007f\xe2F\x964EH\xa6\xaa\xf8\xc1\xe8\xde4b!t\xeb\xb6&\xa0˟\xcc\x14\xe2\x836\x92JAD\xea\x13\b<Jj?U||\xf5Y>\b\xdc\xcf\xc7\x11\xb6Aq\xe8\rV\xbb'4\xc5,\x81$pjF;\xd8\x01\xf5k0}z,Pah(\r\x95H\x80M7\x88N\x19S^\xf0\x9d{$f\xa4\xcdk|\xa1ܢ\x8a\x99Ph\xd1\xf4\xa5\x05R\x16\xb1\xa5\x1c\xa2A[\x18\t-;\f\x84\xb9\x9bi\xc5\xc4\xf3NZ\xf7\x99\xfcU\x18\xde^\n\x14c\xb2\x98\x1c?\x17\xde\xf9\xb6\xf6\xea\v\xd5}\xea\x9f\xde\xeew)\xd0g\xc1\xb0\xcb}\x01\xe5&>)Jz\xfd\x8f\xbb\x8e\xab\xbf\x98\xf0\xe6/\xc9\x00m\x85r\x98pO\x89%a\b\x94\xf6Ь\xc1\xcb\xdd1f\u007fk\x06\x960\xcdYC\x1b\x9a4\xb6eC\x15\xea\x15|H\xcd3\x00\t\xb5\xe7\f#Vu\xd1\xe5,\vDK\xbb\f\x0e\xae\xfd\xe7\xc7K\x00\xbc\xc9\xf4\xaf\x00\x06\r\xf1\x861k(\x8cA\xd9=G\xf7\xc8\xed:\xfc\xc4F\xb1?\xbfx\x1e\xdfgqI\xe2\x13'4\xb4Ȕ\xd8Nq\"\x10\x1c\x94[\x82\xc1\xc8\x1d[\xf4\x92\xd9o\x93C0d\x1a\xe4\\\xbe\r\xd7'\xbd\xe8\xaf_\x1a\xea\xc3\xd3Ҏ{r\xe6\xaa5\xea\xba\xc0\x84\n(\xe5\xce9+d\xc7 \xb0;dP_\xb2\x0e\xf3\xf3\x9fV\xa1Q8\xe1\x908L\x9c1\xa4\xba\a\x10v\xe4\xfb\x1a\x99\r\v\x8d\xde\x1c\xd5\xceǳ\xd90\xa2\x14\f;#\xe9\xa0\xd7k\u007f,\x8b\xb6x\xc8\x1cb\xf9\xe9yZ\xdb\xfd\x8c\x8bw\x04\xfb\x9e\x0e9rb\x92\xc1勠\xf9\x0e\xd2`\xdcθ\x1d\xcb\xcfc\x8f\v\xe5\xb5\xe3\xd6\x06\xf0\x18\xa2ÓK\xd7Y\x9e{\x14M\x12\xd6!\xb6\xda!\x1a\x03:\x89\xaf\xbf\xc45\x91ǡ0'\xb6\xb3\xc2\xfdؕp\xa3\xac\xc7\xf6\xb8\v\xa7ā\xf7\x00\x14\xf5YH\xcc\xfft{i\x82 \xed\\:4\x1b\n\xbfJlFr?\xeeuC\x03\xdc\xf3\xff\xe3\xd3<\xd0U\xd6X\x92\xd0\xe5H\xea\xe7\xfc\xbc<\xa2C\x83\xff\x01\x00\x00\xff\xff]\xcec\x0f\x95\x06\x00\x00",
	},

	"/data/en/zips_format": {
		local:      "data/en/zips_format",
		size:       8,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffR\x86\x00.@\x00\x00\x00\xff\xff\x87\xdd'=\b\x00\x00\x00",
	},

	"/data/ru/characters": {
		local:      "data/ru/characters",
		size:       119,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x04\xc0U\x15\xc3\x00\x00\xc5\xd0\xff\xa8\x19\x83\xbb1Ù\x931\xb7\xb5\xf0\xe2\xa87gr!Wr#w\xf2\xc0\x13y\x92\x17y\x93\x0f\xf9\x92\x1f\xf9\x93\x82\x94\xa4\xc2\x11\x8eq\x82S\x9c\xe1\x1c\x17\xb8\xc4\x15\xaeq\x83[\xdc\xe1\x1e\x0fx\xa4A\x93\x16m:t\xe9\xd1g\xc0\x90:\x00\x00\xff\xff\xb7\x8c\"\x8ew\x00\x00\x00",
	},

	"/data/ru/cities": {
		local:      "data/ru/cities",
		size:       2271,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfflVir\xf2F\x10\xfd\xcf)\xb8\x00wD\xe0\xd8N\xe1\n\xf1Vv\xe4\x05\x03\x95\xf2\xaf\xc4b\x11\x8c\x85\x90\xaf0s\xa3\xbc\xf7zF\x88\xd4\xf7\x8bަ\x97\u05cb\xf0\u007f\xfa\xad?\xf82\f{ A\xf8*d\xbe\xf0G\xbf\rC\xdf\xf85\xc5G\b~|\x01\x12\xb2=\x98\xda\x17!\x13\x1b~\x93\xf1\x06\x0f\x0f\xe1\x06O\xab$vx\xd2\f\xfc;d\"\xfd\xdaW\xe6#\v#\xf8)\xeci\xb8\xe9\xf9[\xbc\xde!\x87K\xbe&s\x80\xc3F\xf1\xb7\x14\x18\xb96\xef\xb7\xf0>\xc5;1wpq@\x05\x85\xff&×\x0ea\x9c\xff\xee\xfb7\xc5\xec8\xfa\x9f~\xa9D\xfe\xf0\x1bi`T\x86)\xc9\x03\xbcma\xb2\xc6KX\xe0m\xd5\x15\xd7\xde\x11\xac;ȕ\xa6߰\x18\xf3O\x11\x04LH\xac\"\x1fY\x1d\xd80\xf1+\x89\x18\x10%\xec\xc3\r\x81\xec\xf9{\x15\xe1\u0095H\xe4t\x850GӔ\xea\t\xba\x01\x80\x1c\x13\xb9O\xf5\x10\xaf\x96\x05\x96,V\xa2\a\xa58J\xed{\bcĕ\xbbGC\x03q˄\xe6\xa3\xffND\xc5\xd8\x02\x82\xd1Wal\xa9>\xaar\xb0x\\\x18\xab0O0g\xe1\xf1\xfd\x13\xc3\xe3\x9d\xeb\"\xfe\f!ǃ\xb95dw\xa7\xc8Ϙ$\xe1a\xec_ \xaf\x95\xc3\x10CØ=\x9f\xe3\xf1>\rIn()\xc4\xf1\x04\xba\xc4)9c\x00-m\xc4j\xfajC\"G\xf0Z\x05\xc6|r\xea\x90\xc0\xc4H4\n)\xd4\xe6\xc8\t\x87\x83\x1a\xbaN\x82\x96\xccL\xce*-\xfd\xdcr#\xc1ImM\x1b\xe4\xd2Y\x8d<\x8eHm\r\xc9m\xc4dnq\xb5\x18\x19\xb5a\xda\xd7ڰ\x9e㹎\xdd/8\x84\x1dQ\x98&$s\x9b\xbap\x8dn\x02\xa20\xa2h\x9c\xb4/\x86)\xc7\xdd:\x01ےb.i\\\xc1\x17\x98\xdb@\xb9pIt^\x85\xbdb0cMD\x9c\xb6W\xd8\xedNc\xf0*}ž\x83\xd1\f\xd5\xea\xa0i%@\xad$/9/I>A\xb2.\xfc\xee\x9d1\xd7ZR\xc0\xf9\x86\x9f\x9dz\xfe\xab\x8d>\xd3.\x10gÞ\xf5\xa2\xa1mF\xa6\x11\x88\xe9%\x05E+\xb6\xd4\xc4\xef\x18\x03# \xb9\xa5\x87w\xad~\xa1D\xd4ʙJ\x8e]\xef\xb3#}<\xab\x88\xddL\x83\xbfgųxH\x12Dh\xfc\x80\x9b\xa2\xa5Ʉ\x8c\xe3\xa92\xbb\x9asM\xd2֎\x85]D\x81*\xdc\xcbG;c3\xf3B\x82\x13МƊ\x82*\r\xdd,vU\xe4\xf8\x04e\x87\x89\x89\x94\xfdx\x10&\xd4Nق(\x90\u05cf\xf6\x00\xae\xfb\xe7\xc7\xf3L7 \x18\x03\x9c\x1d\x0e\xdd\x18*l\xa3\xc0\xf8\xd0\xc5s\xa9\xc9\x1fZ̸\xcd\xf3\xf8\x15\x19\xd2p\xae#Q\x85Ѡ\x83F{\x80\xe6\xf1\xd0D7\x89\xfdA\x98C\xcbZ*d\xd7\xf2`w\xb8-e\xcei\xe7\xaa\xc8\xce\xd6JY\x18\xee\x8e\xea>a\xd4\xc7o\x9b4?i\x11(\xa8\xb5\xbb\xe5)\x8f&\x02R\xa4\xe9\x99\xc7\xed\xe6\x97Ή\xe5)\a!D\xb6:N\x88j\x83ʂر\x8d\tꄌ1+\x8b\xba`uH\"#L\v\xd564\xa3F&\xed\x04,\"\x83&\xb2\x8d&\xa8[\x1d\"\xed\xecCFf\x94\xbe<\vb\x18\t]\x8bRA\xf1Q\xacU(\"-\x91j\xfc8--\x82\xddss\xbd\f\x17|\xfe\xb7\x92\x1f\x85\xc9@\xa7\xe2\xd8]\xb9OxZi>3\x1531\x11\xcfZg8>mq\xe2\x1f\x05]\x96\u007f\x14Wk\xd8\xf3\xff\xa6\xcf\xd6@\xc3\x1e\xeb\xb6Y\xfc\x02+\xa3\xaf\xb3\xf3D\xac\xff\v\x00\x00\xff\xffL\x92o\xa5\xdf\b\x00\x00",
	},

	"/data/ru/colors": {
		local:      "data/ru/colors",
		size:       216,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffD\x8e\xbd\x0e\xc20\f\x84\xf7<u(\x03\x03\v\b\tP\x91\x10HH\xacP\xfe\t\xcd3|~#δ4\x83eߗ\x9c}\xd4\x169و֦\xbc\x02{\x9e\xb4*\x8d\v2\xc9Ɯ\xd5%W\xdcH\xaa\xfe\xe3ҥU\x9d\xd8\xfa\x12\xf9\xee\x82M\x87\x0eZ\x92\xdd`\x95z\x0f7\x82\xa9ȵ,W\xa1\v9\xb0\x13\u007f\x94\xb7\x19o\x1a-\xfd(]\xf4P\x81\x9a\xec\xa3Md\x1b\xee\x1cu!\xfeC\xcd\u007f\xa1|\xfc\x06\x00\x00\xff\xff^6\xf4&\xd8\x00\x00\x00",
	},

	"/data/ru/continents": {
		local:      "data/ru/continents",
		size:       133,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfft\xcc\xc1\r\x85 \x10\x84\xe1;U\xbcRy\xc4;\\,\xc0\x06\f!\xa2\x04E[\xf8\xb7#7r\xf0\xe4i\xb3\xf3M\x86\xc0F\x11o\x98\xc8$\xb2X\x1aQ\xfcO\xe5x\xdeB%\x1af\xd6\x0f\t$\xf9\x8b\xd3 \xb2\xf7\xb1 \xc3룺\xe5\xe4\xea\xe5&Ng,Uoa\xd1\xf0\x0e\x00\x00\xff\xff\u007f\xf5p\x8b\x85\x00\x00\x00",
	},

	"/data/ru/countries": {
		local:      "data/ru/countries",
		size:       3010,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfflVYv\x1b\xb7\x12\xfd\xe7^\xb8G\x0e\x96\x9fΑ_d\r9\xb6I1\"%%֏\xe2f\x930\xc1\xa1\x9b[@o!+ɽ\xb7\nݭ\xe3|\x01u\x01\xd4< }Ne3n&\xcd(\x15\xe9\x98bs=\xe8AN\xeeR\x00\xb1ƍ}ڦ\x9fX+\xc2GAU\xbeu\xc4IlF\xdcVi\x93j\x00\x85\x11\xdbT7#J\x10\xd9LRL\x9bf*\x12\xe8&\x05\a\xab\f\x9d\b9\xdf\xe6\x03n\x98\x18(e\xb2o\xb0\x10<5WFP \xc4ARh.\x05\xb9\xc2\x14=6\xe0\x02P\x80\x05z\x1fp\x9dJM\xc1tl\x92\f\x8ci\xe7\xdb\xe6\x13\x98\xb6'\x15\xf5\xe3\x96vm\xc4\xdf\xcfj\xbd*[\x12\xfc\xaa\x8e\x98\x80,\xa5!l\xbb\x91\x9bw\xb8\uebbe\xa1\n8\x83^$\xa68?\x98#\x86\xe9\x05BƩv\\\u05f6)\x1a\xe9~\xb8\xe5BOB̔\xa4Yp\x80Nk)8\xe9\x02t+#6Yo#\x03̟6\xff\xb7P\xddRw<\xa9\x9b\xff\x81\xeb\x15U\xba\x85\x13\x02\xa0\x8a\xce\x1e\xa4;,k0\xaflK\xfeѶ\xfbl\"\x89\x13.I\xc8]\x87\x96T\x122O\xca3\x87\xa2T\xb8~G\f\xe16\x84\x1aקăҡ\xb5\xe2\x8e\xd2\xd3V\x9e*\x18\xd9;E\xb5R\xa8\v'\x9b\x8f~Yn\xdb\x19q\xdf1\xb9g\xa6\xc2CS\xd3\xff\x1e<O\x12O\xd7\x15\xbf\x00pƘ+\x9f\xfeΜHg:e\x90\xbe\xf4m\xfd\x02\xfc\xa4\x9c[Ø0H_\xd3N\xe1\x8eL%\x92\x8a\x1f\xafڶV\xdcw\x19B6҄\xac\xe3W\xbd=\xe4Me\x9b\xa3.d6P\xeb\x17\xe0\xdcg1\xe9\x8a\xfa\x9b|\x1f\xc8hfQ\x1c2\a$4\x18\xb6S\x91\xb4E6s\xebj\xab\xf9\f\x04\xcbE#[\xbf\xcf\x14ނ\xe5?\xeb\x8awF_\x19\x16U\xe91\xdbk@\xa4\xbf,\b\xb3\xb4HK.('\xf0o\xdd:S,X\xaa(\xf5%\x13\x16\x1c\xa78=vѢeu:(%\xc0\f\xe9\x1b\xba\x88\x91\xf1=Y\xff\xe7\xe3\x99\xe7<\xeam\xd9\xc7&ô\xfdg\xf4\rq)U_\xb4A/\v۔,Y\xe6\xc0\x1c\x1eP\x8b\x99K\xb8\xb7\x81\xb9d\xd5,&\x12\xb43\xe4ʛ\xab[\xc8\xc5\xf3\xaeq\xccUM%\xf9s{\xa1b\xa9\x9aK\xadjZ\xf3\xe67\xd88\x96\xb5k\x15\xc0f\x90\x1e\xc0\xa8\x94'\xd97\xf6=\xa0_\xfb\x0f\x16&\xf5-s\xcc\xc8\xc0\x03\xb8Y&\xb6\xf7\x98Q%#\x92\x89}\x8e\x99\x01\xf9\x04\x9daK\xf5\x19\x17\a(R\x04\xe4#\x1e\xecCN6\x97z\x8b\x9e\x9b\x1f\x04\xd9\xe2\x1e\u007f\x00\xbe\xcb\xc5Ĥ\u007fPWݚ*&\xbbV\xb29˺\x1d1v\n\xd9׀Nd\xb6\x10\xa3\x98\x13h\xc1N\u009c\xe56\xa8>\x8e\xdcF\x0e\x1e\xfa\xa1\xdd\xe6\xfbQc\xa4-6\xea\xbb0MU\x8a>\xba\x16\xb4\xc5Rl\xa1\xe2eJ\xf8\xc0\xf8\x03s\xee\x8d\xcb\xc9\xe2\xfc(\xcd{\x03\xec\xd1|K\xa5\x1e\xbd\x8c\xa4;\x89\xb3\xf1\u007f\xcc\xe2\x94,{\x02\xc1\xccxT\x8d|\xa2OE\x00E\xfb\xd7@t\u007f,\x95\xd5>ז\xe4\xa7&Q\x18\x81\xa9\x99\x03\xbe\xf2H\x96>*G\x06\xb1\x9c\n\xdbVC\x0f(\xdba\xddbO*\xcb \x1a,\xb7\nl[s+1\xb49\xc7f\xb0%Ă\xb9\xb4\xc1\xaa\x9bўk\f\x99\xf2N\xb2\xf8n)O\xc9^\xb1\xcaZ\x9c}c\xc2rk\x11\x94E\xb6t\xe5\xff\x14'\xa2\x92D\x0e5\xc3b\x8e\xf1\xca2ѣ\xd2\x03B\xe7\x98\xd7\xf4\x99K\xad\x93\x932Γw%\xc0ka%\xeb\x15ԕ\x8a2\xe6a\xb9\xd2\xecԿgȆ`\r\u007f\x00\xd7\x15\xea\xa8\xf1}J<٨\xb0P\x19Y\xa9(\\#z|\xc3\b<y\xf6#@O./J\x05\xbeRkbL\xa7F\xe8ۤ\xad\xbe\x16\xbd\x8f\x95\xa0<)\x9f\xfd\x8f\xa5\x1cy\x86\xd85\v\xb4\xaf\u07b3\xf5\xd7\xfcK{\xd6\x14hs\xf3E}$\xe4\x0e\f\x98\x1f\x88\xc0\x83h\xc6\xda\xf6\xa8\xe1yVp\xaf\f\xaa\xa0\xecu7\xc3^l\xdce\xc5\xfe\xf4Ҳ\xc6\x0e\xe0/\x8b\xbb\xfdW\xe1aF\xa5h>x\v|7\xab\xbf\x9bO\xbeK\xaf\xcaܗ3\x80\xe0E\xdeF\v\xe5\xabw\xf5\x8f\xdd\xefNP\xd6\xe5\x95\xe8P\x1d\xbf\xb2\xbe\xf5\x86\xa5W:N\xea\vE}\xb2\x86\xd2\xe6͚\xb2\xfe'\"\xf5\xd5\xca\xd1x\xc377\x82<\x1b\xf97\x1aȒ\xcb\xcf\xfc\x13\xfb\xa1\x94ۛ\xdc\x1f\xb8\x97\x9f\xfe\x1b\x00\x00\xff\xff\xf3G\xd8e\xc2\v\x00\x00",
	},

	"/data/ru/currencies": {
		local:      "data/ru/currencies",
		size:       4636,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x8cX]\x8e\xdcD\x10~\xefS\xf8\x02\xb9\x06\x8a\xe0\x05!x\x80\xd3\xcc\xcf\xeef\xa5\xec2\vBB\n,\xcaF\b\xf1\x12\xe1\xf1\xd8;=\xb6ǾB\xfbF\xd4\xf7U{\xdc\xedm/(\x0f\x99\xb8\xbb\xab\xab\xbe\xfa\xea\xab\xea\xb8\xd2\xd9a5\\\xbbܵ\x99\xfb\xc3=\xb8\xcf\xc6\xe5Õ;ȗ\xf3\xb0v\xb5\xb3\xee\x94M\x9f\x9c\x95u\u05f8}\xbc\u07b8\xca\xd58\xb9r\xed\xb0\vWJ\xf9$\xc6\r\xf7o\xe4[3\xdcM\xab\x87a\x8b\x0f\xe2F\xe5\xce\xdc#\xd7t\x97=\xf9\xb0\xcb䯂\vG\x97\xeb\r\al\xa6-\xbd\xa7sU\xe6zW\xc9?:\xd8(\x86\xf5\xb0\xe1\xb5\rn\t|\x81i\xf9#F\xb0\xef(GV\x8c\xe4$K\xcfqD-\xeḟ\x8d\xd1X\xe5V9\bG\x87ۋo\xe2D\x8e\x9f\xd8\x03\xd3\a\x98\x8eQ)x~\xb8\x96\x9b*\xb9'D\x06\x01\xa8/\xfba+\xe7\xb6\xf2\xcf2\xf2x\xb8b\x18g`\xbb\xf7;\xaaň\xd4\a+pEQ\x87\x9fa\xab\xa3-\xc6\x1fg\x83.\nh\x1a1\xd2\xdd.\xdf$\b\xaf51S\xa6zdSѨ\xe4G\a\x8f%\x17\x93}Y\xdf\xe3J\x83\x13\b^葾\x01Y\xf5\\\x10[\xc7\x14\"í\xdc\x0f4n\"\xd4\xc3\x1d\xef\x10\xe1\x05\x8d\x98'\xb2\x9b\xe9\v\xd0\xfe\x91\xe1\xdc\xe9\xed\x8d8ۊ\xbbK\xa7;\xb0\x8c|\xb4\x1a\xcd\x14\t\x8f#\x00\xd9ǐ\x93L\x15\xefB.\xc9\xd6\x15\x83\xce\xcd\x14B\xe6~\x11bZ\x12d\x83\xda+A\xc9\xd4\x19\xb2\xfd\x99՜`\xd8pOo\xbb0[\xc1\xd9J\x90\xb6tl\x13\xe2\xb8E\x99\xe1\xec\x95|\xe9$\xbd\x13\xa7\x86\xf7Ȓ\xd0\x1bFL\x98\xbb\xcc\xfd){PO\u058c62ގjk\xe8\x12\xf2Zd\xee\t\x80x/\xa5\xa4\xc1M,\x11@:\xc7\x02\xe2\xd9\x02n\x1eȠ\xe3\x05N_d́\x99I\x96\xfc\xa0i\xfd\x1e&\xd2C\x93\xf3\xe4Zw\x14އt\xe9a=G\x00T\x85H\xc0jbv\xa3\x9f\r\x03;\x8f\xe4]\xa0\xb6\xee)Q\xf2p`JH\xc3\vz\xe4\x10$\xb8&\b\xc5<\xe1[MYA\xc4\x0e\x005\xf0\x99G,!\xcf2\xf7sb\x0f\tq\xc3|\x9e\xd5k\xd9\xfd^uB)\x97\xbd\xfd\xee\vC<Jn\xaa\x88\xf8i\x12=$\x01^\xee\f3V\xe8q+U\n\xc4f\x9ar\x8b\xce\x009\x00<6R\xb8\xb91\xc6-\x11\xa6\x14R\xd7ΡVY\xd5*+\xdf\x1a\xba^&kiǤ-i\xa6%be\\\xbfA\xdd\xecĹ\xa8j,1\xcd\x01\xefOc\x8d\xfe\xa3\x9bd\xcdf_~\xfd\xbda\xc0\xe7X\xc9o)E\x9e\xfe\xaa>+V\xdd1\xa4l\x87NY{\xc2vZE\xa1\x95\x15\xef\x17,q\xff\xe3\u007fe\xd2}\x88-e\xd9Wo\xbf1\xac\x8c\x82\xd5\xd8\xf9J\xab\x82\x8c\xc8R7R\x8c?2\xf7\x91\x8aիzS\xee,l\xf3$\x12\aR\xaa\x14o\x92ɫ\xd9l\xd0\x04)\x99a\x1a7#E\xff_@\v\x9e\xcc\xccK\x9c?|K\x8c]\x9f\xb9\xdf\xe4S\xe7묃\xb5G\x11Տ\xc6\xf7\xc4ț\x8b\xe4\xdd\x02\xe97\x9eW\xf5\"c\x1b\xf6\x10\x94@ G\x163CC\xc1\x98u\xe2F\xe7\x89D\x8f\x0e\xc0j\xa9g\xc8L\xfd\x82\x95㨆] q\xa9\xd3N8o\x9c\x8c\x8a\x94vp\xe8[x\x8f\xb6+/\x98\xa5R\x19\x92\x9d\xc4\xf5w\x0e7\x95\x97\x01\x06\xdcJi\xcbx\xa7\xd3ф\xdb;\r\xac\x9d\x9av(\x01\x1b1{\xd0>\xa9Ρi\u0600\t\x84u\x8bj\xe0\x1d\xe3\x8eW\x80\x1fC,\xa2\x1d:*\xc2\x1b\xec\xa8(%as\x8e[\xf8h\xe3\x14\xe8ۥ\xc8\x10\xc6AS\x89\xa8\x8e\x97\x1eR\x87#b\xc5)\xb4V!\x8az\xe0#\x0fX\x1e\xb1\x86b@\x1d\x8e\xfd=S\x99\xa0\xf6\xdcQ339\xa7\xe2h\xb0:\xea@\xc1yc7\xb6Gܷף\xda(*\xf7\x9c\xec\xee\x90\xf0>h\\I\x11g\x90U\xa4\xa4/ղ\x9b'\xfe\xa2½\x9f\xe7\xdap\xee\xd4\x1a\x86\xfc䆸\xaf\x82\xc0|3\x1a\xfb\a\xc5\x0fM\x82\xc3F\xc3q\xa4_\x18\xf1{\x12\xd3Fj2\x0f\xaa\x9f\xb1P\xa2#\x82\xb8\x8c\x16\x14i*\xc0)~\x90\xe4\xbe\xf3[*\x97\x0ey\x89\x90\x03\xdfqo+?\xcf\xf3Z\x84\xd6⍑\xec-k\xf9kK\x80\x8bȼTX\xa3*}b\xfb\\ț\x84\xbf\x9du\xae@\xc0ȇtG\\{~\xe7(\xf8x^\x9cr-\xd6\xef\x981\xc1\xe2\r\xa3\xe9\xe6\xe1q80ڵ^<\xb6f\r\x0f\xce\xfa\xc2Z|V\xc0\xb1\xa0B\xe2\x80\xf4\xb1\xc1'\ve\xab\x8c$\xd7\xf3\xf6\xa0}א\x17\xf6%\x99\xf7\xd4)\xaejk\xadg\xcdW\x95.\xca톎\xd7\x14\x94\xe4\xfb\x90\xfav&\x1dSY\xd6\xf3\x954\xb6˘\xa7\x03^,\x18\x9f<:\x96)-\xd1vE\x82?i\x8dCSc\xa7\xa8[ڽ\xee\x16\xd0܌\xaf\xe6\xd7G\x11q\xbe\xf6\x83[\xa08\xaaׅ\x12f\xeb\xdf\xfe\xe5\xab#M\x14Γ\xfb\xdb=\x18\xff\xac\r\xab,~\x02a\x9a\xdf{\xa5\x1e\xf3\x80Z\x1a\xe7ۊ\xbc\xb9\x8f+yz\xd0\xe2*PD\x9e(+\x8c\x12r\x8c\xacݤh\xc6`\x83\x97\xd5\ay\xa8<d\xee/\xf7 \x83\xf2g\xf7āG\x9fI\xd2B\x04\xec7^\x90m\\\xbf\x11a{f\xd6\x06\xe3\n\xc2\xe3%\x05rx$\xb0\xad\u007f\xc3t\xfe\xbf2X\xbe/]\xf9\x15\xff\xff\"Έ+'\xbe\x05\xaa\x94\xd6\xca3\xf5\x99\\\xc8ib\xfe\xfa\x14$\xee\x91+\x134\xadd\x8b\x84 \x90\xb3q\xf3\xbb\xbc\xd1\xff\r\x00\x00\xff\xff\xee\xbd\u0600\x1c\x12\x00\x00",
	},

	"/data/ru/female_first_names": {
		local:      "data/ru/female_first_names",
		size:       506,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffD\x90]N\xc3@\f\x84\xdf9u\x93\xe7\nP!\x02\xb5\x94\xbf\x03\xa0%?\xb0JZ\xb8\xc2\xf8F\x9d\xb1w\x1bU\xaa\xec\xf1\xf8\xf3dq\x8f\xc16\xc8\xf8\xe7/\xe3\x8ctCi\xc1\x84\xd9\x1a$\n#\xc7.rf\x8d\xb5\xfaG\xb6\xbb\x90\xca\u0099\xfa\x1f\x9bB\xd8qS\f\xc9Wi\nЎ\xca,\xbb\xce\n\xf3\xe0\xe6b{$~c[\xe9\x1dz\x8c\xa4\xcea\xebX$\xeeM\x1e\xd7\xcd]\x1c\xa9e\xc6/Q=\x1d\xad\xa4'\x9e\xe0\xde\xf3\xea\xdfӔ\x9c\xeaĽ'h\u058c\a\x1d\x97\x14M\xaeƃ\xdd\xe2\x9bIz\xdbF3\xe2\xc4\xe9\"\xdbK\xec\x04\xa04\xda9\xf2\xd2\xc8t?\xfcO\xde*\x15\x96\xf8\xb6c\xddx\xbd\xbe\xb37\x1ccP\xf9\xe6$=\xf5LH\x1f\xd2\xe4\xf9\x878\xf0βD\xfd(\x1f\xbdT\xd0'\x8bS\x10\xbc\xb1Vgc\xf6\xa5\x97\"\xe0\x12\x00\x00\xff\xff\xaf\x88\x9c\xa1\xfa\x01\x00\x00",
	},

	"/data/ru/female_last_names": {
		local:      "data/ru/female_last_names",
		size:       1726,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xfflT[n\xdb@\f\xfc\xef\xa9\x1d%\r\n$@\xfcJc\xb8n\\\xa7\x06\x9a~U\x92%X\xd1\xf3\n\xdc\x1bu\xc8}\x90v\xfbax\xc5%\xb9Ù٥\r\x95\x94\xd3@#\xff\u007f\xa2%\xe5\xee\x86\x1a\xea\xdc#\xd5>\xb4\xa7\xdaen\x163\x0e\xd4S\x83\xcfT\xb2\xc3\xe7g\xf4\xf8\xa0.\x86\x8enA\x15\x8d\xb6h\xa4\x16\xbf\x94\xf1G>K\x04\xd2)#Mqw\x8e\xee\x15\xaa\xeb\xb8;\x97\xc4\x16\xd0\xfe\x13\xcaCv(\xfe\x86p\x81_\x153W\xd8k\xe8\xc4xt\xaa\x83˰\x9c\xec\xec\a\x04z O\x81=vKż\xc0ℶ\xa6\xe4;\x1a\xfb\xb9\xf2\xd8x-\xfcdrf\b=\xfbÕc.h\xe3\xe7\xd6\xdd\xd2\x19-kw\u007f\xd9\x16]\xf0?D\xfe\x04\x87{\x04@\xc9y\xc3^\x8f\xb6wJ\xc9+\xceHh\xe7\xd8a\xa4\xd0S\xa7>\xb2\xb4\xf8M\xca5\xa6F\xd9I\xbb\xbc\xa0\x04z*\xde\x1d>2,S\xc6R`\x81\f\xe4\xe4,\xb4\x19\x04g\xf5\x115\nE \x01\x1ar\xb08+ȍ\x18-\xa4\xaf\x04I~Y\xb0\x13$\xad2ËF\x03\xbb\xc0\xec9\x06~`\xd1[\x8d^\xa5\xe1d\x1a\xa2ڌ\xf7\xc2\xde\x05\xf7uT\xb9\x06\xa8\xa7d\x1e\xf6\x80\x88\xd9\x19w\x82\x8fQOx\x06\xd5\x062\x1bp\xe4\x1c\xe5}-]\x1e\x94\xab\x95\xbb\x153\xa7\xdd\xc1\x9e\xb0\x85\xf67\x82c\x91\xee\x8f\x00n#Wor\xe9\xc6p\x17\x93\x003\x9c\xd1]\xbb\xe8Κo+\xc0+!\xda\x00d\xed\x10\xc4ܥ\xa5\x82\v\x19J(\xfe\x8a\xac\xd6\xd8X\xda\x16\xea\xc9ߢ`!\f\x8fz\xaf\x99\xceV\xfd\x96\x8b\x97\xcc\v\xf3.\x96\xa9\\\x86\xb9\xb8\xe5%\x9d\xbf\x18\x01\xb0\x9a\x1e\v\xd1\xef\xe2\xea6\xff>8\x85\\\x9a8\xa0\x94LڡV\xa2v\xc2}\xe7\xed\xe7\xb9͒\x98\x80Y\xea\xbb6\xb3\x8f\xd8\x06\xd7f\xb8\xe2ѻO\x81\xec\x83n\x18J\xb3\x16\x90\xa9U;\x152L\x1ax%\x94\xdd\xca#\x968\x1cb\xe3\xc85\a[\u007f\x0e\x04bc|\xa4\xfb\x02D\x0fJ\xceV<\xe2G\f\xa1w\xb4\x87\x00\x90:<\xe03\xeb\xe7\xa37\x96wڒ\x1f7\xb5\x80\b\x84\x82{\xb9DO\x9c\\\v\xf5\x99\xda.\xc7m4\x14\xfct_\xac#\xd6(\x18\xac\xabĒ,1+\x1a\u07bc+\x8f\r\x11\r\x9a\x03z\xa4\xeao\x00\x00\x00\xff\xff\xc0T\xbbH\xbe\x06\x00\x00",
	},

	"/data/ru/female_patronymics": {
		local:      "data/ru/female_patronymics",
		size:       237,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffT\x8eA\x0e\xc3 \f\x04\xef}5X\xed\x8dC\x9f\xe2V UP\x92/\xac\u007f\x94\x15\t\x91\x11\x17<\f\xeb\xc5\x1b\r\x05\xd5\"\x14\x1d\xd9\x026|y\xd3\a\x9fNPx&\xb2`/\xfc\xb0{k\x06x\x0f݄Nw\x1e\x8dʱYB^\xbe\x13|P\x98,\xde\x0e\xb4\x95\xfb\x93\xdf\xfe\x1c-\xd5\xe2\x82\a\x92+{\xe9*\x1c\xffw\xea\x11\x00\x00\xff\xff\xfc;\xa2\x9c\xed\x00\x00\x00",
	},

	"/data/ru/genders": {
		local:      "data/ru/genders",
		size:       30,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xba0\xe7b\xf3\x85m\x17\x1b/캰\xef\xc2N\xae\v\xd3.l\xbd\xb0\x17\xcc\xdd\x01\xe4\x02\x02\x00\x00\xff\xffdgd\x9f\x1e\x00\x00\x00",
	},

	"/data/ru/languages": {
		local:      "data/ru/languages",
		size:       419,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\\\x90Mn\x830\x10\x85\xf7\x9cڀ`\x85ԟ\x15\xa2TT\xb4]\xe36\xa0\x98\x80\xc9\x15\xde\xdc(o\x90\x90MVx\xbe\x99y|6\x1a8\xc9`1K\x8a\x1b\x1c\xe6\x045\x8fw\"\x1f\xd0+<.X\xb4\b\xf0\x97\x1f\x8f\x11\x8e}1\\\xf8\v\xbd\x0e\x9b\x18\xc9$\xe7\x9a\xc5\"Uh\xbda\xdaӞ\xf0\x97䒆r\xa0\xc2\x16+\f\xf8?;}\xc0J\xc1\xdf{\xd8\x04\x9f\f]1Iyt\x1b\x15 \x8c|\u007fvI/%\xad\xae\x01\xf7\x9cZTT\xf2\x04-C9E\xf3B\xef\xd5S\xcaı=\x13V\x1eO\xea\xefRq&S\x13\xac\x01\u007fsu\xd4\xd0z\u007fa]y\x89/\xd0*<?}\xc7\x1c\xc3\xd2q\xf1\x80\x8f\x00\x00\x00\xff\xffj\xa6\x1d\x9b\xa3\x01\x00\x00",
	},

	"/data/ru/male_first_names": {
		local:      "data/ru/male_first_names",
		size:       430,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffLP\xcbJ\xc4@\x10\xbc\xcfW\x9b\x05o\v꺂\x88\xaf\x05\xcf\x126\x19\x98\xcdd\xe2/T\xff\x91U\xdd;\xe8%\xa9\xc7\xf4\xa3\x1a\xf7\xa8\xc8Xl\xc0\x88\x86\xd9n\xd2?)\xe3\"\xda0\xda\x0e\x1b\xe5\xd2\x05>\f\xf7\x01\x1ba\xb1!\xe1\xc0\x16\xac\x14\x95C\xca\x16\xa5W\x1d\xf8[\xd4F#H\b\xc7?\xaf\x92\xcc$+\r\xfa\x8fl\xde4\xd7\xc5K\b\x1a4]\xa9\x0f\x99b\xb6\x84\xa3\x17\xf8\x1aGo\xb2\xeb\xce\x13\xce|\xe8\xae\xe83)\x93\nx\xb9\xed\x05\xab\xed\xed.\xe1\x85!\x9b\r\xbeZ\xe3\xb7\xe8\xe1+\xc9\xe2IV\x91b\xb7\x14\x18+\xe1M\x91\"I'\x9b'\xe1\x9cw?\xe2\x94\xf0A~&\xac\x82Y{%|\xf2\xdd\x1ak\x9c\xfc`S\x1c\xf3\xc4V\x19?\xe1|\x11\xceq\xae\xefk\x98\xdf\x00\x00\x00\xff\xfft5E\xb6\xae\x01\x00\x00",
	},

	"/data/ru/male_last_names": {
		local:      "data/ru/male_last_names",
		size:       1531,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffdT\xdbn\x1aK\x10|\xf7W\xe3\xc5\a\x1dɖ\xc2\xcd1\"\xd8\x04\a)\xceS\x96\x85\x15xaw\u007f\xa1\xe7\x8fR]==\x03\xc9\x03Zf\xa6/\xd5U5#\v\xa9\xa4\x94V:\xa9\xeed*e\xb8\x97\x93\x9cÓԺ\xb1\x96:\x14a`\xa7\x1b\xb9\xc8\t\x8b\x18\xbc\xc2\xe2?\xe4~\xca\xd96\xb6a\"\a\xe9rx'\r~\xf1\xf47\x17\x15\x96\xb1r'\xbd\x9d\x8cQ\xf1\x80\xac\xdaN\xc6\fi\x00䟍2F2\xed\x1b6w\xf8\x1d,j\x86\xfd\x93쵿\xa3߄\x02\u007f\xfa<\xdf\x06\xcb\vP\xc6\xe5\x1a'\x95\xe3\x9b\xe0\xb3G\xb1\x14\xfc\x8ar\x86\xbf\xb4rs\xce_\xb0\x0f7\x9e\xad\x9ds\xa7\xa1\x8d-\x96a(G\x14\xaa\xc3\xe8\xba\x18\xb2\xf1m\x8d\x1b\xf6\rO\x80\x83\xf3w\xec_P\xec\xc1\x87~C݈l\x8c]E\x05m|\xb2\xad\x8a\x84_\xef\x1cb2$\xec=\xfb\x05\xc1\xd0Ʊ\xad\xf0\xb7\xc0\xa8\xf1t\xaa t\\\x9c\x97*Y\x82\x8c\xfa\x17C\x88\x14RNX<\xc7\xe7\xe8\x90\x16\xb4\b\x03g\xec\\^\x87\xaeع\xf1\xc9U\xf5\x93/W\x91\xb1#\x97w\xf2\x1d\xdfK&\xfd\x8du\xfaT\aii\x8a\x175\x1b\b\xadM\xae\x1a\x18\xbeD\xedUH\xearN\x96\xc2\xc0\x9dW}\x06\x83\t\x9d\xfa\xa6\xd3s's\xce\xecG'b\x16\x86\xf4^<is\xd5%\x04\xbcg\xdfI48\xa15F\xc4;\xefC\x17/I\xe4t\x80\xba\xe7[\v<d\xcf,\t\xf1@\xfe\x12\x1c\x15\x02[\x98\xad\xca\xc3j\x8a6g\xdaWD4\xc9u,\xb6s#\xfd\xa2\x1c;2\xd7\xf9ES\xa2\x1a7JI#\xa4+\xfeA\xc5\x0f\xa1\xc0\x04Zꚪ\x9f\xda\x13\xc8R\xf6\x84j\\ݧ\xd3\xdf7~Gg\xdb \f\xee=\xb3v\"Vd\xf4\xac@\x8d\xb5\"\n\x03H\x95?%\x83\xfcr,\xe0\xeb\xf6\x86!\xf3\x8c\xb7]G\x15\x00\xdd#&\xa0\xbdq+\xec\b9\x8e4#\x1dC\xbe\x1c\x91\x9d\xd6\xcb\x19\x83\xba\xd5Xm\x10\xae\xe2~FK\x03\xc1\xa3\x8f\xbe\xa4\xca6\b7>P\x14\xa4\x86\x91\xbd\x8c\x83쾭\x99B=2\xd5\xd7\xc4e$\xdd\b\x1dy\x93-_\xb3\xce\xf9x\x05\xeac\x1e\xf3G\xf8?\xab:Gh\x9b\x1dA#\xa9T\xaa\x8d=2\xb7\xfeh\xad?J\x02\xa6Q\xf1'\x00\x00\xff\xff\f\xe6\xecL\xfb\x05\x00\x00",
	},

	"/data/ru/male_patronymics": {
		local:      "data/ru/male_patronymics",
		size:       1165,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffl\x93\xcfN\xf30\x10\xc4\xef\xdfS\xb7\xfd@\xbdT\x02J+UP\xfe\x14q\x05J\x12K!N\xd2WX\xbf\x11\xb3Cpv\x1b\xd4C\xad\x9f\xc7\xeb\xf1\xecF\xae%J\x90&\xcd\xe5(\x9dTi&\xbd\x14R\xa7\xe5?\xb3\x15\xf0˰\x93cZ@\x15\xd3\xcac=\xec\x95\xd4u\xa6\"\x14\r.\xaa\xdc\xd14\x83.H;\xean\xb0,\xa5\xa2%\vq\xbc\x86\x9d\x8c\xd6P\xa8G\xbd\xa8\xb6\xdaa\x03\x95\xcdE\x808\\{\xe3k\x80F}ڗ\x03\x02\x1dϕQ\xadC\xdf\x028\xf5\xb0\x81\xea\xba*\xccV\xbaJK\xf8\x98lܪk\x8d\x92\x05\x83\xc5\xea\xa4tP\x9f\r\xd4\xfb\xe7l\x18\x0f\x9e3Vݰ\xaa\x8bhC\xb7\v\x96\xc8G\xb7X\x94\xccmea\xe9Sئ\v>6\x83\x1d\x16\xae%\xbb\xe1H\x18\x81F6,{M\x1b5\xb2\xfc\x8e\xb9\xa9\xe3h!F$͙\xf7\xa4\x8f\xf7\x9a\a_Y\x19\x88+L\x94{\x1cl\xd8\xd8\xd6A\xd4*\xec<\xee\xf1w\t헽\xfdAۯÌ\x8d\xac|\xe4P\x95\xa3\xea\x89͋\x16\x04fj@\x1c>\v\xe3\xfe\x19\xcbON\xe1\xc2\xc3\xd6\xc7x\xe0\xf4\x873\xc410\xfe\x0f?\x11\xfd5f\a~@'_\xf5\x05\x86f\x1cy\x83\x18\x12zb\xea\xbebY\xf9\xce+\x8a\xfc0\xe6\x16ք'\xfc2|÷\xfc\x9f\x17\x99\x06\xbd\xfbA\xfd`\xc2\x05C\xfdEz\xdb\xe4\x1d\xdf\x01\x00\x00\xff\xff[hP\x9b\x8d\x04\x00\x00",
	},

	"/data/ru/months": {
		local:      "data/ru/months",
		size:       148,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffL\x8c1\x0e\x02!\x14\x05{N\xbd\xae\xadQ\n\v\v51\xb1\x171\x14b\xd0+̻\x91\xffC\xa2\xdb1\xc3\xfbÍF&i\xd2&p\xa5\x905\x91x9\x1e\xbb\x9f\x03\x91\x8f\xd9\xf2\xb3<\x03\ami\xce\xfe\xe8\x1f\xd1B\x0f\xad\xb5\xf2\x93\x8b͛f\xed\xb8\x8f\xf6\x99\xba\xc4\x13\xef?\xecm\\-;\xf0\x1b\x00\x00\xff\xff\xe9\xeb\xb4ٔ\x00\x00\x00",
	},

	"/data/ru/phones_format": {
		local:      "data/ru/phones_format",
		size:       26,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffR\xd6UV\x86a \xe2B\xb0\x00\x01\x00\x00\xff\xff\xb5\xe6\x87^\x1a\x00\x00\x00",
	},

	"/data/ru/states": {
		local:      "data/ru/states",
		size:       2756,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\x8cVKR\x1bA\f\xdds\n/\x93\x05w4C\bTA\xc5@H\x852`bS\x14\xab\x04c{`\xf0|\xb8\x82\xfa\n9I\xde{\xdd6\xa4\x98\x91Y@yZ\xaf\xf5yRK\xb2c[\x84C\x9b[\x1e\x06[\xf8(CfS{\u07b2\x13\x9b\x86\x03[Z\x13\xfa8j\u008e\x045\x04a\x17G\x03|\x16\xbcr\x86S\xde^\xcbϭ\xb690\a8L\x98!$\x0f\xd0\u05f7\x85\x15\x107\xdbTo%\xd4\xf3p\x8d)\xad\x82/\xcb\xf5\x01dӰ\x0fAn3^\xba\x83\xca>\xe4\xb0\xf6\x16\x93\xe3b\xfal\xac\xb2b\xcb.\xa3^{\xee\xd9o+\xf9\xdd\xc8x\x03=B\x8e\x01سi\xef\x93\xdd\xdb\x12\xce\xca\xd1\xcf<\xa7)Z\xa9\x81\x18\xf4l\x04S)\x8e\xde\xdf\xfeY\x8f\x141Ψf\x02PF[\xaf\xe1O\x10\xc1L\x9a'a\x97\xbf\xa0\xf5\x06\xa6+і\b\xb9\x05\x96\xb1\xa70\x18\u05fe\xd5\xfa\x19/\x81\xbd\x18\x91(x\x9b\x19\xdcY\xc6Ȗ\x941U?#\xbd\xf6L\x9d\xc0\x1d\xb5`HoE.C\xd6&\x95\x99\x1d\xa6\x06\xae*\x1e\x0f\x83\xec\xb7!\xaeE\\\xd5&Q6*\x15\xd3{\xe9X\x91\xcd\xf0\xd9\xd8\v\xfe\xda#\xb8]\x17\x11\xb2\xd8\"\x1fAZ\xd2C\x92L\xc2\"\xe3;\"\x1a\x99l^\xe5\xe1\br\xc8\xf6\x94\xc99\v(\x99lG\xf2G?V\f\xb2ԅ;Q!\xceUj\xa0\xb1\x1b\xc7\xe7\xe3\xe89U\x85-DX\xe1DpJ\xaah\x0fύ\x1ct[\x8c\xc8FX\x17%\xcfk\xc4\xf1؉:gy\xaaPf\xdd\x18\xbeR\xbe\xaaND|\xed\xec\x05\x85څ\xef\xbf\xd0x\x18\xdd^\r\xe1s\xa5\xf2\xf3\xfc\x1a\x92\xcf\r\x88&e\xbbI\x85܊be\x81M\xaf\x1a\x86n\xf5]\xc0ߏ\xc5~\x01\xd4\v\"\xfb\xda!\xbfT\xffe\xbdxް\x01R\xe6\xc5~\xa9\xa8*W\xcf\x15\xbcy\x84\uf6eb\xfcJ\xcd\xf6\xa38zW@\xe0\xd5\xfb\xc8\xc9\xc8Hc\xa0\xb6\x87\x98\x19\x17W\xba,\\K\xcf\x13\xffwc6r\xf9+ML\x17\x83.\xf0\xe4\xb2=f\xd3^\xb7\xe3\x0e\x84:\x93o)M\xbb\xf8\xe2\x1cT\x9c|\x8b\r\f\x8d\xd9\xc8q\xe014\x91\xe7\x0f\xae\x9eI\xb4\xe7Ƚ7\xc8\xe9\xeau\xedI\xf8Ǝ\xe0\xf8x\xa3\xfb\x83\r\xbd\xecN\xd3a\xc0\xdattݳ\xca!-5ȺP\xdfW\xaf\x01\xfbA\xf8\x02\xc5\v\x11\xae\x99-7\xf0\\z\xa8\x89ZkO\xbe\xda2V\x8f\x17\x8d7\xd6D\x8d\xee\x9amk\xdefҐ\xca~\xb5\xbfLS\xed\xa5a\xcaa8\xd3A\xad\x05\xa9nu\xee\x87&p\x9e6\f!\x8em\xf4\xee.֛\x9cw1w\x91\x819]\xba\x92\xab\xb1Ei&\xebޭJ;\v\x87\xdb\xeaQ\xb5^\xf8\x9b\xf5\x05\xa0\xb8Q\xfd\x89=0\xee>\\8\xb3\xff@ WM\xa9\xe4\x06\xd8j\xea_\x00\x00\x00\xff\xff\x9du\x89T\xc4\n\x00\x00",
	},

	"/data/ru/streets": {
		local:      "data/ru/streets",
		size:       694,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\\\x92\xcbn\xf2@\f\x85\xf7<u\x02\xfa\xf5W\x02\t\xf5\xa2V\n\xa5)\x9b\xb6+\x14.\x03\t)\xe1\x15\x8eߨ\xc7\xceL=\xea\"A\xb6\x0f\xf6\xe7\xe3\xe0\x03\x01W\x99J\x81\x06\xbd,pE#\xcb\t^1\xa0\xe7sd\xf9\x94\x92_\xb8h\xdaU\x1b\x86{\x04\xfe\xbdd\xc9R+\x86e\xaa\xaf\xad\x1e\xa5\r\x9b\xfd\x86k\x86;J\x8b\xbc\xff\v\u007f\x98\x90\xffN\xd1*\x98ue\xaeռ\xca\x02\xd9B\x12Ն\x1aR\xe7\x95\xccp\xf0Ao\xb8\x10o\x89\x9d\x14\xb2p̊\x8ao\x86|\x8f+y\xe9\x91h\a#\x89\xf3j\xe3ܛ\xb6A\xe7ʍM\xd5b\xf2C\xa6L\xddr{z\x06\xd9F\x95mdt\x13l}\xf5\x9a\xc3\x06\x93\x16\xde\xff>\x1a\x94\xad\xb3\xa5k\xad\xad۹\xc9\xca;8Ee\x1b\xfd\xe3\xfb\x9cR\x0f\xac\x97\x84\x1b\x9c\xe4=Ǫe&w\x1c\x9b66\xc6^/\xabF8\xd1s\xb4^\x1b\xeb\xc71\x98\xf0\xe4=\xf5\xaa:\xe4\xef\xf7\"st\xe3\xa9xu\x1ce\xae\x979\xe7\xdeU\xfa\x01\x1a@\x84\xf8d\xe2\xc6'\x8cf=Q\x96x\u007f\x02\x00\x00\xff\xff#>\x03\x17\xb6\x02\x00\x00",
	},

	"/data/ru/weekdays": {
		local:      "data/ru/weekdays",
		size:       117,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff4\x8d;\n\x031\fD{\x9d\xdaI U \xe0\v\x18Wi\xb3\x1f\xc1\xe2\xc5>\xc3Ӎv\\,\xc6\xd2<\xcd\bQ\x18t\x9c]\xff\x8c\x8f\xf4A3r<\x19\x91n\xac\x923\xf37~\xb8\xbcU5\xb1\x19%\xbeB\xc5\xe2=\xdd\x1a/\x16\xbd\xa1\xa10K<hs[\xdd\xe9\xba\xe0v\x05\x00\x00\xff\xff\xcbVS\xadu\x00\x00\x00",
	},

	"/data/ru/weekdays_short": {
		local:      "data/ru/weekdays_short",
		size:       35,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xff\xba0\xff\xc2^\xae\v\x93.6q]Xx\xb1\x81\xeb\xc2r\x10k>\x98{a#H\xa2\x91\v\x10\x00\x00\xff\xff\xber!\xd8#\x00\x00\x00",
	},

	"/data/ru/zips_format": {
		local:      "data/ru/zips_format",
		size:       8,
		compressed: "\x1f\x8b\b\x00\x00\tn\x88\x00\xffR\x86\x00.@\x00\x00\x00\xff\xff\x87\xdd'=\b\x00\x00\x00",
	},

	"/": {
		isDir: true,
	},

	"/data": {
		isDir: true,
	},

	"/data/en": {
		isDir: true,
	},

	"/data/ru": {
		isDir: true,
	},
}
