"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.InputProvider = void 0;

var React = _interopRequireWildcard(require("react"));

var _cliSpinners = require("cli-spinners");

var _core = require("@kui-shell/core");

var _OnPaste = _interopRequireDefault(require("./OnPaste"));

var _OnKeyDown = _interopRequireDefault(require("./OnKeyDown"));

var _OnKeyPress = _interopRequireDefault(require("./OnKeyPress"));

var _context = _interopRequireDefault(require("../../../Client/context"));

var _ActiveISearch = require("./ActiveISearch");

var _BlockModel = require("./BlockModel");

var _DropDown = _interopRequireDefault(require("../../../spi/DropDown"));

var _Icons = _interopRequireDefault(require("../../../spi/Icons"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const strings = (0, _core.i18n)('plugin-client-common');
const strings2 = (0, _core.i18n)('plugin-client-common', 'screenshot');

class InputProvider extends React.PureComponent {
  /** rendered to the left of the input element */
  // eslint-disable-next-line @typescript-eslint/no-empty-function
  status() {}
  /** the "xxx" part of "xxx >" of the prompt */


  promptLeft() {
    return !this.props.noPromptContext && React.createElement(_context.default.Consumer, null, config => !config.noPromptContext && this.props.model && React.createElement("span", {
      className: "repl-context"
    }, this.props.model.cwd));
  }
  /** the ">" part of "xxx >" of the prompt */


  promptRight() {
    // &#x2771; "heavy right-pointing angle bracket ornament"
    // another option: &#x276f; "heavy right-pointing angle quotation mark ornament"
    return React.createElement(_context.default.Consumer, null, config => React.createElement("span", {
      className: "repl-prompt-righty"
    }, config.prompt || '/'));
  }

  isearchPrompt() {
    return React.createElement("div", {
      className: "repl-prompt"
    }, this.state.isearch.render());
  }

  normalPrompt() {
    return React.createElement("div", {
      className: "repl-prompt"
    }, this.promptLeft(), this.promptRight());
  }
  /** the "xxx >" prompt part of the input section */


  prompt() {
    if (this.state && this.state.isearch && this.state.prompt) {
      try {
        return this.isearchPrompt();
      } catch (err) {
        console.error('error rendering i-search', err);
        return this.normalPrompt();
      }
    } else if (!this.props.isPinned) {
      return this.normalPrompt();
    }
  }

  render() {
    return React.createElement("div", {
      className: 'repl-input' + (this.state && this.state.isearch ? ' kui--isearch-active' : '')
    }, React.createElement("div", {
      className: "kui--input-and-context"
    }, this.prompt(), this.props.children, this.input(), this.status()), this.state && this.state.tabCompletion && this.state.tabCompletion.render());
  }

}

exports.InputProvider = InputProvider;

class Input extends InputProvider {
  constructor(props) {
    super(props);
    this.state = {
      execUUID: (0, _BlockModel.hasUUID)(props.model) && props.model.execUUID,
      prompt: undefined,
      spinner: undefined
    };
  }
  /** @return the value of the prompt */


  value() {
    return this.state.prompt && this.state.prompt.value;
  }
  /** Owner wants us to focus on the current prompt */


  doFocus() {
    if (this.state.prompt) {
      this.state.prompt.focus();
    }
  }

  static newSpinner(spinnerDom) {
    let frame = 0;
    return setInterval(function () {
      frame = frame + 1 === _cliSpinners.dots.frames.length ? 0 : frame + 1;
      spinnerDom.innerText = _cliSpinners.dots.frames[frame];
    }, _cliSpinners.dots.interval);
  }

  static updateSpinner(props, state) {
    const spinner = (0, _BlockModel.isProcessing)(props.model) ? state.spinner || state.spinnerDom && Input.newSpinner(state.spinnerDom) : undefined;

    if (!spinner && state.spinner) {
      clearInterval(state.spinner);
    }

    return spinner;
  }

  static getDerivedStateFromProps(props, state) {
    const spinner = Input.updateSpinner(props, state);

    if ((0, _BlockModel.hasUUID)(props.model)) {
      return {
        spinner,
        execUUID: props.model.execUUID
      };
    } else if (state.prompt && (0, _BlockModel.isActive)(props.model) && state.execUUID !== undefined && !state.isearch) {
      // e.g. terminal has been cleared; we need to excise the current
      // <input/> because react aggressively caches these
      return {
        spinner,
        prompt: undefined,
        execUUID: undefined
      };
    }

    return state;
  }
  /** the element that represents the command being/having been/going to be executed */


  input() {
    if (this.props.isPinned) {
      return;
    }

    const active = (0, _BlockModel.isActive)(this.props.model);

    if (active) {
      setTimeout(() => this.state.prompt.focus());
      const kp = active && !this.state.isearch ? _OnKeyPress.default.bind(this) : undefined;
      const kd = active && !this.state.isearch ? _OnKeyDown.default.bind(this) : undefined;
      const ku = active ? _ActiveISearch.onKeyUp.bind(this) : undefined;
      const op = active && !this.state.isearch ? evt => (0, _OnPaste.default)(evt.nativeEvent, this.props.tab, this.state.prompt) : undefined;
      return React.createElement("input", {
        type: "text",
        autoFocus: true,
        autoCorrect: "off",
        autoComplete: "off",
        spellCheck: "false",
        autoCapitalize: "off",
        className: 'repl-input-element' + (this.state.isearch ? ' repl-input-hidden' : ''),
        "aria-label": "Command Input",
        tabIndex: 1,
        placeholder: this.props.promptPlaceholder,
        onBlur: this.props.onInputBlur,
        onFocus: this.props.onInputFocus,
        onMouseDown: this.props.onInputMouseDown,
        onMouseMove: this.props.onInputMouseMove,
        onChange: this.props.onInputChange,
        onClick: this.props.onInputClick,
        onKeyPress: evt => {
          if (kp) kp(evt);
          this.props.onInputKeyPress && this.props.onInputKeyPress(evt);
        },
        onKeyDown: evt => {
          if (kd) kd(evt);
          this.props.onInputKeyDown && this.props.onInputKeyDown(evt);
        },
        onKeyUp: evt => {
          if (ku) ku(evt);
          this.props.onInputKeyUp && this.props.onInputKeyUp(evt);
        },
        onPaste: op,
        ref: c => {
          if (c && !this.state.prompt) {
            c.value = (0, _BlockModel.hasValue)(this.props.model) ? this.props.model.value : '';
            this.setState({
              prompt: c
            });
          }
        }
      });
    } else {
      const value = this.value() || ((0, _BlockModel.hasValue)(this.props.model) ? this.props.model.value : (0, _BlockModel.hasCommand)(this.props.model) ? this.props.model.command : '');

      if ((0, _BlockModel.isProcessing)(this.props.model)) {
        // for processing blocks, we still need an input, albeit
        // readOnly, to handle ctrl+C
        return React.createElement("span", {
          className: "flex-layout flex-fill"
        }, React.createElement("input", {
          className: "repl-input-element",
          readOnly: true,
          value: value,
          onKeyDown: evt => {
            if (evt.key === 'c' && evt.ctrlKey) {
              (0, _core.doCancel)(this.props.tab, this.props._block);
            }
          },
          ref: c => c && c.focus()
        }));
      } else {
        // for "done" blocks, render the value as a plain div
        return React.createElement("div", {
          className: "repl-input-element"
        }, value);
      }
    }
  }
  /** render the time the block started processing */


  timestamp() {
    if (!this.props.isPinned && !(0, _BlockModel.isEmpty)(this.props.model) && ((0, _BlockModel.isProcessing)(this.props.model) || (0, _BlockModel.isFinished)(this.props.model))) {
      return this.props.model.startTime && React.createElement("span", {
        className: "kui--repl-block-timestamp kui--repl-block-right-element"
      }, this.props.model.startTime.toLocaleTimeString());
    }
  }
  /** spinner for processing blocks */


  spinner() {
    if ((0, _BlockModel.isProcessing)(this.props.model)) {
      return React.createElement("span", {
        className: "kui--repl-block-spinner",
        ref: spinnerDom => {
          this.setState({
            spinnerDom
          });
        }
      });
    }
  }
  /** error icon for error blocks */


  errorIcon() {
    if ((0, _BlockModel.isOops)(this.props.model)) {
      return React.createElement(_Icons.default, {
        className: "kui--repl-block-error-icon",
        icon: "Error",
        "data-mode": "error"
      });
    }
  }

  removeAction() {
    return !this.props.willRemove || this.props.isPinned ? [] : [{
      label: strings('Remove'),
      handler: () => this.props.willRemove()
    }];
  }

  screenshotAction() {
    return !this.props.willScreenshot || (0, _core.inBrowser)() ? [] : [{
      label: strings2('Screenshot'),
      handler: () => this.props.willScreenshot()
    }];
  }

  unPinnedAction() {
    return !this.props.isPinned ? [] : [{
      label: strings2('Show as table in terminal'),
      handler: () => this.props.unPin()
    }];
  }
  /** DropDown menu for completed blocks */


  dropdown() {
    if (!(0, _BlockModel.isActive)(this.props.model)) {
      const actions = this.screenshotAction().concat(this.unPinnedAction()).concat(this.removeAction());
      return React.createElement(_DropDown.default, {
        actions: actions,
        className: "kui--repl-block-right-element kui--toolbar-button-with-icon",
        onClose: this.props.willLoseFocus
      });
    }
  }
  /** Close button. Only for pinned blocks for now. */


  close() {
    return this.props.willRemove && this.props.isPinned && React.createElement("a", {
      href: "#",
      onClick: this.props.willRemove,
      className: "kui--repl-block-right-element kui--toolbar-button-with-icon kui--pinned-close-button",
      title: strings('Close watcher')
    }, React.createElement(_Icons.default, {
      icon: "WindowClose"
    }));
  }
  /**
   * Status elements associated with the block; even though these
   * pertain to the Output part of a Block, these are currently placed
   * in the Input area.
   *
   */


  status() {
    return React.createElement("span", {
      className: "repl-prompt-right-elements"
    }, this.spinner(), this.errorIcon(), this.timestamp(), this.dropdown(), this.close());
  }

}

exports.default = Input;